#ifndef TCP_IP_RULE_SET_H
#define TCP_IP_RULE_SET_H

#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <iostream.h>
#include <vector>

 struct TcpIpRule
  {
   enum {ALLOW, DENY};
   int allow_deny;
   struct in_addr netnum;
   struct in_addr netmask;
   TcpIpRule() :  allow_deny(ALLOW) { netnum.s_addr = netmask.s_addr = 0;}
   TcpIpRule(char* netnum_str, char* netmask_str, int set_allow_deny = ALLOW):
    allow_deny(set_allow_deny) 
    {
     inet_aton(netnum_str, &netnum);
     inet_aton(netmask_str, &netmask);
    } 
  
   int match(struct in_addr& host)
    {
     return (host.s_addr & netmask.s_addr) == netnum.s_addr ;
    }

   int match(char* host_str)
    {
     struct in_addr host;
     inet_aton(host_str, &host);
     return match(host);
    }
  
   int allowed(struct in_addr& host)
    {
     return match(host) && allow_deny == ALLOW;
    }
   
   int allowed(char*  host_str)
    {
     return match(host_str) && allow_deny == ALLOW;
    }
  
   int denied(struct in_addr& host)
    {
     return match(host) && allow_deny == DENY;
    }
   
   int denied(char*  host_str)
    {
     return match(host_str) && allow_deny == DENY;
    }
 
   friend ostream& operator <<(ostream& os, TcpIpRule& r);

  };

 class TcpIpRuleSet
  {
   protected:
    vector<TcpIpRule> rules;
    int default_policy;
  
   public:
    enum {ALLOW, DENY};

    TcpIpRuleSet(int init_default_polity = DENY) : 
     default_policy(init_default_polity)
      {
      }

    void add_rule(char* netnum, char* netmask, int allow_deny = TcpIpRule::ALLOW)
     {
      TcpIpRule rule(netnum, netmask, allow_deny);
      //cerr << "inserting rule " << rule << endl;
      rules.insert(rules.end(), rule);
     }

    int ok(struct in_addr& host);
    int ok(char* host_str)
     {
      struct in_addr host;
      inet_aton(host_str, &host);
      return ok(host);
     }

    friend ostream& operator <<(ostream& os, TcpIpRuleSet& r);
  };

#endif
