/***

packet.c - routines to open the raw socket, read socket data and
           adjust the initial packet pointer
           
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <asm/types.h>
#include <curses.h>
#include <panel.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <net/if_arp.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#include <linux/if_fddi.h>
#include <linux/isdn.h>
#include <linux/sockios.h>
#include "deskman.h"
#include "error.h"
#include "options.h"
#include "links.h"
#include "isdntab.h"
#include "packet.h"

void open_socket(int *fd)
{
    int dummy;

    *fd = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_ALL));

    if (*fd < 0) {
	errbox("Unable to open raw socket", ANYKEY_MSG, &dummy);
	return;
    }
}

unsigned short getlinktype(unsigned short family, char *ifname, int isdn_fd,
			   struct isdntab *isdnlist)
{
    unsigned short result = 0;
    struct isdntabent *isdnent;

    switch (family) {
    case ARPHRD_ETHER:
	if (strncmp(ifname, "eth", 3) == 0)
	    result = LINK_ETHERNET;
	else if (strncmp(ifname, "plip", 4) == 0)
	    result = LINK_PLIP;
	else if ((strncmp(ifname, "isdn", 4) == 0) && (isdn_fd != -1)) {
	    isdnent = isdn_table_lookup(isdnlist, ifname, isdn_fd);

	    switch (isdnent->encap) {
	    case ISDN_NET_ENCAP_RAWIP:
		result = LINK_ISDN_RAWIP;
		break;
	    case ISDN_NET_ENCAP_CISCOHDLC:
		result = LINK_ISDN_CISCOHDLC;
		break;
	    default:
		result = LINK_INVALID;
		break;
	    }
	}
	break;
    case ARPHRD_LOOPBACK:
	result = LINK_LOOPBACK;
	break;
    case ARPHRD_SLIP:
    case ARPHRD_CSLIP:
    case ARPHRD_SLIP6:
    case ARPHRD_CSLIP6:
	result = LINK_SLIP;
	break;
    case ARPHRD_PPP:
	result = LINK_PPP;
	break;
    case ARPHRD_FDDI:
	result = LINK_FDDI;
	break;
    default:
	result = LINK_INVALID;
	break;
    }
    return result;
}

void adjustpacket(char *tpacket, unsigned short family,
		  char **packet, char *aligned_buf, int *readlen)
{
    switch (family) {
    case LINK_ETHERNET:
    case LINK_LOOPBACK:
    case LINK_PLIP:
	*packet = tpacket + ETH_HLEN;
	*readlen -= ETH_HLEN;

	/*
	 * Move IP data into an aligned buffer.  96 bytes should be sufficient
	 * for IP and TCP headers with reasonable numbers of options and some
	 * data.
	 */

	memmove(aligned_buf, *packet, min(SNAPSHOT_LEN, *readlen));
	*packet = aligned_buf;
	break;
    case LINK_PPP:
    case LINK_SLIP:
    case LINK_ISDN_RAWIP:
	*packet = tpacket;
	break;
    case LINK_ISDN_CISCOHDLC:
	*packet = tpacket + 4;
	*readlen -= 4;
	break;
    case LINK_FDDI:
	*packet = tpacket + sizeof(struct fddihdr);
	*readlen -= sizeof(struct fddihdr);

	/*
	 * Move IP data into an aligned buffer.  96 bytes should be sufficient
	 * for IP and TCP headers with reasonable numbers of options and some
	 * data.
	 */

	memmove(aligned_buf, *packet, min(SNAPSHOT_LEN, *readlen));
	*packet = aligned_buf;
	break;
    default:
	*packet = (char *) NULL;	/* return a NULL packet to signal */
	break;			/* an unrecognized link protocol */
    }				/* to the caller.  Hopefully, this */
}				/* switch statement will grow. */

void getpacket(int fd, char *buf, struct sockaddr_ll *fromaddr,
	       int *ch, int *br, char *ifname, WINDOW * win)
{
    int fromlen;
    fd_set set;
    struct timeval tv;
    int ss;
    int ir;
    struct ifreq ifr;

    FD_ZERO(&set);
    FD_SET(0, &set);
    FD_SET(fd, &set);

    tv.tv_sec = 0;
    tv.tv_usec = DEFAULT_UPDATE_DELAY;

    do {
	ss = select(fd + 1, &set, 0, 0, &tv);
    } while ((ss < 0) && (errno == EINTR));

    *br = 0;
    *ch = ERR;

    if (FD_ISSET(fd, &set)) {
	fromlen = sizeof(struct sockaddr_pkt);
	*br = recvfrom(fd, buf, 8192, 0,
		       (struct sockaddr *) fromaddr, &fromlen);
	ifr.ifr_ifindex = fromaddr->sll_ifindex;
	ir = ioctl(fd, SIOCGIFNAME, &ifr);
	strcpy(ifname, ifr.ifr_name);
    }
    if (FD_ISSET(0, &set))
	*ch = wgetch(win);
}
