//////////////////////////////////////////////////////////////////
//
// yasocket.cxx
//
// $Id: yasocket.cxx,v 1.1.2.9 2004/05/12 17:46:41 zvision Exp $
//
// Copyright (c) Citron Network Inc. 2002-2003
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Chih-Wei Huang <cwhuang@linux.org.tw>
// initial version: 03/14/2003
//
//////////////////////////////////////////////////////////////////

#include <ptlib.h>
#include <ptlib/sockets.h>
#include "stl_supp.h"
#include "yasocket.h"

#ifdef LARGE_FDSET

// class YaSelectList

bool YaSelectList::Select(SelectType t, const PTimeInterval & timeout)
{
	large_fd_set fdset;

	// add handles to the fdset
	const_iterator i = fds.begin();
	const_iterator endIter = fds.end();
	while (i != endIter)
		fdset.add((*i++)->GetHandle());
	
	fd_set *readfds, *writefds;
	if (t == Read)
		readfds = fdset, writefds = NULL;
	else
		writefds = fdset, readfds = NULL;

	const unsigned long msec = timeout.GetInterval();
	struct timeval tval;
	tval.tv_sec  = msec / 1000;
	tval.tv_usec = (msec - tval.tv_sec * 1000) * 1000;
	const bool r = ::select(maxfd + 1, readfds, writefds, 0, &tval) > 0;
	if (r) {
#if 1
		std::vector<YaSocket*>::iterator last = std::remove_if(
			fds.begin(), fds.end(),
			std::not1(std::compose1(
				std::bind1st(mem_fun(&large_fd_set::has), &fdset), 
				mem_fun(&YaSocket::GetHandle)
				)));
		fds.erase(last, fds.end());
#else
		/* This unrolled implementation of the above code may give
		   another 10-15% of performance gain. As it is not much under normal
		   conditions, I leave it for thouse who want to squeeze a few more
		   calls from their proxies;-)
		   
		   I did some performance tests (Duron 1.1GHz) with simulation of
		   various fds selected sockets coverage (10%, 33%, 50%, 75%, 90%):
		   
		   LARGE_FDSET=1024  - 12% performance gain
		   LARGE_FDSET=4096  - 15% performance gain
		   LARGE_FDSET=16384 - 13% performance gain
		   
		   For LARGE_FDSET=4096 it took less than 1ms to manipulate the fdset
		   and this grows in a linear fashion (LARGE_FDSET=16384 takes a few
		   milliseconds to perform the same task).
		*/
		std::vector<YaSocket*>::reverse_iterator j = fds.rbegin();
		std::vector<YaSocket*>::iterator k = fds.end();
		const std::vector<YaSocket*>::reverse_iterator rendIter = fds.rend();
		bool hasfd = false;
		
		// start from the end of the list, skip consecutive sockets 
		// that were not selected (find the first one selected)
		while (j != rendIter) {
			k--;
			if (fdset.has((*j)->GetHandle())) {
				hasfd = true;
				break;
			} else
				j++;
		}
		// reorder remaining sockets, so non-selected sockets 
		// are moved to the end of the vector
		if (hasfd) {
			while (++j != rendIter) {
				if (!fdset.has((*j)->GetHandle()))
					*j = *k--;
			}
			// at this point the vector [begin(),k] should contain
			// all selected sockets, so erase the remaining vector elements
			fds.erase(++k, fds.end());
		} else
			fds.clear();
#endif
	}
	return r;
}

// class YaSocket
YaSocket::YaSocket() : os_handle(-1)
{
	lastReadCount = lastWriteCount = 0;
}

YaSocket::~YaSocket()
{
	Close();
}

bool YaSocket::Close()
{
	if (!IsOpen())
		return false;

	// send a shutdown to the other end
	::shutdown(os_handle, SHUT_RDWR);
	::close(os_handle);
	os_handle = -1;
	return true;
}

bool YaSocket::Read(void *buf, int sz)
{
	int r = os_recv(buf, sz);
	lastReadCount = ConvertOSError(r, PSocket::LastReadError) ? r : 0;
	return lastReadCount > 0;
}

bool YaSocket::ReadBlock(void *buf, int len)
{
	// lazy implementation, but it is enough for us...
	return Read(buf, len) && (lastReadCount == len);
}

bool YaSocket::Write(const void *buf, int sz)
{
	lastWriteCount = 0;
	if (!CanWrite(writeTimeout.GetInterval())) {
		errno = EAGAIN;
		return ConvertOSError(-1, PSocket::LastWriteError);
	}
	int r = os_send(buf, sz);
	if (ConvertOSError(r, PSocket::LastWriteError)) {
		lastWriteCount = r;
		return lastWriteCount == sz;
	} else
		return false;
}

bool YaSocket::CanRead(
	long timeout
	) const
{
	const int h = os_handle;
	if (h < 0)
		return false;

	YaSelectList::large_fd_set fdset;
	fdset.add(h);

	struct timeval tval;
	tval.tv_sec  = timeout / 1000;
	tval.tv_usec = (timeout - tval.tv_sec * 1000) * 1000;
	return ::select(h + 1, (fd_set*)fdset, NULL, NULL, &tval) > 0;
}

bool YaSocket::CanWrite(
	long timeout
	) const
{
	const int h = os_handle;
	if (h < 0)
		return false;
		
	YaSelectList::large_fd_set fdset;
	fdset.add(h);

	struct timeval tval;
	tval.tv_sec  = timeout / 1000;
	tval.tv_usec = (timeout - tval.tv_sec * 1000) * 1000;
	return ::select(h + 1, NULL, (fd_set*)fdset, NULL, &tval) > 0;
}

void YaSocket::GetLocalAddress(Address & addr) const
{
	WORD pt;
	GetLocalAddress(addr, pt);
}

void YaSocket::GetLocalAddress(Address & addr, WORD & pt) const
{
	sockaddr_in inaddr;
	socklen_t insize = sizeof(inaddr);
	if (::getsockname(os_handle, (struct sockaddr*)&inaddr, &insize) == 0) {
		addr = inaddr.sin_addr;
		pt = ntohs(inaddr.sin_port);
	}
}

PString YaSocket::GetErrorText(PSocket::ErrorGroup group) const
{
	return PSocket::GetErrorText(GetErrorCode(group));
}

bool YaSocket::ConvertOSError(int libReturnValue, PSocket::ErrorGroup group)
{
	if (libReturnValue < 0 && errno == EAGAIN) {
		lastErrorCode[group] = PSocket::Timeout;
		lastErrorNumber[group] = errno;
		return false;
	} else
		return PSocket::ConvertOSError(libReturnValue, lastErrorCode[group], lastErrorNumber[group]);
}

bool YaSocket::SetNonBlockingMode()
{
	if (!IsOpen())
		return false;
	int cmd = 1;
	if (::ioctl(os_handle, FIONBIO, &cmd) == 0 && ::fcntl(os_handle, F_SETFD, 1) == 0)
		return true;
	Close();
	return false;

}

// class YaTCPSocket
YaTCPSocket::YaTCPSocket(WORD pt)
{
	peeraddr.sin_family = AF_INET;
	SetPort(pt);
}

void YaTCPSocket::GetPeerAddress(Address & addr) const
{
	addr = peeraddr.sin_addr;
}

void YaTCPSocket::GetPeerAddress(Address & addr, WORD & pt) const
{
	addr = peeraddr.sin_addr;
	pt = ntohs(peeraddr.sin_port);
}

int YaTCPSocket::os_recv(void *buf, int sz)
{
	return ::recv(os_handle, buf, sz, 0);
}

int YaTCPSocket::os_send(const void *buf, int sz)
{
	return ::send(os_handle, buf, sz, 0);
}

// class YaUDPSocket
YaUDPSocket::YaUDPSocket()
{
	sendaddr.sin_family = AF_INET;
}

bool YaUDPSocket::Listen(unsigned, WORD pt)
{
	return Listen(INADDR_ANY, 0, pt);
}

bool YaUDPSocket::Listen(const Address & addr, unsigned, WORD pt)
{
	os_handle = ::socket(PF_INET, SOCK_DGRAM, 0);
	if (!ConvertOSError(os_handle))
		return false;

	SetNonBlockingMode();
	SetSendAddress(addr, pt);
	int value = 1;
	return ConvertOSError(::bind(os_handle, (struct sockaddr *)&sendaddr, sizeof(sendaddr))) && ConvertOSError(::setsockopt(os_handle, SOL_SOCKET, SO_REUSEADDR, &value, sizeof(value)));
}

void YaUDPSocket::GetLastReceiveAddress(Address & addr, WORD & pt) const
{
	addr = recvaddr.sin_addr;
	pt = ntohs(recvaddr.sin_port);
}

void YaUDPSocket::SetSendAddress(const Address & addr, WORD pt)
{
	sendaddr.sin_addr = addr;
	sendaddr.sin_port = htons(pt);
}

bool YaUDPSocket::ReadFrom(void *buf, PINDEX len, Address & addr, WORD pt)
{
	const bool result = Read(buf, len);
	if (result)
		GetLastReceiveAddress(addr, pt);
	return result;
}

bool YaUDPSocket::WriteTo(const void *buf, PINDEX len, const Address & addr, WORD pt)
{
	SetSendAddress(addr, pt);
	return Write(buf, len);
}

int YaUDPSocket::os_recv(void *buf, int sz)
{
	socklen_t addrlen = sizeof(recvaddr);
	return ::recvfrom(os_handle, buf, sz, 0, (struct sockaddr *)&recvaddr, &addrlen);
}

int YaUDPSocket::os_send(const void *buf, int sz)
{
	return ::sendto(os_handle, buf, sz, 0, (struct sockaddr *)&sendaddr, sizeof(sendaddr));
}

#else

bool SocketSelectList::Select(SelectType t, const PTimeInterval & timeout)
{
	SocketSelectList dumb, *rlist, *wlist;
	if (t == Read)
		rlist = this, wlist = &dumb;
	else
		wlist = this, rlist = &dumb;
#if PTRACING
	PSocket::Errors r = PSocket::Select(*rlist, *wlist, timeout);
	PTRACE_IF(3, r != PSocket::NoError, "ProxyH\tSelect error: " << r);
#else
	PSocket::Select(*rlist, *wlist, timeout);
#endif
	return !IsEmpty();
}

PSocket *SocketSelectList::operator[](int i) const
{
	typedef PSocket::SelectList PSocketSelectList; // stupid VC...
	return &PSocketSelectList::operator[](i);
}

#endif // LARGE_FDSET
