/*
 * sqlacct.cxx
 *
 * SQL accounting module for GNU Gatekeeper
 *
 * Copyright (c) 2004, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: sqlacct.cxx,v $
 * Revision 1.1.2.3  2004/05/12 14:00:48  zvision
 * Header file usage more consistent. Solaris std::map problems fixed.
 * Compilation warnings removed. VSNET2003 project files added. ANSI.h removed.
 *
 * Revision 1.1.2.2  2004/04/24 10:31:57  zvision
 * Use baseclass GetConfigSectionName
 *
 * Revision 1.1.2.1  2004/04/23 16:01:16  zvision
 * New direct SQL accounting module (SQLAcct)
 *
 */
#ifdef HAS_ACCT

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <h225.h>
#include "gk_const.h"
#include "h323util.h"
#include "Toolkit.h"
#include "RasTbl.h"
#include "gkacct.h"
#include "gksql.h"
#include "sqlacct.h"


SQLAcct::SQLAcct( 
	PConfig& cfg,
	const PString& moduleName,
	const char* cfgSecName
	) 
	: GkAcctLogger(cfg, moduleName, cfgSecName), m_sqlConn(NULL), 
	m_gkName(Toolkit::Instance()->GKName())
{
	SetSupportedEvents(SQLAcctEvents);	

	const PString& cfgSec = GetConfigSectionName();
	
	const PString driverName = cfg.GetString(cfgSec, "Driver", "");
	if (driverName.IsEmpty()) {
		PTRACE(1, "GKACCT\t" << GetName() << " module creation failed: "
			"no SQL driver selected"
			);
		return;
	}
	
	GkSQLCreator* driverCreator = GkSQLCreator::FindDriver(driverName);
	if (driverCreator == NULL) {
		PTRACE(1, "GKACCT\t" << GetName() << " module creation failed: "
			"could not find " << driverName << " database driver"
			);
		return;
	}

	m_sqlConn = driverCreator->CreateConnection(moduleName);
	if (!m_sqlConn->Initialize(&cfg, cfgSec)) {
		delete m_sqlConn;
		m_sqlConn = NULL;
		PTRACE(2, "GKACCT\t" << GetName() << " module creation failed: "
			"could not connect to the database"
			);
		return;
	}
	
	m_startQuery = cfg.GetString(cfgSec, "StartQuery", "");
	if (m_startQuery.IsEmpty() 
		&& (GetEnabledEvents() & GetSupportedEvents() & AcctStart) == AcctStart)
		PTRACE(1, "GKACCT\t" << GetName() << " module creation failed: "
			"no start query configured"
			);
	else
		PTRACE(4, "GKACCT\t" << GetName() << " start query: " << m_startQuery);
	
	m_startQueryAlt = cfg.GetString(cfgSec, "StartQueryAlt", "");
	if (!m_startQueryAlt) 
		PTRACE(4, "GKACCT\t" << GetName() << " alternative start query: " 
			<< m_startQueryAlt
			);

	m_updateQuery = cfg.GetString(cfgSec, "UpdateQuery", "");
	if (m_updateQuery.IsEmpty() 
		&& (GetEnabledEvents() & GetSupportedEvents() & AcctUpdate) == AcctUpdate)
		PTRACE(1, "GKACCT\t" << GetName() << " module creation failed: "
			"no update query configured"
			);
	else
		PTRACE(4, "GKACCT\t" << GetName() << " update query: " << m_updateQuery);

	m_stopQuery = cfg.GetString(cfgSec, "StopQuery", "");
	if (m_stopQuery.IsEmpty() 
		&& (GetEnabledEvents() & GetSupportedEvents() & AcctStop) == AcctStop)
		PTRACE(1, "GKACCT\t" << GetName() << " module creation failed: "
			"no stop query configured"
			);
	else
		PTRACE(4, "GKACCT\t" << GetName() << " stop query: " << m_stopQuery);
	
	m_stopQueryAlt = cfg.GetString(cfgSec, "StopQueryAlt", "");
	if (!m_stopQueryAlt) 
		PTRACE(4, "GKACCT\t" << GetName() << " alternative stop query: " 
			<< m_stopQueryAlt
			);
}

SQLAcct::~SQLAcct()
{
	delete m_sqlConn;
}

void SQLAcct::SetupQueryParams(
	/// query parameters (name => value) associations
	std::map<PString, PString>& params,
	/// call (if any) associated with an accounting event being logged
	callptr& call
	) const
{
	PIPSocket::Address addr;
	WORD port = 0;
	time_t t;
	
	params["g"] = m_gkName;
	params["n"] = PString(call->GetCallNumber());
	params["d"] = call->GetDuration();
	params["c"] = call->GetDisconnectCause();
	params["s"] = call->GetAcctSessionId();
	params["CallId"] = ::AsString(call->GetCallIdentifier().m_guid);
	params["ConfId"] = ::AsString(call->GetConferenceIdentifier());
	
	t = call->GetSetupTime();
	if (t)
		params["setup-time"] = AsString(t);
	t = call->GetConnectTime();
	if (t)
		params["connect-time"] = AsString(t);
	t = call->GetDisconnectTime();
	if (t)
		params["disconnect-time"] = AsString(t);
	
	if (call->GetSrcSignalAddr(addr, port)) {
		params["caller-ip"] = addr.AsString();
		params["caller-port"] = port;
	}
	
	PString srcInfo = call->GetSourceInfo();
	params["src-info"] = srcInfo;

	// Get User-name
	if (!(srcInfo.IsEmpty() || srcInfo == "unknown")) {
		const PINDEX index = srcInfo.FindOneOf(":");
		if( index != P_MAX_INDEX )
			srcInfo = srcInfo.Left(index);
	}
	
	endptr callingEP = call->GetCallingParty();
	PString userName;
		
	if (callingEP && callingEP->GetAliases().GetSize() > 0)
		userName = GetBestAliasAddressString(
			callingEP->GetAliases(),
			H225_AliasAddress::e_h323_ID
			);
	else if (!srcInfo)
		userName = srcInfo;
	else if (addr.IsValid())
		userName = addr.AsString();
		
	if (!userName)
		params["u"] = userName;

	PString stationId = call->GetCallingStationId();
		
	if (stationId.IsEmpty() && callingEP && callingEP->GetAliases().GetSize() > 0)
		stationId = GetBestAliasAddressString(
			callingEP->GetAliases(),
			H225_AliasAddress::e_dialedDigits,
			H225_AliasAddress::e_partyNumber,
			H225_AliasAddress::e_h323_ID
			);
					
	if (stationId.IsEmpty())
		stationId = srcInfo;
			
	if (stationId.IsEmpty() && addr.IsValid() && port)
		stationId = ::AsString(addr, port);

	if (!stationId)
		params["Calling-Station-Id"] = stationId;
		
	addr = (DWORD)0;
	port = 0;
		
	if (call->GetDestSignalAddr(addr, port)) {
		params["callee-ip"] = addr.AsString();
		params["callee-port"] = port;
	}

	PString destInfo = call->GetDestInfo();
	params["dest-info"] = destInfo;

	stationId = call->GetCalledStationId();
	
	if (stationId.IsEmpty() && !(destInfo.IsEmpty() || destInfo == "unknown")) {
		const PINDEX index = destInfo.FindOneOf(":");
		stationId = (index == P_MAX_INDEX) ? destInfo : destInfo.Left(index);
	}
		
	if (stationId.IsEmpty()) {
		endptr calledEP = call->GetCalledParty();
		if (calledEP && calledEP->GetAliases().GetSize() > 0)
			stationId = GetBestAliasAddressString(
				calledEP->GetAliases(),
				H225_AliasAddress::e_dialedDigits,
				H225_AliasAddress::e_partyNumber,
				H225_AliasAddress::e_h323_ID
				);
	}
	
	if (stationId.IsEmpty() && addr.IsValid() && port)
		stationId = ::AsString(addr, port);
		
	if (!stationId)
		params["Called-Station-Id"] = stationId;
}

GkAcctLogger::Status SQLAcct::LogAcctEvent(
	GkAcctLogger::AcctEvent evt, 
	callptr& call
	)
{
	if ((evt & GetEnabledEvents() & GetSupportedEvents()) == 0)
		return Next;
		
	if (!call) {
		PTRACE(1, "GKACCT\t" << GetName() << " - missing call info for event" << evt);
		return Fail;
	}
	
	const long callNumber = call->GetCallNumber();
		
	if (m_sqlConn == NULL) {
		PTRACE(2, "GKACCT\t" << GetName() << " failed to store accounting "
			"data (event: " << evt << ", call: " << callNumber 
			<< "): SQL connection not active"
			);
		return Fail;
	}
	
	PString query, queryAlt;
	if (evt == AcctStart) {
		query = m_startQuery;
		queryAlt = m_startQueryAlt;
	} else if (evt == AcctUpdate)
		query = m_updateQuery;
	else if (evt == AcctStop) {
		query = m_stopQuery;
		queryAlt = m_stopQueryAlt;
	}
	
	if (query.IsEmpty()) {
		PTRACE(2, "GKACCT\t" << GetName() << " failed to store accounting "
			"data (event: " << evt << ", call: " << callNumber 
			<< "): SQL query is empty"
			);
		return Fail;
	}

	std::map<PString, PString> params;
	SetupQueryParams(params, call);
	GkSQLResult* result = m_sqlConn->ExecuteQuery(query, params);
	if (result == NULL)
		PTRACE(2, "GKACCT\t" << GetName() << " failed to store accounting "
			"data (event: " << evt << ", call: " << callNumber 
			<< "): timeout or fatal error"
			);
	
	if (result) {
		if (result->IsValid()) {
			if (result->GetNumRows() < 1)
				PTRACE(4, "GKACCT\t" << GetName() << " failed to store accounting "
					"data (event: " << evt << ", call: " << callNumber 
					<< "): no rows have been updated"
					);
		} else
			PTRACE(2, "GKACCT\t" << GetName() << " failed to store accounting "
				"data (event: " << evt << ", call: " << callNumber 
				<< "): (" << result->GetErrorCode() << ") "
				<< result->GetErrorMessage()
				);
	} else if (!queryAlt) {
		result = m_sqlConn->ExecuteQuery(queryAlt, params);
		if (result == NULL)
			PTRACE(2, "GKACCT\t" << GetName() << " failed to store accounting "
				"data (event: " << evt << ", call: " << callNumber 
				<< "): timeout or fatal error"
				);
		else {
			if (result->IsValid()) {
				if (result->GetNumRows() < 1)
					PTRACE(4, "GKACCT\t" << GetName() << " failed to store accounting "
						"data (event: " << evt << ", call: " << callNumber 
						<< "): no rows have been updated"
						);
			} else
				PTRACE(2, "GKACCT\t" << GetName() << " failed to store accounting "
					"data (event: " << evt << ", call: " << callNumber 
					<< "): (" << result->GetErrorCode() << ") "
					<< result->GetErrorMessage()
					);
		}
	}

	const bool succeeded = result != NULL && result->IsValid();	
	delete result;
	return succeeded ? Ok : Fail;
}

namespace {
GkAcctFactory<SQLAcct> SQL_ACCT("SQLAcct");
}

#endif /* HAS_ACCT */
