/*
 * radacct.h
 *
 * RADIUS protocol accounting module for GNU Gatekeeper. 
 *
 * Copyright (c) 2003, Quarcom FHU, Michal Zygmuntowicz
 *
 * This work is published under the GNU Public License (GPL)
 * see file COPYING for details.
 * We also explicitely grant the right to link this code
 * with the OpenH323 library.
 *
 * $Log: radacct.h,v $
 * Revision 1.1.2.8  2004/05/12 17:46:40  zvision
 * Fixes CVS keyword substitution. New ParseEmailAliases config option.
 *
 * Revision 1.1.2.7  2004/05/12 14:00:46  zvision
 * Header file usage more consistent. Solaris std::map problems fixed.
 * Compilation warnings removed. VSNET2003 project files added. ANSI.h removed.
 *
 * Revision 1.1.2.6  2003/12/21 12:20:35  zvision
 * Fixed conditional compilation
 *
 * Revision 1.1.2.5  2003/11/24 22:06:15  zvision
 * Small optimizations (NASIdentifier and localInterfaceAddr variables added)
 *
 * Revision 1.1.2.4  2003/09/14 21:13:48  zvision
 * Changes due to generic accounting API redesign
 *
 * Revision 1.1.2.3  2003/07/31 22:58:48  zvision
 * Added Framed-IP-Address attribute and improved h323-disconnect-cause handling
 *
 * Revision 1.1.2.2  2003/07/03 15:30:40  zvision
 * Added cvs Log keyword
 *
 */
#ifndef RADACCT_H
#define RADACCT_H "#(@) $Id: radacct.h,v 1.1.2.8 2004/05/12 17:46:40 zvision Exp $"
#if defined(HAS_RADIUS) && defined(HAS_ACCT)

#include "gkacct.h"
#include "radproto.h"


class RadAcct : public GkAcctLogger
{
	PCLASSINFO(RadAcct,GkAcctLogger)
public:
	enum Constants
	{
		/// events recognized by this module
		RadAcctEvents = AcctStart | AcctStop | AcctUpdate | AcctOn | AcctOff,
		CiscoVendorId = 9
	};
	
	/** Create GkAcctLogger for RADIUS protocol
	*/
	RadAcct( 
		/// accounting logger settings
		PConfig& cfg, 
		/// name from Gatekeeper::Acct section
		const PString& acctName,
		const char* cfgSecName = NULL
		);
		
	/// Destroy the accounting logger
	virtual ~RadAcct();

	virtual Status LogAcctEvent(
		AcctEvent evt,
		callptr& call
		);

protected:		
	/** Called before Accounting-Request PDU is send.
		Can be used to introduce additional attributes etc.
		
		@return
		TRUE to proceed, FALSE to fail and not send this pdu.
	*/
	virtual BOOL OnSendPDU(
		RadiusPDU& pdu, /// PDU to be sent
		AcctEvent evt, /// accounting event being processed
		callptr& call /// call associated with this request (if any)
		);

	virtual BOOL OnReceivedPDU(
		RadiusPDU& pdu, /// PDU received from RADIUS server
		AcctEvent evt, /// accounting event being processed
		callptr& call /// call associated with this response (if any)
		);
		
private:
	/* No copy constructor allowed */
	RadAcct( const RadAcct& );
	/* No operator= allowed */
	RadAcct& operator=( const RadAcct& );
	
private:
	/// array of configured RADIUS server names
	PStringArray radiusServers;
	/// shared secret for gk client<->RADIUS server authorization
	PString sharedSecret;
	/// default port that will be used for sending RADIUS acct
	/// requests
	WORD acctPort;
	/// base port number for UDP client socket allocation
	WORD portBase;
	/// max port number for UDP client socket allocation
	WORD portMax;
	/// timeout (ms) for a single RADIUS request
	unsigned requestTimeout;
	/// timeout (ms) for RADIUS requests IDs to be unique
	unsigned idCacheTimeout;
	/// timeout (ms) for unused sockets to be deleted
	unsigned socketDeleteTimeout;
	/// how many times to transmit a single request (1==no retransmission)
	/// to a single RADIUS server
	unsigned numRequestRetransmissions;
	/// retransmission fashion: 
	/// 	FALSE - do #numRequestRetransmissions# for server A,
	///				then do #numRequestRetransmissions# for server B, etc.
	///		TRUE - transmit request to server A, then to server B, etc.
	///				the whole procedure repeat #numRequestRetransmissions# times
	BOOL roundRobin;
	/// if TRUE Cisco VSAs are appended to the RADIUS packets
	BOOL appendCiscoAttributes;
	/// append IP address of the calling endpoint
	BOOL includeFramedIp;
	/// Local interface RADIUS client should be bound to (multihomed hosts)
	PString localInterface;	
	/// IP address of the local interface
	PIPSocket::Address localInterfaceAddr;
	/// NAS identifier (GK name)
	PString NASIdentifier;
	/// Fixed value for User-Name attribute in outgoing requests
	PString fixedUsername;
	/// RADIUS protocol client class associated with this authenticator
	RadiusClient* radiusClient;
};

#endif /* defined(HAS_RADIUS) && defined(HAS_ACCT) */
#endif /* RADACCT_H */
