//////////////////////////////////////////////////////////////////
//
// gkauth.h
//
// Gatekeeper authentication modules
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
//
// History:
//      2001/09/19      initial version (Chih-Wei Huang)
//
//////////////////////////////////////////////////////////////////

#ifndef GKAUTH_H
#define GKAUTH_H "#(@) $Id: gkauth.h,v 1.6.2.15 2004/05/12 17:46:40 zvision Exp $"

#include <map>
#include <list>
#include "RasTbl.h"

class H225_GatekeeperRequest;
class H225_RegistrationRequest;
class H225_UnregistrationRequest;
class H225_AdmissionRequest;
class H225_BandwidthRequest;
class H225_DisengageRequest;
class H225_LocationRequest;
class H225_InfoRequest;
class H225_ArrayOf_ClearToken;
class H225_ArrayOf_CryptoH323Token;
class H225_ArrayOf_AliasAddress;
class H225_ArrayOf_AuthenticationMechanism;
class H225_ArrayOf_PASN_ObjectId;
class H235_AuthenticationMechanism;
class PASN_ObjectId;
class H235Authenticators;
class Q931;
class H225_Setup_UUIE;

class CacheManager;

class GkAuthenticator {
public:
	enum Control {
		e_Optional,
		e_Required,
		e_Sufficient
	};

	enum Status {
		e_ok = 1,	// the request is authenticated
		e_fail = -1,	// the request should be rejected
		e_next = 0	// the request is undetermined
	};

	enum {
		e_GRQ = 0x0001,
		e_RRQ = 0x0002,
		e_URQ = 0x0004,
		e_ARQ = 0x0008,
		e_BRQ = 0x0010,
		e_DRQ = 0x0020,
		e_LRQ = 0x0040,
		e_IRQ = 0x0080,
		e_ALL = 0x00FF,
		e_Setup = 0x1000
	};

	GkAuthenticator(PConfig *, const char *authName = "default");
	virtual ~GkAuthenticator();

	template<class RasType> bool CheckRas(PBYTEArray &rawPDU, const RasType & req, unsigned & reason)
	{
	        setLastReceivedRawPDU(rawPDU);
		if (checkFlag & RasValue(req)) {
			int r = Check(req, reason);
			if (r == e_ok) {
				PTRACE(4, "GkAuth\t" << name << " check ok");
				if (controlFlag != e_Required)
					return true;
			} else if (r == e_fail) {
				PTRACE(2, "GkAuth\t" << name << " check failed");
				return false;
			}
		}
		// try next rule
		return (next) ? next->CheckRas(rawPDU, req, reason) : true;
	}

	bool CheckRas(
		/// received raw H.225 ARQ message
		PBYTEArray &rawPDU, 
		/// decoded H.225 ARQ message
		H225_RegistrationRequest& req, 
		/// RRJ code if the authentication fails
		unsigned& rejectReason,
		/// the address which the RRQ has been received from
		const PIPSocket::Address& rxaddr
		);
		
	bool CheckRas(
		/// received raw H.225 ARQ message
		PBYTEArray &rawPDU, 
		/// decoded H.225 ARQ message
		H225_AdmissionRequest& req, 
		/// ARJ code if authentication fails
		unsigned& rejectReason,
		/// call duration limit to be set (-1 for no limit)
		long& callDurationLimit
		);
		
	/** Authenticate/Authorize Setup signalling message.
		
		@return
		true if the message has been authenticated, false otherwise
		(if authentication failed or call duration limit is 0)
	*/
	bool CheckSig( 
		/// received Q.931 Setup message
		Q931& q931pdu, 
		/// received H.225 Setup message
		H225_Setup_UUIE& setup, 
		/// CallRec for the call being authenticated
		callptr& call,
		/// Q.931 cause to set, if authentication fails
		unsigned& releaseCompleteCause, 
		/// call duration limit to be set (-1 for no limit)
		long& callDurationLimit
		);
	
	const char *GetName() { return name; }

	/** @return
		TRUE if this authenticator provides H.235 compatible security.
		It simply checks if h235Authenticators list is not empty.
	*/
	virtual BOOL IsH235Capable() const;
	
	/** If the authenticator supports H.235 security,
		this call returns H.235 security capabilities
		associated with it. It scans list pointed by h235Authenticators.
		
		@return
		TRUE is H.235 security is supported and capabilities
		has been set.
	*/
	virtual BOOL GetH235Capability(
		/// append supported authentication mechanism to this array
		H225_ArrayOf_AuthenticationMechanism& mechanisms,
		/// append supported algorithm OIDs for the given authentication
		/// mechanism
		H225_ArrayOf_PASN_ObjectId& algorithmOIDs
		) const;

	/** Check if this authenticator supports the given
		H.235 capability (mechanism+algorithmOID) by scanning
		list pointed by h235Authenticators.
		
		@return
		TRUE if the capability is supported.
	*/
	virtual BOOL IsH235Capability(
		/// authentication mechanism
		const H235_AuthenticationMechanism& mechanism,
		/// algorithm OID for the given authentication mechanism
		const PASN_ObjectId& algorithmOID
		) const;

	/** @return
		Control flag determining authenticator behaviour
		(optional,sufficient,required).
	*/
	Control GetControlFlag() const { return controlFlag; }

	/** @return
		Next authenticator on the list.
	*/
	GkAuthenticator* GetNext() const { return next; }
	
protected:
	// the second argument is the reject reason, if any
	virtual int Check(const H225_GatekeeperRequest &, unsigned &);
	virtual int Check(H225_RegistrationRequest &, unsigned &);
	virtual int Check(const H225_UnregistrationRequest &, unsigned &);
	virtual int Check(H225_AdmissionRequest &, unsigned &);
	virtual int Check(
		/// received H.225 ARJ message
		H225_AdmissionRequest& request, 
		/// ARJ reason to set, if authentication failed
		unsigned& rejectReason, 
		/// call duration limit to set (-1 for no duration limit)
		long& callDurationLimit
		);
	virtual int Check(const H225_BandwidthRequest &, unsigned &);
	virtual int Check(const H225_DisengageRequest &, unsigned &);
	virtual int Check(const H225_LocationRequest &, unsigned &);
	virtual int Check(const H225_InfoRequest &, unsigned &);
	/** Authenticate/Authorize Setup signalling message.
	
		@return
		e_fail - authentication failed
		e_ok - authenticated with this authenticator
		e_next - authentication could not be determined
	*/
	virtual int Check(
		/// received Q.931 Setup message
		Q931& q931pdu, 
		/// received H.225 Setup message
		H225_Setup_UUIE& setup,
		/// CallRec for the call being authenticated
		callptr& call,
		/// Q.931 cause to set, if authentication failed
		unsigned& releaseCompleteCause, 
		/// call duration limit to set (-1 for no duration limit)
		long& callDurationLimit
		);

	int RasValue(const H225_GatekeeperRequest &)     { return e_GRQ; }
	int RasValue(const H225_RegistrationRequest &)   { return e_RRQ; }
	int RasValue(const H225_UnregistrationRequest &) { return e_URQ; }
	int RasValue(const H225_AdmissionRequest &)      { return e_ARQ; }
	int RasValue(const H225_BandwidthRequest &)      { return e_BRQ; }
	int RasValue(const H225_DisengageRequest &)      { return e_DRQ; }
	int RasValue(const H225_LocationRequest &)       { return e_LRQ; }
	int RasValue(const H225_InfoRequest &)           { return e_IRQ; }

	PBYTEArray& getLastReceivedRawPDU(){ return m_lastReceivedRawPDU; }

	Control controlFlag;
	Status defaultStatus;
	PConfig *config;

	H235Authenticators* h235Authenticators;
	
private:
	const char *name;
	int checkFlag;

	GkAuthenticator *next;
	static GkAuthenticator *head;

	GkAuthenticator(const GkAuthenticator &);
	GkAuthenticator & operator=(const GkAuthenticator &);
	
	void setLastReceivedRawPDU(PBYTEArray &rawPDU){ m_lastReceivedRawPDU = rawPDU; }
	
	PBYTEArray m_lastReceivedRawPDU;

	friend class GkAuthenticatorList;
};

class SimplePasswordAuth : public GkAuthenticator {
public:
	SimplePasswordAuth(PConfig *, const char *);
	~SimplePasswordAuth();

protected:
	virtual int Check(const H225_GatekeeperRequest &, unsigned &);
	virtual int Check(H225_RegistrationRequest &, unsigned &);
	virtual int Check(const H225_UnregistrationRequest &, unsigned &);
	virtual int Check(H225_AdmissionRequest &, unsigned &);
	virtual int Check(const H225_BandwidthRequest &, unsigned &);
	virtual int Check(const H225_DisengageRequest &, unsigned &);
	virtual int Check(const H225_LocationRequest &, unsigned &);
	virtual int Check(const H225_InfoRequest &, unsigned &);

	virtual PString GetPassword(const PString & id);

	virtual bool CheckAliases(const PString &);
	virtual bool CheckTokens(const H225_ArrayOf_ClearToken &);
	virtual bool CheckCryptoTokens(const H225_ArrayOf_CryptoH323Token &);

	template<class RasType> int doCheck(const RasType & req)
	{
		if (req.HasOptionalField(RasType::e_cryptoTokens))
			return CheckCryptoTokens(req.m_cryptoTokens) ? e_ok : e_fail;
	 	else if (req.HasOptionalField(RasType::e_tokens))
			return CheckTokens(req.m_tokens) ? e_ok : e_fail;
		return (controlFlag == e_Optional) ? e_next : e_fail;
	}

#if defined HAS_WLDAP
	bool InternalGetPassword(PString & passwd);
#else
	bool InternalGetPassword(const PString & id, PString & passwd);
#endif // HAS_WLDAP

protected:
	int filled;
	bool checkid;
	const H225_ArrayOf_AliasAddress *aliases;
	CacheManager* m_cache;
};

class GkAuthInitializer {
public:
	GkAuthInitializer(const char *);
	virtual ~GkAuthInitializer();
	// virtual constructor
	virtual GkAuthenticator *CreateAuthenticator(PConfig *) = 0;
	bool Compare(PString n) const;

protected:
	const char *name;
};

template<class GkAuth> class GkAuthInit : public GkAuthInitializer {
public:
	GkAuthInit(const char *n) : GkAuthInitializer(n) {}
	virtual GkAuthenticator *CreateAuthenticator(PConfig *config)
	{ return new GkAuth(config, name); }
};

class GkAuthenticatorList {
public:
	GkAuthenticatorList(PConfig *);
	virtual ~GkAuthenticatorList();

	template<class RasType> bool Check(const RasType & req, unsigned & reason)
	{
		return (GkAuthenticator::head) ? GkAuthenticator::head->CheckRas(getLastReceivedRawPDU(), req, reason) : true;
	}
	
	bool Check(
		/// received H.225 RRQ message
		H225_RegistrationRequest& request, 
		/// RRJ reason to set, if the authentication fails
		unsigned& rejectReason,
		/// the address which the RRQ has been received from
		const PIPSocket::Address& rxaddr
		)
	{
		return (GkAuthenticator::head) ? GkAuthenticator::head->CheckRas(
			getLastReceivedRawPDU(), request, rejectReason, rxaddr
			) : true;
	}
	
	bool Check(
		/// received H.225 ARQ message
		H225_AdmissionRequest& request, 
		/// ARJ reason to set, if authentication fails
		unsigned& rejectReason, 
		/// call duration limit to set (-1 for no duration limit)
		long& callDurationLimit
		)
	{
		callDurationLimit = -1;
		return (GkAuthenticator::head) ? GkAuthenticator::head->CheckRas(
			getLastReceivedRawPDU(), request, rejectReason, callDurationLimit
			) : true;
	}

	/** Authenticate/Authorize Setup signalling message. 
		Iterate through all authenticators.
		
		@return
		true if the message has been authenticated, false if not
		or if call duration limit is 0
	*/
	bool CheckSig(
		/// received Q.931 Setup message
		Q931& q931pdu,
		/// received H.225 Setup message
		H225_Setup_UUIE& setup, 
		/// CallRec for the call being authenticated
		callptr& call,
		/// Q.931 cause value to set, if authentication fails
		unsigned& releaseCompleteCause,
		/// call duration limit to set (-1 for no duration limit)
		long& callDurationLimit
		)
	{
		callDurationLimit = -1;
		return (GkAuthenticator::head != NULL) ? GkAuthenticator::head->CheckSig( 
			q931pdu, setup, call, releaseCompleteCause, callDurationLimit 
			) : true;
	}
	
	virtual void setLastReceivedRawPDU(PBYTEArray &rawPDU){ m_lastReceivedRawPDU = rawPDU; }

	/** @return
		Head of the authenticators list. Use GkAuthenticator::GetNext()
		to traverse the list.
	*/
	GkAuthenticator* GetHead() const { return GkAuthenticator::head; }

	void GetH235Capabilities(
		H225_ArrayOf_AuthenticationMechanism& mechanisms,
		H225_ArrayOf_PASN_ObjectId& algorithmOIDs
		) const;
	
private:
	GkAuthenticatorList(const GkAuthenticatorList &);
	GkAuthenticatorList & operator=(const GkAuthenticatorList &);

	virtual PBYTEArray& getLastReceivedRawPDU(){ return m_lastReceivedRawPDU; }
	
	PBYTEArray m_lastReceivedRawPDU;
	
	/// the most common authentication capabilities 
	/// shared by all authenticators on the list
	H225_ArrayOf_AuthenticationMechanism* m_mechanisms;
	H225_ArrayOf_PASN_ObjectId* m_algorithmOIDs;
};


#endif  // GKAUTH_H

