//////////////////////////////////////////////////////////////////
//
// gkauth.cxx
//
// $Id
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
//      2003/10/23      LDAP support for Window (HAS_WLDAP) added (Franz J Ehrengruber)
//      2001/09/19      initial version (Chih-Wei Huang)
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <h323pdu.h>
#include <h225.h>
#include <h235.h>
#include <h235auth.h>
#include "gk_const.h"
#include "h323util.h"
#include "stl_supp.h"
#include "Toolkit.h"
#include "RasTbl.h"
#include "gksql.h"
#include "gkauth.h"

#if defined(HAS_LDAP)		// shall use LDAP (UNIX)
  #include "gkldap.h"
#endif
#if defined(HAS_WLDAP)		// shall use WLDAP (Windows)
  #include "gk_wldap.h"
#endif

namespace {
const char* const GkAuthSectionName = "Gatekeeper::Auth";
}

GkAuthenticator *GkAuthenticator::head = 0;

//////////////////////////////////////////////////////////////////////
// Definition of authentication rules

class AliasAuth : public GkAuthenticator {
public:
	AliasAuth(PConfig *, const char *);

protected:
	virtual int Check(const H225_GatekeeperRequest &, unsigned &);
	virtual int Check(H225_RegistrationRequest &, unsigned &);
	virtual int Check(const H225_UnregistrationRequest &, unsigned &);
	virtual int Check(H225_AdmissionRequest &, unsigned &);
	virtual int Check(const H225_BandwidthRequest &, unsigned &);
	virtual int Check(const H225_DisengageRequest &, unsigned &);
	virtual int Check(const H225_LocationRequest &, unsigned &);
	virtual int Check(const H225_InfoRequest &, unsigned &);

	virtual bool doCheck(const H225_ArrayOf_TransportAddress &, const PString &);
	virtual bool AuthCondition(const H225_TransportAddress & SignalAdr, const PString &);

private:
	/** @returns the value for a given alias from section #RasSrv::RRQAuth# 
	    in ini-file
	 */
	virtual PString GetConfigString(const PString & alias);
};

#if HAS_MYSQL

#include <mysql.h>

class MySQLPasswordAuth : public SimplePasswordAuth
{
public:
	MySQLPasswordAuth(
		PConfig* cfg, 
		const char* authName
		);
	virtual ~MySQLPasswordAuth();

protected:
	virtual PString GetPassword(
		const PString& alias
		);
		
protected:
	GkSQLConnection* m_sqlConn;
	PString m_query;
};

class MySQLAliasAuth : public AliasAuth
{
public:
	MySQLAliasAuth(
		PConfig* cfg, 
		const char* authName
		);
	
	virtual ~MySQLAliasAuth();

protected:
	virtual PString GetConfigString(
		const PString& alias
		);

protected:
	GkSQLConnection* m_sqlConn;
	CacheManager* m_cache;
	PString m_query;
};


#endif // HAS_MYSQL

/// Generic SQL authenticator for H.235 enabled endpoints
class SQLPasswordAuth : public SimplePasswordAuth
{
public:
	/// build authenticator reading settings from the config
	SQLPasswordAuth(
		/// config to read settings from
		PConfig* cfg,
		/// name for this authenticator and for the config section to read settings from
		const char* authName
		);
	
	virtual ~SQLPasswordAuth();

protected:
	/** Override from SimplePasswordAuth.
	
	    @return
	    Clear text password associated with the given alias
	    or an empty string if no password is found.
	*/
	virtual PString GetPassword(
		/// alias to check the password for
		const PString& alias
		);
		
protected:
	/// connection to the SQL database
	GkSQLConnection* m_sqlConn;
	/// parametrized query string for password retrieval
	PString m_query;
};

/// Generic SQL authenticator for alias/IP based authentication
class SQLAliasAuth : public AliasAuth
{
public:
	/// build authenticator reading settings from the config
	SQLAliasAuth(
		/// config to read settings from
		PConfig* cfg, 
		/// name for this authenticator and for the config section to read settings from
		const char* authName
		);
	
	virtual ~SQLAliasAuth();

protected:
	/** Override from AliasAuth.
	
	    @return
	    Auth condition string associated with the given alias
	    or an empty string if no match is found for the given alias.
	*/
	virtual PString GetConfigString(
		/// alias to retrieve the auth condition string for
		const PString& alias
		);

protected:
	/// connection to the SQL database
	GkSQLConnection* m_sqlConn;
	/// auth condition string cache
	CacheManager* m_cache;
	/// parametrized query string for the auth condition string retrieval
	PString m_query;
};


#if ((defined(__GNUC__) && __GNUC__ <= 2) && !defined(WIN32))
#include <unistd.h>
#include <procbuf.h>

class ExternalPasswordAuth : public SimplePasswordAuth {
public:
	ExternalPasswordAuth(PConfig *, const char *);

	virtual PString GetPassword(const PString &);
private:
	bool ExternalInit();

	PString Program;
};

namespace {
GkAuthInit<ExternalPasswordAuth> ExternalPasswordAuthFactory("ExternalPasswordAuth");
}

#endif

// LDAP authentification
#if defined(HAS_LDAP)		// shall use LDAP

class LDAPPasswordAuth : public SimplePasswordAuth {
public:
  LDAPPasswordAuth(PConfig *, const char *);
  virtual ~LDAPPasswordAuth();

  virtual PString GetPassword(const PString &alias);
  
  virtual int Check(H225_RegistrationRequest & rrq, unsigned & reason);
};

// ISO 14882:1998 (C++), ISO9899:1999 (C), ISO9945-1:1996 (POSIX) have a
// very clear oppinion regarding user symbols starting or ending with '_'

class LDAPAliasAuth : public AliasAuth {
public:
	LDAPAliasAuth(PConfig *, const char *);
	virtual ~LDAPAliasAuth();
	
	virtual int Check(H225_RegistrationRequest & rrq, unsigned &);

private:
	/** Searchs for an alias in LDAP and converts it to a valid config
	    string (the expected return value from LDAP is only an IP-address!).
	    @returns config-string (format: see description in ini-file)
	 */
	virtual PString GetConfigString(const PString &alias);
};


#endif // HAS_LDAP

#if defined HAS_WLDAP

class LDAPPasswordAuth : public SimplePasswordAuth {
public:
  LDAPPasswordAuth(PConfig *, const char *);
  virtual ~LDAPPasswordAuth();

  /** Searchs for an h323id/e164 alias in LDAP then:
	    1) mandatory: retrieves the H235Password attribute 
	    2) optional: retrieves the AccountStatus attribute and converts 
		   it to a valid config string. (format: see description in ini-file)
		3) optionally: retrieves the h323id/e164 alias attributes
	    @returns the H323Password attribute or an empty string;
  */
  virtual PString GetPassword(const PString &alias);
  
  /** calls SimplePasswordAuth::Check(rrq, reason) and then:
	    1) optionally converts AccountStatus attribute to a valid 
		   config string. (format: see description in ini-file) and 
		   calls doCheckAccountStatus(rrq.m_callSignalAddress, cfgString)
		   to verify the account status
		2) optionally: checks the supplied h323id/e164 alias are valid.
	    @returns (result) ? e_ok : e_fail;
  */
  virtual int Check(H225_RegistrationRequest & rrq, unsigned & reason);


private:

	// see above
	virtual bool doCheckAccountStatus(const H225_ArrayOf_TransportAddress & addrs, const PString & cfgString);
	
	// see above
	virtual bool AuthCondition(const H225_TransportAddress & SignalAdr, const PString & Condition);
	
	int attribute_count;
	PString epAccountStatus_value;
	PString epH323ID_value;
	PStringList epTelephoneNo_value;
	PString epH235Password_value;
};

// ISO 14882:1998 (C++), ISO9899:1999 (C), ISO9945-1:1996 (POSIX) have a
// very clear oppinion regarding user symbols starting or ending with '_'

class LDAPAliasAuth : public AliasAuth {
public:
	LDAPAliasAuth(PConfig *, const char *);
	virtual ~LDAPAliasAuth();
	
	/** Searchs for an h323id/e164 alias in LDAP then:
	    1) mandatory: retrieves the AccountStatus attribute and converts 
		   it to a valid config string. (format: see description in ini-file)
		2) optionally: retrieves the h323id/e164 alias attributes and checks
		   if the aliases are valid.
	    @returns (AliasFoundInLDAP) ? e_ok : defaultStatus;
	 */
	virtual int Check(H225_RegistrationRequest & rrq, unsigned &);

};


#endif // HAS_WLDAP


// Initial author: Michael Rubashenkkov  2002/01/14 (GkAuthorize)
// Completely rewrite by Chih-Wei Huang  2002/05/01
class AuthObj;
class AuthRule;
class PrefixAuth : public GkAuthenticator {
public:
	PrefixAuth(PConfig *, const char *);
	~PrefixAuth();

	typedef std::map< PString, AuthRule *, greater<PString> > Rules;

private:
	virtual int Check(const H225_GatekeeperRequest &, unsigned &);
	virtual int Check(H225_RegistrationRequest &, unsigned &);
	virtual int Check(const H225_UnregistrationRequest &, unsigned &);
	virtual int Check(H225_AdmissionRequest &, unsigned &); 
	virtual int Check(const H225_BandwidthRequest &, unsigned &);
	virtual int Check(const H225_DisengageRequest &, unsigned &);
	virtual int Check(const H225_LocationRequest &, unsigned &);
	virtual int Check(const H225_InfoRequest &, unsigned &);

	virtual int doCheck(const AuthObj &);
	
	Rules prefrules;
};



class CacheManager 
{
public:
	CacheManager(
		long ttl /// cache expiration timeout (seconds)
		) : m_ttl(ttl) {}

	bool Retrieve(const PString & key, PString & value);
	void Save(const PString & key, const PString & value);
	
	void SetTimeout(
		long newTimeout /// new cache expiration timeout
		)
	{
		m_ttl = newTimeout;
	}

private:
	std::map<PString, PString> m_cache;
	std::map<PString, long> m_ctime;
	// 0 means don't cache, -1 means never expires
	long m_ttl; // seconds

	CacheManager(const CacheManager &);
	CacheManager & operator=(const CacheManager &);
};      

bool CacheManager::Retrieve(const PString & key, PString & value)
{
	std::map<PString, PString>::iterator iter = m_cache.find(key);
	if (iter == m_cache.end())
		return false;
	if (m_ttl > 0) {
		std::map<PString, long>::iterator i = m_ctime.find(key);
		if ((time(NULL) - i->second) > m_ttl)
			return false; // cache expired
	}
	value = iter->second;
	PTRACE(5, "GkAuth\tCache found for " << key);
	return true;
}

void CacheManager::Save(const PString & key, const PString & value)
{
	if (m_ttl != 0) {
		m_cache[key] = value;
		m_ctime[key] = time(NULL);
	}
}

//////////////////////////////////////////////////////////////////////

GkAuthenticator::GkAuthenticator(PConfig *cfg, const char *authName) 
	: config(cfg), h235Authenticators(NULL), name(authName), checkFlag(e_ALL)
{
	PStringArray control(config->GetString(GkAuthSectionName, name, "").Tokenise(";,"));
	if (PString(name) == "default")
		controlFlag = e_Sufficient,
		defaultStatus = Toolkit::AsBool(control[0]) ? e_ok : e_fail;
	else if (control[0] *= "optional")
		controlFlag = e_Optional, defaultStatus = e_next;
	else if (control[0] *= "required")
		controlFlag = e_Required, defaultStatus = e_fail;
	else
		controlFlag = e_Sufficient, defaultStatus = e_fail;

	if (control.GetSize() > 1) {
		checkFlag = 0;
		std::map<PString, int> rasmap;
		rasmap["GRQ"] = e_GRQ, rasmap["RRQ"] = e_RRQ,
		rasmap["URQ"] = e_URQ, rasmap["ARQ"] = e_ARQ,
		rasmap["BRQ"] = e_BRQ, rasmap["DRQ"] = e_DRQ,
		rasmap["LRQ"] = e_LRQ, rasmap["IRQ"] = e_IRQ,
		rasmap["Setup"] = e_Setup;
		for (PINDEX i=1; i < control.GetSize(); ++i) {
			if (rasmap.find(control[i]) != rasmap.end())
				checkFlag |= rasmap[control[i]];
		}
	}
	
	next = head;
	head = this;

	PTRACE(1, "GkAuth\tAdd " << name << " rule with flag " << hex << checkFlag << dec);
}

GkAuthenticator::~GkAuthenticator()
{
	PTRACE(1, "GkAuth\tRemove " << name << " rule");
	delete next;  // delete whole list recursively
	delete h235Authenticators;
}

bool GkAuthenticator::CheckRas(
	PBYTEArray &rawPDU, 
	H225_RegistrationRequest& rrq, 
	unsigned& rejectReason,
	const PIPSocket::Address& rxaddr
	)
{
	setLastReceivedRawPDU(rawPDU);
	if (checkFlag & RasValue(rrq)) {
		int r = Check(rrq, rejectReason);
		if (r == e_ok) {
			PTRACE(4, "GkAuth\t" << name << " check ok");
			if (controlFlag != e_Required)
				return true;
		} else if (r == e_fail) {
			PTRACE(2, "GkAuth\t" << name << " check failed");
			return false;
		}
	}
	// try next rule
	return next == NULL || next->CheckRas(rawPDU,rrq,rejectReason,rxaddr);
}

bool GkAuthenticator::CheckRas(
	PBYTEArray &rawPDU, 
	H225_AdmissionRequest& req, 
	unsigned& rejectReason,
	long& callDurationLimit
	)
{
	callDurationLimit = -1;
	setLastReceivedRawPDU(rawPDU);
	if (checkFlag & RasValue(req)) {
		int r = Check(req, rejectReason, callDurationLimit);
		if( callDurationLimit == 0 ) {
			PTRACE(2,"GkAuth\t" << name << " - call duration limit is 0");
			return false;
		}
		if (r == e_ok) {
			PTRACE(4, "GkAuth\t" << name << " check ok");
			if (controlFlag != e_Required)
				return true;
		} else if (r == e_fail) {
			PTRACE(2, "GkAuth\t" << name << " check failed");
			return false;
		}
	}
	// try next rule
	if( next ) {
		long limit = -1;
		if( !next->CheckRas(rawPDU,req,rejectReason,limit) )
			return false;
		if( callDurationLimit >= 0 && limit >= 0 )
			callDurationLimit = PMIN(limit,callDurationLimit);
		else
			callDurationLimit = PMAX(limit,callDurationLimit);
	}
	if( callDurationLimit == 0 ) {
		PTRACE(2,"GkAuth\t" << name << " - call duration limit is 0");
		return false;
	}
	return true;
}
		
bool GkAuthenticator::CheckSig( 
	/// received Q.931 Setup message
	Q931& q931pdu, 
	/// received H.225 Setup message
	H225_Setup_UUIE& setup, 
	/// CallRec for the call being authenticated
	callptr& call,
	/// Q.931 cause to set, if authentication fails
	unsigned& releaseCompleteCause, 
	/// call duration limit (should be set initially to -1,
	/// what means no limit)
	long& callDurationLimit
	)
{
	callDurationLimit = -1;
	if( checkFlag & e_Setup ) {
		const int r = Check( q931pdu, setup, call, releaseCompleteCause, callDurationLimit );
		if( callDurationLimit == 0 ) {
			PTRACE(2,"GkAuth\t"<<name<<" - call duration limit is 0");
			return false;
		}
		if( r == e_ok ) {
			PTRACE(4,"GkAuth\t"<<name<<" Q.931 Setup check ok");
			if( controlFlag != e_Required )
				return true;
		} else if( r == e_fail ) {
			PTRACE(2,"GkAuth\t"<<name<<" Q.931 Setup check failed");
			return false;
		}
	}
	
	// try next rule
	if( next ) {
		long limit = -1;
		if( !next->CheckSig( q931pdu, setup, call, releaseCompleteCause, limit ) )
			return false;
		if( callDurationLimit >= 0 && limit >= 0 )
			callDurationLimit = PMIN(limit,callDurationLimit);
		else
			callDurationLimit = PMAX(limit,callDurationLimit);
	}
	if( callDurationLimit == 0 ) {
		PTRACE(2,"GkAuth\t"<<name<<" - call duration limit is 0");
		return false;
	}
	return true;
}

int GkAuthenticator::Check(const H225_GatekeeperRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(H225_RegistrationRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_UnregistrationRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(H225_AdmissionRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(
	H225_AdmissionRequest& request, 
	unsigned& rejectReason,
	long& /*callDurationLimit*/
	)
{
	// for backward compatibility with derived authenticators
	// that do not understand call duration limit feature
	return Check(request,rejectReason);
}

int GkAuthenticator::Check(const H225_BandwidthRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_DisengageRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_LocationRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_InfoRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check( Q931&, H225_Setup_UUIE&, callptr&, unsigned&, long& )
{
	return defaultStatus;
}

BOOL GkAuthenticator::GetH235Capability(
	H225_ArrayOf_AuthenticationMechanism& mechanisms,
	H225_ArrayOf_PASN_ObjectId& algorithmOIDs
	) const
{
	if( h235Authenticators == NULL )
		return FALSE;
		
	for( int i = 0; i < h235Authenticators->GetSize(); i++ )
		(*h235Authenticators)[i].SetCapability(mechanisms,algorithmOIDs);
		
	return TRUE;
}		

BOOL GkAuthenticator::IsH235Capability(
	const H235_AuthenticationMechanism& mechanism,
	const PASN_ObjectId& algorithmOID
	) const
{
	if( h235Authenticators == NULL )
		return FALSE;
		
	for( int i = 0; i < h235Authenticators->GetSize(); i++ )
		if( (*h235Authenticators)[i].IsCapability(mechanism,algorithmOID) )
			return TRUE;
		
	return FALSE;
}

BOOL GkAuthenticator::IsH235Capable() const
{
	return (h235Authenticators != NULL) && (h235Authenticators->GetSize() > 0);
}


const char *passwdsec = "Password";

// SimplePasswordAuth
SimplePasswordAuth::SimplePasswordAuth(PConfig *cfg, const char *authName)
      : GkAuthenticator(cfg, authName), aliases(0)
{
	filled = config->GetInteger(passwdsec, "KeyFilled", 0);
	checkid = Toolkit::AsBool(config->GetString(passwdsec, "CheckID", "0"));
	m_cache = new CacheManager(config->GetInteger(passwdsec, "PasswordTimeout", -1));
	
	if( h235Authenticators == NULL )
		h235Authenticators = new H235Authenticators;
		
	H235Authenticator* authenticator;
	
	authenticator = new H235AuthSimpleMD5;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
	authenticator = new H235AuthCAT;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
/*
#if P_SSL
	authenticator = new H235AuthProcedure1;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
#endif
*/
}

SimplePasswordAuth::~SimplePasswordAuth()
{
	delete m_cache;
}

int SimplePasswordAuth::Check(const H225_GatekeeperRequest & grq, unsigned &)
{
	return doCheck(grq);
}

int SimplePasswordAuth::Check(H225_RegistrationRequest & rrq, unsigned &)
{
	if (checkid) {
		if (!rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
			return e_fail;
		aliases = &rrq.m_terminalAlias;
	}
	return doCheck(rrq);
}

int SimplePasswordAuth::Check(const H225_UnregistrationRequest & urq, unsigned &)
{
	return doCheck(urq);
}

int SimplePasswordAuth::Check(H225_AdmissionRequest & arq, unsigned &)
{
	return doCheck(arq);
}

int SimplePasswordAuth::Check(const H225_BandwidthRequest & brq, unsigned &)
{
	return doCheck(brq);
}

int SimplePasswordAuth::Check(const H225_DisengageRequest & drq, unsigned &)
{
	return doCheck(drq);
}

int SimplePasswordAuth::Check(const H225_LocationRequest & lrq, unsigned &)
{
	return doCheck(lrq);
}

int SimplePasswordAuth::Check(const H225_InfoRequest & drq, unsigned &)
{
	return doCheck(drq);
}

PString SimplePasswordAuth::GetPassword(const PString & id)
{
	return Toolkit::CypherDecode(id, config->GetString(passwdsec, id, ""), filled);
}

#if defined HAS_WLDAP

// ini file section name
extern const char *ldap_attr_name_sec; // [GkLDAP::LDAPAttributeNames]
extern const char *ldap_auth_sec;      // [GkLDAP::Settings]

bool SimplePasswordAuth::InternalGetPassword(PString & passwd)
{
	PStringList AliasList;
	// alias search presedence e164/h323id
	if (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapPasswAuthSearchPrecedenceE164", "0"))) {
		PString h323id;
		for (PINDEX j = 0; j < aliases->GetSize(); ++j) {
			if ((*aliases)[j].GetTag() == H225_AliasAddress::e_dialedDigits) {
				const PString & e164 = H323GetAliasAddressString((*aliases)[j]);
				PTRACE(5, "GkAuth\tSimplePasswordAuth::InternalGetPassword() e164 '" << e164 <<"'");
				AliasList.AppendString(e164);
			}
			else if ((*aliases)[j].GetTag() == H225_AliasAddress::e_h323_ID) {
				h323id = H323GetAliasAddressString((*aliases)[j]);
				PTRACE(5, "GkAuth\tSimplePasswordAuth::InternalGetPassword() h323id '" << h323id <<"'");
			}
		}
		AliasList.AppendString(h323id);
	}
	else {
		for (PINDEX j = 0; j < aliases->GetSize(); ++j) {
			PTRACE(5, "GkAuth\tSimplePasswordAuth::InternalGetPassword() H323GetAliasAddressString((*aliases)[j]) '" << H323GetAliasAddressString((*aliases)[j]) <<"'");
			AliasList.AppendString(H323GetAliasAddressString((*aliases)[j]));
		}
	}

	// step through the list of aliases and search
	// LDAP for a password. If the returned string
	// 'passwd' is not empty we return result.
	bool result = false;
	for (PINDEX i = 0; i < AliasList.GetSize(); i++) {
		passwd = "";
		result = m_cache->Retrieve(AliasList[i], passwd);
		if (!result) {
			passwd = GetPassword(AliasList[i]);
			PTRACE(5, "GkAuth\tSimplePasswordAuth::InternalGetPassword() passwd '" << passwd <<"'");
			if(!passwd.IsEmpty()) {
				aliases = 0;
				return result;
			}
		}
	}
	aliases = 0;
	return result;
}

#else // any authenticator

bool SimplePasswordAuth::InternalGetPassword(const PString & id, PString & passwd)
{
	bool result = m_cache->Retrieve(id, passwd);
	if (!result)
		passwd = GetPassword(id);
	return result;
}

#endif // HAS_WLDAP

bool SimplePasswordAuth::CheckAliases(const PString & id)
{
	bool r = false;
	for (PINDEX i = 0; i < aliases->GetSize(); i++)
		if (H323GetAliasAddressString((*aliases)[i]) == id) {
			r = true;
			break;
		}
#if !defined HAS_WLDAP
	aliases = 0;
#endif
	return r;
}

static const char OID_CAT[] = "1.2.840.113548.10.1.2.1";

bool SimplePasswordAuth::CheckTokens(const H225_ArrayOf_ClearToken & tokens)
{
	for (PINDEX i=0; i < tokens.GetSize(); ++i) {
		H235_ClearToken & token = tokens[i];
		// check for Cisco Access Token
		if( token.m_tokenOID == OID_CAT )
		{
			if( !token.HasOptionalField(H235_ClearToken::e_generalID) )
				return false;
				
			PString id = token.m_generalID;
			if (aliases && !CheckAliases(id))
				return false;
				
			PString passwd;
			bool cached = false;
#if defined HAS_WLDAP
			cached = InternalGetPassword(passwd);
#else
			cached = InternalGetPassword(id, passwd);
#endif // HAS_WLDAP

			H235AuthCAT authCAT;
			authCAT.SetLocalId(id);
			authCAT.SetPassword(passwd);
			if (authCAT.ValidateClearToken(token) == H235Authenticator::e_OK) {
				PTRACE(4, "GkAuth\t" << id << " password match (CAT)");
				if (!cached)
					m_cache->Save(id, passwd);
				return true;
			}
			return false;
		}
		if (token.HasOptionalField(H235_ClearToken::e_generalID) &&
		    token.HasOptionalField(H235_ClearToken::e_password)) {
			PString id = token.m_generalID;
			if (aliases && !CheckAliases(id))
				return false;
			PString passwd, tokenpasswd = token.m_password;
			bool cached = false;
#if defined HAS_WLDAP
			cached = InternalGetPassword(passwd);
#else
			cached = InternalGetPassword(id, passwd);
#endif
			if (passwd == tokenpasswd) {
				PTRACE(4, "GkAuth\t" << id << " password match");
				if (!cached)
					m_cache->Save(id, passwd);
				return true;
			}
		}
	}
	return false;
}

bool SimplePasswordAuth::CheckCryptoTokens(const H225_ArrayOf_CryptoH323Token & tokens)
{
	for (PINDEX i = 0; i < tokens.GetSize(); ++i) {
		if (tokens[i].GetTag() == H225_CryptoH323Token::e_cryptoEPPwdHash) {
			H225_CryptoH323Token_cryptoEPPwdHash & pwdhash = tokens[i];
			PString id = AsString(pwdhash.m_alias, FALSE);
			if (aliases && !CheckAliases(id))
				return false;
			PString passwd;
			bool cached = false;
#if defined HAS_WLDAP
			cached = InternalGetPassword(passwd);
#else
			cached = InternalGetPassword(id, passwd);
#endif // HAS_WLDAP

			H235AuthSimpleMD5 authMD5;
			authMD5.SetLocalId(id);
			authMD5.SetPassword(passwd);
			PBYTEArray nullPDU;
			if (authMD5.ValidateCryptoToken(tokens[i], nullPDU) == H235Authenticator::e_OK) {
				PTRACE(4, "GkAuth\t" << id << " password match (MD5)");
				if (!cached)
					m_cache->Save(id, passwd);
				return true;
			}
#if P_SSL
		} else if (tokens[i].GetTag() == H225_CryptoH323Token::e_nestedcryptoToken){
			H235_CryptoToken & nestedCryptoToken = tokens[i];
			H235_CryptoToken_cryptoHashedToken & cryptoHashedToken = nestedCryptoToken;
			H235_ClearToken & clearToken = cryptoHashedToken.m_hashedVals;
			PString gk_id = clearToken.m_generalID;
			//assumption: sendersID == endpoint alias (RRQ)
			PString ep_alias = clearToken.m_sendersID; 
			if (aliases && !CheckAliases(ep_alias))
				return false;
			PString passwd;
			bool cached = false;
#if defined HAS_WLDAP
			cached = InternalGetPassword(passwd);
#else
			cached = InternalGetPassword(ep_alias, passwd);
#endif // HAS_WLDAP
			//if a password is not found: senderID == endpointIdentifier?
			if (passwd.IsEmpty()){
			 	//get endpoint by endpointIdentifier
				H225_EndpointIdentifier ep_id;
				ep_id = clearToken.m_sendersID;
				endptr ep = RegistrationTable::Instance()->FindByEndpointId(ep_id);
				if(!ep){
					return false;
				}
				//check all endpoint aliases for a password
				H225_ArrayOf_AliasAddress ep_aliases = ep->GetAliases();
				for (PINDEX i = 0; i < ep_aliases.GetSize(); i++){
					ep_alias = H323GetAliasAddressString(ep_aliases[i]);
					cached = InternalGetPassword(ep_alias, passwd);
					if (!passwd)
						break;
				}
			}
			H235AuthProcedure1 authProcedure1;
			authProcedure1.SetLocalId(gk_id);
			authProcedure1.SetPassword(passwd);
			if (authProcedure1.ValidateCryptoToken(tokens[i], getLastReceivedRawPDU()) == H235Authenticator::e_OK) {
				PTRACE(4, "GkAuth\t" << ep_alias << " password match (SHA-1)");
				if (!cached)
					m_cache->Save(ep_alias, passwd);
				return true;
			}
#endif

		}
	}
	return false;
}


#if HAS_MYSQL

// MySQLPasswordAuth
MySQLPasswordAuth::MySQLPasswordAuth(
	PConfig* cfg, 
	const char* authName
	)
	: SimplePasswordAuth(cfg, authName), m_sqlConn(NULL)
{
	GkSQLCreator* driverCreator = GkSQLCreator::FindDriver("MySQL");
	if (driverCreator == NULL) {
		PTRACE(1, GetName() << "\tModule creation failed: could not find driver for MySQL database");
		return;
	}
	
	if (m_cache)
		m_cache->SetTimeout(cfg->GetInteger("MySQLAuth", "CacheTimeout", -1));
		
	const PString password = cfg->GetString("MySQLAuth", "Password", "");
	
	PString host = cfg->GetString("MySQLAuth", "Host", "localhost");
	if (cfg->GetInteger("MySQLAuth", "Port", -1) > 0)
		host += ":" + PString(cfg->GetInteger("MySQLAuth", "Port", MYSQL_PORT));
		
	m_sqlConn = driverCreator->CreateConnection(authName);
	if (!m_sqlConn->Initialize(
			host,
			cfg->GetString("MySQLAuth", "Database", "mysql"),
			cfg->GetString("MySQLAuth", "User", "mysql"),
			password.IsEmpty() ? (const char*)NULL : (const char*)password,
			1, 1)) {
		delete m_sqlConn;
		m_sqlConn = NULL;
		PTRACE(2, GetName() << "\tModule creation failed: could not connect to the database");
		return;
	}
	
	const PString table = cfg->GetString("MySQLAuth", "Table", "");
	const PString passwordField = cfg->GetString("MySQLAuth", "PasswordField", "");
	const PString aliasField = cfg->GetString("MySQLAuth", "IDField", "");
	
	if (table.IsEmpty() || passwordField.IsEmpty() || aliasField.IsEmpty()) {
		PTRACE(1, GetName() << "\tCannot build query: Table, PasswordField or IDField not specified");
		return;
	}
	
	m_query = "SELECT " + passwordField + " FROM " + table + " WHERE " 
		+ aliasField + " = '%1'";
		
	const PString extraCrit = cfg->GetString("MySQLAuth", "ExtraCriterion", "");
	if (!extraCrit.IsEmpty())
		m_query += " AND " + extraCrit;
		
	PTRACE(4, GetName() << "\tConfigured query: " << m_query);
}

MySQLPasswordAuth::~MySQLPasswordAuth()
{
	delete m_sqlConn;
}

PString MySQLPasswordAuth::GetPassword(
	const PString& alias
	)
{
	PString password;
	
	if (m_sqlConn == NULL) {
		PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
			<< "' failed: SQL connection not active"
			);
		return password;
	}
	
	if (m_query.IsEmpty()) {
		PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
			<< "' failed: Query string not configured"
			);
		return password;
	}
	
	PStringArray params;
	params += alias;
	GkSQLResult* result = m_sqlConn->ExecuteQuery(m_query, &params);
	if (result == NULL)
		PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
			<< "' failed: Timeout or fatal error"
			);
	else {
		if (result->IsValid()) {
			PStringArray fields;
			if (result->GetNumRows() < 1)
				PTRACE(4, GetName() << "\tPassword not found for alias '" << alias << '\'');
			else if (result->GetNumFields() < 1)
				PTRACE(4, GetName() << "\tBad-formed query: no columns found in the result set");
			else if ((!result->FetchRow(fields)) || fields.GetSize() < 1)
				PTRACE(4, GetName() << "\tPassword query for alias '" << alias 
					<< "' failed: could not fetch the result row"
					);
			else
				password = fields[0];
		} else
			PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
				<< "' failed (" << result->GetErrorCode() << "): "
				<< result->GetErrorMessage()
				);
		delete result;
	}
	
	return password;
}

// MySQLAliasAuth
MySQLAliasAuth::MySQLAliasAuth(
	PConfig* cfg, 
	const char* authName
	)
	: AliasAuth(cfg, authName), m_sqlConn(NULL), m_cache(NULL)
{
	GkSQLCreator* driverCreator = GkSQLCreator::FindDriver("MySQL");
	if (driverCreator == NULL) {
		PTRACE(1, GetName() << "\tModule creation failed: could not find driver for MySQL database");
		return;
	}
	
	m_cache = new CacheManager(cfg->GetInteger(authName, "CacheTimeout", -1));
	
	const PString password = cfg->GetString(authName, "Password", "");
	
	PString host = cfg->GetString(authName, "Host", "localhost");
	if (cfg->GetInteger(authName, "Port", -1) > 0)
		host += ":" + PString(cfg->GetInteger(authName, "Port", MYSQL_PORT));
		
	m_sqlConn = driverCreator->CreateConnection(authName);
	if (!m_sqlConn->Initialize(
			host,
			cfg->GetString(authName, "Database", "mysql"),
			cfg->GetString(authName, "User", "mysql"),
			password.IsEmpty() ? (const char*)NULL : (const char*)password,
			1, 1)) {
		delete m_sqlConn;
		m_sqlConn = NULL;
		PTRACE(2, GetName() << "\tModule creation failed: could not connect to the database");
		return;
	}
	
	const PString table = cfg->GetString(authName, "Table", "");
	const PString ipField = cfg->GetString(authName, "IPField", "");
	const PString aliasField = cfg->GetString(authName, "IDField", "");
	
	if (table.IsEmpty() || ipField.IsEmpty() || aliasField.IsEmpty()) {
		PTRACE(1, GetName() << "\tCannot build query: Table, IPField or IDField not specified");
		return;
	}
	
	m_query = "SELECT " + ipField + " FROM " + table + " WHERE " 
		+ aliasField + " = '%1'";
		
	const PString extraCrit = cfg->GetString(authName, "ExtraCriterion", "");
	if (!extraCrit.IsEmpty())
		m_query += " AND " + extraCrit;
		
	PTRACE(4, GetName() << "\tConfigured query: " << m_query);
}

MySQLAliasAuth::~MySQLAliasAuth()
{
	delete m_cache;
	delete m_sqlConn;
}

PString MySQLAliasAuth::GetConfigString(
	const PString& alias
	)
{
	PString authCondition;
	
	if (m_sqlConn == NULL) {
		PTRACE(2, GetName() << "\tQuery for alias '" << alias 
			<< "' failed: SQL connection not active"
			);
		return authCondition;
	}
	
	if (m_query.IsEmpty()) {
		PTRACE(2, GetName() << "\tQuery for alias '" << alias 
			<< "' failed: Query string not configured"
			);
		return authCondition;
	}
	
	if (m_cache == NULL || m_cache->Retrieve(alias, authCondition))
		return authCondition;
		
	PStringArray params;
	params += alias;
	GkSQLResult* result = m_sqlConn->ExecuteQuery(m_query, &params);
	if (result == NULL)
		PTRACE(2, GetName() << "\tQuery for alias '" << alias 
			<< "' failed: Timeout or fatal error"
			);
	else {
		if (result->IsValid()) {
			PStringArray fields;
			if (result->GetNumRows() < 1)
				PTRACE(4, GetName() << "\tEntry not found for alias '" << alias << '\'');
			else if (result->GetNumFields() < 1)
				PTRACE(4, GetName() << "\tBad-formed query: no columns found in the result set");
			else if ((!result->FetchRow(fields)) || fields.GetSize() < 1)
				PTRACE(4, GetName() << "\tQuery for alias '" << alias 
					<< "' failed: could not fetch the result row"
					);
			else {
				authCondition = fields[0];
				m_cache->Save(alias, authCondition);
			}
		} else
			PTRACE(2, GetName() << "\tQuery for alias '" << alias 
				<< "' failed (" << result->GetErrorCode() << "): "
				<< result->GetErrorMessage()
				);
		delete result;
	}
	return authCondition;
}

#endif // HAS_MYSQL

SQLPasswordAuth::SQLPasswordAuth(
	PConfig* cfg, 
	const char* authName
	)
	: SimplePasswordAuth(cfg, authName), m_sqlConn(NULL)
{
	const PString driverName = cfg->GetString(authName, "Driver", "");
	if (driverName.IsEmpty()) {
		PTRACE(1, GetName() << "\tModule creation failed: no SQL driver selected");
		return;
	}
	
	GkSQLCreator* driverCreator = GkSQLCreator::FindDriver(driverName);
	if (driverCreator == NULL) {
		PTRACE(1, GetName() << "\tModule creation failed: could not find " 
			<< driverName << " database driver"
			);
		return;
	}

	if (m_cache)
		m_cache->SetTimeout(cfg->GetInteger(authName, "CacheTimeout", 0));
		
	m_sqlConn = driverCreator->CreateConnection(authName);
	if (!m_sqlConn->Initialize(cfg, authName)) {
		delete m_sqlConn;
		m_sqlConn = NULL;
		PTRACE(2, GetName() << "\tModule creation failed: could not connect to the database");
		return;
	}
	
	m_query = cfg->GetString(authName, "Query", "");
	if (m_query.IsEmpty())
		PTRACE(1, GetName() << "\tModule creation failed: no query configured");
	else
		PTRACE(4, GetName() << "\tConfigured query: " << m_query);
}

SQLPasswordAuth::~SQLPasswordAuth()
{
	delete m_sqlConn;
}

PString SQLPasswordAuth::GetPassword(
	const PString& alias
	)
{
	PString password;
	
	if (m_sqlConn == NULL) {
		PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
			<< "' failed: SQL connection not active"
			);
		return password;
	}
	
	if (m_query.IsEmpty()) {
		PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
			<< "' failed: Query string not configured"
			);
		return password;
	}
	
	PStringArray params;
	params += alias;
	params += Toolkit::GKName();
	GkSQLResult* result = m_sqlConn->ExecuteQuery(m_query, &params);
	if (result == NULL)
		PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
			<< "' failed: Timeout or fatal error"
			);
	else {
		if (result->IsValid()) {
			PStringArray fields;
			if (result->GetNumRows() < 1)
				PTRACE(4, GetName() << "\tPassword not found for alias '" << alias << '\'');
			else if (result->GetNumFields() < 1)
				PTRACE(4, GetName() << "\tBad-formed query: no columns found in the result set");
			else if ((!result->FetchRow(fields)) || fields.GetSize() < 1)
				PTRACE(4, GetName() << "\tPassword query for alias '" << alias 
					<< "' failed: could not fetch the result row"
					);
			else
				password = fields[0];
		} else
			PTRACE(2, GetName() << "\tPassword query for alias '" << alias 
				<< "' failed (" << result->GetErrorCode() << "): "
				<< result->GetErrorMessage()
				);
		delete result;
	}
	
	return password;
}

SQLAliasAuth::SQLAliasAuth(
	PConfig* cfg, 
	const char* authName
	)
	: AliasAuth(cfg, authName), m_sqlConn(NULL), m_cache(NULL)
{
	const PString driverName = cfg->GetString(authName, "Driver", "");
	if (driverName.IsEmpty()) {
		PTRACE(1, GetName() << "\tModule creation failed: no SQL driver selected");
		return;
	}
	
	GkSQLCreator* driverCreator = GkSQLCreator::FindDriver(driverName);
	if (driverCreator == NULL) {
		PTRACE(1, GetName() << "\tModule creation failed: could not find " 
			<< driverName << " database driver"
			);
		return;
	}

	m_cache = new CacheManager(config->GetInteger(authName, "CacheTimeout", 0));
	
	m_sqlConn = driverCreator->CreateConnection(authName);
	if (!m_sqlConn->Initialize(cfg, authName)) {
		delete m_sqlConn;
		m_sqlConn = NULL;
		PTRACE(2, GetName() << "\tModule creation failed: could not connect to the database");
		return;
	}
	
	m_query = cfg->GetString(authName, "Query", "");
	if (m_query.IsEmpty())
		PTRACE(1, GetName() << "\tModule creation failed: no query configured");
	else
		PTRACE(4, GetName() << "\tConfigured query: " << m_query);
}

SQLAliasAuth::~SQLAliasAuth()
{
	delete m_sqlConn;
	delete m_cache;
}

PString SQLAliasAuth::GetConfigString(
	const PString& alias
	)
{
	PString authCondition;
	
	if (m_sqlConn == NULL) {
		PTRACE(2, GetName() << "\tQuery for alias '" << alias 
			<< "' failed: SQL connection not active"
			);
		return authCondition;
	}
	
	if (m_query.IsEmpty()) {
		PTRACE(2, GetName() << "\tQuery for alias '" << alias 
			<< "' failed: Query string not configured"
			);
		return authCondition;
	}

	if (m_cache == NULL || m_cache->Retrieve(alias, authCondition))
		return authCondition;
	
	PStringArray params;
	params += alias;
	params += Toolkit::GKName();
	GkSQLResult* result = m_sqlConn->ExecuteQuery(m_query, &params);
	if (result == NULL)
		PTRACE(2, GetName() << "\tQuery for alias '" << alias 
			<< "' failed: Timeout or fatal error"
			);
	else {
		if (result->IsValid()) {
			PStringArray fields;
			if (result->GetNumRows() < 1)
				PTRACE(4, GetName() << "\tEntry not found for alias '" << alias << '\'');
			else if (result->GetNumFields() < 1)
				PTRACE(4, GetName() << "\tBad-formed query: no columns found in the result set");
			else if ((!result->FetchRow(fields)) || fields.GetSize() < 1)
				PTRACE(4, GetName() << "\tQuery for alias '" << alias 
					<< "' failed: could not fetch the result row"
					);
			else {
				authCondition = fields[0];
				m_cache->Save(alias, authCondition);
			}
		} else
			PTRACE(2, GetName() << "\tQuery for alias '" << alias 
				<< "' failed (" << result->GetErrorCode() << "): "
				<< result->GetErrorMessage()
				);
		delete result;
	}
	return authCondition;
}

#if ((defined(__GNUC__) && __GNUC__ <= 2) && !defined(WIN32))
// ExternalPasswordAuth

ExternalPasswordAuth::ExternalPasswordAuth(PConfig * cfg, const char * authName)
      : SimplePasswordAuth(cfg, authName)
{
	ExternalInit();
}

bool ExternalPasswordAuth::ExternalInit()
{
	const char *ExternalSec = "ExternalAuth";

	// Read the configuration
	Program = config->GetString(ExternalSec, "PasswordProgram", "");
	
	return true;
}

PString ExternalPasswordAuth::GetPassword(const PString & id)
{
	const int BUFFSIZE = 256;
	char buff[BUFFSIZE] = "";
	if (!Program) {
		procbuf proc(Program + " " + id, ios::in);
		istream istr(&proc);
		istr.getline(buff, BUFFSIZE);
	} else {
		PTRACE(2, "GkAuth\tProgram is not defined");
	}
	return PString(buff);
}

#endif // class ExternalPasswordAuth

// LDAP authentification
#if defined(HAS_LDAP)

LDAPPasswordAuth::LDAPPasswordAuth(PConfig * cfg, const char * authName)
  : SimplePasswordAuth(cfg, authName)
{
}

LDAPPasswordAuth::~LDAPPasswordAuth()
{
}

PString LDAPPasswordAuth::GetPassword(const PString & alias)
{
	PStringList attr_values;
	using namespace lctn; // LDAP config tags and names
	// get pointer to new answer object
	if(GkLDAP::Instance()->getAttribute(alias, H235PassWord, attr_values) && 
	   !attr_values.IsEmpty()){
		return attr_values[0];
	}
	return "";
}  

int LDAPPasswordAuth::Check(H225_RegistrationRequest & rrq, unsigned & reason)
{
	int result = SimplePasswordAuth::Check(rrq, reason);
	if(result == e_ok) {
		// check if all aliases in RRQ exists in LDAP entry
		const H225_ArrayOf_AliasAddress & aliases = rrq.m_terminalAlias;
		if(!GkLDAP::Instance()->validAliases(aliases)) {
			result = e_fail;
		}
	}
	return result;
}


LDAPAliasAuth::LDAPAliasAuth(PConfig *cfg, const char *authName) : AliasAuth(cfg, authName)
{
}

LDAPAliasAuth::~LDAPAliasAuth()
{
}

PString LDAPAliasAuth::GetConfigString(const PString &alias)
{
	PStringList attr_values;
	using namespace lctn; // LDAP config tags and names
	// get pointer to new answer object
	if (GkLDAP::Instance()->getAttribute(alias, IPAddress, attr_values) && (!attr_values.IsEmpty())) {
		PString ip = attr_values[0];
    		if(!ip.IsEmpty()){
      			PString port = GK_DEF_ENDPOINT_SIGNAL_PORT;    
			return "sigip:" + ip + ":" + port;
		}
	}
	return "";
}

int LDAPAliasAuth::Check(H225_RegistrationRequest & rrq, unsigned & reason)
{
	int result = AliasAuth::Check(rrq, reason);
	if(result == e_ok) {
		// check if all aliases in RRQ exists in LDAP entry
		const H225_ArrayOf_AliasAddress & aliases = rrq.m_terminalAlias;
		if(!GkLDAP::Instance()->validAliases(aliases)) {
      			result = e_fail;
    		}
	}
  	return result;
}

#endif // HAS_LDAP

#if defined HAS_WLDAP

LDAPPasswordAuth::LDAPPasswordAuth(PConfig * cfg, const char * authName) : SimplePasswordAuth(cfg, authName)
{
}

LDAPPasswordAuth::~LDAPPasswordAuth()
{
}

PString LDAPPasswordAuth::GetPassword(const PString & alias) // called by SimplePasswordAuth::InternalGetPassword()
{ 
	LDAPAnswer * answer = new LDAPAnswer;
	bool CheckAliases = false;
	bool CheckStatus = false;
	PStringArray attributes;

	// attributes to retrieve
	using namespace lctn;
	attributes += LDAPAttrTags[H235PassWord];
	if (CheckStatus = (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapPasswAuthCheckAccountStatus", "0"))))
		attributes += LDAPAttrTags[AccountStatus];
	if (CheckAliases = (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapAliasAuthCheckAliases", "0")))) {
		attributes += LDAPAttrTags[H323ID];
		attributes += LDAPAttrTags[TelephonNo];
	}
	attribute_count = GkLDAP::Instance()->getAttributes(alias, attributes, answer);

	// don't bother to get attr values
	// if attribute_count is zero
	if(attribute_count) {
		// extract attribute values
		PStringList attrib_values;
		int attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, H235PassWord, attrib_values));
		if(!attrib_values.IsEmpty())
			epH235Password_value = attrib_values[0];

		if(CheckStatus) {
			attrib_values.RemoveAll();
			attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, AccountStatus, attrib_values));
			if(!attrib_values.IsEmpty())
				epAccountStatus_value = attrib_values[0];
		}

		if(CheckAliases) {
			attrib_values.RemoveAll();
			attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, H323ID, attrib_values));
			if(!attrib_values.IsEmpty())
				epH323ID_value = attrib_values[0];

			attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, TelephonNo, epTelephoneNo_value));
		}
	}
	delete answer;

	// success
	if(attribute_count && !epH235Password_value.IsEmpty())
		return epH235Password_value;

	// failed
	return "";
}  

int LDAPPasswordAuth::Check(H225_RegistrationRequest & rrq, unsigned & reason)
{
	// reset data class members
	attribute_count = 0;
	epAccountStatus_value = "";
	epH323ID_value = "";
	epTelephoneNo_value.RemoveAll();
	epH235Password_value = "";

	// reqired, sets global aliases
	checkid = true;

	// get the password for alias
	int result = SimplePasswordAuth::Check(rrq, reason);

	// success - do further checks
	if(result == e_ok && attribute_count > 0) {
		// check account status
		// Note: empty authentication attribute values indicate the
		//       password was cached, so the check below would fail!
		if (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapPasswAuthCheckAccountStatus", "0"))) {
			if(!epAccountStatus_value.IsEmpty()) {
				const PString cfgString = epAccountStatus_value;
				PTRACE(4, "Gk\tLDAPPasswordAuth::Check() '" << cfgString << "' " << epH323ID_value);
				if (!cfgString) {
					if (doCheckAccountStatus(rrq.m_callSignalAddress, cfgString)) {
						result = e_ok;
					} 
					else {
						PTRACE(4, "Gk\tLDAPAliasAuth::Check() condition '" << cfgString << "' rejected endpoint " << epH323ID_value);
						return e_fail;
					}
				}
			}
		} // check account status
	
		if (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapPasswAuthCheckAliases", "0"))) {
			// check if all aliases in RRQ exists in LDAP entry only if
			// autentication attribute values are valid. see note below
			// Note: empty authentication attribute values indicate the
			//       password was cached, so the check below would fail!
			if(!epH323ID_value.IsEmpty() &&
			   !epTelephoneNo_value.IsEmpty()) {
				const H225_ArrayOf_AliasAddress & aliases = rrq.m_terminalAlias;
				if(!GkLDAP::Instance()->validateAliases(aliases, epH323ID_value, epTelephoneNo_value)) {
					PTRACE(4, "Gk\tLDAPPasswordAuth::Check() validateAliases() failed");
					return e_fail;
				}
				PTRACE(4, "Gk\tLDAPPasswordAuth::Check() validateAliases() success");
			}
		}
	} // result == e_ok & attribute_count
	
	return result;
}

bool LDAPPasswordAuth::doCheckAccountStatus(const H225_ArrayOf_TransportAddress & addrs, const PString & cfgString)
{
	const PStringArray conditions(cfgString.Tokenise("&|", FALSE));
	for (PINDEX i = 0; i < conditions.GetSize(); ++i) {
		for (PINDEX j = 0; j < addrs.GetSize(); ++j) {
			if (AuthCondition(addrs[j], conditions[i])) {
				PTRACE(4, "Gk\tLDAPPasswordAuth::doCheckAccountStatus() '" << conditions[i] << "' applied successfully for endpoint " << AsDotString(addrs[j]));
				return true;
			}
		}
	}
	return false;
}

bool LDAPPasswordAuth::AuthCondition(const H225_TransportAddress & SignalAdr, const PString & Condition)
{
	const bool ON_ERROR = false; // return value on parse error in condition

	const PStringArray rule = Condition.Tokenise(":", FALSE);
	if (rule.GetSize() < 1) {
		PTRACE(1, "LDAPPasswordAuth::AuthCondition() Errornous RRQAuth rule: " << Condition);
		return ON_ERROR;
	}
	
	// 
	// condition = rule[0]:rule[1]... = rName:params...
	//
	
	const PString &rName = rule[0];

 	if (rName=="confirm" || rName=="allow") {
 		return true;
 	}
 	else if (rName=="reject" || rName=="deny" || rName=="forbid") {
 		return false;
 	}
	//
	// condition 'sigaddr' example:
	//   sigaddr:.*ipAddress .* ip = .* c3 47 e2 a2 .*port = 1720.*
	//
	else if (rName=="sigaddr") {
		if(rule.GetSize() < 2)
			return false;
		return Toolkit::MatchRegex(AsString(SignalAdr), rule[1]) != 0;
	}
	//
	// condition 'sigip' example:
	//   sigip:195.71.129.69:1720
	//
	else if (rName=="sigip") {
		if (rule.GetSize() < 2)
			return false;
		PIPSocket::Address ip;
		PIPSocket::GetHostAddress(rule[1], ip);
		WORD port = (rule.GetSize() < 3) ? GK_DEF_ENDPOINT_SIGNAL_PORT : (WORD)rule[2].AsInteger();
		return (SignalAdr == SocketToH225TransportAddr(ip, port));
	} else {
		PTRACE(4, "Gk\tLDAPPasswordAuth::AuthCondition() Unknown RRQAuth condition: " << Condition);
		return ON_ERROR;
	}

	// not reached...
	return false;
}

LDAPAliasAuth::LDAPAliasAuth(PConfig *cfg, const char *authName) : AliasAuth(cfg, authName)
{
}

LDAPAliasAuth::~LDAPAliasAuth()
{
}

int LDAPAliasAuth::Check(H225_RegistrationRequest & rrq, unsigned & reason)
{
	int result;
	bool AliasFoundInLDAP = false;

	if (!rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
		return defaultStatus;

	const H225_ArrayOf_AliasAddress & NewAliases = rrq.m_terminalAlias;
	PStringList AliasList;

	// alias search presedence e164/h323id
	if (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapAliasAuthSearchPrecedenceE164", "0"))) {
		PString h323id;
		for (PINDEX j = 0; j < NewAliases.GetSize(); ++j) {
			if (NewAliases[j].GetTag() == H225_AliasAddress::e_dialedDigits)
				AliasList.AppendString(H323GetAliasAddressString(NewAliases[j]));
			else if (NewAliases[j].GetTag() == H225_AliasAddress::e_h323_ID)
				h323id = H323GetAliasAddressString(NewAliases[j]);
		}
		AliasList.AppendString(h323id);
	}
	else {
		for (PINDEX j = 0; j < NewAliases.GetSize(); ++j) {
			AliasList.AppendString(H323GetAliasAddressString(NewAliases[j]));
		}
	}

	// step through alias list and search LDAP
	for (PINDEX i = 0; i < AliasList.GetSize(); ++i) {
		// attributes values
		int attribute_count;
		PString epAccountStatus_value;
		PString epH323ID_value;
		PStringList epTelephoneNo_value;

		LDAPAnswer * answer = new LDAPAnswer;
		bool CheckAliases = false;
		PStringArray attributes;

		// attributes to return
		using namespace lctn;
		attributes += LDAPAttrTags[AccountStatus];
		if (CheckAliases = (Toolkit::AsBool(GkConfig()->GetString(ldap_auth_sec, "LdapAliasAuthCheckAliases", "0")))) {
			attributes += LDAPAttrTags[H323ID];
			attributes += LDAPAttrTags[TelephonNo];
		}
		attribute_count = GkLDAP::Instance()->getAttributes(AliasList[i], attributes, answer);

		// don't bother to get attr values
		// if attribute_count is zero
		if(attribute_count) {
			// extract attribute values
			PStringList attrib_values;
			int attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, AccountStatus, attrib_values));
			if(!attrib_values.IsEmpty())
				epAccountStatus_value = attrib_values[0];

			if(CheckAliases) {
				attrib_values.RemoveAll();
				attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, H323ID, attrib_values));
				if(!attrib_values.IsEmpty())
					epH323ID_value = attrib_values[0];

				attrvalues_num = (GkLDAP::Instance()->getAttributeValues(answer, TelephonNo, epTelephoneNo_value));
			}
		}
		delete answer;

		// LDAP success
		if(attribute_count > 0 && !epAccountStatus_value.IsEmpty()) {
			const PString cfgString = epAccountStatus_value;
			PTRACE(5, "Gk\tLDAPAliasAuth::Check() '" << cfgString << "' " << AliasList[i]);
			if (!cfgString) {
				if (doCheck(rrq.m_callSignalAddress, cfgString)) {
					AliasFoundInLDAP = true;
					result = e_ok;

					// check if all aliases in RRQ exists in LDAP entry = second step
					if(CheckAliases) {
						if(!epH323ID_value.IsEmpty() &&
						   !epTelephoneNo_value.IsEmpty()) {
							const H225_ArrayOf_AliasAddress & aliases = rrq.m_terminalAlias;
							if(!GkLDAP::Instance()->validateAliases(aliases, epH323ID_value, epTelephoneNo_value)) {
								PTRACE(4, "Gk\tLDAPAliasAuth::Check() validateAliases() failed");
								return e_fail;
							}
							PTRACE(4, "Gk\tLDAPAliasAuth::Check() validateAliases() success");
						}
						else {
							PTRACE(4, "Gk\tLDAPAliasAuth::Check() one or more aliases missing");
							return e_fail;
						}
					}
					break;
				} 
				else {
					PTRACE(4, "Gk\tLDAPAliasAuth::Check() condition '" << cfgString << "' rejected endpoint " << AliasList[i]);
					return e_fail;
				}
			}
		} // if 
	} // for
	
	// return success/failure 
	result = (AliasFoundInLDAP) ? e_ok : defaultStatus;
	return result;
}
#endif // HAS_WLDAP

// AliasAuth
AliasAuth::AliasAuth(PConfig *cfg, const char *authName) : GkAuthenticator(cfg, authName)
{
}

int AliasAuth::Check(const H225_GatekeeperRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(H225_RegistrationRequest & rrq, unsigned &)
{
	bool AliasFoundInConfig = false;

	if (!rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
		return defaultStatus;

	const H225_ArrayOf_AliasAddress & NewAliases = rrq.m_terminalAlias;

	// alias is the config file entry of this endpoint
	for (PINDEX i = 0; i <= NewAliases.GetSize(); ++i) {
		const PString alias = (i < NewAliases.GetSize()) ? AsString(NewAliases[i], FALSE) : PString("default");
		const PString cfgString = GetConfigString(alias);
		if (!cfgString) {
			if (doCheck(rrq.m_callSignalAddress, cfgString)) {
				AliasFoundInConfig = true;
				break;
			} else {
				PTRACE(4, "Gk\tRRQAuth condition '" << cfgString << "' rejected endpoint " << alias);
				return e_fail;
			}
		}
	}
	return (AliasFoundInConfig) ? e_ok : defaultStatus;
}

int AliasAuth::Check(const H225_UnregistrationRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(H225_AdmissionRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_BandwidthRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_DisengageRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_LocationRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_InfoRequest &, unsigned &)
{
	return e_next;
}

PString AliasAuth::GetConfigString(const PString & alias)
{
	return config->GetString("RasSrv::RRQAuth", alias, "");
}

bool AliasAuth::doCheck(const H225_ArrayOf_TransportAddress & addrs, const PString & cfgString)
{
	const PStringArray conditions(cfgString.Tokenise("&|", FALSE));
	for (PINDEX i = 0; i < conditions.GetSize(); ++i) {
		for (PINDEX j = 0; j < addrs.GetSize(); ++j) {
			if (AuthCondition(addrs[j], conditions[i])) {
				PTRACE(4, "Gk\tRRQAuth condition '" << conditions[i] << "' applied successfully for endpoint " << AsDotString(addrs[j]));
				return true;
			}
		}
	}
	return false;
}

bool AliasAuth::AuthCondition(const H225_TransportAddress & SignalAdr, const PString & Condition)
{
	const bool ON_ERROR = false; // return value on parse error in condition

	const PStringArray rule = Condition.Tokenise(":", FALSE);
	if (rule.GetSize() < 1) {
		PTRACE(1, "Errornous RRQAuth rule: " << Condition);
		return ON_ERROR;
	}
	
	// 
	// condition = rule[0]:rule[1]... = rName:params...
	//
	
	const PString &rName = rule[0];

 	if (rName=="confirm" || rName=="allow") {
 		return true;
 	}
 	else if (rName=="reject" || rName=="deny" || rName=="forbid") {
 		return false;
 	}
	//
	// condition 'sigaddr' example:
	//   sigaddr:.*ipAddress .* ip = .* c3 47 e2 a2 .*port = 1720.*
	//
	else if (rName=="sigaddr") {
		if(rule.GetSize() < 2)
			return false;
		return Toolkit::MatchRegex(AsString(SignalAdr), rule[1]) != 0;
	}
	//
	// condition 'sigip' example:
	//   sigip:195.71.129.69:1720
	//
	else if (rName=="sigip") {
		if (rule.GetSize() < 2)
			return false;
		PIPSocket::Address ip;
		PIPSocket::GetHostAddress(rule[1], ip);
		WORD port = (rule.GetSize() < 3) ? GK_DEF_ENDPOINT_SIGNAL_PORT : (WORD)rule[2].AsInteger();
		return (SignalAdr == SocketToH225TransportAddr(ip, port));
	} else {
		PTRACE(4, "Unknown RRQAuth condition: " << Condition);
		return ON_ERROR;
	}

	// not reached...
	return false;
}

// Help classes for PrefixAuth
static const char* const prfflag="prf:";
static const char* const allowflag="allow";
static const char* const denyflag="deny";
static const char* const ipflag="ipv4:";
static const char* const aliasflag="alias:";

class AuthObj { // abstract class
public:
	virtual ~AuthObj() {}

	virtual bool IsValid() const { return true; }

	virtual PStringArray GetPrefixes() const = 0;

	virtual PIPSocket::Address GetIP() const = 0;
	virtual PString GetAliases() const = 0;
};

class RRQAuthObj : public AuthObj {
public:
	RRQAuthObj(const H225_RegistrationRequest & ras) : rrq(ras) {}

	virtual PStringArray GetPrefixes() const;

	virtual PIPSocket::Address GetIP() const;
	virtual PString GetAliases() const;

private:
	const H225_RegistrationRequest & rrq;
};

class ARQAuthObj : public AuthObj {
public:
	ARQAuthObj(const H225_AdmissionRequest & ras);

	virtual bool IsValid() const { return ep; }

	virtual PStringArray GetPrefixes() const;

	virtual PIPSocket::Address GetIP() const;
	virtual PString GetAliases() const;

private:
	const H225_AdmissionRequest & arq;
	endptr ep;
};

ARQAuthObj::ARQAuthObj(const H225_AdmissionRequest & ras) : arq(ras)
{
	ep = RegistrationTable::Instance()->FindByEndpointId(arq.m_endpointIdentifier);
}

PStringArray ARQAuthObj::GetPrefixes() const
{
	PStringArray array;
	if (arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo))
		if (PINDEX ss = arq.m_destinationInfo.GetSize() > 0) {
			array.SetSize(ss);
			for (PINDEX i = 0; i < ss; ++i)
				array[i] = AsString(arq.m_destinationInfo[i], FALSE);
		}
	if (array.GetSize() == 0)
		// let empty destinationInfo match the ALL rule
		array.SetSize(1);

	return array;
}

PIPSocket::Address ARQAuthObj::GetIP() const
{
	PIPSocket::Address result;
	const H225_TransportAddress & addr = (arq.HasOptionalField(H225_AdmissionRequest::e_srcCallSignalAddress)) ?
		arq.m_srcCallSignalAddress : ep->GetCallSignalAddress();
	GetIPFromTransportAddr(addr, result);
	return result;
}

PString ARQAuthObj::GetAliases() const
{
	return AsString(ep->GetAliases());
}

class LRQAuthObj : public AuthObj {
public:
	LRQAuthObj(const H225_LocationRequest & ras);

	virtual PStringArray GetPrefixes() const;

	virtual PIPSocket::Address GetIP() const;
	virtual PString GetAliases() const;

private:
	const H225_LocationRequest & lrq;
	PIPSocket::Address ipaddress;
};

LRQAuthObj::LRQAuthObj(const H225_LocationRequest & ras) : lrq(ras)
{
	GetIPFromTransportAddr(lrq.m_replyAddress, ipaddress);
}

PStringArray LRQAuthObj::GetPrefixes() const
{
	PStringArray array;
	if (PINDEX ss = lrq.m_destinationInfo.GetSize() > 0) {
		array.SetSize(ss);
		for (PINDEX i = 0; i < ss; ++i)
			array[i] = AsString(lrq.m_destinationInfo[i], FALSE);
	}
	return array;
}

PIPSocket::Address LRQAuthObj::GetIP() const
{
	return ipaddress;
}

PString LRQAuthObj::GetAliases() const
{
	return (lrq.HasOptionalField(H225_LocationRequest::e_sourceInfo)) ? AsString(lrq.m_sourceInfo) : PString();
}


class AuthRule {
public:
	enum Result {
		e_nomatch,
		e_allow,
		e_deny
	};

	AuthRule(Result f, bool r) : priority(1000), fate(f), inverted(r), next(0) {}
	virtual ~AuthRule() { delete next; }

	virtual bool Match(const AuthObj &) = 0;
	int Check(const AuthObj &);
	
	bool operator<(const AuthRule & o) const { return priority < o.priority; }
	void SetNext(AuthRule *n) { next = n; }

//	virtual PString GetName() const { return PString(); }

protected:
	int priority; // the lesser the value, the higher the priority

private:
	Result fate;
	bool inverted;
	AuthRule *next;
};

int AuthRule::Check(const AuthObj & aobj)
{
//	PTRACE(3, "auth\t" << GetName());
	return (Match(aobj) ^ inverted) ? fate : (next) ? next->Check(aobj) : e_nomatch;
}

inline void delete_rule(PrefixAuth::Rules::value_type r)
{
	delete r.second;
}

class NullRule : public AuthRule {
public:
	NullRule() : AuthRule(e_nomatch, false) {}
	virtual bool Match(const AuthObj &) { return false; }
};

class IPv4AuthRule : public AuthRule {
public:
	IPv4AuthRule(Result, const PString &, bool);

private:
	virtual bool Match(const AuthObj &);
//	virtual PString GetName() const { return network.AsString() + "/" + PString(PString::Unsigned, 32-priority); }

	PIPSocket::Address network, netmask;
};

IPv4AuthRule::IPv4AuthRule(Result f, const PString & cfg, bool r) : AuthRule(f, r)
{
	Toolkit::GetNetworkFromString(cfg, network, netmask);
	DWORD n = ~PIPSocket::Net2Host(DWORD(netmask));
	for (priority = 0; n; n >>= 1)
		++priority;
}

bool IPv4AuthRule::Match(const AuthObj & aobj)
{
	return ((aobj.GetIP() & netmask) == network);
}

class AliasAuthRule : public AuthRule {
public:
	AliasAuthRule(Result f, const PString & cfg, bool r) : AuthRule(f, r), pattern(cfg) { priority = -1; }

private:
	virtual bool Match(const AuthObj &);
//	virtual PString GetName() const { return pattern; }

	PString pattern;
};

bool AliasAuthRule::Match(const AuthObj & aobj)
{
	return (aobj.GetAliases().FindRegEx(pattern) != P_MAX_INDEX);
}

inline bool is_inverted(const PString & cfg, PINDEX p)
{
	return (p > 1) ? cfg[p-1] == '!' : false;
}

inline bool comp_authrule_priority(AuthRule *a1, AuthRule *a2)
{
	return *a1 < *a2;
}

// PrefixAuth
PrefixAuth::PrefixAuth(PConfig *cfg, const char *authName)
      : GkAuthenticator(cfg, authName)
{
	int ipfl = strlen(ipflag), aliasfl = strlen(aliasflag);
	PStringToString cfgs=cfg->GetAllKeyValues("PrefixAuth");
	for (PINDEX i = 0; i < cfgs.GetSize(); ++i) {
		PString key = cfgs.GetKeyAt(i);
		if (key *= "default") {
			defaultStatus = Toolkit::AsBool(cfgs.GetDataAt(i)) ? e_ok : e_fail;
			continue;
		} else if (key *= "ALL") {
			// use space (0x20) as the key so it will be the last resort
			key = " ";
		}
		if (prefrules.find(key) != prefrules.end())
			continue; //rule already exists? ignore

		PStringArray rules = cfgs.GetDataAt(i).Tokenise("|", FALSE);
		PINDEX sz = rules.GetSize();
		if (sz < 1)
			continue;
		//AuthRule *rls[sz];
		AuthRule **rls = new AuthRule *[sz];
		for (PINDEX j = 0; j < sz; ++j) {
			PINDEX pp;
			// if not allowed, assume denial
			AuthRule::Result ft = (rules[j].Find(allowflag) != P_MAX_INDEX) ? AuthRule::e_allow : AuthRule::e_deny;
			if ((pp=rules[j].Find(ipflag)) != P_MAX_INDEX)
				rls[j] = new IPv4AuthRule(ft, rules[j].Mid(pp+ipfl), is_inverted(rules[j], pp));
			else if ((pp=rules[j].Find(aliasflag)) != P_MAX_INDEX)
				rls[j] = new AliasAuthRule(ft, rules[j].Mid(pp+aliasfl), is_inverted(rules[j], pp));
			else
				rls[j] = new NullRule;
		}

		// sort the rules by priority
		stable_sort(rls, rls + sz, comp_authrule_priority);
		for (PINDEX k = 1; k < sz; ++k)
			rls[k-1]->SetNext(rls[k]);
		prefrules[key] = rls[0];
		delete [] rls;
	}
}

PrefixAuth::~PrefixAuth()
{
	for_each(prefrules.begin(), prefrules.end(), delete_rule);
}

int PrefixAuth::Check(const H225_GatekeeperRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(H225_RegistrationRequest & /*rrq*/, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_UnregistrationRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(H225_AdmissionRequest & arq, unsigned &)
{
	return CallTable::Instance()->FindCallRec(arq.m_callIdentifier) ? e_ok : doCheck(ARQAuthObj(arq));
}

int PrefixAuth::Check(const H225_BandwidthRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_DisengageRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_LocationRequest & lrq, unsigned &)
{
	return doCheck(LRQAuthObj(lrq));
}

int PrefixAuth::Check(const H225_InfoRequest &, unsigned &)
{
	return e_next;
}

struct comp_pref { // function object
	comp_pref(const PString & s) : value(s) {}
	bool operator()(const PrefixAuth::Rules::value_type & v) const;
	const PString & value;
};

inline bool comp_pref::operator()(const PrefixAuth::Rules::value_type & v) const
{
	return (value.Find(v.first) == 0) || (v.first *= " ");
}

int PrefixAuth::doCheck(const AuthObj & aobj)
{
	if (!aobj.IsValid())
		return e_fail;
	PStringArray ary(aobj.GetPrefixes());
	for (PINDEX i = 0; i < ary.GetSize(); ++i) {
		// find the first match rule
		// since prefrules is descendently sorted
		// it must be the most specific prefix
		for (Rules::iterator j = prefrules.begin(); j != prefrules.end(); ++j) {
			Rules::iterator iter = find_if(j, prefrules.end(), comp_pref(ary[i]));
			if (iter == prefrules.end())
				break;
			switch (iter->second->Check(aobj))
			{
				case AuthRule::e_allow:
					return e_ok;
				case AuthRule::e_deny:
					return e_fail;
				default: // try next prefix...
					j = iter;
			}
		}
	}
	return defaultStatus;
}

namespace {
std::list<GkAuthInitializer *> *AuthNameList;
}

GkAuthInitializer::GkAuthInitializer(const char *n) : name(n)
{
	static std::list<GkAuthInitializer *> aList;
	AuthNameList = &aList;

	AuthNameList->push_back(this);
}

GkAuthInitializer::~GkAuthInitializer()
{
}

bool GkAuthInitializer::Compare(PString n) const
{
	return n == name;
}

GkAuthenticatorList::GkAuthenticatorList(PConfig *cfg)
{
	PStringList authList(cfg->GetKeys(GkAuthSectionName));

	for (PINDEX i=authList.GetSize(); i-- > 0; ) {
		PString authName(authList[i]);
		std::list<GkAuthInitializer *>::iterator Iter =
			find_if(AuthNameList->begin(), AuthNameList->end(),
				bind2nd(mem_fun(&GkAuthInitializer::Compare), authName));
		if (Iter != AuthNameList->end())
			(*Iter)->CreateAuthenticator(cfg);
#if PTRACING
		else
			PTRACE(1, "GkAuth\tUnknown auth " << authName << ", ignore!");
#endif
	}
	
	m_mechanisms = new H225_ArrayOf_AuthenticationMechanism;
	m_algorithmOIDs = new H225_ArrayOf_PASN_ObjectId;
	GkAuthenticator* authenticator = GetHead();
	BOOL found = FALSE;
			
	// scan all authenticators that are either "required" or "sufficient"
	// (skip "optional") and fill #mechanisms# and #algorithmOIDs# arrays
	// with H.235 capabilities that are supported by all these authenticators
	while( authenticator )
	{
		if( authenticator->IsH235Capable() 
			&& ((authenticator->GetControlFlag() == GkAuthenticator::e_Required)
				|| (authenticator->GetControlFlag() == GkAuthenticator::e_Sufficient)) )
		{
			if( m_mechanisms->GetSize() == 0 )
			{
				// append H.235 capability to empty arrays
				authenticator->GetH235Capability(
					*m_mechanisms, *m_algorithmOIDs
					);
				// should never happen, but we should check just for a case				
				if( m_algorithmOIDs->GetSize() == 0 )
					m_mechanisms->RemoveAll();
				else
					found = TRUE;
				authenticator = authenticator->GetNext();
				continue;
			}
			
			// Already have H.235 capabilities - check the current
			// authenticator if it supports any of the capabilities.
			// Remove capabilities that are not supported
			
			H225_ArrayOf_AuthenticationMechanism matchedMechanisms;
			
			int i, j, k;
			
			for( i = 0; i < m_algorithmOIDs->GetSize(); i++ )
			{
				BOOL matched = FALSE;
			
				for( j = 0; j < m_mechanisms->GetSize(); j++ )
					if( authenticator->IsH235Capability(
							(*m_mechanisms)[j], (*m_algorithmOIDs)[i]
							) )
					{
						for( k = 0; k < matchedMechanisms.GetSize(); k++ )
							if( matchedMechanisms[k].GetTag() == (*m_mechanisms)[j].GetTag() )
								break;
						if( k == matchedMechanisms.GetSize() )
						{
							matchedMechanisms.SetSize(k+1);
							matchedMechanisms[k].SetTag((*m_mechanisms)[j].GetTag());
						}
						matched = TRUE;
					}
				
				if( !matched )
				{
					PTRACE(5,"GkAuth\tRemoved from GCF list algorithm OID: "<<(*m_algorithmOIDs)[i]);
					m_algorithmOIDs->RemoveAt(i);
					i--;
				}
			}
			
			for( i = 0; i < m_mechanisms->GetSize(); i++ )
			{
				for( j = 0; j < matchedMechanisms.GetSize(); j++ )
					if( (*m_mechanisms)[i].GetTag() == matchedMechanisms[j].GetTag() )
						break;
				if( j == matchedMechanisms.GetSize() )
				{
					PTRACE(5,"GkAuth\tRemoved from GCF list mechanism: "<<(*m_mechanisms)[i]);
					m_mechanisms->RemoveAt(i);
					i--;
				}
			}
			
			if( (m_mechanisms->GetSize() == 0) || (m_algorithmOIDs->GetSize() == 0) )
			{
				PTRACE(4,"GkAuth\tConflicting H.235 capabilities are active"
					<<" - GCF will not select any particular capability"
					);
				m_mechanisms->RemoveAll();
				m_algorithmOIDs->RemoveAll();
				break;
			}
		}
		authenticator = authenticator->GetNext();
	}

	// Scan "optional" authenticators if the above procedure has not found
	// any H.235 capabilities or has found more than one
	if( (!found) || (m_mechanisms->GetSize() > 1) || (m_algorithmOIDs->GetSize() > 1) )
	{
		authenticator = GetHead();
		while( authenticator )
		{
			if( authenticator->IsH235Capable() 
				&& (authenticator->GetControlFlag() == GkAuthenticator::e_Optional) )
			{
				if( m_mechanisms->GetSize() == 0 )
				{
					authenticator->GetH235Capability(
						(*m_mechanisms), (*m_algorithmOIDs)
						);
					if( m_algorithmOIDs->GetSize() == 0 )
						m_mechanisms->RemoveAll();
					else
						found = TRUE;
					authenticator = authenticator->GetNext();
					continue;
				}
			
				H225_ArrayOf_AuthenticationMechanism matchedMechanisms;
			
				int i, j, k;
			
				for( i = 0; i < m_algorithmOIDs->GetSize(); i++ )
				{
					BOOL matched = FALSE;
			
					for( j = 0; j < m_mechanisms->GetSize(); j++ )
						if( authenticator->IsH235Capability(
								(*m_mechanisms)[j], (*m_algorithmOIDs)[i]
								) )
						{
							for( k = 0; k < matchedMechanisms.GetSize(); k++ )
								if( matchedMechanisms[k].GetTag() == (*m_mechanisms)[j].GetTag() )
									break;
							if( k == matchedMechanisms.GetSize() )
							{
								matchedMechanisms.SetSize(k+1);
								matchedMechanisms[k].SetTag((*m_mechanisms)[j].GetTag());
							}
							matched = TRUE;
						}
				
					if( !matched )
					{
						PTRACE(5,"GkAuth\tRemoved from GCF list algorithm OID: "<<(*m_algorithmOIDs)[i]);
						m_algorithmOIDs->RemoveAt(i);
						i--;
					}
				}
			
				for( i = 0; i < m_mechanisms->GetSize(); i++ )
				{
					for( j = 0; j < matchedMechanisms.GetSize(); j++ )
						if( (*m_mechanisms)[i].GetTag() == matchedMechanisms[j].GetTag() )
							break;
					if( j == matchedMechanisms.GetSize() )
					{
						PTRACE(5,"GkAuth\tRemoved from GCF list mechanism: "<<(*m_mechanisms)[i]);
						m_mechanisms->RemoveAt(i);
						i--;
					}
				}
			
				if( (m_mechanisms->GetSize() == 0) || (m_algorithmOIDs->GetSize() == 0) )
				{
					PTRACE(4,"GkAuth\tConflicting H.235 capabilities are active"
						<<" - GCF will not select any particular capability"
						);
					m_mechanisms->RemoveAll();
					m_algorithmOIDs->RemoveAll();
					break;
				}
			}
			authenticator = authenticator->GetNext();
		}
	}
	
	if( PTrace::CanTrace(5) )
		if( (m_mechanisms->GetSize() > 0) && (m_algorithmOIDs->GetSize() > 0) )
		{
			ostream& strm = PTrace::Begin(5,__FILE__,__LINE__);
			strm <<"GkAuth\tH.235 capabilities selected for GCF:\n";
			strm <<"\tAuthentication mechanisms: \n";
			int i;
			for( i = 0; i < m_mechanisms->GetSize(); i++ )
				strm << "\t\t" << (*m_mechanisms)[i] << '\n';
			strm <<"\tAuthentication algorithm OIDs: \n";
			for( i = 0; i < m_algorithmOIDs->GetSize(); i++ )
				strm << "\t\t" << (*m_algorithmOIDs)[i] << '\n';
			PTrace::End(strm);
		}
}

GkAuthenticatorList::~GkAuthenticatorList()
{	
	delete GkAuthenticator::head;
	GkAuthenticator::head = 0;
	
	delete m_mechanisms;
	delete m_algorithmOIDs;
}

void GkAuthenticatorList::GetH235Capabilities(
	H225_ArrayOf_AuthenticationMechanism& mechanisms,
	H225_ArrayOf_PASN_ObjectId& algorithmOIDs
	) const
{
	mechanisms = *m_mechanisms;
	algorithmOIDs = *m_algorithmOIDs;
}

namespace {
GkAuthInit<GkAuthenticator> DefaultAuthFactory("default");
GkAuthInit<AliasAuth> AliasAuthFactory("AliasAuth");
GkAuthInit<PrefixAuth> PrefixAuthFactory("PrefixAuth");
GkAuthInit<SimplePasswordAuth> SimplePasswordAuthFactory("SimplePasswordAuth");
GkAuthInit<SQLPasswordAuth> SQLPasswordAuthFactory("SQLPasswordAuth");
GkAuthInit<SQLAliasAuth> SQLAliasAuthFactory("SQLAliasAuth");
#if HAS_MYSQL
GkAuthInit<MySQLPasswordAuth> MySQLPasswordAuthFactory("MySQLPasswordAuth");
GkAuthInit<MySQLAliasAuth> MySQLAliasAuthFactory("MySQLAliasAuth");
#endif
#if defined(HAS_LDAP) || defined(HAS_WLDAP)
GkAuthInit<LDAPPasswordAuth> LDAPPasswordAuthFactory("LDAPPasswordAuth");
GkAuthInit<LDAPAliasAuth> LDAPAliasAuthFactory("LDAPAliasAuth");
#endif
}
