/////////////////////////////////////////////////////////////////////////////
//
// gk_wldap_interface.h
//
// Copyright (C) 2003 Franz J Ehrengruber <franz@iptelenet.com>
//  
// PURPOSE OF THIS FILE: Define the non-opaque part to the MS LDAP-C-API
//                       Windows LDAP integration (wldap.h/wldap32.lib)
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  
//
// History:
//      2003/10/01      initial version (Franz J Ehrengruber)
//                      based on gk_ldap_interface.h (Nils Bokermann)
//
/////////////////////////////////////////////////////////////////////////////

/*
 * Non opaque part of the MS LDAP API
 */

#ifndef __GK_WLDAP_H
#define __GK_WLDAP_H "#(@) $Id: gk_wldap_interface.h,v 1.1.2.3 2004/05/12 17:46:40 zvision Exp $"

#ifdef HAS_WLDAP

#include <vector>
#include <ptlib.h>
#include <winldap.h> // !! DO NOT MOVE !!
#include <Winber.h>

//////////////////////////////
// Initialize an LDAP session 
//////////////////////////////

// the`LDAP session handle
typedef struct {
	LDAP * ld;
} GK_LDAP;

// The ldap_init function initializes a session with an LDAP server.
// Parameters
//		hostname 
//		[in] A domain name, or a space-separated list of host names 
//		or dotted strings representing the IP address of hosts running
//		an LDAP server to which to connect. Each host name in the list 
//		can include an optional port number which is separated from the 
//		host itself with a colon (:) character. See the Remarks below 
//		about the use of the LDAP_OPT_AREC_EXCLUSIVE option when connecting
//		to Active Directory servers. 
//	portno 
//		[in] Contains the TCP port number to which to connect. Set to LDAP_PORT
//		to obtain the default port, 389. This parameter is ignored if a host 
//		name includes a port number. 
//	Return Values
//		If the function succeeds, it returns a session handle, in the form of a
//		pointer to an LDAP data structure.
//		If the function fails, it returns NULL. Use LdapGetLastError to retrieve
//		the error code.
GK_LDAP *gk_ldap_init (const char *hostname, int portno);

// depricated   bind.cxx
// gk_ldap_open initializes the LDAP * session handle. As in @ref{ldap_init} it {\em must}
// be passed to all following calls. Unlike gk_ldap_init gk_ldap_open will open a connection to
// a gk_ldap server.
// @see gk_ldap_init, gk_ldap_bind
GK_LDAP *gk_ldap_open (const char *hostname, int portno);

////////////////////////////////////////////
// connecting to an LDAP server witout bind
////////////////////////////////////////////

// The ldap_connect function establishes a connection with the server
// Parameters
//	ld 
//		[in] The session handle. 
//	timeout 
//		[in] The number of seconds to spend in an attempt to establish
//		a connection before timing out. If NULL, the function uses a default
//		 timeout value. 
//	Return Values
//		If the function succeeds, the return value is LDAP_SUCCESS.
//		If the function fails, it returns an error code. See Return
//		Values for more information.
ULONG gk_ldap_connect(GK_LDAP *ld, struct l_timeval* timeout);

/////////////////////
// binding/unbinding
/////////////////////

// Bind with sasl mechanism
int gk_ldap_sasl_bind (GK_LDAP *ld, char const *dn, char const *mechanism,
		       struct berval const *cred, LDAPControl **serverctrls,
		       LDAPControl **clientctrls, int *msgidp);

/// Bind with sasl mechanism. Synchronous version.
int gk_ldap_sasl_bind_s (GK_LDAP *ld, char const *dn, char const *mechanism,
			 struct berval const *cred, LDAPControl **serverctrls,
			 LDAPControl **clientctrls, struct berval **servercredp);

/// Bind with simple pasword.
int gk_ldap_simple_bind (GK_LDAP *ld, char const *dn, char const *passwd);

/// Bind with simple pasword. Synchronous version.
int gk_ldap_simple_bind_s (GK_LDAP *ld,  char const *dn, char const *passwd);

// depricated
/// Old bind. For backward compatibility
int gk_ldap_bind (GK_LDAP *ld, char const *dn, char const *cred, int method);

/// Old bind. For backward compatibility. Synchronous version.
int gk_ldap_bind_s (GK_LDAP *ld,  char const *dn, char const *cred, int method);

/// Unbind Request. Send an unbind request and terminate the connection.
int gk_ldap_unbind (GK_LDAP *ld);

/// Unbind Request. Send an unbind request and terminate the connection.
int gk_ldap_unbind_s (GK_LDAP *ld);

///////////////////
// option handling
///////////////////

// Get an option from a connection. The user has to provide the right datatype
// and cast it to void.
int gk_ldap_get_option (GK_LDAP *ld, int option, void *outvalue);

// Set an option in a connection. The user has to provide the right datatype
// and cast it to void.
int gk_ldap_set_option (GK_LDAP *ld, int option, void const *outvalue);

//////////////////
// LDAP searching
//////////////////

//  Comlete search request. 
//    @param base the search base
//    @param scope the search scope
//    @param filter the search filter 
//    @param attrs array of attributes to be fetched or NULL
//    @param attrsonly 0 if values \em and attributes should be fetched, in any other
//    case only attributes will be fetched.
//    @param timeout Servertimeout
//    @param msgidp pointer to the messageID
int gk_ldap_search_ext (GK_LDAP *ld,  
						char const *base,
			            int scope,  
						char const *filter, 
						char **attrs, 
						int attrsonly, 
			            LDAPControl **serverctrls, 
						LDAPControl **clientctrls,
			            ULONG timeout, 
						ULONG sizelimit, 
						ULONG *msgidp);

//  Complete search request (synchronous). 
//    @param res the search results
//    @see gk_ldap_search_ext for details.
int gk_ldap_search_ext_s (GK_LDAP *ld,  
						  char const *base, 
						  int scope,  
						  char const *filter,
			              char **attrs, 
						  int attrsonly, 
						  LDAPControl 
						  **serverctrls,
			              LDAPControl **clientctrls, 
						  struct l_timeval *timeout,
			              ULONG sizelimit, 
						  LDAPMessage **res);

//  Simple search request. 
//    @param base the search base
//    @param scope the search scope
//    @param filter the search filter 
//    @param attrs array of attributes to be fetched or NULL
//    @param attrsonly 0 if values \em and attributes should be fetched, in any other
//    case only attributes will be fetched.
//    @return messageID
int gk_ldap_search (GK_LDAP *ld, char const *base, int scope, char const *filter, char **attrs,
		    int attrsonly);

//  Simple search request (synchronous). @see gk_ldap_search for details.
//    @param res the search results
//    @return LDAP_SUCCESS if search was successful, a GK_LDAP error otherwise.
int gk_ldap_search_s (GK_LDAP *ld, char const *base, int scope, char const *filter, char **attrs,
		      int attrsonly, LDAPMessage **res);

//  Simple search request with timeout.
//    @param timeout overall timeout to get the messages
//    @see gk_ldap_search_s
int gk_ldap_search_st (GK_LDAP *ld, 
					   char const *base, 
					   int scope, 
					   char const *filter, 
					   char **attrs,
		               int attrsonly, 
					   struct l_timeval *timeout, 
					   LDAPMessage **res);

//////////////////////////////////////////////
// comparing between local values and entries
/////////////////////////////////////////////

//  Compare. 
//    @param attr array of attributes to compare
//    @param bvalue array of bervals (values to compare)
//    @param msgidp pointer to the messageid
//    @return LDAP_SUCCESS on success. Error othterwise
int gk_ldap_compare_ext (GK_LDAP *ld, 
						 char const *dn, 
						 char const *attr,
						 char const *value,
			             struct berval *bvalue, 
						 LDAPControl **serverctrls,
			             LDAPControl **clientctrls, 
						 ULONG *msgidp);

//  Compare (synchronous).
//    @see gk_ldap_compare_ext for details.
//    @return LDAP_RESULT
int gk_ldap_compare_ext_s (GK_LDAP *ld, 
						   char const *dn, 
						   char const *attr,
						   char const *value,
			               struct berval *bvalue, 
						   LDAPControl **serverctrls,
			               LDAPControl **clientctrls);

//  Compare asynchronous. 
//    @param attr array of attributes to compare
//    @param bvalue array of values
//    @return MessageID
int gk_ldap_compare (GK_LDAP *ld, char const *dn, char const *attr, char *const value);

//  Compare synchronous
int gk_ldap_compare_s (GK_LDAP *ld, char const *dn, char const *attr, char *const value);

/////////////////////
// Modifying entries
/////////////////////

//  Complete modification Request.
//    @param mods Array of modifications
//    @param msgidp Pointer to the messageid
//    @return LDAP_SUCCESS on success. Error othterwise
int gk_ldap_modify_ext (GK_LDAP *ld, char const *dn, LDAPMod **mods, LDAPControl **serverctrls,
			LDAPControl **clientctrls, ULONG *msgidp);

//  Complete modification Request (synchronous).
//    @see gk_ldap_modify_ext for details
int gk_ldap_modify_ext_s (GK_LDAP *ld, char const *dn, LDAPMod **mods,
			  LDAPControl **serverctrls, LDAPControl **clientctrls );

// deprecated
/// Simple modification request.
int gk_ldap_modify (GK_LDAP *ld, char const *dn, LDAPMod **mods);

/// Simple modification request (synchronous).
int gk_ldap_modify_s (GK_LDAP *ld, char const *dn, LDAPMod **mods);

// Modifying names of entries
int gk_ldap_rename (GK_LDAP *ld, 
					char const *dn, 
					char const *newrdn, 
					char const *newSuperior, 
					int deleteoldrdn,
					LDAPControl **servercontrols, 
					LDAPControl **clientcontrols, 
					int *msgidp);

int gk_ldap_rename_s (GK_LDAP *ld, char const *dn, char const *newrdn,
		      char *newSuperior, int deleteoldrdn, LDAPControl **servercontrols,
		      LDAPControl **clientcontrols);

// deprecated with LDAPv3
//  Modify the relative distinguished name of an entry.
//    @param deleteoldrdn 0 if not to delete the old rdn.
//    @return MessageID
int gk_ldap_modrdn2 (GK_LDAP *ld, char const *dn, char const *newrdn, int deleteoldrdn);

//  Modify the relative distinguished name of an entry (synchronous).
//    @see gk_ldap_modrdn2 for details
//    @return LDAP_SUCCESS on success. Error otherwise
int gk_ldap_modrdn2_s (GK_LDAP *ld, char const *dn, char const *newrdn, int deleteoldrdn);

//  Modify the relative distinguished name of an entry.
//    The old RDN will be deleted.
//    @see gk_ldap_modrdn2
int gk_ldap_modrdn (GK_LDAP *ld, char const *dn, char const *newrdn);

//  Modify the relative distinguished name of an entry (synchronous).
//    The old RDN will be deleted.
//    @see gk_ldap_modrdn2
//    @return LDAP_SUCCESS on success. Error otherwise

int gk_ldap_modrdn_s (GK_LDAP *ld, char const *dn, char const *newrdn);

//////////////////////////
// Adding/Deleting entries
//////////////////////////

//  Add an entry.
//    @param attrs Array of attribute/values to add. The (*attr)->mod_op is ignored. 
//    Only the decision wether to take the berval or the char is done.
//    @param msgidp pointer to the MessageID
//    @return LDAP_SUCCESS on success. Error otherwise
int gk_ldap_add_ext (GK_LDAP *ld, char const *dn, LDAPMod **attrs,
		     LDAPControl **serverctrls, LDAPControl **clientctrls, ULONG *msgidp);

//  Add an entry (synchronous).
//    @see gk_ldap_add_ext for details.
int gk_ldap_add_ext_s (GK_LDAP *ld, char const *dn, LDAPMod **attrs,
		       LDAPControl **serverctrls, LDAPControl **clientctrls);

//   Add an entry.
//    @see gk_ldap_add_ext for details.
int gk_ldap_add (GK_LDAP *ld, char const *dn, LDAPMod **attrs);

//  Add an entry (synchronous).
//    @see gk_ldap_add_ext for details.
//
int gk_ldap_add_s (GK_LDAP *ld, char const *dn, LDAPMod **attrs );
//int gk_ldap_add_s (GK_LDAP *ld,  char *dn, LDAPMod **attrs );

//  Delete an entry.
//    @param msgidp pointer to the MessageID
//    @return LDAP_SUCCESS on success. Error otherwise
int gk_ldap_delete_ext (GK_LDAP *ld,  char const *dn, LDAPControl **serverctrls,
			LDAPControl **clientctrls, ULONG *msgidp);

//  Delete an entry (synchronous).
//    @see gk_ldap_delete_ext for details.
int gk_ldap_delete_ext_s (GK_LDAP *ld, char const *dn, LDAPControl **serverctrls,
			  LDAPControl **clientctrls);

//  Delete an entry.
//    @see gk_ldap_delete_ext for details.
int gk_ldap_delete (GK_LDAP *ld, char const *dn );

//  Delete an entry (synchronous).
//    @see gk_ldap_delete_ext for details.
int gk_ldap_delete_s (GK_LDAP *ld, char const *dn );

////////////////////////////
// Extended LDAP operations
////////////////////////////

int gk_ldap_extended_operation (GK_LDAP *ld, char const *reqoid,
				struct berval *reqdata, LDAPControl **serverctrls,
				LDAPControl **clientctrls, ULONG *msgidp);

int gk_ldap_extended_operation_s (GK_LDAP *ld, char const *reqoid,
				  struct berval *reqdata, LDAPControl **serverctrls,
				  LDAPControl **clientctrls, char **retoidp, struct berval **retdatap);

int gk_ldap_parse_extended_result (GK_LDAP *ld, LDAPMessage *res,
				   char **retoidp, struct berval **retdatap, int freeit);

/////////////////////////
// Abandoning operations
/////////////////////////

// Abandon.
int gk_ldap_abandon (GK_LDAP *ld, ULONG msgid);

//////////////////////////////
// Obtaining/Handling results
//////////////////////////////

//  Getting a LDAPMessage pointer.
//    @param msgid MessageID.
//    @param all 0 if not all messages should be fetched.
//    @param timeout overall timeout for the operation.
//    @param result pointer to LDAPMessage *
//    @return type of the message fetched.    
//
int gk_ldap_result (GK_LDAP *ld, ULONG msgid, int all, struct l_timeval  *timeout,
		    LDAPMessage **result);

// Handling results
//   Parse a LDAPMessage.
//    @param matcheddnp the matched DN
//    @param errmsg textual message the server returned.
//    @param freeit 0 if the LDAPMessage should not be freed. 
int gk_ldap_parse_result (GK_LDAP *ld, 
						  LDAPMessage *res, 
			              ULONG *errcodep, 
						  char **matcheddnp, 
						  char **errmsgp,
			              char ***referralsp, 
						  LDAPControl ***serverctrls,
			              int freeit);

/// Get the textual representation of a numeric error.
const char * gk_ldap_err2string (int err);

// deprecated
/// Get the error from a LDAPMessage.
int gk_ldap_result2error (GK_LDAP *ld, LDAPMessage *res, int freeit);

// deprecated
/// Print the textual error with a "string" in front.
void gk_ldap_perror (GK_LDAP *ld,  char const *s);

/// get the last LDAP error
int gk_LdapGetLastError(void);

///////////////////////////////////////////////////
// Stepping through lists of entries or references
///////////////////////////////////////////////////

/// Get first (LDAP) entry of chain.
LDAPMessage * gk_ldap_first_entry (GK_LDAP *ld, LDAPMessage *chain);

/// Get next (LDAP) entry of chain.
LDAPMessage * gk_ldap_next_entry (GK_LDAP *ld, LDAPMessage *entry);

/// Count (LDAP) entries of chain.
int gk_ldap_count_entries (GK_LDAP *ld, LDAPMessage *chain);

/// stepping through LDAP reference  
LDAPMessage * gk_ldap_first_reference (GK_LDAP *ld, LDAPMessage *chain);

/// stepping through LDAP reference  
LDAPMessage * gk_ldap_next_reference (GK_LDAP *ld, LDAPMessage *ref);

/// get the refernce count
int gk_ldap_count_references (GK_LDAP *ld, LDAPMessage *chain);

///////////////////////////////////
// Stepping through the attributes
///////////////////////////////////

/// Get the first attribute from a LDAPMessage.
char * gk_ldap_first_attribute (GK_LDAP *ld, LDAPMessage *entry, BerElement **ber);

/// Get the next attribute from a LDAPMessage.
char * gk_ldap_next_attribute (GK_LDAP *ld, LDAPMessage *entry, BerElement *ber);

/////////////////////////////////////////
// Retrieving the values of an attribute
/////////////////////////////////////////

/// Get all values for an attribute from a LDAPMessage (as strings).
char ** gk_ldap_get_values (GK_LDAP *ld, LDAPMessage *entry, char const *target);

/// Get all values for an attribute from a LDAPMessage (as bervals).
struct berval ** gk_ldap_get_values_len (GK_LDAP *ld, LDAPMessage *entry,
					 char const *target);

/// Count values returned by gk_ldap_get_values
int gk_ldap_count_values (char **vals);

/// Count values returned by gk_ldap_get_values_len
int gk_ldap_count_values_len (struct berval **vals);

/// get the DN of a LDAPMessage as a string.
char * gk_ldap_get_dn (GK_LDAP *ld, LDAPMessage *entry);

/// Get the user friendly notation of a DN.
char * gk_ldap_dn2ufn ( char const *dn);

/// Get all entries from a DN.
char ** gk_ldap_explode_dn (char const *dn, int notypes);

// Parsing references
int gk_ldap_parse_reference (GK_LDAP *ld, 
							 LDAPMessage *ref,
							 char ***referrals);

/////////////////
// freeing memory
/////////////////

//  name Freeing memory returned by the gk_ldap API
void gk_ldap_control_free (LDAPControl *ctrl);
void gk_ldap_controls_free (LDAPControl **ctrl);

/// Frees the structure LDAPMessage.
int gk_ldap_msgfree (LDAPMessage *lm);

/// Frees the char ** returned by gk_ldap_get_values.
int gk_ldap_value_free (char **vals);

/// Frees the struct berval ** returned by gk_ldap_get_values_len.
int gk_ldap_value_free_len (struct berval **vals);

/// Frees any char * returned by gk_ldap-API functions
void gk_ldap_memfree (char const *mem);

/// Frees BerElement structure
void gk_ldap_berfree(BerElement* pBerElement, int fbuf);

#endif // HAS_WLDAP

#endif // __GK_WLDAP_H
//
// End of gk_ldap_interface.h
//
