//////////////////////////////////////////////////////////////////
//
// ProxyThread.h
//
// Copyright (c) Citron Network Inc. 2001-2003
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// initial author: Chin-Wei Huang <cwhuang@linux.org.tw>
// initial version: 12/7/2001
//
//////////////////////////////////////////////////////////////////

#ifndef PROXYTHREAD_H
#define PROXYTHREAD_H "#(@) $Id: ProxyThread.h,v 1.7.2.30 2004/05/17 14:26:09 zvision Exp $"

#include <list>
#include <vector>
#include "thread.h"
#include "yasocket.h"

class ProxySocket;
class ProxyListener;
class ProxyConnectThread;
class ProxyHandleThread;
class HandlerList;

// abstract interface of a proxy socket
class ProxySocket {
public:
	enum Result {
		NoData,
		Connecting,
		Forwarding,
		Closing,
		Error
	};

	ProxySocket(IPSocket *, const char *);
	virtual ~ProxySocket() = 0; // abstract class

	const char *Type() const { return type; }
	PString Name() const { return name; }

	virtual Result ReceiveData();
	virtual bool ForwardData();
	virtual bool TransmitData(const PBYTEArray &);
	virtual bool TransmitData(const BYTE *, WORD);
	virtual bool EndSession();

	bool IsSocketOpen() const { return self->IsOpen(); }
	bool CloseSocket() { return IsSocketOpen() ? self->Close() : false; }

	bool Flush();
	bool CanFlush() const { return (qsize > 0) && IsSocketOpen(); }

	bool IsBlocked() const { return blocked; }
	void MarkBlocked(bool b) { blocked = b; }
	bool IsConnected() const { return connected; }
	void SetConnected(bool c) { connected = c; }
	bool IsDeletable() const { return deletable; }
	void SetDeletable() { deletable = true; }
	ProxyHandleThread *GetHandler() const { return handler; }
	void SetHandler(ProxyHandleThread *h) { handler = h; }
	void AddToSelectList(SocketSelectList& sl) { sl.Append(self); }

#ifdef LARGE_FDSET
	bool IsReadable(long timeout = 0) const { return self->CanRead(timeout); }
	bool IsWriteable(long timeout = 0) const { return self->CanWrite(timeout); }
#else
	bool IsReadable(long timeout = 0) const 
	{ 
		return SocketSelectList(self).Select(SocketSelectList::Read, timeout); 
	}
	
	bool IsWriteable(long timeout = 0) const 
	{ 
		return SocketSelectList(self).Select(SocketSelectList::Write, timeout); 
	}
#endif

protected:
	virtual bool WriteData(const BYTE *, WORD);
	virtual bool FlushData(const BYTE *, WORD);
	bool ErrorHandler(PSocket::ErrorGroup);
	void SetName(const PIPSocket::Address & ip, WORD pt);
	IPSocket* Self() const { return self; }

	int GetQueueSize() const { return qsize; }	
	void PutBackPacket(const BYTE* buf, WORD len)
	{
		queueMutex.Wait();
		queue.push_front(new PBYTEArray(buf, len));
		++qsize;
		queueMutex.Signal();
	}
	
	void QueuePacket(const BYTE* buf, WORD len)
	{
		queueMutex.Wait();
		queue.push_back(new PBYTEArray(buf, len));
		++qsize;
		queueMutex.Signal();
	}
	
	PBYTEArray* PopQueuedPacket()
	{
		PBYTEArray* packet = NULL;
		queueMutex.Wait();
		if (!queue.empty()) {
			packet = queue.front();
			queue.pop_front();
			--qsize;
		}
		queueMutex.Signal();
		return packet;
	}
	
	BYTE *wbuffer;
	WORD wbufsize, buflen;

private:
	// these should not be ever used
	ProxySocket();
	ProxySocket( const ProxySocket& );
	ProxySocket& operator=( const ProxySocket& );
	
	IPSocket *self;
	std::list<PBYTEArray *> queue;
	int qsize;

	PString name;
	const char *type;

	bool blocked, connected, deletable;
	PMutex writeMutex, queueMutex;
	ProxyHandleThread *handler;
};

#ifdef LARGE_FDSET
class TCPProxySocket : public YaTCPSocket, public ProxySocket {
#else
class TCPProxySocket : public PTCPSocket, public ProxySocket {
	PCLASSINFO( TCPProxySocket, PTCPSocket )
#endif
public:
	TCPProxySocket(const char *, TCPProxySocket * = 0, WORD = 0);
	virtual ~TCPProxySocket();

	// override from class ProxySocket
	virtual bool ForwardData();
	virtual bool TransmitData(const PBYTEArray &);

	// override from class PTCPSocket
	virtual BOOL Accept(PSocket &);
	virtual BOOL Connect(const Address &, WORD, const Address &);
	virtual BOOL Connect(WORD, const Address &);
	virtual BOOL Connect(const Address &);

	// new virtual function
	virtual TCPProxySocket *ConnectTo() = 0;

protected:
	bool ReadTPKT();

	TCPProxySocket *remote;
	PBYTEArray buffer;

private:
	// these should not be ever user
	TCPProxySocket();
	TCPProxySocket( const TCPProxySocket& );
	TCPProxySocket& operator=( const TCPProxySocket& );
	
	bool InternalWrite(const PBYTEArray &);
	bool SetMinBufSize(WORD);

	BYTE *bufptr;
};

class ProxyListener : public MyPThread {
public:
	PCLASSINFO ( ProxyListener, MyPThread )

	ProxyListener(HandlerList *, const PIPSocket::Address &, WORD, unsigned);
	virtual ~ProxyListener();
	virtual bool Open(unsigned);

	// override from class MyPThread
	virtual void Close();
	virtual void Exec();

	WORD GetPort() const { return m_port; }
				
protected:
	PTCPSocket *m_listener;
	PIPSocket::Address m_interface; 
	WORD m_port;

private:
	TCPProxySocket *CreateSocket();

	HandlerList *m_handler;
};

class ProxyHandleThread : public MyPThread {
public:
	PCLASSINFO ( ProxyHandleThread, MyPThread )

	typedef std::list<ProxySocket *>::iterator iterator;
	typedef std::list<ProxySocket *>::const_iterator const_iterator;
	typedef std::list<ProxyConnectThread *>::iterator citerator;
	typedef std::list<ProxyConnectThread *>::const_iterator const_citerator;

	ProxyHandleThread() : lcHandler(0) {}
	ProxyHandleThread(PINDEX);
	virtual ~ProxyHandleThread();

	void Insert(ProxySocket *);
	void InsertLC(ProxySocket *socket) { lcHandler->Insert(socket); }
	void MoveTo(ProxyHandleThread *, ProxySocket *);
	void Remove(iterator);
	void Remove(ProxySocket *socket);
	void SetID(const PString & i) { id = i; }
	void ConnectTo(ProxySocket *);
	bool CloseUnusedThreads();

	// override from class MyPThread
	virtual void Exec();

private:
	void FlushSockets();
	void RemoveSockets();
	void BuildSelectList(SocketSelectList &);
	ProxyConnectThread *FindConnectThread();
	
	std::list<ProxySocket *> sockList, removedList;
	std::list<PTime *> removedTime;
	mutable PReadWriteMutex mutex;
	PMutex removedMutex;
	std::list<ProxyConnectThread *> connList;
	mutable PReadWriteMutex connMutex;
	ProxyHandleThread *lcHandler;
	PString id;
};

class HandlerList {     
public:         
	HandlerList(PIPSocket::Address = INADDR_ANY);
	~HandlerList();

	void LoadConfig();
	void Insert(ProxySocket *);
	void Check();

	WORD GetCallSignalPort() const { return listenerThread->GetPort(); }

private:
	void CloseListener();

	std::vector<ProxyHandleThread *> handlers;
	ProxyListener *listenerThread;
	unsigned currentHandler;
	PMutex mutex;
	PIPSocket::Address GKHome;
	WORD GKPort;
};


#endif // __proxythread_h__

