/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "PollReactor.h"

#ifdef HAVE_POLL_REACTOR

#include "EventHandler.h"
#include "ReactorHelpers.h"
#include "SynchFactory.h"
#include "MtGuard.h"
#include "AtomicOps.h"
#include <ace/Lock.h>
#include <cassert>
#include <cstdlib>
#include <cerrno>
#include <new> // for bad_alloc
#include <limits>

#define EXCLUSIVE_WAITSET_ACCESS()    \
MtGuard<ACE_Lock> guard(*m_ptrMutex); \
if (m_isInsideDemux) wakeup();        \
MtGuard<ACE_Lock> demux_guard(*m_ptrDemuxMutex);

using namespace std;
using namespace ReactorHelpers;

static short const POLL_READ = POLLIN|POLLERR|POLLHUP|POLLNVAL;
static short const POLL_WRITE = POLLOUT;
static short const POLL_EXCEPT = POLLPRI;

static unsigned const HANDLER_NEW = 1;
static unsigned const HANDLER_NOT_NEW = 0;
/*
When registering a handler, we store HANDLER_NEW in Handler::flags.
In updateWaitSet() we set all active handlers to HANDLER_NOT_NEW.
When dispatching events, we skip handlers with flags == HANDLER_NEW.
That's done to prevent a situation like this:
1. Multiple events have occured on a handle.
2. The first event is dispatched.
3. The event handler unregisters itself and closes the handle.
4. Another handle is created with the same value as the one closed.
5. The new handle is registered in the reactor.
6. The rest of the events ocurred on the old handle are dispatched
to the new handler, which is wrong.
*/


static pollfd handlerToPollFD(Handler const& handler);

static short translateToPollEvents(Reactor::IOEvents events);


PollReactor::PollReactor(
	AbstractSynchFactory const& synch_factory, bool wakeup_on_signal)
:	m_ptrHandlers(new HandlerRepository),
	m_ptrTimers(new TimerQueue),
	m_ptrMutex(synch_factory.createMutex()),
	m_ptrDemuxMutex(synch_factory.createMutex()),
	m_wakeupPipe(),
	m_waitSet(m_wakeupPipe.getReadHandle()),
	m_dispatchWave(0),
	m_ioIter(0),
	m_wakeupOnSignal(wakeup_on_signal),
	m_isInsideDemux(false),
	m_isStopped(false),
	m_isStoppedAtomic(0)
{
}

PollReactor::~PollReactor()
{
}

ReactorHandlerId
PollReactor::findHandler(ACE_HANDLE handle) const
{
	return m_ptrHandlers->findByHandle(handle);
}

ReactorHandlerId
PollReactor::registerHandler(
	ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events)
{
	assert(handler);
	
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	if (m_isInsideDemux) {
		wakeup();
	}
	
	m_waitSet.ensureCapacity(m_ptrHandlers->size() + 1);
	Handler new_handler(handle, handler, events, HANDLER_NEW, 0);
	return m_ptrHandlers->add(new_handler);
}

void
PollReactor::unregisterHandler(ReactorHandlerId const& id)
{
	assert(id);
	EXCLUSIVE_WAITSET_ACCESS();
	// It's neccessary to make the other thread return from poll(),
	// as the client may close the handle right after we return.
	
	m_ptrHandlers->remove(id);
	m_waitSet.reduceCapacity(m_ptrHandlers->size());
}

void
PollReactor::unregisterAllHandlers()
{
	EXCLUSIVE_WAITSET_ACCESS();
	
	m_ptrHandlers->clear();
	m_ptrHandlers->first_inactive = m_ptrHandlers->end();
	m_waitSet.trim(0);
	m_waitSet.reduceCapacity(0);
	m_ioIter.reset(0);
}

void
PollReactor::enableEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	m_ptrHandlers->setEvents(it, it->events | events);
}

void
PollReactor::disableEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	m_ptrHandlers->setEvents(it, it->events & ~events);
}

void
PollReactor::setEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	m_ptrHandlers->setEvents(it, events);
}

Reactor::IOEvents
PollReactor::getEvents(ReactorHandlerId const& id) const
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	return it->events;
}

ReactorTimerId
PollReactor::registerTimer(EventHandlerPtr const& handler, ACE_Time_Value const* timeout)
{
	assert(handler);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	TimerQueue* tq = m_ptrTimers.get();
	ACE_Time_Value was_earliest_time = tq->getEarliestTime();
	ReactorTimerId id = tq->add(handler, timeout);
	assert(id);
	
	if (m_isInsideDemux && tq->getEarliestTime() < was_earliest_time) {
		wakeup();
	}
	
	return id;
}

void
PollReactor::rescheduleTimer(ReactorTimerId const id, ACE_Time_Value const* timeout)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	TimerQueue* tq = m_ptrTimers.get();
	ACE_Time_Value was_earliest_time = tq->getEarliestTime();
	tq->reschedule(id, timeout);
	
	if (m_isInsideDemux && tq->getEarliestTime() < was_earliest_time) {
		wakeup();
	}
}

void
PollReactor::unregisterTimer(ReactorTimerId const& id)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	m_ptrTimers->erase(TimerQueue::iterFromId(id));
}

void
PollReactor::unregisterAllTimers()
{
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	m_ptrTimers->clear();
}

Reactor::Status
PollReactor::handleEvents()
{
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	if (m_isStopped) {
		m_isStopped = AtomicOps::add(&m_isStoppedAtomic, 0);
		if (m_isStopped) {
			return STOPPED;
		}
	}
	
	unsigned const old_dispatch_wave = m_dispatchWave;
	continueIODispatching();
	m_ptrTimers->continueDispatching(*m_ptrMutex, m_isStopped);
	if (m_isStopped) {
		return STOPPED;
	}
	if (old_dispatch_wave != m_dispatchWave) {
		// This means a recursive call to handleEvents() has
		// already called poll() and did event dispatching,
		// so there is no reason to do that again.
		return SUCCESS;
	}

	{
		MtAntiGuard<ACE_Lock> anti_guard(*m_ptrMutex);
		m_beforeSleepSignal.emit(); // may add / remove handlers
	}
	if (m_isStopped) {
		return STOPPED;
	}
	
	updateWaitSet();
	bool woken_up = false;
	bool interrupted = false;
	int res = 0;
	{
		DemultiplexerGuard guard(*m_ptrMutex, *m_ptrDemuxMutex, m_isInsideDemux);
		while (true) {
			res = m_waitSet.poll(getPollTimeout(), woken_up);
			if (woken_up) {
				m_wakeupPipe.deactivate();
			}
			if (res < 0 && errno == EINTR) {
				if (m_wakeupOnSignal) {
					interrupted = true;
				} else {
					continue;
				}
			}
			break;
		}
	}
	if (res < 0) {
		assert(!woken_up);
		return interrupted ? INTERRUPTED : FAILURE;
	}
	
	++m_dispatchWave;
	m_ioIter.reset(res);
	m_ptrTimers->updateDispatchThreshold();
	
	if (woken_up) {
		m_isStopped = AtomicOps::add(&m_isStoppedAtomic, 0);
		if (m_isStopped) {
			return STOPPED;
		}
	}
	
	if (res > 0) {
		continueIODispatching();
	}
	m_ptrTimers->continueDispatching(*m_ptrMutex, m_isStopped);
	
	return m_isStopped ? STOPPED : SUCCESS;
}

Reactor::Status
PollReactor::runEventLoop()
{
	Status status = SUCCESS;
	do {
		status = handleEvents();
	} while (status == SUCCESS);
	return status;
}

void
PollReactor::wakeup()
{
	m_wakeupPipe.activate();
}

void
PollReactor::stop()
{
	if (AtomicOps::set(&m_isStoppedAtomic, 1) == 0) {
		wakeup();
	}
}

void
PollReactor::restart()
{
	AtomicOps::set(&m_isStoppedAtomic, 0);
}

sigc::signal<void>&
PollReactor::beforeSleepSignal()
{
	return m_beforeSleepSignal;
}

void
PollReactor::updateWaitSet()
{
	assert(m_waitSet.capacity() >= m_ptrHandlers->size());
	m_waitSet.trim(0); // trim doesn't reduce capacity
	HandlerRepository::iterator it = m_ptrHandlers->begin();
	HandlerRepository::iterator const end = m_ptrHandlers->first_inactive;
	for (; it != end; ++it) {
		m_waitSet.append(handlerToPollFD(*it));
		it->flags = HANDLER_NOT_NEW;
	}
}

int
PollReactor::getPollTimeout() const
{
	ACE_Time_Value const target_time = m_ptrTimers->getEarliestTime();
	if (target_time == ACE_Time_Value::max_time) {
		return -1;
	}
	
	ACE_Time_Value const cur_time = m_ptrTimers->getCurrentTime();
	if (cur_time + ACE_Time_Value(0, 2) >= target_time) {
		/*
		These 2 microseconds compensate the effects of
		TimerQueue::updateDispatchThreshold() and
		TimerQueue::ensureTimeAfterThreshold().
		*/
		return 0;
	}
	
	/*
	We round the timeout upwards to a millisecond, to prevent a situation
	where poll() returns on timeout, but no timers are dispatched.
	*/
	ACE_Time_Value timeout = target_time - cur_time;
	timeout.usec(timeout.usec() + (999 - timeout.usec() % 1000));
	int const max_int = std::numeric_limits<int>::max();
	ACE_Time_Value const max_timeout(max_int / 1000, (max_int % 1000) * 1000);
	if (timeout >= max_timeout) {
		return max_int;
	} else {
		return timeout.msec();
	}
}

void
PollReactor::continueIODispatching()
{
	while (!m_isStopped && m_ioIter.next(m_waitSet)) {
		assert(m_ioIter.pos < m_waitSet.size());
		ACE_HANDLE const handle = m_waitSet[m_ioIter.pos].fd;
		
		typedef HandlerRepository::HandleIdx HandleIdx;
		HandleIdx& idx = m_ptrHandlers->get<HandleTag>();
		HandleIdx::iterator it = idx.find(handle);
		if (it != idx.end() && it->flags == HANDLER_NOT_NEW) {
			dispatchIO(*it);
		}
	}
}

void
PollReactor::dispatchIO(Handler const& handler)
{
	/*
	Unfortunately we can't dispatch all events in one go, because
	event handlers may erase the current handler, may advance m_ioIter,
	and may even indirectly call updateWaitSet().
	*/
	
	ACE_HANDLE handle = handler.handle;
	EventHandlerBase* eh = handler.handler.get();
	if (m_ioIter.phase == 0 && (handler.events & WRITE)) {
		dispatchIOEvent(handle, eh, &EventHandlerBase::handleWrite);
	} else if (m_ioIter.phase == 1 && (handler.events & EXCEPT)) {
		dispatchIOEvent(handle, eh, &EventHandlerBase::handleExcept);
	} else if (m_ioIter.phase == 2 && (handler.events & READ)) {
		dispatchIOEvent(handle, eh, &EventHandlerBase::handleRead);
	}
}

void
PollReactor::dispatchIOEvent(
	ACE_HANDLE handle, EventHandlerBase* eh, HandlerFuncPtr func)
{
	MtAntiGuard<ACE_Lock> anti_guard(*m_ptrMutex);
	(eh->*func)(handle);
}

pollfd handlerToPollFD(Handler const& handler)
{
	pollfd fd;
	fd.fd = handler.handle;
	fd.events = translateToPollEvents(handler.events);
	fd.revents = 0;
	return fd;
}

short translateToPollEvents(Reactor::IOEvents events)
{
	short res = 0;
	if (events & PollReactor::READ) {
		res |= POLL_READ;
	}
	if (events & PollReactor::WRITE) {
		res |= POLL_WRITE;
	}
	if (events & PollReactor::EXCEPT) {
		res |= POLL_EXCEPT;
	}
	return res;
}


/*===================== PollReactor::IODispatchIter =====================*/

bool
PollReactor::IODispatchIter::next(PollWaitSet const& ws)
{
	if (todo == 0) {
		return false;
	}
	
	assert(pos < ws.size());
	short revents = ws[pos].revents;

	switch (phase) {
		case -1:
		case_m1:
		phase = 0;
		if (revents & POLL_WRITE) {
			break;
		}
		// fall through
		case 0:
		phase = 1;
		if (revents & POLL_EXCEPT) {
			break;
		}
		// fall through
		case 1:
		phase = 2;
		if (revents & POLL_READ) {
			break;
		}
		// fall through
		case 2:
		phase = 0;
		++pos;
		if (revents != 0) {
			if (--todo == 0) {
				return false;
			}
		}
		assert(pos < ws.size());
		revents = ws[pos].revents;
		goto case_m1;
	}
	
	return true;
}


/*=========================== PollWaitSet ==============================*/

PollWaitSet::PollWaitSet(ACE_HANDLE wakeup_handle)
:	m_size(0),
	m_capacity(0),
	m_fdArray(0)
{
	resizeFdArray(0);
	assert(m_fdArray);
	
	m_fdArray[0].fd = wakeup_handle;
	m_fdArray[0].events = POLLIN;
	m_fdArray[0].revents = 0;
}

PollWaitSet::~PollWaitSet()
{
	assert(m_fdArray);
	free(m_fdArray);
}

void
PollWaitSet::ensureCapacity(size_t capacity)
{
	if (m_capacity < capacity) {
		resizeFdArray(capacity);
	}
}

void
PollWaitSet::reduceCapacity(size_t capacity)
{
	if (m_capacity > capacity && m_size <= capacity) {
		resizeFdArray(capacity);
	}
}

void
PollWaitSet::trim(size_t to)
{
	if (m_size > to) {
		m_size = to;
		// pollfd structures don't need destruction
	}
}

void
PollWaitSet::append(pollfd const& fd)
{
	assert(m_capacity >= m_size);
	if (m_capacity == m_size) {
		resizeFdArray(m_size + 1);
	}
	(*this)[m_size] = fd;
	++m_size;
}

int
PollWaitSet::poll(int timeout, bool& woken_up)
{
	int res = ::poll(m_fdArray, m_size + 1, timeout);
	woken_up = (res > 0 && m_fdArray[0].revents);
	if (woken_up) {
		--res;
	}
	return res;
}

void
PollWaitSet::resizeFdArray(size_t user_requested_size)
{
	size_t const current_capacity = m_capacity + 1;
	size_t const requested_size = user_requested_size + 1;
	size_t const new_capacity = adviceFdArraySize(requested_size);
	if (new_capacity < current_capacity / 2) {
		assert(m_fdArray);
		void* mem = realloc(m_fdArray, new_capacity);
		if (mem != m_fdArray) {
			/*
			I hope there are no platforms that allocate new
			memory when you ask to truncate an existing block.
			Just updating m_fdArray is not good enough, because
			then we lose exception-safety (allocation may fail).
			*/
			abort();
		}
		m_capacity = new_capacity - 1;
	} else if (requested_size > current_capacity || !m_fdArray) {
		void* mem = malloc(new_capacity);
		if (!mem) {
			throw std::bad_alloc();
		}
		free(m_fdArray);
		m_fdArray = (pollfd*)mem;
		m_capacity = new_capacity - 1;
	} else {
		assert(requested_size <= current_capacity);
	}
}

size_t
PollWaitSet::adviceFdArraySize(size_t requested_size)
{
	size_t size = requested_size + requested_size / 2;
	if (size < sizeof(pollfd) * 4) {
		size = sizeof(pollfd) * 4;
	}
	return size;
}

#endif // HAVE_POLL_REACTOR
