/*
    BFilter - a web proxy which removes banner ads
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HEADERPARSER_H_
#define HEADERPARSER_H_

#include "HeaderLexerCore.h"
#include "HeaderParserListener.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include <stddef.h>

class HeaderParser : private HeaderLexerCore
{
	friend class HeaderLexerCore::Actions;
public:
	HeaderParser();
	
	void consume(
		SplittableBuffer& hdr_value,
		HeaderParserListener* listener);
private:
	class EnvironmentManager;
	friend class HeaderParser::EnvironmentManager;

	void processError(size_t pos);
	
	void processElementName(Iterator const& begin, Iterator const& end);
	
	void processNullElementValue();
	
	void processElementValue(Iterator const& begin, Iterator const& end);
	
	void appendElementValue(Iterator const& begin, Iterator const& end);
	
	void appendElementValueQuoted(char ch);
	
	void appendElementValueDone();
	
	void setProcessingParam(bool val);
	
	static char unquoteChar(char ch);
private:
	SBOutStream m_valueStream;
	HeaderParserListener* m_pListener;
	bool m_isProcessingParam;
};


inline void
HeaderParser::processError(size_t pos)
{
	m_pListener->processError(pos);
}

inline void
HeaderParser::processElementName(Iterator const& begin, Iterator const& end)
{
	if (m_isProcessingParam) {
		m_pListener->processParamName(begin, end);
	} else {
		m_pListener->processElementName(begin, end);
	}
}

inline void
HeaderParser::processNullElementValue()
{
	if (m_isProcessingParam) {
		m_pListener->processParamNullValue();
	} else {
		m_pListener->processElementNullValue();
	}
}

inline void
HeaderParser::processElementValue(Iterator const& begin, Iterator const& end)
{
	if (m_isProcessingParam) {
		m_pListener->processParamValue(begin, end);
	} else {
		m_pListener->processElementValue(begin, end);
	}
}

inline void
HeaderParser::appendElementValue(Iterator const& begin, Iterator const& end)
{
	m_valueStream.data().append(begin, end);
}

inline void
HeaderParser::appendElementValueQuoted(char ch)
{
	m_valueStream << unquoteChar(ch);
}

inline void
HeaderParser::appendElementValueDone()
{
	SplittableBuffer data;
	m_valueStream.swapData(data);
	if (m_isProcessingParam) {
		m_pListener->processParamValue(data.begin(), data.end());
	} else {
		m_pListener->processParamValue(data.begin(), data.end());
	}
}

inline void
HeaderParser::setProcessingParam(bool val)
{
	m_isProcessingParam = val;
}

#endif
