/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "BasicConfigDialog.h"
#include "Application.h"
#include "OperationLog.h"
#include "GlobalState.h"
#include "Conf.h"
#include "StringUtils.h"
#include "ProxyDescriptor.h"
#include "InetAddr.h"
#include "CompiledImages.h"
#include "EffectiveFileTimestamps.h"
#include "types.h"
#include <boost/lexical_cast.hpp>
#include <gtkmm/entry.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/box.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/button.h>
#include <gtkmm/alignment.h>
#include <gtkmm/label.h>
#include <gtkmm/frame.h>
#include <gtkmm/stock.h>
#include <gtkmm/messagedialog.h>
#include <list>
#include <sstream>
#include <cassert>
#include <time.h>

using namespace std;

BasicConfigDialog* BasicConfigDialog::m_spInstance = 0;

BasicConfigDialog::BasicConfigDialog()
{
	set_title("Basic Configuration");
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	set_border_width(5);
	
	Gtk::VBox* vbox = manage(new Gtk::VBox);
	add(*vbox);
	
	Gtk::Alignment* align1 = manage(new Gtk::Alignment(0.5, 0.5, 0.0, 0.0));
	vbox->pack_start(*align1);
	
	Gtk::HBox* hbox1 = manage(new Gtk::HBox);
	align1->add(*hbox1);
	
	hbox1->set_border_width(5);
	hbox1->set_spacing(10);
	hbox1->pack_start(*manage(new Gtk::Label("Listen Port")));
	
	m_pListenPort = manage(new Gtk::Entry);
	hbox1->pack_start(*m_pListenPort);
	m_pListenPort->set_size_request(50);
	
	Gtk::Frame* fwd_frame = manage(new Gtk::Frame(" Forwarding "));
	vbox->pack_start(*fwd_frame);
	Gtk::VBox* vbox2 = manage(new Gtk::VBox);
	fwd_frame->add(*vbox2);
	
	vbox2->set_border_width(5);
	vbox2->set_spacing(2);
	
	m_pForward = manage(new Gtk::CheckButton("Forward to another proxy"));
	vbox2->pack_start(*m_pForward);
	Gtk::HBox* hbox2 = manage(new Gtk::HBox);
	vbox2->pack_start(*hbox2, Gtk::PACK_SHRINK);
	hbox2->set_spacing(5);
	
	m_pProxyHost = manage(new Gtk::Entry);
	hbox2->pack_start(*m_pProxyHost);
	m_pProxyHost->set_size_request(130);
	hbox2->pack_start(*manage(new Gtk::Label(":")), Gtk::PACK_SHRINK);
	m_pProxyPort = manage(new Gtk::Entry);
	hbox2->pack_start(*m_pProxyPort);
	m_pProxyPort->set_size_request(50);
	
	Gtk::Alignment* spacer1(new Gtk::Alignment);
	vbox->pack_start(*spacer1);
	spacer1->set_size_request(3, 3);
	
	m_pTrayAnimation = manage(new Gtk::CheckButton("Tray icon animation"));
	vbox->add(*m_pTrayAnimation);
	
	Gtk::Alignment* spacer2 = manage(new Gtk::Alignment);
	vbox->pack_start(*spacer2, Gtk::PACK_SHRINK);
	spacer2->set_size_request(5, 5);
	
	Gtk::Alignment* bbox_alignment = manage(new Gtk::Alignment(0.5, 0.5, 0.0, 0.0));
	vbox->pack_start(*bbox_alignment, Gtk::PACK_SHRINK);
	
	Gtk::HButtonBox* button_box = manage(new Gtk::HButtonBox);
	bbox_alignment->add(*button_box);
	button_box->set_spacing(10);
	
	Gtk::Button *cancel_btn = manage(new Gtk::Button(Gtk::Stock::CANCEL));
	button_box->pack_start(*cancel_btn, Gtk::PACK_SHRINK);
	cancel_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &BasicConfigDialog::hide)
	);
	
	Gtk::Button *ok_btn = manage(new Gtk::Button(Gtk::Stock::OK));
	button_box->pack_start(*ok_btn, Gtk::PACK_SHRINK);
	ok_btn->signal_clicked().connect(
		sigc::mem_fun(*this, &BasicConfigDialog::onApply)
	);
	
	loadConfig();
	cancel_btn->grab_focus();
	show_all_children();
	
	Application::instance()->destroySignal().connect(
		sigc::mem_fun(*this, &BasicConfigDialog::hide)
	);
}

BasicConfigDialog::~BasicConfigDialog()
{
	assert(m_spInstance);
	m_spInstance = 0;
}

void
BasicConfigDialog::showWindow()
{
	if (m_spInstance) {
		m_spInstance->present();
	} else {
		(m_spInstance = new BasicConfigDialog)->show();
	}
}

void
BasicConfigDialog::on_hide()
{
	delete this;
}

void
BasicConfigDialog::loadConfig()
{
	Config config = GlobalState::ReadAccessor()->config();
	list<InetAddr> listen_addrs = config.getListenAddresses();
	if (listen_addrs.size() > 1) {
		m_pListenPort->set_text(Glib::ustring());
		showWarning(
			"Your current config has multiple listen addresses.\n"
			"You should be using the Advanced Configuration dialog."
		);
	} else if (listen_addrs.size() == 1) {
		m_pListenPort->set_text(
			StringUtils::fromNumber(listen_addrs.front().getPort())
		);
	} else {
		m_pListenPort->set_text(Glib::ustring());
	}
	m_pTrayAnimation->set_active(config.isTrayAnimationEnabled());
	m_pForward->set_active(config.isNextHopProxyEnabled());
	ProxyDescriptor pdesc = config.getNextHopProxy();
	m_pProxyHost->set_text(pdesc.getProxyAddr().getHost());
	if (pdesc.getProxyAddr().getPort() <= 0) {
		m_pProxyPort->set_text(Glib::ustring());
	} else {
		m_pProxyPort->set_text(
			StringUtils::fromNumber(pdesc.getProxyAddr().getPort())
		);
	}	
}

void
BasicConfigDialog::onApply()
{
	bool forwarding = m_pForward->get_active();
	uint16_t listen_port = 0;
	try {
		string port = StringUtils::trimStr(m_pListenPort->get_text().c_str());
		listen_port = boost::lexical_cast<uint16_t>(port);
	} catch (boost::bad_lexical_cast& e) {}
	if (listen_port == 0) {
		showError("Listen port must be an integer in range of [1, 65535]");
		m_pListenPort->grab_focus();
		return;
	}
	
	string proxy_host = StringUtils::trimStr(m_pProxyHost->get_text().c_str());	
	if (forwarding && proxy_host.empty()) {
		showError("Remote proxy host is not given");
		m_pProxyHost->grab_focus();
		return;
	}
	
	uint16_t proxy_port = 0;
	try {
		string port = StringUtils::trimStr(m_pProxyPort->get_text().c_str());
		proxy_port = boost::lexical_cast<uint16_t>(port);
	} catch (boost::bad_lexical_cast& e) {}
	if (forwarding && proxy_port == 0) {
		showError("Remote proxy port must be an integer in range of [1, 65535]");
		m_pProxyPort->grab_focus();
		return;
	}
	
	Config const old_config(GlobalState::ReadAccessor()->config());
	Config new_config(old_config);
	list<InetAddr> listen_addrs;
	listen_addrs.push_back(InetAddr("127.0.0.1", listen_port));
	new_config.setListenAddresses(listen_addrs);
	new_config.setTrayAnimationEnabled(m_pTrayAnimation->get_active());
	new_config.setNextHopProxyEnabled(forwarding);
	InetAddr proxy_addr(proxy_host, -1);
	if (proxy_port > 0) {
		proxy_addr.setPort(proxy_port);
	}
	new_config.setNextHopProxy(ProxyDescriptor(proxy_addr));
	
	Application& app = *Application::instance();
	Log* log = OperationLog::instance();
	log->appendRecord("Applying new config ... ");
	size_t num_records = log->getNumRecords();
#if 0
	if (new_config == old_config) {
		log->appendToLastRecord(
			"no changes",
			log->getSuccessStyle()
		);
		hide();
		return;
	}
#endif
	
	if (!app.applyConfig(new_config)) {
		restoreOldConfig(old_config);
		app.showLogDialog();
		return;
	}
	
	ConfigFile new_config_file(app.configFile());
	new_config_file.updateWith(new_config);
	ostringstream content;
	content << new_config_file;
	
	time_t mtime = 0;
	if (!app.writeFile(app.getConfigFileName(), content.str(), &mtime)) {
		restoreOldConfig(old_config);
		app.showLogDialog();
		return;
	}
	
	app.configFile().swap(new_config_file);
	EffectiveFileTimestamps::config = mtime;
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	
	hide();
}

bool
BasicConfigDialog::restoreOldConfig(Config const& config)
{
	Application& app = *Application::instance();
	Log* log = OperationLog::instance();
	log->appendRecord("Restoring the old config ... ");
	size_t num_records = log->getNumRecords();
	
	if (!app.applyConfig(config)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord("done", log->getSuccessStyle());
	}
	return true;
}

void
BasicConfigDialog::showError(Glib::ustring const& msg)
{
	Gtk::MessageDialog dialog(
		*this, msg,
		false, /* use_markup */
		Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK,
		true /* modal */
	);
	dialog.run();
}

void
BasicConfigDialog::showWarning(Glib::ustring const& msg)
{
	Gtk::MessageDialog dialog(
		*this, msg,
		false, /* use_markup */
		Gtk::MESSAGE_WARNING, Gtk::BUTTONS_OK,
		true /* modal */
	);
	dialog.run();
}
