/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Daemon.h"
#include "Conf.h"
#include "ConfError.h"
#include "ConfErrorHandler.h"
#include "ConfigFile.h"
#include "UrlPatterns.h"
#include "UrlPatternsFile.h"
#include "ContentFilters.h"
#include "FilterGroupTag.h"
#include "ContentFiltersFile.h"
#include "CombinedUrlPatterns.h"
#include "RegexFilterDescriptor.h"
#include "GlobalState.h"
#include "ConnAcceptor.h"
#include "InetAddr.h"
#include "IntrusivePtr.h"
#include "RefCountableSAP.h"
#include "binreloc/prefix.h"
#include <ace/Dirent.h>
#include <ace/INET_Addr.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/Signal.h>
#include <boost/program_options.hpp>
#include <boost/tokenizer.hpp>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <deque>
#include <list>
#include <cstdlib>

namespace po = boost::program_options;
using namespace std;

class Daemon::ConfigErrorHandler : public ConfErrorHandler
{
public:
	ConfigErrorHandler(std::string const& fname)
	: m_fileName(fname), m_hasCriticalErrors(false) {}
	
	virtual bool handleError(ConfError const& err);
	
	bool hasCriticalErrors() const { return m_hasCriticalErrors; }
private:
	std::string m_fileName;
	bool m_hasCriticalErrors;
};


class Daemon::ConfigLoader
{
public:
	bool load(std::string const& confdir);
	
	Config const& getConfig() const { return m_config; }
private:
	Config m_config;
};


class Daemon::UrlPatternsLoader
{
public:
	bool load(std::string const& confdir);
	
	CombinedUrlPatterns const& getCombinedPatterns() const {
		return m_combinedPatterns;
	}
private:
	bool load(std::string const& confdir,
		std::string const& fname, UrlPatterns& target);
	
	CombinedUrlPatterns m_combinedPatterns;
};


class Daemon::ContentFiltersLoader
{
public:
	bool load(std::string const& confdir);
	
	ContentFilters const& getFilters() const { return m_filters; }
private:
	void loadEnabledFilterList(
		ContentFilters& filters, std::string const& abs_fname);
	
	void appendEnabledFilters(ContentFilters const& filters);
	
	ContentFilters m_filters;
};


/* ==================== Daemon::ConfigErrorHandler =======================*/

bool
Daemon::ConfigErrorHandler::handleError(ConfError const& err)
{
	cerr << '[' << m_fileName;
	if (err.getLine()) {
		cerr << ':' << err.getLine();
		if (err.getCol()) {
			cerr << ':' << err.getCol();
		}
	}
	cerr << "] ";
	if (err.getType() == ConfError::T_WARNING) {
		cerr << "Warning: ";
	} else {
		m_hasCriticalErrors = true;
	}
	cerr << err.getMessage();
	cerr << endl;
	return true;
}


/* ====================== Daemon::ConfigLoader =========================*/

bool
Daemon::ConfigLoader::load(std::string const& confdir)
{
	string const fname("config");
	string const abs_fname(confdir+"/"+fname);
	ifstream istrm(abs_fname.c_str());
	if (istrm.fail()) {
		cerr << "Could not open " << abs_fname << endl;
		return false;
	}
	ostringstream ostrm;
	ostrm << istrm.rdbuf();
	if (istrm.bad()) {
		cerr << "Error reading " << abs_fname << endl;
		return false;
	}
	
	ConfigFile cfile;
	ConfigErrorHandler eh(fname);
	cfile.load(ostrm.str(), m_config, eh);
	
	if (eh.hasCriticalErrors()) {
		return false;
	}
	
	return true;
}


/* ==================== Daemon::UrlPatternsLoader =======================*/

bool
Daemon::UrlPatternsLoader::load(std::string const& confdir)
{
	if (!load(confdir, "urls", m_combinedPatterns.standardPatterns())) {
		return false;
	}
	if (!load(confdir, "urls.local", m_combinedPatterns.localPatterns())) {
		return false;
	}
	return true;
}

bool
Daemon::UrlPatternsLoader::load(
	std::string const& confdir,
	std::string const& fname, UrlPatterns& target)
{
	string const abs_fname(confdir+"/"+fname);
	ifstream istrm(abs_fname.c_str());
	if (istrm.fail()) {
		cerr << "Could not open " << abs_fname << endl;
		return false;
	}
	ostringstream ostrm;
	ostrm << istrm.rdbuf();
	if (istrm.bad()) {
		cerr << "Error reading " << abs_fname << endl;
		return false;
	}
	
	UrlPatternsFile cfile;
	ConfigErrorHandler eh(fname);
	cfile.load(ostrm.str(), target, eh);
	
	if (eh.hasCriticalErrors()) {
		return false;
	}
	
	return true;
}


/* ====================== Daemon::ContentFilterLoader =========================*/

bool
Daemon::ContentFiltersLoader::load(std::string const& confdir)
{
	string const dirname(confdir+"/filters");
	ACE_Dirent dir;
	if (dir.open(dirname.c_str()) == -1) {
		cerr << "Could not open directory " << dirname << endl;
		return false;
	}
	
	deque<string> fnames;
	
	for (dirent* ent; (ent = dir.read()); ) {
		string const fname(ent->d_name);
		if (fname.find('.') != string::npos) {
			// skip files with extensions
			continue;
		}
		fnames.push_back(fname);
	}
	std::sort(fnames.begin(), fnames.end());
	
	bool critical_errors = false;	
	
	for (; !fnames.empty(); fnames.pop_front()) {
		string const fname = fnames.front();
		string const abs_fname(dirname+"/"+fname);
		ifstream istrm(abs_fname.c_str());
		if (istrm.fail()) {
			cerr << "Could not open " << abs_fname << endl;
			critical_errors = true;
			continue;
		}
		ostringstream ostrm;
		ostrm << istrm.rdbuf();
		if (istrm.bad()) {
			cerr << "Error reading " << abs_fname << endl;
			critical_errors = true;
			continue;
		}
		
		ContentFilters filters;
		ContentFiltersFile cfile;
		ConfigErrorHandler eh(fname);
		cfile.load(ostrm.str(), filters, eh, FilterGroupTag::create());
		
		if (eh.hasCriticalErrors()) {
			critical_errors = true;
			continue;
		}
		
		loadEnabledFilterList(filters, abs_fname+".enabled");
		appendEnabledFilters(filters);
	}
	
	m_filters.sortByOrder();
	return !critical_errors;
}

void
Daemon::ContentFiltersLoader::loadEnabledFilterList(
	ContentFilters& filters, std::string const& abs_fname)
{
	ifstream istrm(abs_fname.c_str());
	if (istrm.fail()) {
		return;
	}
	ostringstream ostrm;
	ostrm << istrm.rdbuf();
	if (istrm.bad()) {
		return;
	}
	
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("\r\n");
	Tokenizer tokens(ostrm.str(), sep);
	set<string> enabled_set(tokens.begin(), tokens.end());
	bool const all_enabled = (enabled_set.find("*") != enabled_set.end());
	
	typedef ContentFilters::FilterList FilterList;
	FilterList& list = filters.filters();
	FilterList::iterator it = list.begin();
	FilterList::iterator const end = list.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor& filter = **it;
		bool const enabled = all_enabled ||
			(enabled_set.find(filter.name()) != enabled_set.end());
		filter.setEnabled(enabled);
	}
}

void
Daemon::ContentFiltersLoader::appendEnabledFilters(ContentFilters const& filters)
{
	typedef ContentFilters::FilterList FilterList;
	FilterList const& list = filters.filters();
	FilterList::const_iterator it = list.begin();
	FilterList::const_iterator const end = list.end();
	for (; it != end; ++it) {
		if ((*it)->isEnabled()) {
			m_filters.filters().push_back(*it);
		}
	}
}


/*=========================== Daemon ==================================*/

volatile bool Daemon::m_sSigtermReceived = false;
volatile bool Daemon::m_sSigintReceived = false;

bool
Daemon::init(std::string const& confdir)
{
	if (!loadConfiguration(confdir)) {
		return false;
	}
	
	if (!bind()) {
		return false;
	}
	return true;
}

bool
Daemon::run(bool nodaemon)
{
	if (!nodaemon) {
		if (ACE::daemonize("/", false) == -1) {
			cerr << "Could not enter the daemon mode" << endl;
			return false;
		}
	}
	
	{
		ACE_Sig_Action sigpipe_action(SIG_IGN, SIGPIPE);
		ACE_Sig_Action sigterm_action(&Daemon::sigtermHandler, SIGTERM);
		ACE_Sig_Action sigint_action(&Daemon::sigintHandler, SIGINT);
	}
	
	while (!m_sSigtermReceived && !m_sSigintReceived) {
		m_acceptor.acceptConnections();
	}
	return true;
}

bool
Daemon::loadConfiguration(std::string const& confdir)
{
	ConfigLoader config_loader;
	if (!config_loader.load(confdir)) {
		return false;
	}
	
	UrlPatternsLoader patterns_loader;
	if (!patterns_loader.load(confdir)) {
		return false;
	}
	
	ContentFiltersLoader cf_loader;
	if (!cf_loader.load(confdir)) {
		return false;
	}
	
	GlobalState::WriteAccessor global_state;
	global_state->config() = config_loader.getConfig();
	global_state->urlPatterns() = patterns_loader.getCombinedPatterns();
	global_state->contentFilters() = cf_loader.getFilters();
	
	return true;
}

bool
Daemon::bind()
{
	typedef RefCountableSAP<ACE_SOCK_Acceptor> Acceptor;
	typedef IntrusivePtr<Acceptor> AcceptorPtr;
	
	bool ok = true;
	deque<AcceptorPtr> acceptors;
	list<InetAddr> addrs = GlobalState::ReadAccessor()->config().getListenAddresses();
	for (; !addrs.empty(); addrs.pop_front()) {
		InetAddr const& addr = addrs.front();
		ACE_INET_Addr resolved_addr;
		if (!addr.resolve(resolved_addr)) {
			ok = false;
			cerr << "Could not resolve listen address \"" << addr << '"' << endl;
			continue;
		}
		AcceptorPtr acceptor(new Acceptor);
		if (acceptor->open(resolved_addr, true) == -1) {
			ok = false;
			cerr << "Could not bind to \"" << addr << '"' << endl;
			continue;
		}
		acceptors.push_back(acceptor);
	}
	if (!ok) {
		return false;
	}
	if (acceptors.empty()) {
		cerr << "No addresses to listen on!" << endl;
		return false;
	}
	
	for (; !acceptors.empty(); acceptors.pop_front()) {
		m_acceptor.add(acceptors.front());
	}
	
	return true;
}

void
Daemon::sigtermHandler(int)
{
	m_sSigtermReceived = true;
}

void
Daemon::sigintHandler(int)
{
	m_sSigintReceived = true;
}


/* =============================== main ================================*/

// ACE #define's main() to include its own initialization
int main(int argc, char **argv)
{
	bool nodaemon = false;
	std::string confdir = string(SYSCONFDIR) + "/bfilter";
	
	try {
		po::options_description desc("Allowed options");
		desc.add_options()
			("help,h", "Print help message")
			("version,v", "Print version")
			("confdir,c", po::value<string>(&confdir), "Set custom config directory")
			("nodaemon,n", po::bool_switch(&nodaemon), "Disable background daemon mode")
		;
		po::variables_map vm;
		po::store(po::parse_command_line(argc, argv, desc), vm);
		po::notify(vm);
		if (vm.count("help")) {
			std::cout << desc << std::endl;
			return 0;
		}
		if (vm.count("version")) {
			std::cout << BFILTER_VERSION << std::endl;
			return 0;
		}
	} catch (std::exception& e) {
		std::cerr << e.what() << std::endl;
		exit(EXIT_FAILURE);
	}
	
	Daemon daemon;
	if (!daemon.init(confdir)) {
		exit(EXIT_FAILURE);
	}
	if (!daemon.run(nodaemon)) {
		exit(EXIT_FAILURE);
	}
	
	return 0;
}
