/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "UrlPatterns.h"
#include "TextPattern.h"
#include "BString.h"
#include <ace/config.h>
#include <boost/regex.hpp>
#include <stdexcept>

using namespace std;

UrlPatterns::UrlPatterns()
{
}

UrlPatterns::UrlPatterns(UrlPatterns const& other)
:	m_forbidPatterns(other.m_forbidPatterns),
	m_allowPatterns(other.m_allowPatterns),
	m_substPatterns(other.m_substPatterns),
	m_hintPatterns(other.m_hintPatterns)
{
}

UrlPatterns::~UrlPatterns()
{
}

UrlPatterns&
UrlPatterns::operator=(UrlPatterns const& other)
{
	m_forbidPatterns = other.m_forbidPatterns;
	m_allowPatterns = other.m_allowPatterns;
	m_substPatterns = other.m_substPatterns;
	m_hintPatterns = other.m_hintPatterns;
	return *this;
}

void
UrlPatterns::clear()
{
	m_forbidPatterns.clear();
	m_allowPatterns.clear();
	m_substPatterns.clear();
	m_hintPatterns.clear();
}

void
UrlPatterns::addForbidPattern(PatternPtr const& pattern)
{
	m_forbidPatterns.push_back(pattern);
}

void
UrlPatterns::addAllowPattern(PatternPtr const& pattern)
{
	m_allowPatterns.push_back(pattern);
}

void
UrlPatterns::addSubstitutionPattern(PatternPtr const& pattern, Substitution subst)
{
	m_substPatterns.push_back(SubstPatterns::value_type(pattern, subst));
}

void
UrlPatterns::addHintPattern(PatternPtr const& pattern, int hint)
{
	m_hintPatterns.push_back(HintPatterns::value_type(pattern, hint));
}

bool
UrlPatterns::isForbidden(BString const& subj) const
{
	ForbidPatterns::const_iterator it = m_forbidPatterns.begin();
	ForbidPatterns::const_iterator const end = m_forbidPatterns.end();
	for (; it != end; ++it) {
		try {
			bool const matched = boost::regex_match(
				subj.begin(), subj.end(), (*it)->regex()
			);
			if (matched) {
				return true;
			}
		} catch (std::runtime_error& e) {}
	}
	return false;
}

bool
UrlPatterns::isExplicitlyAllowed(BString const& subj) const
{
	AllowPatterns::const_iterator it = m_allowPatterns.begin();
	AllowPatterns::const_iterator const end = m_allowPatterns.end();
	for (; it != end; ++it) {
		try {
			bool const matched = boost::regex_match(
				subj.begin(), subj.end(), (*it)->regex()
			);
			if (matched) {
				return true;
			}
		} catch (std::runtime_error& e) {}
	}
	return false;
}

UrlPatterns::Substitution
UrlPatterns::getSubstitutionFor(BString const& subj) const
{
	SubstPatterns::const_iterator it = m_substPatterns.begin();
	SubstPatterns::const_iterator const end = m_substPatterns.end();
	for (; it != end; ++it) {
		try {
			bool const matched = boost::regex_match(
				subj.begin(), subj.end(), it->first->regex()
			);
			if (matched) {
				return it->second;
			}
		} catch (std::runtime_error& e) {}
	}
	return NO_SUBSTITUTION;
}

int
UrlPatterns::getHintFor(BString const& subj) const
{
	HintPatterns::const_iterator it = m_hintPatterns.begin();
	HintPatterns::const_iterator const end = m_hintPatterns.end();
	for (; it != end; ++it) {
		try {
			bool const matched = boost::regex_match(
				subj.begin(), subj.end(), it->first->regex()
			);
			if (matched) {
				return it->second;
			}
		} catch (std::runtime_error& e) {}
	}
	return 0;
}
