/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "WorkerThreadPool.h"
#include "ServiceContext.h"
#include "Reactor.h"
#include "ReactorFactory.h"
#include "SynchFactory.h"
#include "Client.h"
#include "ScopedIncDec.h"
#include "Debug.h"
#include <ace/Thread_Manager.h>
#include <ace/Time_Value.h>
#include <ace/Reverse_Lock_T.h>
#include <ace/OS_NS_sys_time.h>
#include <memory>
#include <stdexcept>

using namespace std;

class WorkerThreadPool::ReactorRegistration
{
public:
	ReactorRegistration(Reactor* reactor, ReactorList& list);
	
	~ReactorRegistration();
private:
	ReactorList& m_rList;
	ReactorList::iterator m_iter;
};


WorkerThreadPool::WorkerThreadPool()
:	m_cond(m_mutex),
	m_lastThreadCreateTime(0),
	m_numWaiters(0),
	m_isClosing(false)
{
}

WorkerThreadPool::~WorkerThreadPool()
{
	{
		ACE_GUARD(Mutex, guard, m_mutex);
		
		m_isClosing = true;
		m_cond.broadcast();
		
		ReactorList::iterator it = m_runningReactors.begin();
		ReactorList::iterator const end = m_runningReactors.end();
		for (; it != end; ++it) {
			(*it)->stop();
		}
	}
	m_threadManager.wait();
}

void
WorkerThreadPool::handleConnection(
	SocketPtr const& socket, ACE_INET_Addr const& client_addr)
{
	ACE_GUARD(Mutex, guard, m_mutex);
	
	m_requestQueue.push_back(Connection(socket, client_addr));
	
	if (m_numWaiters == 0) {
		m_lastThreadCreateTime = ACE_OS::gettimeofday();
		if (m_threadManager.spawn(&WorkerThreadPool::runThread,
		    this, THR_SCOPE_SYSTEM|THR_DETACHED) == -1) {
			DEBUGLOG("could not spawn a new thread");
			m_requestQueue.pop_back();
		}
	} else {
		m_cond.signal();
	}
}

int
WorkerThreadPool::wait()
{
	ScopedIncDec<volatile int> waiters_manager(m_numWaiters);
	
	ACE_Time_Value tv(WAIT_FOR_NEW_TASK);
	tv += ACE_OS::gettimeofday();
	return m_cond.wait(&tv);
}

void
WorkerThreadPool::runService()
{
	ACE_GUARD(Mutex, guard, m_mutex);
	
	if (m_isClosing) {
		return;
	}
	
	auto_ptr<Reactor> reactor;
	try {
		reactor = ReactorFactory::createBestReactor(SynchFactory<ACE_NULL_SYNCH>());
	} catch (Reactor::Exception& e) {
		DEBUGLOG2("Could not initialize the reactor: " << e.what());
		if (!m_requestQueue.empty()) {
			m_requestQueue.pop_front();
		}
		return;
	}
	
	ReactorRegistration reactor_registration(reactor.get(), m_runningReactors);
	ServiceContext service_context(*reactor);
	
	while (true) {
		if (m_isClosing) {
			break;
		}
		while (m_requestQueue.empty()) {
			if (m_numWaiters >= WAITERS_KEEP_ALIVE && ACE_OS::gettimeofday()
			    > m_lastThreadCreateTime + ACE_Time_Value(WAIT_FOR_NEW_TASK)) {
				return;
			}
			if (wait() == -1) {
				if (m_numWaiters >= WAITERS_KEEP_ALIVE) {
					return;
				}
			}
			if (m_isClosing) {
				return;
			}
		}
		
		SocketPtr socket = m_requestQueue.front().first;
		service_context.clientAddr() = m_requestQueue.front().second;
		m_requestQueue.pop_front();
		
		{
			typedef ACE_Reverse_Lock<Mutex> ReverseLock;
			ReverseLock reverse_lock(m_mutex);
			ACE_GUARD(ReverseLock, unguard, reverse_lock);
			
			Client client(service_context, socket);
			if (reactor->runEventLoop() == Reactor::FAILURE) {
				return;
			}
			
			service_context.reset();
			reactor->restart(); // in case it have been stopped
		}
	}
}

ACE_THR_FUNC_RETURN
WorkerThreadPool::runThread(void* arg)
{
	static_cast<WorkerThreadPool*>(arg)->runService();
	return 0;
}


/*================ WorkerThreadPool::ReactorRegistration ===============*/

WorkerThreadPool::ReactorRegistration::ReactorRegistration(
	Reactor* reactor, ReactorList& list)
:	m_rList(list),
	m_iter(list.insert(list.end(), reactor))
{
}

WorkerThreadPool::ReactorRegistration::~ReactorRegistration()
{
	m_rList.erase(m_iter);
}
