/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef RESPONSEFILTERCHAIN_H_
#define RESPONSEFILTERCHAIN_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AbstractResponseHandler.h"
#include "RequestPtr.h"
#include "DownloadProgress.h"
#include "NonCopyable.h"
#include "IntrusivePtr.h"
#include "FilterGroupTag.h"
#include <memory>
#include <string>
#include <deque>
#include <set>
#include <map>
#include <utility>

class AbstractResponseFilter;
class HttpResponseMetadata;
class HttpRequestMetadata;
class ResponseErrorInitiator;
class ServiceContext;
class ErrorDescriptor;
class FilterTryList;
class JsFilterContext;

class ResponseFilterChain : public AbstractResponseHandler
{
	DECLARE_NON_COPYABLE(ResponseFilterChain)
public:
	ResponseFilterChain(
		ServiceContext& context,
		ConstRequestPtr const& request,
		std::auto_ptr<FilterTryList> filter_try_list,
		IntrusivePtr<AbstractResponseHandler> const& final_recipient,
		ResponseErrorInitiator& error_initiator);
	
	virtual ~ResponseFilterChain();

	virtual void processProvisionalResponse(std::auto_ptr<HttpResponseMetadata> metadata);
	
	virtual void processResponseMetadata(std::auto_ptr<HttpResponseMetadata> metadata);
	
	// postcondition: data is empty
	virtual void processBodyData(SplittableBuffer& data, bool eof, DownloadProgress const& progress);
	
	virtual void processError(std::auto_ptr<ErrorDescriptor> edesc);

	ServiceContext& getContext() { return m_rContext; }

	HttpRequestMetadata const& getRequest() const { return *m_ptrRequest; }

	ConstRequestPtr const& getRequestPtr() const { return m_ptrRequest; }

	AbstractResponseHandler& getFinalRecipient() { return *m_ptrFinalRecipient; }

	ResponseErrorInitiator& getErrorInitiator() { return m_rErrorInitiator; }

	void appendFilter(IntrusivePtr<AbstractResponseFilter> const& filter);
	
	IntrusivePtr<AbstractResponseFilter> nextFilter(HttpResponseMetadata const& metadata);
	
	bool isFlagSet(FilterGroupTag const& group, std::string const& flag) const;
	
	void setFlag(FilterGroupTag const& group, std::string const& flag);
	
	void clearFlag(FilterGroupTag const& group, std::string const& flag);
	
	DownloadProgress const& getDownloadProgress() const { return m_downloadProgress; }
	
	JsFilterContext& getJsContextFor(FilterGroupTag const& group);
private:
	class Terminator;
	class RcJsFilterContext;

	typedef void (*Constructor)(ResponseFilterChain&, HttpResponseMetadata const&);
	typedef std::pair<FilterGroupTag, std::string> FilterFlag;
	
	ServiceContext& m_rContext;
	ConstRequestPtr m_ptrRequest;
	std::auto_ptr<FilterTryList> m_ptrFilterTryList;
	IntrusivePtr<AbstractResponseHandler> const m_ptrFinalRecipient;
	ResponseErrorInitiator& m_rErrorInitiator;
	IntrusivePtr<AbstractResponseFilter> m_ptrTerminator;
	IntrusivePtr<AbstractResponseFilter> m_ptrFirstFilter;
	std::deque<IntrusivePtr<AbstractResponseFilter> > m_filterQueue;
	std::set<FilterFlag> m_flags;
	std::map<FilterGroupTag, IntrusivePtr<RcJsFilterContext> > m_jsContexts;
	DownloadProgress m_downloadProgress;
};

#endif
