/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPFETCHER_H_
#define HTTPFETCHER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "SplittableBuffer.h"
#include "IntrusivePtr.h"
#include "RequestPtr.h"
#include <stddef.h>
#include <limits>
#include <memory>

class Server;
class AbstractRequestHandler;
class HttpResponseMetadata;
class ErrorDescriptor;
class ResponseFilterChain;

class HttpFetcher : private NonCopyable
{
public:
	enum Status { IN_PROGRESS, COMPLETE, FAILED };
	
	HttpFetcher(Server& server, ConstRequestPtr const& request,
		size_t max_body_size = std::numeric_limits<size_t>::max(),
		size_t max_fetch_size = std::numeric_limits<size_t>::max());
	// The body may be decompressed while being received, so max_body_size
	// is typically bigger than max_fetch_size.
	
	~HttpFetcher();
	
	Status getStatus() const { return m_status; }
	
	std::auto_ptr<HttpResponseMetadata>& responseMetadata() { return m_ptrResponseMetadata; }
	
	std::auto_ptr<HttpResponseMetadata> const& responseMetadata() const { return m_ptrResponseMetadata; }
	
	SplittableBuffer& responseBody() { return m_responseBody; }
	
	SplittableBuffer const& responseBody() const { return m_responseBody; }
	
	size_t getReceivedBodySize() const { return m_receivedBodySize; }
	
	std::auto_ptr<ErrorDescriptor>& errorDescriptor() { return m_ptrErrDesc; }
	
	std::auto_ptr<ErrorDescriptor> const& errorDescriptor() const { return m_ptrErrDesc; }
private:
	class ResponseHandler;
	friend class HttpFetcher::ResponseHandler;
	
	void handleData(SplittableBuffer& data, bool eof);
	
	void handleError(std::auto_ptr<ErrorDescriptor> edesc);
	
	static void createSizeLimiterFilter(
		ResponseFilterChain& chain,
		HttpResponseMetadata const& metadata, size_t limit);
	
	Server& m_rServer;
	ConstRequestPtr m_ptrRequest;
	IntrusivePtr<ResponseHandler> m_ptrResponseHandler;
	IntrusivePtr<AbstractRequestHandler> m_ptrRequestHandler;
	Status m_status;
	size_t m_maxBodySize;
	size_t m_maxFetchSize;
	size_t m_receivedBodySize;
	SplittableBuffer m_responseBody;
	std::auto_ptr<HttpResponseMetadata> m_ptrResponseMetadata;
	std::auto_ptr<ErrorDescriptor> m_ptrErrDesc;
};

#endif
