/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTMLNODE_H_
#define HTMLNODE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RefCountable.h"
#include "RefCounter.h"
#include "IntrusivePtr.h"
#include "SplittableBuffer.h"
#include <map>
#include <string>

class BString;
class HtmlNode;

typedef IntrusivePtr<HtmlNode> HtmlNodePtr;

class HtmlNode : public RefCountable<RefCounter<ACE_NULL_SYNCH> >
{
public:
	enum NodeType {
		DOCUMENT_NODE,
		FRAGMENT_NODE,
		ELEMENT_NODE,
		TEXT_NODE
	};
	typedef std::map<BString, BString> AttributeMap;
protected:
	HtmlNode();
	
	HtmlNode(HtmlNode* parent); // to be used only by HtmlNodeWithChildren::Sentinel
public:
	virtual ~HtmlNode();
	
	virtual HtmlNodePtr clone() const = 0;
	
	virtual NodeType getNodeType() const = 0;
	
	virtual SplittableBuffer const& getContent() const = 0;
	
	virtual void setContent(SplittableBuffer const& content) = 0;
	
	virtual void appendContent(
		SplittableBuffer::ByteIterator const& begin,
		SplittableBuffer::ByteIterator const& end) = 0;
	
	virtual void appendContent(BString const& text) = 0;
	
	virtual BString const& getTagName() const = 0;
	
	virtual void setTagName(BString const& name) = 0;
	
	virtual int getTagType() const = 0;
	
	virtual void setTagType(int type) = 0;
	
	virtual bool isClosed() const = 0;
	
	virtual void setClosed(bool closed) = 0;
	
	virtual bool hasAdMark() const = 0;
	
	virtual void setAdMark(bool mark) = 0;
	
	virtual AttributeMap const& getRawAttributes() const = 0;
	
	virtual bool hasAttribute(BString const& name) const = 0;
	
	virtual bool isBooleanAttribute(BString const& name) const = 0;
	
	virtual BString getRawAttribute(BString const& name) const = 0;
	
	virtual BString getAttribute(BString const& name) const = 0;
	
	virtual void setRawAttribute(BString const& name, BString const& value) = 0;
	
	virtual void setAttribute(BString const& name, BString const& value) = 0;
	
	virtual void setBooleanAttribute(BString const& name) = 0;
	
	virtual void removeAttribute(BString const& name) = 0;
	
	virtual bool hasChildren() const = 0;
	
	virtual HtmlNode* getFirstChild() = 0;
	
	virtual HtmlNode const* getFirstChild() const = 0;
	
	virtual HtmlNode* getLastChild() = 0;
	
	virtual HtmlNode const* getLastChild() const = 0;
	
	virtual void appendChild(HtmlNode* node) = 0;
	
	virtual void prependChild(HtmlNode* node) = 0;
	
	virtual void removeAllChildren() = 0;
	
	void insertAfter(HtmlNode* node);
	
	void insertBefore(HtmlNode* node);
	
	void detachFromParent();
	
	HtmlNode* getParent() { return m_pParent; }
	
	HtmlNode const* getParent() const { return m_pParent; }
	
	HtmlNode* getNextSibling();
	
	HtmlNode const* getNextSibling() const;
	
	HtmlNode* getPrevSibling();
	
	HtmlNode const* getPrevSibling() const;
protected:
	static void makeSentinelPair(HtmlNode* left, HtmlNode* right);
private:
	HtmlNode* m_pParent;
	HtmlNode* m_pPrev;
	HtmlNode* m_pNext;
};

#endif

