/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CompressorResponseFilter.h"
#include "GzipCompressor.h"
#include "ResponseFilterChain.h"
#include "ResponseErrorInitiator.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include "HttpResponseMetadata.h"
#include "HttpRequestMetadata.h"
#include "HttpStatusLine.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "ErrorResponse.h"
#include "ErrorDescriptor.h"
#include "ErrorCodes.h"
#include "StringUtils.h"

using namespace std;

CompressorResponseFilter::CompressorResponseFilter(ResponseFilterChain& chain)
:	ResponseFilterBase(chain),
	m_isEOF(false)
{
}

CompressorResponseFilter::~CompressorResponseFilter()
{
}

void
CompressorResponseFilter::processMetadata(auto_ptr<HttpResponseMetadata> metadata)
{
	BString const cenc_str("Content-Encoding");
	BString const gzip_str("gzip");
	HttpHeader* hdr = metadata->headers().getHeaderPtr(cenc_str);
	if (!hdr) {
		metadata->headers().setHeader(HttpHeader(cenc_str, gzip_str));
	} else {
		HttpHeaderStructure structure(*hdr);
		structure.appendElement(HttpHeaderElement(gzip_str));
		structure.commitChanges(*hdr);
	}
	metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_UNKNOWN);
	outputMetadata(metadata);
}

void
CompressorResponseFilter::processBodyData(SplittableBuffer& data, bool eof)
{
	m_isEOF |= eof;
	m_compressor.consume(data, m_isEOF);
	SplittableBuffer output;
	size_t written = 0;
	
	do {
		written = m_compressor.retrieveOutput(output, OUTPUT_LIMIT);
		if (written != 0) {
			outputBodyData(output, false);
		}
		if (m_compressor.isError() && !isErrorRaised()) {
			string problem("error compressing response");
			raiseError(auto_ptr<ErrorDescriptor>(
				new ErrorDescriptor(
					ErrorCodes::ERROR_COMPRESSING_RESPONSE, problem,
					sigc::bind(
						sigc::ptr_fun(&ErrorResponse::errGenericError),
						getRequest().requestLine().getURI().toString(),
						problem, int(HttpStatusLine::SC_INTERNAL_SERVER_ERROR)
					)
				)
			));
			return;
		}
	} while (written != 0);
	
	if (m_isEOF) {
		outputBodyData(output, m_isEOF);
	}
}
