/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef CLIENT_H_
#define CLIENT_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "HttpRequestParser.h"
#include "SplittableBuffer.h"
#include "RefCountableSAP.h"
#include "IntrusivePtr.h"
#include "StrongPtr.h"
#include "NonCopyable.h"
#include "Reactor.h"
#include "EventHandler.h"
#include "NullRefCounter.h"
#include "Timer.h"
#include <ace/SOCK_Stream.h>
#include <sigc++/sigc++.h>
#include <memory>

class ServiceContext;
class Server;
class HttpRequestMetadata;
class AbstractRequestHandler;
class URI;

class Client :
	private HttpRequestParser,
	private EventHandler<NullRefCounter>,
	public sigc::trackable
{
	DECLARE_NON_COPYABLE(Client)
public:
	typedef IntrusivePtr<RefCountableSAP<ACE_SOCK_Stream> > SocketPtr;
	
	Client(ServiceContext& context, SocketPtr const& client_stream);
	
	virtual ~Client();
private:
	enum { READ_BUF_SIZE = 8192 };
	enum { MAX_WRITE_BUF_SIZE = 24000 };
	enum { READ_TIMEOUT = 100 };
	enum { WRITE_TIMEOUT = 100 };
	enum { IDLE_TIMEOUT = 20 };
	
	enum Flags {
		SESSION_TERMINATION_SCHEDULED = 1,
		READING_FORBIDDEN = 2, // after a parse error
		READING_SUSPENDED = 4,
		RESPONSE_IN_PROGRESS = 8,
		WRITE_BUFFER_FULL = 16
	};
	
	typedef WeakPtr<ACE_NULL_SYNCH, Client, Client*> ClientWeakPtr;
	
	class ResponseHandler;
	friend class Client::ResponseHandler;
	class ScopedSuspender;
	friend class Client::ScopedSuspender;
	
	void processOutputData();
	
	size_t writeOutputData();
	
	void processInputData(bool eof);
	
	void processResponseData(SplittableBuffer& data, bool eof, long response_id);
	
	bool tryDiscardResponseOutput(long response_id, uintmax_t discard_size);
	
	bool isRequestInProgress() const;
	
	bool isResponseInProgress() const { return m_flags & RESPONSE_IN_PROGRESS; }
	
	void setResponseInProgress(bool val);
	
	bool isIdle() const;
	
	bool isReadingForbidden() const { return m_flags & READING_FORBIDDEN; }
	
	void suspend() { m_flags |= READING_SUSPENDED; }
	
	void resume() { m_flags &= ~READING_SUSPENDED; }
	
	void scheduleSessionTermination() { m_flags |= SESSION_TERMINATION_SCHEDULED; }
	
	bool isSessionTerminationScheduled() const
	{ return m_flags & SESSION_TERMINATION_SCHEDULED; }
	
	void onReadTimeout();
	
	void onWriteTimeout();
	
	void onIdleTimeout();
	
	void terminateSession();
	
	void updateState();
	
	virtual void eventGotMetadata(
		std::auto_ptr<HttpRequestMetadata> metadata, bool is_persistent);
	
	virtual void handleRead(ACE_HANDLE);
	
	virtual void handleWrite(ACE_HANDLE);
	
	ServiceContext& m_rContext;
	SocketPtr m_ptrClientStream;
	SplittableBuffer m_inputData;
	SplittableBuffer m_outputData;
	bool m_isEndOfOutput;
	IntrusivePtr<AbstractRequestHandler> m_ptrRequestHandler;
	std::auto_ptr<URI> m_ptrRequestURI;
	ReactorHandlerId m_handlerId;
	Timer m_readTimer;
	Timer m_writeTimer;
	Timer m_idleTimer;
	Reactor::IOEvents m_eventMask;
	int m_flags;
	int m_isWaitingForWrite;
	long m_responseIdCounter;
	long m_lastResponseToProduceOutput;
	StrongPtr<ACE_NULL_SYNCH, Client, Client*> m_ptrSelf;
};


inline void
Client::setResponseInProgress(bool val)
{
	if (val) {
		m_flags |= RESPONSE_IN_PROGRESS;
	} else {
		m_flags &= ~RESPONSE_IN_PROGRESS;
	}
}

#endif
