/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AdSuspectList.h"
#include "HeuristicScore.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include "StringUtils.h"
#include "URI.h"
#include <string>
#include <cassert>

using namespace std;

AdSuspectList::AdSuspectList()
{
}

AdSuspectList::~AdSuspectList()
{
}

bool
AdSuspectList::isSameHost() const
{
	BString host;
	SuspectList::const_iterator iter(m_suspects.begin());
	SuspectList::const_iterator end(m_suspects.end());
	if (iter != end) {
		host = (*iter)->getLocation().getHost();
		++iter;
	}
	for (; iter != end; ++iter) {
		BString host2 = (*iter)->getLocation().getHost();
		if (!StringUtils::ciEqual(host.begin(), host.end(), host2.begin(), host2.end())) {
			return false;
		}
	}
	return true;
}

bool
AdSuspectList::isConsistentSize() const
{
	int width = -1;
	int height = -1;
	SuspectList::const_iterator iter(m_suspects.begin());
	SuspectList::const_iterator const end(m_suspects.end());
	for (; iter != end; ++iter) {
		if (width == -1) {
			width = (*iter)->getWidth();
		} else if (width != (*iter)->getWidth()) {
			return false;
		}
		if (height == -1) {
			height = (*iter)->getHeight();
		} else if (height != (*iter)->getHeight()) {
			return false;
		}
	}
	return true;
}

AdSuspect const*
AdSuspectList::findMainSuspect() const
{
	if (m_suspects.empty()) {
		return 0;
	}
	SuspectList::const_iterator iter(m_suspects.begin());
	SuspectList::const_iterator const end(m_suspects.end());
	AdSuspect const* ms = iter->get();
	++iter;
	for (; iter != end; ++iter) {
		if ((*iter)->getScore().getNumericScore() > ms->getScore().getNumericScore()) {
			ms = iter->get();
		}
	}
	return ms;
}

int
AdSuspectList::markForAnalyzing(SplittableBuffer const& script_code,
	SplittableBuffer& result, HeuristicScore::Status min_status) const
{
	auto_ptr<AdSuspectList::HostSet> hosts = getUniqueHosts(min_status);
	if (hosts->empty()) {
		result.append(script_code);
		return 0;
	}
	SplittableBuffer buf1, buf2;
	HostSet::iterator it(hosts->begin());
	HostSet::iterator const end(hosts->end());
	int num_marks = markSingleHost(script_code, buf1, *it);
	++it;
	for (; it != end; ++it) {
		num_marks += markSingleHost(buf1, buf2, *it);
		buf1.clear();
		if (++it == end) {
			break;
		}
		num_marks += markSingleHost(buf2, buf1, *it);
		buf2.clear();
	}
	result.appendDestructive(buf1.empty() ? buf2 : buf1);
	return num_marks;
}

auto_ptr<AdSuspectList::HostSet>
AdSuspectList::getUniqueHosts(HeuristicScore::Status min_status) const
{
	auto_ptr<HostSet> hosts(new HostSet);
	SuspectList::const_iterator iter(m_suspects.begin());
	SuspectList::const_iterator const end(m_suspects.end());
	for (; iter != end; ++iter) {
		if ((*iter)->getScore().getStatus() >= min_status) {
			BString const& host = (*iter)->getLocation().getHost();
			if (!host.empty()) {
				hosts->insert(host);
			}
		}
	}
	return hosts;
}

int
AdSuspectList::markSingleHost(SplittableBuffer const& script_code,
	SplittableBuffer& result, BString const& host)
{
	assert(!host.empty());
	
	int num_marks = 0;
	SplittableBuffer::ByteIterator base(script_code.begin());
	SplittableBuffer::ByteIterator const end(script_code.end());
	while (true) {
		SplittableBuffer::ByteIterator pos = script_code.ciFind(base, end, host);
		if (pos == end) {
			break;
		}
		pos += host.size();
		result.append(base, pos);
		base = pos;
		result.append(BString("/bf-analyze"));
		++num_marks;
	}
	result.append(base, end);
	return num_marks;
}
