/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "JsEnvironment.h"
#include "JsRuntime.h"
#include "JsRequestScope.h"
#include "JsRequestSuspender.h"
#include "WeakPtr.h"
#include "Debug.h"
#include "jsapi.h"
#include <ace/Singleton.h>
#include <ace/Synch.h>
#include <string>
#include <cstring>
#include <cctype>
#include <sstream>
#include <algorithm>

using namespace std;


class JsEnvironment::Context
{
public:
	Context();
	
	~Context();
	
	JSContext* getContext() { return m_pContext; }
	
	bool executeScript(std::string const& script,
		std::string const& filename, char const* js_version=0);
	
	void setListener(ListenerAccessor const& listener) {
		m_listenerAccessor = listener;
	}
	
	void removeListener() { m_listenerAccessor = ListenerAccessor(); }
private:
	enum { STACK_SIZE = 15*1024 };
	enum { BRANCH_LIMIT = 1000 };
	
	static Context* getEnvContext(JSContext* cx);
	
	static void errorReporter(JSContext *cx, const char *msg, JSErrorReport *report);
	
	static JSBool branchCallback(JSContext *cx, JSScript *script);
	
	static JSBool cookieGetter(JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool cookieSetter(JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool onloadSetter(JSContext *cx, JSObject *obj, jsval id, jsval *vp);
	
	static JSBool javaEnabled(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool cookieEnabled(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool alert(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool documentWrite(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool documentWriteln(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool documentGetElementById(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool imageConstructor(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	static JSBool doNothing(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	JSBool defineProperty(JSObject *obj, char const* name, jsval value,
		JSPropertyOp getter, JSPropertyOp setter, uintN flags);
	
	JSBool defineProperty(JSObject* obj, char const* name, jsval value);
	
	JSFunction* defineFunction(JSObject *obj, char const* name,
		JSNative call, uintN nargs, uintN flags);
	
	JSFunction* defineFunction(JSObject *obj, char const* name, JSNative call, uintN nargs);
	
	JSObject* defineObject(JSObject *obj, char const *name,
		JSClass *clasp, JSObject *proto, uintN flags);
	
	JSObject* defineObject(JSObject *obj, char const *name, JSClass *clasp);
	
	void setJsVersion(char const* version=0);
	
	static JSClass m_sWindowClass;
	static JSClass m_sDocumentClass;
	static JSClass m_sBodyClass;
	static JSClass m_sNavigatorClass;
	static JSClass m_sLocationClass;
	static JSClass m_sScreenClass;
	static JSClass m_sImageClass;
	static JSClass m_sFormClass;
	static JSClass m_sMimetypeClass;
	static JSClass m_sPluginClass;
	ListenerAccessor m_listenerAccessor;
	JSContext* m_pContext;
	JSObject* m_pWindowObj;
	JSObject* m_pDocumentObj;
	JSObject* m_pFormArray;
	int m_branchCount;
	string m_documentCookie;
};


struct JsEnvironment::VersionPair
{
	char const* strv;
	JSVersion jsv;
};


struct JsEnvironment::VersionComparator
{
	VersionComparator() {}
	
	bool operator()(VersionPair const& lhs, char const* rhs) const {
		return strcmp(lhs.strv, rhs) < 0;
	}
	
	bool operator()(char const* lhs, VersionPair const& rhs) const {
		return strcmp(lhs, rhs.strv) < 0;
	}
};


/*===================== JsEnvironment::Context =========================*/

JSClass JsEnvironment::Context::m_sWindowClass = {
	"Window", JSCLASS_HAS_PRIVATE,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sDocumentClass = {
	"HTMLDocument", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sBodyClass = {
	"HTMLBodyElement", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sNavigatorClass = {
	"Navigator", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sLocationClass = {
	"Location", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sScreenClass = {
	"Screen", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sImageClass = {
	"HTMLImageElement", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sFormClass = {
	"HTMLFormElement", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sMimetypeClass = {
	"MimeType", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

JSClass JsEnvironment::Context::m_sPluginClass = {
	"Plugin", 0,
	JS_PropertyStub, JS_PropertyStub, JS_PropertyStub, JS_PropertyStub,
	JS_EnumerateStub, JS_ResolveStub, JS_ConvertStub, JS_FinalizeStub
};

inline JSBool
JsEnvironment::Context::defineProperty(
	JSObject *obj, char const* name, jsval value,
	JSPropertyOp getter, JSPropertyOp setter, uintN flags)
{
	return JS_DefineProperty(
		m_pContext, obj, name, value, getter, setter, flags
	);
}

inline JSBool
JsEnvironment::Context::defineProperty(
	JSObject* obj, char const* name, jsval value)
{
	return JS_DefineProperty(
		m_pContext, obj, name, value, 0, 0,
		JSPROP_ENUMERATE|JSPROP_READONLY|JSPROP_PERMANENT
	);
}

inline JSFunction*
JsEnvironment::Context::defineFunction(
	JSObject *obj, char const* name,
	JSNative call, uintN nargs, uintN flags)
{
	return JS_DefineFunction(m_pContext, obj, name, call, nargs, flags);
}

inline JSFunction*
JsEnvironment::Context::defineFunction(
	JSObject *obj, char const* name, JSNative call, uintN nargs)
{
	return JS_DefineFunction(
		m_pContext, obj, name, call, nargs,
		JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
}

inline JSObject*
JsEnvironment::Context::defineObject(
	JSObject *obj, char const *name,
	JSClass *clasp, JSObject *proto, uintN flags)
{
	return JS_DefineObject(m_pContext, obj, name, clasp, proto, flags);
}

inline JSObject*
JsEnvironment::Context::defineObject(
	JSObject *obj, char const *name, JSClass *clasp)
{
	return JS_DefineObject(
		m_pContext, obj, name, clasp, 0,
		JSPROP_ENUMERATE|JSPROP_READONLY|JSPROP_PERMANENT
	);
}

JsEnvironment::Context::Context()
:	m_branchCount(0)
{
	JSRuntime* runtime = JsRuntime::rep();
	m_pContext = JS_NewContext(runtime, STACK_SIZE);
	JsRequestScope rscope(m_pContext);
	
	JS_SetErrorReporter(m_pContext, &errorReporter);
	JS_SetBranchCallback(m_pContext, &branchCallback);
	
	m_pWindowObj = JS_NewObject(m_pContext, &m_sWindowClass, 0, 0);
	JS_InitStandardClasses(m_pContext, m_pWindowObj);
	JS_SetPrivate(m_pContext, m_pWindowObj, this);
	
	defineProperty(m_pWindowObj, "self", OBJECT_TO_JSVAL(m_pWindowObj));
	defineProperty(m_pWindowObj, "window", OBJECT_TO_JSVAL(m_pWindowObj));
	defineProperty(m_pWindowObj, "top", OBJECT_TO_JSVAL(m_pWindowObj));
	defineProperty(m_pWindowObj, "parent", OBJECT_TO_JSVAL(m_pWindowObj));
	defineProperty(m_pWindowObj, "history", OBJECT_TO_JSVAL(JS_NewArrayObject(m_pContext, 0, 0)));
	defineFunction(m_pWindowObj, "alert", &alert, 1);
	defineFunction(m_pWindowObj, "open", &doNothing, 1);
	defineFunction(m_pWindowObj, "setTimeout", &doNothing, 2);
	defineFunction(m_pWindowObj, "setInterval", &doNothing, 2);
	defineFunction(m_pWindowObj, "clearTimeout", &doNothing, 1);
	defineFunction(m_pWindowObj, "clearInterval", &doNothing, 1);
	defineFunction(m_pWindowObj, "focus", &doNothing, 0);
	defineFunction(m_pWindowObj, "blur", &doNothing, 0);
	defineFunction(m_pWindowObj, "resizeTo", &doNothing, 2);
	defineFunction(m_pWindowObj, "moveTo", &doNothing, 2);
	defineFunction(m_pWindowObj, "Image", &imageConstructor, 0);
	defineProperty(m_pWindowObj, "onload", JSVAL_VOID,
		0, &onloadSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	
	JSObject* location_obj = defineObject(m_pWindowObj, "location", &m_sLocationClass);
	defineProperty(location_obj, "hostname", JS_GetEmptyStringValue(m_pContext));
	defineProperty(location_obj, "protocol", STRING_TO_JSVAL(JS_NewStringCopyZ(m_pContext, "http:")));
	defineProperty(location_obj, "host", JS_GetEmptyStringValue(m_pContext));
	defineProperty(location_obj, "port", INT_TO_JSVAL(80));
	defineProperty(location_obj, "pathname", STRING_TO_JSVAL(JS_NewStringCopyZ(m_pContext, "/")));
	defineProperty(location_obj, "hash", JS_GetEmptyStringValue(m_pContext));
	defineProperty(location_obj, "href", JS_GetEmptyStringValue(m_pContext));
	defineProperty(location_obj, "search", JS_GetEmptyStringValue(m_pContext));
	JSObject* nav_obj = defineObject(m_pWindowObj, "navigator", &m_sNavigatorClass);
	defineFunction(nav_obj, "javaEnabled", &javaEnabled, 0);
	defineFunction(nav_obj, "cookieEnabled", &cookieEnabled, 0);
	defineProperty(nav_obj, "platform", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Win32")
	));
	defineProperty(nav_obj, "appCodeName", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Mozilla")
	));
	defineProperty(nav_obj, "appName", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Netscape")
	));
	defineProperty(nav_obj, "appVersion", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "5.0 (Windows; en-US)")
	));
	defineProperty(nav_obj, "userAgent", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Mozilla/5.0 (Windows; U; Win98; en-US) Netscape6/6.0")
	));
	
	JSObject* mimetypes_array = JS_NewArrayObject(m_pContext, 0, 0);
	defineProperty(nav_obj, "mimeTypes", OBJECT_TO_JSVAL(mimetypes_array));
	JSObject* plugins_array = JS_NewArrayObject(m_pContext, 0, 0);
	defineProperty(nav_obj, "plugins", OBJECT_TO_JSVAL(plugins_array));
	JSObject* flash_mimetype_obj = defineObject(
		mimetypes_array, "application/x-shockwave-flash", &m_sMimetypeClass
	);
	jsval flash_mimetype_jsval = OBJECT_TO_JSVAL(flash_mimetype_obj);
	JS_SetElement(m_pContext, mimetypes_array, 0, &flash_mimetype_jsval);
	defineProperty(flash_mimetype_obj, "description", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Shockwave Flash 6.0")
	));
	defineProperty(flash_mimetype_obj, "type", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "application/x-shockwave-flash")
	));
	defineProperty(flash_mimetype_obj, "suffixes", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "swf")
	));
	JSObject *flash_plugin_obj = defineObject(
		flash_mimetype_obj, "enabledPlugin", &m_sPluginClass
	);
	jsval flash_plugin_jsval = OBJECT_TO_JSVAL(flash_plugin_obj);
	JS_SetElement(m_pContext, plugins_array, 0, &flash_plugin_jsval);
	defineProperty(plugins_array, "Shockwave Flash", flash_plugin_jsval);
	defineProperty(flash_plugin_obj, "name", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Shockwave Flash")
	));
	defineProperty(flash_plugin_obj, "description", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "Shockwave Flash 6.0")
	));
	defineProperty(flash_plugin_obj, "length", INT_TO_JSVAL(1));
	defineFunction(flash_plugin_obj, "refresh", &doNothing, 0);
	
	JSObject* screen_obj = defineObject(m_pWindowObj, "screen", &m_sScreenClass);
	defineProperty(screen_obj, "width", INT_TO_JSVAL(1024));
	defineProperty(screen_obj, "height", INT_TO_JSVAL(768));
	defineProperty(screen_obj, "availWidth", INT_TO_JSVAL(1024));
	defineProperty(screen_obj, "availHeight", INT_TO_JSVAL(738));
	defineProperty(screen_obj, "colorDepth", INT_TO_JSVAL(16));
	defineProperty(screen_obj, "pixelDepth", INT_TO_JSVAL(16));
	
	JSObject *frames_obj = JS_NewArrayObject(m_pContext, 0, 0);
	defineProperty(m_pWindowObj, "frames", OBJECT_TO_JSVAL(frames_obj));
	
	m_pDocumentObj = defineObject(m_pWindowObj, "document", &m_sDocumentClass);
	defineProperty(m_pDocumentObj, "cookie", JS_GetEmptyStringValue(m_pContext),
		&cookieGetter, &cookieSetter, JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	defineProperty(m_pDocumentObj, "location", OBJECT_TO_JSVAL(location_obj));
	defineProperty(m_pDocumentObj, "domain", JS_GetEmptyStringValue(m_pContext));
	defineProperty(m_pDocumentObj, "referrer", STRING_TO_JSVAL(
		JS_NewStringCopyZ(m_pContext, "http://whatever")
	));
	JSObject* images_array = JS_NewArrayObject(m_pContext, 0, 0);
	defineProperty(m_pDocumentObj, "images", OBJECT_TO_JSVAL(images_array));
	defineProperty(m_pDocumentObj, "URL", JS_GetEmptyStringValue(m_pContext));
	defineFunction(m_pDocumentObj, "write", &documentWrite, 1);
	defineFunction(m_pDocumentObj, "writeln", &documentWriteln, 1);
	defineFunction(m_pDocumentObj, "getElementById", &documentGetElementById, 1);
	defineFunction(m_pDocumentObj, "open", &doNothing, 0);
	defineFunction(m_pDocumentObj, "close", &doNothing, 0);
	
	m_pFormArray = JS_NewArrayObject(m_pContext, 0, 0);
	defineProperty(m_pDocumentObj, "forms", OBJECT_TO_JSVAL(m_pFormArray));
	
	JSObject* body_obj = defineObject(m_pDocumentObj, "body", &m_sBodyClass);
	defineProperty(body_obj, "clientWidth", INT_TO_JSVAL(1024));
	defineProperty(body_obj, "clientHeight", INT_TO_JSVAL(602));
	defineProperty(body_obj, "innerText", JS_GetEmptyStringValue(m_pContext));
	defineProperty(body_obj, "innerHTML", JS_GetEmptyStringValue(m_pContext));
}

JsEnvironment::Context::~Context()
{
	JS_DestroyContext(m_pContext);
}

bool
JsEnvironment::Context::executeScript(
	std::string const& script,
	std::string const& filename, char const* version)
{
	JsRequestScope rscope(m_pContext);
	JSVersion version_saved = JS_GetVersion(m_pContext);
	setJsVersion(version);
	jsval rval;
	JSBool res = JS_EvaluateScript(
		m_pContext, m_pWindowObj,
		script.c_str(), script.length(),
		filename.c_str(), 1, &rval
	);
	JS_SetVersion(m_pContext, version_saved);
	return (res == JS_TRUE);
}

// We don't use RequestScope anywhere below, because the code below is
// called either by Context constructor or by Context::executeScript,
// and is protected by their own request scopes.

void
JsEnvironment::Context::setJsVersion(char const* version)
{
	static VersionPair const versions[] = {
		{ "1.0", JSVERSION_1_0 },
		{ "1.1", JSVERSION_1_1 },
		{ "1.2", JSVERSION_1_2 },
		{ "1.3", JSVERSION_1_3 },
		{ "1.4", JSVERSION_1_4 },
		{ "1.5", JSVERSION_1_5 }
	};
	
	JSVersion jsv = JSVERSION_DEFAULT;
	if (version) {
		VersionPair const* end = versions + sizeof(versions)/sizeof(versions[0]);
		VersionComparator comp;
		VersionPair const* it = std::lower_bound(versions, end, version, comp);
		if (it != end && !comp(version, *it)) {
			jsv = it->jsv;
		}
	}
	JS_SetVersion(m_pContext, jsv);
}

JsEnvironment::Context*
JsEnvironment::Context::getEnvContext(JSContext* cx)
{
	return static_cast<JsEnvironment::Context*>(
		JS_GetPrivate(cx, JS_GetGlobalObject(cx))
	);
}

void
JsEnvironment::Context::errorReporter(
	JSContext* cx, char const* msg, JSErrorReport* report)
{
#if defined(DEBUG) && 1
	DEBUGLOG("========= JS ERROR =========\n" << msg
		<< " [" << report->filename << ':' << report->lineno << ']');
	if (!report->linebuf) {
		return;
	}
	string line(report->linebuf);
	std::replace(line.begin(), line.end(), '\t', ' ');
	DEBUGLOG(line);
	line.resize(0);
	line.append(report->tokenptr - report->linebuf, ' ');
	line += '^';
	DEBUGLOG(line);
#endif
}

JSBool
JsEnvironment::Context::branchCallback(JSContext* cx, JSScript* script)
{
	Context* context = getEnvContext(cx);
	if (++context->m_branchCount < BRANCH_LIMIT) {
		return JS_TRUE;
	} else {
		// infinite loop?
		return JS_FALSE; // terminate the script
	}
}

JSBool
JsEnvironment::Context::cookieGetter(
	JSContext* cx, JSObject* obj, jsval id, jsval* vp)
{
	Context* context = getEnvContext(cx);
	JSString* cookie = JS_NewStringCopyN(
		cx, context->m_documentCookie.c_str(),
		context->m_documentCookie.length()
	);
	*vp = STRING_TO_JSVAL(cookie);
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::cookieSetter(
	JSContext* cx, JSObject* obj, jsval id, jsval* vp)
{
	Context* context = getEnvContext(cx);
	JSString* str = JS_ValueToString(cx, *vp);
	if (str) {
		string cookie(JS_GetStringBytes(str));
		if (context->m_documentCookie.empty()) {
			context->m_documentCookie = cookie;
		} else {
			// quick and dirty solution to convince some
			// scripts that getting/setting a cookie works
			context->m_documentCookie += "; ";
			context->m_documentCookie += cookie;
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::onloadSetter(
	JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	if (!JSVAL_IS_NULL(vp)) {
		Listener* listener = getEnvContext(cx)->m_listenerAccessor.get();
		if (listener) {
			listener->processOnLoadAssignment();
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::javaEnabled(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_FALSE;
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::cookieEnabled(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_TRUE;
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::alert(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	if (argc > 0) {
		JSString* str = JS_ValueToString(cx, argv[0]);
		if (str) {
			char const* data = JS_GetStringBytes(str);
			DEBUGLOG("JS alert: " << data);
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::documentWrite(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	if (argc > 0) {
		JSString* str = JS_ValueToString(cx, argv[0]);
		if (str) {
			char const* data = JS_GetStringBytes(str);
			Listener* listener = getEnvContext(cx)->m_listenerAccessor.get();
			if (listener) {
				JsRequestSuspender suspender(cx);
				listener->processJsOutput(data, strlen(data));
			}
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::documentWriteln(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	if (argc > 0) {
		static char const crlf[] = { '\r', '\n' };
		JSString* str = JS_ValueToString(cx, argv[0]);
		if (str) {
			char const* data = JS_GetStringBytes(str);
			Listener* listener = getEnvContext(cx)->m_listenerAccessor.get();
			if (listener) {
				JsRequestSuspender suspender(cx);
				listener->processJsOutput(data, strlen(data));
				listener->processJsOutput(crlf, sizeof(crlf));
			}
		}
	}
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::documentGetElementById(
	JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	*rval = JSVAL_NULL;
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::imageConstructor(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	JSObject* image_obj = JS_NewObject(cx, &Context::m_sImageClass, 0, 0);
	if (!image_obj) {
		return JS_FALSE;
	}
	getEnvContext(cx)->defineProperty(image_obj, "complete", JSVAL_TRUE);
	*rval = OBJECT_TO_JSVAL(image_obj);
	return JS_TRUE;
}

JSBool
JsEnvironment::Context::doNothing(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	return JS_TRUE;
}


/*============================ JsEnvironment ==============================*/

JsEnvironment::JsEnvironment()
:	m_ptrContext(new Context)
{
}

JsEnvironment::~JsEnvironment()
{
}

bool
JsEnvironment::executeScript(std::string const& script,
	std::string const& filename, char const* version)
{
	return m_ptrContext->executeScript(script, filename, version);
}

void
JsEnvironment::setListener(ListenerAccessor const& listener)
{
	return m_ptrContext->setListener(listener);
}

void
JsEnvironment::removeListener()
{
	m_ptrContext->removeListener();
}
