/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef STRINGUTILS_H_
#define STRINGUTILS_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "BString.h"
#include "BSInputStream.h"
#include <string>
#include <cstddef>
#include <cstring>
#include <cctype>
#include <iterator>
#include <algorithm>
#include <sstream>

class StringUtils
{
public:
	template<typename Iter1, typename Iter2>
	static bool equal(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2) {
		return equalGeneric(begin1, end1, begin2, end2);
	}
	
	template<typename V1, typename V2>
	static bool equal(V1* begin1, V1* end1, V2* begin2, V2* end2);
	
	template<typename Iter1, typename Iter2>
	static bool ciEqual(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	template<typename Iter>
	static bool ciEqual(const std::string& str, Iter begin, Iter end) {
		return ciEqual(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool ciEqual(Iter begin, Iter end, const std::string& str) {
		return ciEqual(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool ciEqual(const std::string& str1, const std::string& str2);
	
	static bool ciEqual(const char* str1, const char* str2);
	
	template<typename Iter1, typename Iter2>
	static bool ciLess(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	template<typename Iter1, typename EndPred1, typename Iter2, typename EndPred2>
	static bool ciLess(Iter1 begin1, EndPred1 not_end1, Iter2 begin2, EndPred2 not_end2);
	
	template<typename Iter>
	static bool ciLess(const std::string& str, Iter begin, Iter end) {
		return ciLess(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool ciLess(Iter begin, Iter end, const std::string& str) {
		return ciLess(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool ciLess(const std::string& str1, const std::string& str2);
	
	static bool ciLess(const char* str1, const char* str2);
	
	template<typename Iter1, typename Iter2>
	static bool ciLessEqual(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	template<typename Iter1, typename EndPred1, typename Iter2, typename EndPred2>
	static bool ciLessEqual(Iter1 begin1, EndPred1 not_end1, Iter2 begin2, EndPred2 not_end2);
	
	template<typename Iter>
	static bool ciLessEqual(const std::string& str, Iter begin, Iter end) {
		return ciLessEqual(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool ciLessEqual(Iter begin, Iter end, const std::string& str) {
		return ciLessEqual(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool ciLessEqual(const std::string& str1, const std::string& str2);
	
	static bool ciLessEqual(const char* str1, const char* str2);
	
	template<typename Iter>
	static void toLower(Iter begin, Iter end);
	
	static void toLower(std::string& str) {
		toLower(str.begin(), str.end());
	}
	
	template<typename Iter>
	static std::string toLowerStr(Iter begin, Iter end);
	
	static std::string toLowerStr(const std::string& str) {
		return toLowerStr(str.data(), str.data()+str.length());
	}
	
	static BString toLowerBString(const BString& str);
	
	template<typename Iter>
	static void toUpper(Iter begin, Iter end);
	
	static void toUpper(std::string& str) {
		toUpper(str.begin(), str.end());
	}
	
	template<typename Iter>
	static std::string toUpperStr(Iter begin, Iter end);
	
	static std::string toUpperStr(const std::string& str) {
		return toUpperStr(str.data(), str.data()+str.length());
	}
	
	static BString toUpperBString(const BString& str);
	
	template<typename ST, typename Iter>
	static std::string join(ST separator, Iter begin, Iter end);
	
	template<typename ST, typename Iter, typename Pred>
	static std::string join(ST separator, Iter begin, Iter end, Pred pred);
	
	template<typename Iter>
	static Iter find(Iter begin, Iter end, char ch) {
		return findGeneric(begin, end, ch);
	}
	
	template<typename V>
	static V* find(V* begin, V* end, char ch);
	
	template<typename Iter>
	static Iter rfind(Iter begin, Iter end, char ch);
	
	template<typename Iter1, typename Iter2>
	static bool startsWith(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2) {
		return startsWithGeneric(begin1, end1, begin2, end2);
	}
	
	template<typename V1, typename V2>
	static bool startsWith(V1* begin1, V1* end1, V2* begin2, V2* end2);
	
	template<typename Iter>
	static bool startsWith(const std::string& str, Iter begin, Iter end) {
		return startsWith(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool startsWith(Iter begin, Iter end, const std::string& str) {
		return startsWith(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool startsWith(const std::string& str1, const std::string& str2);
	
	template<typename Iter1, typename Iter2>
	static bool endsWith(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2) {
		return endsWithGeneric(begin1, end1, begin2, end2);
	}
	
	template<typename V1, typename V2>
	static bool endsWith(V1* begin1, V1* end1, V2* begin2, V2* end2);
	
	template<typename Iter>
	static bool endsWith(const std::string& str, Iter begin, Iter end) {
		return endsWith(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool endsWith(Iter begin, Iter end, const std::string& str) {
		return endsWith(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool endsWith(const std::string& str1, const std::string& str2);
	
	template<typename Iter1, typename Iter2>
	static bool ciStartsWith(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	template<typename Iter>
	static bool ciStartsWith(const std::string& str, Iter begin, Iter end) {
		return ciStartsWith(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool ciStartsWith(Iter begin, Iter end, const std::string& str) {
		return ciStartsWith(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool ciStartsWith(const std::string& str1, const std::string& str2);
	
	template<typename Iter1, typename Iter2>
	static bool ciEndsWith(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	template<typename Iter>
	static bool ciEndsWith(const std::string& str, Iter begin, Iter end) {
		return ciEndsWith(str.data(), str.data()+str.length(), begin, end);
	}
	
	template<typename Iter>
	static bool ciEndsWith(Iter begin, Iter end, const std::string& str) {
		return ciEndsWith(begin, end, str.data(), str.data()+str.length());
	}
	
	static bool ciEndsWith(const std::string& str1, const std::string& str2);
	
	static std::string replace(const std::string& search,
		const std::string& replace, const std::string& subject);
	
	template<typename Iter>
	static Iter ltrim(Iter begin, Iter end);
	
	template<typename Iter>
	static Iter rtrim(Iter begin, Iter end);
	
	template<typename Iter>
	static std::string trimStr(Iter begin, Iter end) {
		return trimStrGeneric(begin, end);
	}
	
	template<typename V>
	static std::string trimStr(V* begin, V* end);
	
	static std::string trimStr(const std::string& str) {
		return trimStr(str.data(), str.data()+str.length());
	}
	
	template<typename T>
	static std::string fromNumber(T number);
	
	template<typename T>
	static T toNumber(const std::string& str, T dflt=0);
	
	template<typename T>
	static T toNumber(const BString& str, T dflt=0);
	
	template<typename T, typename Iter>
	static T parseUnsigned(Iter begin, Iter& end);
	
	template<typename T>
	static T parseUnsigned(const std::string& str);
	
	template<typename T, typename Iter>
	static T parseUnsignedHex(Iter begin, Iter& end);
private:
	template<typename Iter1, typename Iter2>
	static bool equalGeneric(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	static bool equalCharPtr(const void* begin1, const void* end1, const void* begin2, const void* end2);
	
	template<typename Iter>
	static Iter findGeneric(Iter begin, Iter end, char ch);
	
	static void* findCharPtr(const void* begin, const void* end, char ch);
	
	template<typename Iter1, typename Iter2>
	static bool startsWithGeneric(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	static bool startsWithCharPtr(const void* begin1, const void* end1, const void* begin2, const void* end2);
	
	template<typename Iter1, typename Iter2>
	static bool endsWithGeneric(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2);
	
	static bool endsWithCharPtr(const void* begin1, const void* end1, const void* begin2, const void* end2);
	
	template<typename Iter>
	static std::string trimStrGeneric(Iter begin, Iter end);
	
	static std::string trimStrCharPtr(const void* begin, const void* end);
};


namespace StringUtilsHelpers
{
	template<typename T>
	struct IsKindOfChar
	{
		enum { Result = 0 };
	};
	
	template<>
	struct IsKindOfChar<char>
	{
		enum { Result = 1 };
	};
	
	template<>
	struct IsKindOfChar<signed char>
	{
		enum { Result = 1 };
	};
	
	template<>
	struct IsKindOfChar<unsigned char>
	{
		enum { Result = 1 };
	};
	
	template<typename T>
	struct IsKindOfChar<T const> : public IsKindOfChar<T>
	{
	};
}


template<typename V1, typename V2>
inline bool
StringUtils::equal(V1* begin1, V1* end1, V2* begin2, V2* end2)
{
	if (StringUtilsHelpers::IsKindOfChar<V1>::Result && StringUtilsHelpers::IsKindOfChar<V2>::Result) {
		return equalCharPtr(begin1, end1, begin2, end2);
	} else {
		return equalGeneric(begin1, end1, begin2, end2);
	}
}

template<typename Iter1, typename Iter2>
bool
StringUtils::equalGeneric(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	while (begin1 != end1 && begin2 != end2) {
		if (char(*begin1) != char(*begin2)) {
			return false;
		}
		++begin1;
		++begin2;
	}
	return (begin1 == end1 && begin2 == end2);
}



template<typename Iter1, typename Iter2>
bool
StringUtils::ciEqual(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	using namespace std;
	
	while (begin1 != end1 && begin2 != end2) {
		if (tolower(static_cast<unsigned char>(*begin1))
		    != tolower(static_cast<unsigned char>(*begin2))) {
			return false;
		}
		++begin1;
		++begin2;
	}
	return (begin1 == end1 && begin2 == end2);
}

template<typename Iter1, typename Iter2>
bool
StringUtils::ciLess(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	using namespace std;
	
	while (begin1 != end1 && begin2 != end2) {
		int diff = tolower(static_cast<unsigned char>(*begin1))
			- tolower(static_cast<unsigned char>(*begin2));
		if (diff) {
			return diff < 0;
		}
		++begin1;
		++begin2;
	}
	return (begin1 == end1 && begin2 != end2);
}

template<typename Iter1, typename EndPred1, typename Iter2, typename EndPred2>
bool
StringUtils::ciLess(Iter1 begin1, EndPred1 not_end1, Iter2 begin2, EndPred2 not_end2)
{
	using namespace std;

	while (not_end1(begin1) && not_end2(begin2)) {
		int diff = tolower(static_cast<unsigned char>(*begin1))
			- tolower(static_cast<unsigned char>(*begin2));
		if (diff) {
			return diff < 0;
		}
		++begin1;
		++begin2;
	}
	return (!not_end1(begin1) && not_end2(begin2));
}

template<typename Iter1, typename Iter2>
bool
StringUtils::ciLessEqual(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	using namespace std;
	
	while (begin1 != end1 && begin2 != end2) {
		int diff = tolower(static_cast<unsigned char>(*begin1))
			- tolower(static_cast<unsigned char>(*begin2));
		if (diff > 0) {
			return false;
		}
		++begin1;
		++begin2;
	}
	return (begin1 == end1);
}

template<typename Iter1, typename EndPred1, typename Iter2, typename EndPred2>
bool
StringUtils::ciLessEqual(Iter1 begin1, EndPred1 not_end1, Iter2 begin2, EndPred2 not_end2)
{
	using namespace std;
	
	while (not_end1(begin1) && not_end1(begin2)) {
		int diff = tolower(static_cast<unsigned char>(*begin1))
			- tolower(static_cast<unsigned char>(*begin2));
		if (diff > 0) {
			return false;
		}
		++begin1;
		++begin2;
	}
	return (!not_end1(begin1));
}

template<typename Iter>
void
StringUtils::toLower(Iter begin, Iter end)
{
	using namespace std;
	
	for (; begin != end; ++begin) {
		*begin = tolower(static_cast<unsigned char>(*begin));
	}
}

template<typename Iter>
std::string
StringUtils::toLowerStr(Iter begin, Iter end)
{
	using namespace std;
	
	std::string res;
	res.resize(std::distance(begin, end));
	std::string::iterator iter = res.begin();
	for (; begin != end; ++begin, ++iter) {
		*iter = tolower(static_cast<unsigned char>(*begin));
	}
	return res;
}

template<typename Iter>
void
StringUtils::toUpper(Iter begin, Iter end)
{
	using namespace std;
	
	for (; begin != end; ++begin) {
		*begin = toupper(static_cast<unsigned char>(*begin));
	}
}

template<typename Iter>
std::string
StringUtils::toUpperStr(Iter begin, Iter end)
{
	using namespace std;
	
	std::string res;
	res.resize(std::distance(begin, end));
	std::string::iterator iter = res.begin();
	for (; begin != end; ++begin, ++iter) {
		*iter = toupper(static_cast<unsigned char>(*begin));
	}
	return res;
}

template<typename ST, typename Iter>
std::string
StringUtils::join(ST separator, Iter begin, Iter end)
{
	std::ostringstream strm;
	if (begin != end) {
		strm << *begin;
		++begin;
	}
	for (; begin != end; ++begin) {
		strm << separator << *begin;
	}
	return strm.str();
}

template<typename ST, typename Iter, typename Pred>
std::string
StringUtils::join(ST separator, Iter begin, Iter end, Pred pred)
{
	std::ostringstream strm;
	if (begin != end) {
		strm << pred(*begin);
		++begin;
	}
	for (; begin != end; ++begin) {
		strm << separator << pred(*begin);
	}
	return strm.str();
}

template<typename V>
inline V*
StringUtils::find(V* begin, V* end, char ch)
{
	if (StringUtilsHelpers::IsKindOfChar<V>::Result) {
		return static_cast<V*>(findCharPtr(begin, end, ch));
	} else {
		return findGeneric(begin, end, ch);
	}
}

template<typename Iter>
Iter
StringUtils::findGeneric(Iter begin, Iter end, char ch)
{
	for (; begin != end && char(*begin) != ch; ++begin) {}
	return begin;
}

template<typename Iter>
Iter
StringUtils::rfind(Iter begin, Iter end, char ch)
{
	Iter it(end);
	while (it != begin) {
		--it;
		if (char(*it) == ch) {
			return it;
		}
	}
	return end;
}

template<typename V1, typename V2>
inline bool
StringUtils::startsWith(V1* begin1, V1* end1, V2* begin2, V2* end2)
{
	if (StringUtilsHelpers::IsKindOfChar<V1>::Result && StringUtilsHelpers::IsKindOfChar<V2>::Result) {
		return startsWithCharPtr(begin1, end1, begin2, end2);
	} else {
		return startsWithGeneric(begin1, end1, begin2, end2);
	}
}

template<typename Iter1, typename Iter2>
bool
StringUtils::startsWithGeneric(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	while (true) {
		if (begin2 == end2) {
			return true;
		} else if (begin1 == end1) {
			break;
		}
		if (char(*begin1) != char(*begin2)) {
			break;
		}
		++begin1;
		++begin2;
	}
	return false;
}

template<typename V1, typename V2>
inline bool
StringUtils::endsWith(V1* begin1, V1* end1, V2* begin2, V2* end2)
{
	if (StringUtilsHelpers::IsKindOfChar<V1>::Result && StringUtilsHelpers::IsKindOfChar<V2>::Result) {
		return endsWithCharPtr(begin1, end1, begin2, end2);
	} else {
		return endsWithGeneric(begin1, end1, begin2, end2);
	}
}

template<typename Iter1, typename Iter2>
bool
StringUtils::endsWithGeneric(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	while (true) {
		if (begin2 == end2) {
			return true;
		} else if (begin1 == end1) {
			break;
		}
		--end1;
		--end2;
		if (char(*end1) != char(*end2)) {
			break;
		}
	}
	return false;
}

template<typename Iter1, typename Iter2>
bool
StringUtils::ciStartsWith(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	using namespace std;
	
	while (true) {
		if (begin2 == end2) {
			return true;
		} else if (begin1 == end1) {
			break;
		}
		if (tolower(static_cast<unsigned char>(*begin1)) !=
		    tolower(static_cast<unsigned char>(*begin2))) {
			break;
		}
		++begin1;
		++begin2;
	}
	return false;
}

template<typename Iter1, typename Iter2>
bool
StringUtils::ciEndsWith(Iter1 begin1, Iter1 end1, Iter2 begin2, Iter2 end2)
{
	using namespace std;
	
	while (true) {
		if (begin2 == end2) {
			return true;
		} else if (begin1 == end1) {
			break;
		}
		--end1;
		--end2;
		if (tolower(static_cast<unsigned char>(*end1)) !=
		    tolower(static_cast<unsigned char>(*end2))) {
			break;
		}
	}
	return false;
}

template<typename Iter>
Iter
StringUtils::ltrim(Iter begin, Iter end)
{
	using std::isspace; // isspace may be a macro
	while (begin != end && isspace(static_cast<unsigned char>(*begin))) {
		++begin;
	}
	return begin;
}

template<typename Iter>
Iter
StringUtils::rtrim(Iter begin, Iter end)
{
	using std::isspace;
	while (begin != end) {
		--end;
		if (!isspace(static_cast<unsigned char>(*end))) {
			return ++end;
		}
	}
	return end;
}

template<typename V>
inline std::string
StringUtils::trimStr(V* begin, V* end)
{
	if (StringUtilsHelpers::IsKindOfChar<V>::Result) {
		return trimStrCharPtr(begin, end);
	} else {
		return trimStrGeneric(begin, end);
	}
}

template<typename Iter>
std::string
StringUtils::trimStrGeneric(Iter begin, Iter end)
{
	begin = ltrim(begin, end);
	end = rtrim(begin, end);
	std::string res;
	res.reserve(std::distance(begin, end));
	while (begin != end) {
		res += *begin;
		++begin;
	}
	return res;
}

template<typename T>
std::string
StringUtils::fromNumber(T number)
{
	std::ostringstream strm;
	strm << number;
	return strm.str();
}

template<typename T>
T
StringUtils::toNumber(const std::string& str, T dflt)
{
	std::istringstream strm(str);
	strm >> dflt;
	return dflt;
}

template<typename T>
T
StringUtils::toNumber(const BString& str, T dflt)
{
	BSInputStream strm(str);
	strm >> dflt;
	return dflt;
}

template<typename T, typename Iter>
T
StringUtils::parseUnsigned(Iter begin, Iter& end)
{
	T res = 0;
	for (; begin != end; ++begin) {
		uint8_t ch = *begin;
		if (ch >= uint8_t('0') && ch <= uint8_t('9')) {
			res = (res << 3) + (res << 1) + (ch - uint8_t('0'));
		} else {
			end = begin;
			break;
		}
	}
	return res;
}

template<typename T>
inline T
StringUtils::parseUnsigned(const std::string& str)
{
	const char* end = str.data() + str.length();
	return parseUnsigned<T>(str.data(), end);
}

template<typename T, typename Iter>
T
StringUtils::parseUnsignedHex(Iter begin, Iter& end)
{
	T res = 0;
	for (; begin != end; ++begin) {
		uint8_t ch = *begin;
		if (ch >= uint8_t('0') && ch <= uint8_t('9')) {
			res = (res << 4) + (ch - uint8_t('0'));
		} else if (ch >= uint8_t('a') && ch <= uint8_t('f')) {
			res = (res << 4) + 10 + (ch - uint8_t('a'));
		} else if (ch >= uint8_t('A') && ch <= uint8_t('F')) {
			res = (res << 4) + 10 + (ch - uint8_t('A'));
		} else {
			break;
		}
	}
	return res;
}

#endif
