/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SBOutStreamBuf.h"
#include "DataChunk.h"
#include "BString.h"
#include <algorithm>

using namespace std;

/*
Note that we don't touch pbase(), pptr() and epptr() at all.
This way the base class won't try to do any writes by itself,
but instead delegate them to overflow() and xsputn().
*/

inline BString*
SBOutStreamBuf::getAdjacentBlock()
{
	BString* block = m_buf.lastSpan();
	return (block && block->chunk() == m_ptrChunk
		&& block->end() == m_pCur) ? block : 0;
}

SBOutStreamBuf::SBOutStreamBuf(size_t chunk_size)
:	m_chunkSize(chunk_size),
	m_pCur(0),
	m_pEnd(0)
{
}

SBOutStreamBuf::~SBOutStreamBuf()
{
	// here we could try to shrink m_ptrChunk using DataChunk::resize,
	// but the underlying realloc() doesn't guarantee it won't move data elsewhere
}

void
SBOutStreamBuf::swapData(SplittableBuffer& with)
{
	SplittableBuffer temp;
	temp.appendDestructive(with);
	with.appendDestructive(m_buf);
	m_buf.appendDestructive(temp);
}

void
SBOutStreamBuf::clear()
{
	m_buf.clear();
	m_ptrChunk.reset(0);
	m_pCur = 0;
	m_pEnd = 0;
}


SBOutStreamBuf::int_type
SBOutStreamBuf::overflow(int_type c)
{
	if (traits_type::eq_int_type(c, traits_type::eof())) {
		return traits_type::not_eof(c);
	} else {
		if (m_pCur == m_pEnd) {
			nextChunk();
		}
		*m_pCur = traits_type::to_char_type(c);
		if (BString* block = getAdjacentBlock()) {
			block->trimBack(-1);
		} else {
			m_buf.append(m_ptrChunk, m_pCur, m_pCur + 1);
		}
		++m_pCur;
		return c;
	}
}

std::streamsize
SBOutStreamBuf::xsputn(char_type const* s, std::streamsize n)
{
	ptrdiff_t todo = n;
	while (todo != 0) {
		ptrdiff_t remaining = m_pEnd - m_pCur;
		if (remaining == 0) {
			nextChunk();
		}
		ptrdiff_t len = std::min(todo, remaining);
		traits_type::copy(m_pCur, s, len);
		if (BString* block = getAdjacentBlock()) {
			block->trimBack(-len);
		} else {
			m_buf.append(m_ptrChunk, m_pCur, m_pCur + len);
		}
		m_pCur += len;
		s += len;
		todo -= len;
	}
	return n;
}

void
SBOutStreamBuf::nextChunk()
{
	auto_ptr<DataChunk> chunk(DataChunk::create(m_chunkSize));
	m_pCur = (char_type*)chunk->getDataAddr();
	m_pEnd = m_pCur + chunk->getDataSize();
	m_ptrChunk.reset(chunk.release());
}
