/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AtomicOps.h"
#include "AtomicOpsConfig.h"

#ifdef ATOMIC_OPS_X86_GNUC

int32_t AtomicOps::set(int32_t volatile* mem, int32_t newval)
{
	int32_t tmp = newval;
	asm volatile (
		"xchgl %0, %1\n"  // xchg already has lock semantics
		
		: "+r"(tmp), "+m"(*mem)  // %0 : tmp, %1 : *mem
		:
		: "memory"
	);
	return tmp;
}

int32_t AtomicOps::add(int32_t volatile* mem, int32_t delta)
{
	int32_t newval;
	asm volatile (
		"movl %2, %0\n"         // newval = delta
		"lock; xaddl %0, %1\n"  // newval = *mem, *mem += delta
		"addl %2, %0\n"         // newval += delta
		
		: "=&r"(newval), "+m"(*mem) // %0 : newval, %1 : *mem
		: "r"(delta)                // %2 : delta
		: "memory", "cc"
	);
	return newval;
}

#endif

#ifdef ATOMIC_OPS_X86_MSC

#pragma warning (push)
#pragma warning (disable: 4035) // disable the "no return value" warning

int32_t AtomicOps::set(int32_t volatile* mem, int32_t newval)
{
	__asm {
		mov eax, newval
		mov edx, mem
		xchg eax, [edx]
	};
	// return value is already in EAX
}

int32_t AtomicOps::add(int32_t volatile* mem, int32_t delta)
{
	__asm {
		mov eax, delta
		mov edx, mem
		lock xadd [edx], eax
		add eax, delta
	};
	// return value is already in EAX
}

#pragma warning (pop)

#endif

#ifdef ATOMIC_OPS_PPC_GNUC

int32_t AtomicOps::set(int32_t volatile* mem, int32_t newval)
{
	int32_t oldval;                                   
	asm volatile (
		"1:\n"                 // retry label
		"lwarx  %0, 0, %2\n"   // oldval = *mem, reserve
#ifdef __PPC405__
		"sync\n"
#endif
		"stwcx. %1, 0, %2\n"   // if (<reservation valid>) *mem = newval
		"bne-   1b\n"          // else goto 1
		
		: "=&r"(oldval)         // %0 : oldval
		: "r"(newval), "b"(mem) // %1 : newval, %2: *mem
		: "memory", "cc"
	);
	return oldval;
}

int32_t AtomicOps::add(int32_t volatile* mem, int32_t delta)
{
	int32_t oldval, newval;                                   
	asm volatile (
		"1:\n"                 // retry label
		"lwarx  %0, 0, %2\n"   // oldval = *mem, reserve
		"add    %1, %0, %3\n"  // newval = oldval + delta
#ifdef __PPC405__
		"sync\n"
#endif
		"stwcx. %1, 0, %2\n"   // if (<reservation valid>) *mem = newval
		"bne-   1b\n"          // else goto 1
		
		: "=&r"(oldval), "=&r"(newval) // %0 : oldval, %1 : newval 
		: "b"(mem), "r"(delta)         // %2 : *mem, %3 : delta
		: "memory", "cc"
	);
	return newval;
}

#endif

#ifdef ATOMIC_OPS_GENERIC

#include <ace/Synch.h>
#include <ace/Singleton.h>
#include <assert.h>

namespace AtomicOps {

class MutexArray
{
public:
	ACE_Thread_Mutex& getMutexFor(void volatile* address) {
		return m_mutexes[((uintptr_t)address / sizeof(void*)) % NUM_MUTEXES];
	}
private:
	enum { NUM_MUTEXES = 8 };
	
	ACE_Thread_Mutex m_mutexes[NUM_MUTEXES];
};

typedef ACE_Unmanaged_Singleton<MutexArray, ACE_Recursive_Thread_Mutex> MutexArraySingleton;

int32_t set(int32_t volatile* mem, int32_t newval)
{
	ACE_Guard<ACE_Thread_Mutex> guard(
		MutexArraySingleton::instance()->getMutexFor(mem)
	);
	assert(guard.locked());
	
	int32_t oldval = *mem;
	*mem = newval;
	return oldval;
}

int32_t add(int32_t volatile* mem, int32_t delta)
{
	ACE_Guard<ACE_Thread_Mutex> guard(
		MutexArraySingleton::instance()->getMutexFor(mem)
	);
	assert(guard.locked());
	
	int32_t newval = *mem + delta;
	*mem = newval;
	return newval;
}

} // namespace AtomicOps

#endif
