/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#ifndef	_H_JFS_METAPAGE
#define _H_JFS_METAPAGE	

#include <linux/pagemap.h>
#include <linux/jfs/jfs_lock.h>

typedef struct metapage {
	/* Common logsyncblk prefix (see jfs_logmgr.h) */
	uint16			xflag;
	int16			lid;
	int32			lsn;
	CDLL_ENTRY(logsyncblk)	synclist;
	/* End of logsyncblk prefix */

	unsigned long		flag;		/* See Below */
	unsigned long		count;		/* Reference count */
	void			*data;		/* Data pointer */

	/* list management stuff */
	struct metapage		*hash_prev;
	struct metapage		*hash_next;	/* Also used for free list */
	/*
	 * mapping & index become redundant, but we need these here to
	 * add the metapage to the hash before we have the real page
	 */
#ifdef kern22
	struct inode		*mapping;
#else
	struct address_space	*mapping;
#endif
	unsigned long		index;
	wait_queue_head_t	wait;
	struct list_head	io_list;	/* I/O list */
	void (*end_io) (void *);		/* End-I/O function */
	void			*endio_arg;	/* end_io argument */

	/* implementation */
	struct page		*page;
	unsigned long		logical_size;

	/* Journal management stuff to come... */
	int32			clsn;
	unsigned int		nohomeok;
	struct jfs_log		*log;
}	metapage_t;

#ifndef kern22
/*
 * Direct-access address space operations
 */
extern struct address_space_operations direct_aops;
#endif kern22

/* metapage flag */
#define META_LOCKED	0x00000001
#define META_ABSOLUTE	0x00000002
#define META_FREE	0x00000004
#define META_DIRTY	0x00000008
#define META_SYNC	0x00000010
#define META_DISCARD	0x00000020

#define mark_metapage_dirty(mp) { mp->flag |= META_DIRTY; }

/* function prototypes */
extern metapage_t *__get_metapage(
	struct inode *inode,
	unsigned long lblock,
	unsigned int size,
	unsigned long flag,
	unsigned long new);

#define read_metapage(inode, lblock, size, flag)\
	 __get_metapage(inode, lblock, size, flag, FALSE)

#define get_metapage(inode, lblock, size, flag)\
	 __get_metapage(inode, lblock, size, flag, TRUE)

extern void release_metapage(metapage_t *);

#define flush_metapage(mp) \
{\
	(mp)->flag |= (META_DIRTY|META_SYNC|META_DISCARD);\
	release_metapage(mp);\
}

extern void __write_metapage(metapage_t *);

#ifdef kern22
#define __sync_metapage(mp)
#else
#define __sync_metapage(mp) \
	generic_buffer_fdatasync((struct inode *)mp->mapping->host,\
				 mp->page->index, mp->page->index + 1)
#endif

#define write_metapage(mp) \
{\
	(mp)->flag |= META_DIRTY;\
	release_metapage(mp);\
}

#define discard_metapage(mp) \
{\
	(mp)->flag &= ~META_DIRTY;\
	(mp)->flag |= META_DISCARD;\
	release_metapage(mp);\
}
extern void invalidate_metapage(metapage_t *);

extern void hold_metapage(metapage_t *);

#endif /* _H_JFS_METAPAGE */
