/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#ifndef _H_JFS_LOCK
#define _H_JFS_LOCK

#ifdef kern22
#include <asm/spinlock.h>
#include <linux/sched.h>
#else
#include <linux/spinlock.h>
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,18)
typedef struct wait_queue * wait_queue_head_t;
#define init_waitqueue_head init_waitqueue
#define init_MUTEX(sem) *(sem) = MUTEX
#define DECLARE_MUTEX_LOCKED(sem) struct semaphore sem = MUTEX_LOCKED
#define DECLARE_WAITQUEUE(wait, current)\
	struct wait_queue wait = { current, NULL }
#endif

/*
 *	jfs_lock.h
 *
 * JFS lock definition for globally referenced locks
 */

/*
 * Change History :
 *
 */

/*
 *	event
 */
typedef struct {
	wait_queue_head_t wq;
	int32 done;
} event_t;

#define event_init(event) { event->done = 1; init_waitqueue_head(&event->wq); }

#define EVENT_SLEEP(event) {\
	event->done = 0;\
	wait_event(event->wq, event->done);\
}

#define EVENT_WAKEUP(event) {\
	event->done = 1;\
	wake_up(&event->wq);\
}

/*
 *	lock
 */
/* spin lock: thread-interrupt/interrupt-interrupt */
#define	XSPINLOCK_T	spinlock_t	/* uint32 */
#define XSPINLOCK_ALLOC(Lock,Flag,Class,Occurrence)
#define XSPINLOCK_FREE(Lock)
#define XSPINLOCK_INIT(Lock) spin_lock_init(Lock)
#define XSPINLOCK_LOCK(Lock) spin_lock_irq(Lock)
#define XSPINLOCK_UNLOCK(xipl,Lock) spin_unlock_irq(Lock)
#define XSPINLOCK_NOLOCK(xipl,Lock)

/* spin lock: thread-thread */
#define	SPINLOCK_T	spinlock_t	/* uint32 */
#define SPINLOCK_ALLOC(Lock,Flag,Class,Occurrence)
#define SPINLOCK_FREE(Lock)
#define SPINLOCK_INIT(Lock) spin_lock_init(Lock)
#define SPINLOCK_LOCK(Lock) spin_lock(Lock)
#define SPINLOCK_UNLOCK(Lock) spin_unlock(Lock)

#define	MUTEXLOCK_T	struct semaphore
#define MUTEXLOCK_ALLOC(Lock,Flag,Class,Occurrence)
#define MUTEXLOCK_FREE(Lock)
#ifdef kern22
#define MUTEXLOCK_INIT(Lock) *(Lock) = MUTEX
#else
#define MUTEXLOCK_INIT(Lock) init_MUTEX(Lock)
#endif
#define MUTEXLOCK_LOCK(Lock) down(Lock)
#define MUTEXLOCK_UNLOCK(Lock) up(Lock)
#define MUTEXLOCK_NOLOCK(Lock)

/* readers/writer lock: thread-thread */
#ifdef kern22
#define JFS_EXCLUSIVE_LOCKED	1
extern spinlock_t jfs_rwlock_lock;

#define	RDWRLOCK_T	jfs_rwlock_t
#define RDWRLOCK_ALLOC(Lock,Flag,Class,Occurrence)
#define RDWRLOCK_FREE(Lock)
#define RDWRLOCK_INIT(Lock)\
{\
	init_waitqueue_head(&(Lock)->wq);\
	(Lock)->flag = (Lock)->readers = (Lock)->writers = 0;\
}

static inline void READ_LOCK(jfs_rwlock_t *Lock)
{
	DECLARE_WAITQUEUE(wait, current);

	spin_lock(&jfs_rwlock_lock);
	if ((Lock->flag & JFS_EXCLUSIVE_LOCKED) ||
	    (Lock->readers && Lock->writers)) {
		add_wait_queue(&Lock->wq, &wait);
		while((Lock->flag & JFS_EXCLUSIVE_LOCKED) ||
		      (Lock->readers && Lock->writers)) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			spin_unlock(&jfs_rwlock_lock);
			schedule();
			spin_lock(&jfs_rwlock_lock);
		}
		remove_wait_queue(&Lock->wq, &wait);
	}
	Lock->readers++;
	spin_unlock(&jfs_rwlock_lock);
}

static inline void READ_UNLOCK(jfs_rwlock_t *Lock)
{
	uint16 nreaders;
	spin_lock(&jfs_rwlock_lock);
	nreaders = --(Lock)->readers;
	spin_unlock(&jfs_rwlock_lock);
	if (nreaders == 0)
		wake_up(&Lock->wq);
}

static inline void WRITE_LOCK(jfs_rwlock_t *Lock)
{
	DECLARE_WAITQUEUE(wait, current);

	spin_lock(&jfs_rwlock_lock);
	if ((Lock->flag & JFS_EXCLUSIVE_LOCKED) || Lock->readers) {
		Lock->writers++;
		add_wait_queue(&Lock->wq, &wait);
		while((Lock->flag & JFS_EXCLUSIVE_LOCKED) || Lock->readers) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			spin_unlock(&jfs_rwlock_lock);
			schedule();
			spin_lock(&jfs_rwlock_lock);
		}
		remove_wait_queue(&Lock->wq, &wait);
		Lock->writers--;
	}
	Lock->flag |= JFS_EXCLUSIVE_LOCKED;
	spin_unlock(&jfs_rwlock_lock);
}

#define	WRITE_UNLOCK(Lock)\
{\
	(Lock)->flag &= ~JFS_EXCLUSIVE_LOCKED;\
	wake_up(&(Lock)->wq);\
}
#else /* ! kern22 ( >= 2.4.0 )  */
typedef struct rw_semaphore jfs_rwlock_t;
#define RDWRLOCK_T jfs_rwlock_t
#define RDWRLOCK_ALLOC(Lock,Flag,Class,Occurrence)
#define RDWRLOCK_FREE(Lock)
#define RDWRLOCK_INIT(Lock) init_rwsem(Lock)
#define READ_LOCK(Lock) down_read(Lock)
#define READ_UNLOCK(Lock) up_read(Lock)
#define WRITE_LOCK(Lock) down_write(Lock)
#define WRITE_UNLOCK(Lock) up_write(Lock)
#endif /* kern22 */

#define IREAD_LOCK(ip)		READ_LOCK(&(ip)->i_jfs_rdwrlock)
#define IREAD_UNLOCK(ip)	READ_UNLOCK(&(ip)->i_jfs_rdwrlock)
#define IWRITE_LOCK(ip)		WRITE_LOCK(&(ip)->i_jfs_rdwrlock)
#define IWRITE_UNLOCK(ip)	WRITE_UNLOCK(&(ip)->i_jfs_rdwrlock)
#define IWRITE_LOCK_LIST	iwritelocklist

void iwritelocklist(int, ... );

#ifdef _STILL_TO_PORT
/*
 *	Event
 *
 * APIRET APIENTRY KernBlock(uint32 EventID, uint32 Timeout, 
 *			     uint32 Flags,
 *			     void *pLock, 
 *			     uint32 *pData);
 * . thread is inserted on the event list hashed by EventID;
 * . if pLock is not NULL, its lock type is specified by Flags, and
 *   it is released before blocked, and reacquired when resumed
 *   (unless T_NORELOCK option is specified);
 *
 * APIRET APIENTRY KernWakeup(uint32 EventID, uint32 Flags, 
 *			      uint32 *pNumThreads, uint32 Data);
 * . there's no event anchor to test whether there are any waiting threads;
 * . Flags specifies the wakeup group and whether to return result;
 */
#define EVENT_NULL	0
#define	EVENT_WAIT	-1

/* timeout */
#define	TIMEOUT_FOREVER		-1

/* lock type flag for KernBlock() */
#define	T_XSPINLOCK		BLOCK_SPINLOCK
#define	T_SPINLOCK		BLOCK_SPINLOCK
#define	T_TSPINLOCK		BLOCK_SPINLOCK
#define	T_MUTEXLOCK		BLOCK_EXCLUSIVE_MUTEX
#define	T_READLOCK		BLOCK_SHARED_MUTEX
#define	T_WRITELOCK		BLOCK_EXCLUSIVE_MUTEX

#define	T_NORELOCK		BLOCK_NOACQUIRE

#ifdef	_JFS_OS2
/* thread-interrupt */

/* It will be extremely rare that KernBlock will return with *(Event) not
 * set to EVENT_NULL.  Therefore, the case where we will block only once must
 * be optimal.
 */
#define	XEVENT_SLEEP(Event,Lock,Flag)\
{\
	ASSERT((Flag) & T_XSPINLOCK);\
	(*(Event))++;\
	while (1)\
	{\
		KernBlock((ULONG)(Event),TIMEOUT_FOREVER,\
			  (Flag)|BLOCK_UNINTERRUPTABLE,Lock,NULL);\
		if (*(Event) == EVENT_NULL)\
			break;\
		if ((Flag) & T_NORELOCK)\
		{\
			XSPINLOCK_LOCK(Lock);\
			if (*(Event) == EVENT_NULL)\
			{\
				XSPINLOCK_UNLOCK(0,Lock);\
				break;\
			}\
		}\
	}\
}
#define	XEVENT_SLEEP_RC(Event,Lock,Flag,rc)\
{\
	ASSERT((Flag) & T_XSPINLOCK);\
	(*(Event))++;\
	while (1)\
	{\
		KernBlock((ULONG)(Event),TIMEOUT_FOREVER,\
			  (Flag)|BLOCK_UNINTERRUPTABLE,Lock,(ULONG *)&(rc));\
		if (*(Event) == EVENT_NULL)\
			break;\
		if ((Flag) & T_NORELOCK)\
		{\
			XSPINLOCK_LOCK(Lock);\
			if (*(Event) == EVENT_NULL)\
			{\
				XSPINLOCK_UNLOCK(0,Lock);\
				break;\
			}\
		}\
	}\
}
#define	XEVENT_WAKEUP(Event)\
{\
	(*(Event)) = EVENT_NULL;\
	KernWakeup((ULONG)(Event),0,NULL,0);\
}
#define	XEVENT_WAKEUP_RC(Event,rc)\
{\
	(*(Event)) = EVENT_NULL;\
	KernWakeup((ULONG)(Event),WAKEUP_DATA,NULL,rc);\
}

/* thread-thread */
#define	EVENT_SLEEP(Event,Lock,Flag)\
{\
	ASSERT((Flag) & T_MUTEXLOCK);\
	(*(Event))++;\
	while (1)\
	{\
		KernBlock((ULONG)(Event),TIMEOUT_FOREVER,\
			  (Flag)|BLOCK_UNINTERRUPTABLE,Lock,NULL);\
		if (*(Event) == EVENT_NULL)\
			break;\
		if ((Flag) & T_NORELOCK)\
		{\
			MUTEXLOCK_LOCK(Lock);\
			if (*(Event) == EVENT_NULL)\
			{\
				MUTEXLOCK_UNLOCK(Lock);\
				break;\
			}\
		}\
	}\
}
/* Retry logic must exist outside the macro to protect from spurrious signals
 */
#define	EVENT_SLEEP_ONCE(Event,Lock,Flag)\
{\
	ASSERT((Flag) & T_MUTEXLOCK);\
	(*(Event))++;\
	KernBlock((ULONG)(Event),TIMEOUT_FOREVER,\
		  (Flag)|BLOCK_UNINTERRUPTABLE,Lock,NULL);\
}
#define	EVENT_SLEEP_RC(Event,Lock,Flag,rc)\
{\
	ASSERT((Flag) & T_MUTEXLOCK);\
	(*(Event))++;\
	while (1)\
	{\
		KernBlock((ULONG)(Event),TIMEOUT_FOREVER,\
			  (Flag)|BLOCK_UNINTERRUPTABLE,Lock,(ULONG *)&(rc));\
		if (*(Event) == EVENT_NULL)\
			break;\
		if ((Flag) & T_NORELOCK)\
		{\
			MUTEXLOCK_LOCK(Lock);\
			if (*(Event) == EVENT_NULL)\
			{\
				MUTEXLOCK_UNLOCK(Lock);\
				break;\
			}\
		}\
	}\
}
#define	EVENT_WAKEUP(Event)\
{\
	(*(Event)) = EVENT_NULL;\
	KernWakeup((ULONG)(Event),0,NULL,0);\
}
#define	EVENT_WAKEUP_RC(Event,rc)\
{\
	(*(Event)) = EVENT_NULL;\
	KernWakeup((ULONG)(Event),WAKEUP_DATA,NULL,rc);\
}

/*
 *	inode lock (per inode)
 */
#define INODE_LOCK(ip)		SPINLOCK_LOCK(&((ip)->i_nodelock))
#define INODE_UNLOCK(ip)	SPINLOCK_UNLOCK(&((ip)->i_nodelock))

/*
 *	filelock (per inode)
 */
#define	FILELOCK_LOCK(ip)	TSPINLOCK_LOCK(&((ip)->i_gnode.gn_byte_lock))
#define	FILELOCK_UNLOCK(ip)	TSPINLOCK_UNLOCK(&((ip)->i_gnode.gn_byte_lock))
#endif				/* _JFS_OS2 */


/*
 *	file system global lock
 *	-----------------------
 */
extern MUTEXLOCK_T jfsLock;
#define JFS_LOCK()		MUTEXLOCK_LOCK(&jfsLock)
#define JFS_UNLOCK()		MUTEXLOCK_UNLOCK(&jfsLock)

/*
 *	inode read/write lock (per inode)
 */
#define IREAD_LOCK(ip)	\
{			\
	READ_LOCK(&((ip)->i_rdwrlock));	\
}
#define IREAD_UNLOCK(ip)	\
{				\
	READ_UNLOCK(&((ip)->i_rdwrlock));	\
}

#define IWRITE_LOCK(ip)	\
{\
	WRITE_LOCK(&((ip)->i_rdwrlock));\
	if (ip->i_xlock & FSXLOCK)\
	{\
		do {\
			printf("fsxlock:%d\n", ip->i_number);\
		        ip->i_xlock |= FSXWANT;\
			EVENT_SLEEP(&(ip)->i_fsevent, &(ip)->i_rdwrlock, T_WRITELOCK);\
		} while (ip->i_xlock & FSXLOCK);\
	}\
}

#define IWRITE_LOCK_TRY(ip)	WRITE_LOCK_TRY(&((ip)->i_rdwrlock))
#define IWRITE_UNLOCK(ip)	\
{				\
	WRITE_UNLOCK(&((ip)->i_rdwrlock));	\
}
#define IWRITE_LOCK_LIST	iwritelocklist

/*
 *      aggregate-wide rename lock
 */
#define RENAME_LOCK(ipmnt)      MUTEXLOCK_LOCK(&((ipmnt)->i_renamelock))
#define RENAME_UNLOCK(ipmnt)    MUTEXLOCK_UNLOCK(&((ipmnt)->i_renamelock))

#endif				/* _STILL_TO_PORT */

#endif				/* _H_JFS_LOCK */
