/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
#ifndef	_H_JFS_BUFFER
#define _H_JFS_BUFFER

#include <linux/jfs/jfs_filsys.h>
/*
 * This file defines functions that interface with linux's buffer cache.
 *
 * Although JFS supports multiple "block sizes".  It really is architected
 * on a 4K page size.  Metadata and most file data are stored in 4K pages.
 * Only the last fragment of a file, or the first leaf page of a directory,
 * which are under 4K in size may begin on a block boundary other than 4K.
 * In essence, JFS's "blocksize" is actually a fragment size, where the
 * real blocksize is fixed at 4K.
 *
 * For this reason, JFS's blocksize will appear externally (to the kernel)
 * as 4K.  Logic with JFS will deal with the fragments not aligned on 4K
 * boundaries.
 */

/*
 * Perform buffer cache operations on relative blocks within an inode
 */
struct buffer_head *fread(struct inode *, int64);
struct buffer_head *fgetblk(struct inode*, int64);


/*
 * read logical block
 *
 * If the block size is less that 4K, the data pointer may not point to the
 * beginning of the pages data (b_data).
 */
extern __inline__ struct buffer_head *LREAD(
struct super_block	*sb,
int64			lblock,
void			**pdata,
uint32			size)
{
	struct buffer_head *bp;

#ifdef _JFS_4K
	bp = bread(sb->s_dev, (int)lblock, PSIZE);
	if (bp)
		*pdata = bp->b_data;
#else /* ! _JFS_4K */
	int	 page;
	uint	 offset;

	page = lblock >> sb->s_jfs_l2nbperpage;
	offset = ((uint32)lblock & (sb->s_jfs_nbperpage - 1))
		 << sb->s_jfs_l2bsize;
	if (offset + size > PSIZE)
	{
		printk("LREAD: metadata crossing page boundary\n");
		return NULL;
	}
	bp = bread(sb->s_dev, page, PSIZE);
	if (bp)
		*pdata = (char *)bp->b_data + offset;
#endif /* ! _JFS_4K */

	return bp;
}
#endif /* _H_JFS_BUFFER */
