/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include "jfs_types.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "devices.h"
#include "debug.h"


/*
 * NAME: ujfs_get_dev_size
 *
 * FUNCTION: Uses the device driver interface to determine the raw capacity of
 *      the specified device.
 *
 * PRE CONDITIONS:
 *
 * POST CONDITIONS:
 *
 * PARAMETERS:
 *      device  - device
 *      size    - filled in with size of device; not modified if failure occurs
 *      hidden  - filled in with size of hidden part of device; not modified if
 *                failure occurs
 *
 * NOTES:
 *
 * DATA STRUCTURES:
 *
 * RETURNS: 0 if successful; anything else indicates failures
 */
int32 ujfs_get_dev_size( HFILE  device,
                         int64  *size,
                         int64  *hidden)
{   

    off_t  Starting_Position;           /* The position within the file/device upon entry to this function. */
    off_t  Current_Position = 16777215; /* The position we are attempting to read from. */
    off_t  Last_Valid_Position = 0;     /* Last position we could successfully read from. */
    off_t  First_Invalid_Position;      /* The first invalid position we attempted to read from/seek to. */
    off_t  Seek_Result;                 /* The value returned by lseek. */
    size_t Read_Result;                 /* The value returned by read. */
    char   Test_Byte;                   /* Used when validating a position that we successfully used lseek to get to. */

    /* There doesn't seem to be any system call that can be used to determine the size of the
       device which we are attempting to format.  It has been suggested that a binary search
       be used to find the last byte in the partition.  It seems like this might work since
       an lseek to a position within the partition does not return an error while an lseek
       to a position beyond the end of the partition does.  Of course, this proves that
       something in the operating system knows where the end of the partition is, but I can't
       find a way to access that information.  Since I have run out of other ideas, I guess 
       I'll give it a try.                                                                       */

    /* Save the starting position so that we can restore it when we are done! */
    Starting_Position = lseek(device,0,SEEK_CUR);
    if ( Starting_Position < 0 )
        return ERROR_SEEK;

    /* Find a position beyond the end of the partition.  We will start by attempting to seek to and read the
       16777216th byte in the partition.  We start here because a JFS partition must be at least this big.  If
       it is not, then we can not format it as JFS.                                                            */
    do
    {

      /* Seek to the location we wish to test. */
      Seek_Result = lseek(device,Current_Position,SEEK_SET);
      if ( Seek_Result == Current_Position )
      {

          /* Can we read from this location? */
          Read_Result = read(device,&Test_Byte,1);
          if ( Read_Result == 1 )
          {

            /* The current test position is valid.  Save it for future reference. */
            Last_Valid_Position = Current_Position;

            /* Lets calculate the next location to test. */
            Current_Position = ( ( Current_Position + 1 ) * 2 ) - 1;
              
          }

      }

    } while ( ( Seek_Result == Last_Valid_Position ) && ( Read_Result == 1) );

    /* We have exited the while loop, which means that Current Position is beyond the end of the partition
       or is unreadable due to a hardware problem (bad block).  Since the odds of hitting a bad block are very
       low, we will ignore that condition for now.  If time becomes available, then this issue can be revisited. */

    /* Is the drive greater than 16MB? */
    if ( Last_Valid_Position == 0 )
    {

        /* The drive is not larger than 16MB!  Return 0 for the size to ensure that we don't format it. */
        *size = 0;
        
    }
    else
    {

      /* The drive is larger than 16MB.  Now we must find out exactly how large. */

      /* We now have a point within the partition and one beyond it.  The end of the partition must
         lie between the two.  We will use a binary search to find it.                              */

      /* Setup for the binary search. */
      First_Invalid_Position = Current_Position;
      Current_Position = Last_Valid_Position + ( ( Current_Position - Last_Valid_Position ) / 2);
      
      /* Iterate until the difference between the last valid position and the first invalid position is 2 or less. */
      while ( ( First_Invalid_Position - Last_Valid_Position ) > 2 )
      {

        /* Seek to the location we wish to test. */
        Seek_Result = lseek(device,Current_Position,SEEK_SET);
        if ( Seek_Result == Current_Position )
        {

            /* Can we read from this location? */
            Read_Result = read(device,&Test_Byte,1);
            if ( Read_Result == 1)
            {

              /* The current test position is valid.  Save it for future reference. */
              Last_Valid_Position = Current_Position;

              /* Lets calculate the next location to test. It should be half way between the current
                 test position and the first invalid position that we know of.                         */
              Current_Position = Current_Position + ( ( First_Invalid_Position - Last_Valid_Position ) / 2);

            }

        }
        else
          Read_Result = 0;

        if ( Read_Result != 1 )
        {

          /* Out test position is beyond the end of the partition.  It becomes our first known invalid position. */
          First_Invalid_Position = Current_Position;

          /* Our new test position should be half way between our last known valid position and our current test position. */
          Current_Position = Last_Valid_Position + ( ( Current_Position - Last_Valid_Position ) / 2);

        }
        
      }

      /* The size of the drive should be Last_Valid_Position + 1 as Last_Valid_Position is an offset from the beginning of the partition. */
      *size = Last_Valid_Position + 1;
        
    }
       
    /* Restore the original position. */
    if ( Starting_Position != lseek(device,Starting_Position,SEEK_SET) )
        return ERROR_SEEK;

    /* There are no hidden sectors (that I know of) under Linux. */
    *hidden = 0;

    return NO_ERROR;

}

/*
 * NAME: ujfs_open_device
 *
 * FUNCTION: Open the specified device and return the handle and  the
 *           struct DPB information.
 *
 * PRE CONDITIONS:
 *
 * POST CONDITIONS:
 *
 * PARAMETERS:
 *      Device          - name of device to open
 *      FileHandle      - Filled in with handle specified device
 *      SectorSize      - Filled in with sector size of specified device; not
 *                        modified if failure occurs
 *      mode            - Indicates to open read-only, or read-write exclusive
 *
 * NOTES:
 *
 * DATA STRUCTURES:
 *
 * RETURNS: 0 for success; anything else indicates a failure
 */
int32 ujfs_open_device( char    *Device,
                        PHFILE  FileHandle,
                        int32  *SectorSize,
                        int32   mode )
{

    /* Open the device. */
    switch ( mode )
    {
        case RDWR_EXCL :
                         *FileHandle = open(Device,O_RDWR | O_EXCL,0);  /* Open the device with exclusive access in read/write mode. */
                         break;
        case READONLY : 
                        *FileHandle = open(Device,O_RDONLY,0);         /* Open the device in read only mode. */
                        break;
        default: return ERROR_INVALID_FUNCTION;
                 break;
    }

    if ( *FileHandle < 0 )
        return ERROR_FILE_NOT_FOUND;

    /* We need to get the sector size of the device.  I am not sure this is necessary at the current time. */
    *SectorSize = 512;  /* Use OS/2 value for now. */

    return NO_ERROR;
}

#ifdef _JFS_OS2
/*
 * NAME: ujfs_beginformat
 *
 * FUNCTION: unmounts current FSD, and forces JFS to mount the drive
 *
 * PRE CONDITIONS: Drive has been opened and locked
 *
 * POST CONDITIONS: OS/2 and IFS are aware that a format is occurring
 *
 * PARAMETERS:
 *      device  - file handle of an opened device
 *
 * NOTES: This function must be followed by a call to ujfs_redeterminemedia
 *        before the drive is closed.
 *
 * DATA STRUCTURES:
 *
 * RETURNS: return code from DosDevIOCtl
 */
int32 ujfs_beginformat(HFILE device)
{
    unsigned char   CommandInfo = 0;
    unsigned long   ParmLen = sizeof(CommandInfo);
    char            FSDname[] = "JFS";
    unsigned long   DataLen = sizeof(FSDname);

    return DosDevIOCtl(device, IOCTL_DISK, DSK_BEGINFORMAT, &CommandInfo,
                       sizeof(CommandInfo), &ParmLen, FSDname,
                       sizeof(FSDname), &DataLen);
}

/*
 * NAME: ujfs_redeterminemedia
 *
 * FUNCTION: unmounts current FSD, and causes OS/2 to rebuild VPB and mount
 *           the filesystem
 *
 * PRE CONDITIONS: Drive has been opened and locked
 *
 * POST CONDITIONS: File system is mounted - format condition cleared if present
 *
 * PARAMETERS:
 *      device  - file handle of an opened device
 *
 * NOTES:
 *
 * DATA STRUCTURES:
 *
 * RETURNS: return code from DosDevIOCtl
 */
int32 ujfs_redeterminemedia(HFILE device)
{
    unsigned char   CommandInfo = 0;
    unsigned long   ParmLen = sizeof(CommandInfo);
    unsigned char   Reserved = 0;
    unsigned long   DataLen = sizeof(Reserved);

    return DosDevIOCtl(device, IOCTL_DISK, DSK_REDETERMINEMEDIA,
                       &CommandInfo, sizeof(CommandInfo), &ParmLen,
                       &Reserved, sizeof(Reserved), &DataLen);
}
#endif  /* _JFS_OS2 */

/*
 * NAME: ujfs_rw_diskblocks
 *
 * FUNCTION: Read/Write specific number of bytes for an opened device.
 *
 * PRE CONDITIONS:
 *
 * POST CONDITIONS:
 *
 * PARAMETERS:
 *      dev_ptr         - file handle of an opened device to read/write
 *      disk_offset     - byte offset from beginning of device for start of disk
 *                        block read/write
 *      disk_count      - number of bytes to read/write
 *      data_buffer     - On read this will be filled in with data read from
 *                        disk; on write this contains data to be written
 *      mode            - GET: read; PUT: write; VRFY: verify
 *
 * NOTES: A disk address is formed by {#cylinder, #head, #sector}
 *      In order to call this routine, the device must be opened by
 *      calling ujfs_open_device() so that the static global Datap
 *      and ptrklay are properly inited.
 *
 *      Also the DSK_READTRACK and DSK_WRITETRACK is a track based
 *      function. If it needs to read/write crossing track boundary,
 *      additional calls are used.
 *
 * DATA STRUCTURES:
 *
 * RETURNS:
 */
int32 ujfs_rw_diskblocks( HFILE dev_ptr,
                          int64 disk_offset,
                          int32 disk_count,
                          void  *data_buffer,
                          int32 mode )
#ifdef _JFS_OS2
{

    uint32              actual;
    int32               rc;
    ULONG ulPhys_sector;
    ULONG ulNumSectors;
    ULONG ulSectorsPerCylinder;
    USHORT ulSecsPerTrk;
    USHORT numSecs, sCylinder, sHead, sFirstSec;
    uint32 ulbytesPsec;
    unsigned long       ParmLen ;
    struct DPB *Pdpb;
    TRACKLAYOUT *tmp_layout;
    char *ptr;

    Pdpb = &Datap;
    ulbytesPsec = Pdpb->dev_bpb.bytes_per_sector;
    if ( (disk_offset % ulbytesPsec) || (disk_count % ulbytesPsec ) )
    {
        DBG_ERROR(("Internal Error: %s(%d): disk_offset or disk_count is wrong \n",
                   __FILE__, __LINE__ ))
        return ERROR_INVALID_PARAMETER;
    }
    tmp_layout = ptrklay;
    ulSecsPerTrk =  Pdpb->dev_bpb.sectors_per_track;
    ParmLen = sizeof(TRACKLAYOUT);
    ulPhys_sector = disk_offset / Pdpb->dev_bpb.bytes_per_sector;
    ulPhys_sector += Pdpb->dev_bpb.hidden_sectors;
    ulSectorsPerCylinder = ulSecsPerTrk * Pdpb->dev_bpb.number_of_heads;
    sCylinder = (USHORT)((ulPhys_sector) / ulSectorsPerCylinder);
    if ( sCylinder > Pdpb->number_of_tracks )
    {
        DBG_ERROR(("Internal error: %s(%d): Cylinder number %d beyond the maximum\n",
                   __FILE__, __LINE__, sCylinder ))
        return ERROR_INVALID_PARAMETER;
    }
    ptr = (char *)data_buffer;
    ulNumSectors = disk_count / Pdpb->dev_bpb.bytes_per_sector;
    sHead = (USHORT)((ulPhys_sector % ulSectorsPerCylinder) / ulSecsPerTrk);
    sFirstSec = (USHORT)((ulPhys_sector % ulSectorsPerCylinder) % ulSecsPerTrk);
    ptrklay->bCommand = 0;
    ptrklay->usCylinder = sCylinder;
    while ( ptrklay->usCylinder < Pdpb->number_of_tracks )
    {
        ptrklay->usHead = sHead;
        while ( ptrklay->usHead  < Pdpb->dev_bpb.number_of_heads )
        {
            ptrklay->usFirstSector =  sFirstSec;
            numSecs = ulSecsPerTrk - ptrklay->usFirstSector;
            ptrklay->cSectors =
            (numSecs > ulNumSectors) ? ulNumSectors : numSecs;
            actual = ptrklay->cSectors * ulbytesPsec ;
            switch ( mode )
            {
                case GET:
                    rc = DosDevIOCtl(dev_ptr,
                                     IOCTL_DISK,
                                     DSK_READTRACK,
                                     (PVOID)ptrklay,
                                     ParmLen,
                                     &ParmLen,
                                     (void *)ptr,
                                     (ULONG)actual,
                                     (PULONG)&actual);

#if 0
                    rc = DosRead(dev_ptr, data_buffer, (int32)disk_count,
                                 (PULONG)&actual);
                    if ((rc == NO_ERROR) && (actual < disk_count))
                        rc = ERROR_READ_FAULT;
#endif
                    break;
                case PUT:
                    rc = DosDevIOCtl(dev_ptr,
                                     IOCTL_DISK,
                                     DSK_WRITETRACK,
                                     (PVOID)ptrklay,
                                     ParmLen,
                                     &ParmLen,
                                     (void *)ptr,
                                     (ULONG)actual,
                                     (PULONG)&actual);
#if 0
                    rc = DosWrite(dev_ptr, data_buffer, (int32)disk_count,
                                  (PULONG)&actual);
                    if ((rc == NO_ERROR) && (actual < disk_count))
                        rc = ERROR_WRITE_FAULT;
#endif
                    break;
                case VRFY:
                    rc = DosDevIOCtl(dev_ptr,
                                     IOCTL_DISK,
                                     DSK_VERIFYTRACK,
                                     (PVOID)ptrklay,
                                     ParmLen,
                                     &ParmLen,
                                     (void *)ptr,
                                     (ULONG)actual,
                                     (PULONG)&actual);
                    break;
                default:
                    DBG_ERROR(("Internal error: %s(%d): bad mode: %d\n",
                               __FILE__, __LINE__, mode))
                    rc = ERROR_INVALID_HANDLE;
                    break;
            }

            if ( rc != 0 )
            {
                DBG_ERROR(("Internal error: %s(%d): error %d\n",
                           __FILE__, __LINE__, rc))
                return rc;
            }
            ulNumSectors -= ptrklay->cSectors;
            if ( ulNumSectors == 0 )
                goto outloop;
            else
            {
                ptr += actual;
                ptrklay->usHead++;
                sFirstSec = 0; /* for the next track, starting from
                                * sector 0
                                */
            }
        }
        ptrklay->usCylinder++;
        sHead = 0;  /* for the next cylinder, starting from head 0 */
    }
    if ( ulNumSectors > 0 )
    {
        DBG_ERROR(("Internal error: %s(%d): too many sectors to r/w\n",
                   __FILE__, __LINE__ ))
        return ERROR_INVALID_PARAMETER;
    }
    outloop:
    return rc;
}
#elif _JFS_AIX
{
    uint32              actual;
    int32               rc;

    if ((disk_offset >> 32) || (disk_count >> 32))
    {
        DBG_ERROR(("Device Driver interface cannot yet handle large address\n"))
        return EPERM;
    }

    rc = DosSetFilePtr(dev_ptr, (int32)disk_offset, FILE_BEGIN,
                       (PULONG)&actual);
    if ((rc == NO_ERROR) && (actual != disk_offset))
        rc = ERROR_SEEK;
    if (rc)
        return rc;

    switch ( mode )
    {
        case GET:
            rc = DosRead(dev_ptr, data_buffer, (int32)disk_count,
                         (PULONG)&actual);
            if ((rc == NO_ERROR) && (actual < disk_count))
                rc = ERROR_READ_FAULT;
            break;
        case PUT:
            rc = DosWrite(dev_ptr, data_buffer, (int32)disk_count,
                          (PULONG)&actual);
            if ((rc == NO_ERROR) && (actual < disk_count))
                rc = ERROR_READ_FAULT;
            break;
        default:
            DBG_ERROR(("Internal error: %s(%d): bad mode: %d\n", __FILE__,
                       __LINE__, mode))
            rc = ERROR_INVALID_HANDLE;
            break;
    }

    if ( rc != 0 )
    {
        DBG_ERROR(( "Internal error: %s(%d): error %d\n", __FILE__,
                    __LINE__, rc))
    }
    return rc;
}
#endif  /* elif _JFS_AIX */

#ifdef _LINUX
{
    off_t      Actual_Location;
    size_t     Bytes_Transferred;

    Actual_Location = lseek(dev_ptr,disk_offset, SEEK_SET);
    if ( ( Actual_Location < 0 ) || ( Actual_Location != disk_offset ) )
        return ERROR_SEEK;

    switch ( mode )
    {
        case GET:
                  Bytes_Transferred = read(dev_ptr,data_buffer,disk_count);
            break;
        case PUT:
                  Bytes_Transferred = write(dev_ptr,data_buffer,disk_count);
            break;
        default:
            DBG_ERROR(("Internal error: %s(%d): bad mode: %d\n", __FILE__,
                       __LINE__, mode))
            return ERROR_INVALID_HANDLE;
            break; /* Keep the compiler happy. */
    }

    if ( Bytes_Transferred != disk_count )
    {
        if ( mode == GET )
          return ERROR_READ_FAULT;
        else
          return ERROR_WRITE_FAULT;
    }

    return NO_ERROR;
}

#endif /* _LINUX */

/*
 * NAME: ujfs_close
 *
 * FUNCTION: Close the specified device and free the space for
 *           track layout table.
 *
 * PRE CONDITIONS:
 *
 * POST CONDITIONS:
 *
 * PARAMETERS:
 *      Device          - File handle of the opened device
 *
 * NOTES:
 *
 * DATA STRUCTURES:
 *
 * RETURNS: 0 for success; anything else indicates a failure
 */
int32 ujfs_close( HFILE device)
{
#ifdef _JFS_OS2

    char            CommandInfo = '\0';
    unsigned long   ParmLen = sizeof(CommandInfo);
    unsigned long   DataLen = sizeof(CommandInfo);

    DosDevIOCtl(device, IOCTL_DISK, DSK_UNLOCKDRIVE, &CommandInfo,
                sizeof(CommandInfo), &ParmLen, &CommandInfo,
                sizeof(CommandInfo), &DataLen);
    free(ptrklay);
    return DosClose(device);
#endif  /* _JFS_OS2 */

#ifdef _LINUX

    if ( close(device) )
      return ERROR_INVALID_HANDLE;

    return NO_ERROR;

#endif

}


#ifdef _JFS_OS2
/*
 * NAME: ujfs_rw_mbr
 *
 * FUNCTION: Reads/writes the master boot record for a partition.
 *
 * PARAMETERS:
 *      dev_ptr - Device handle
 *      ourmbr  - Master boot record to write, or read into
 *      mode    - GET: read; PUT: write
 *
 * NOTES: Reads or writes to cylinder 0, head 0, sector 0 of the partition.
 *
 * RETURNS: 0 for success; Other indicates failure
 */
static int32
ujfs_rw_mbr(HFILE       dev_ptr,
            struct mbr  *ourmbr,
            int32       mode)
{
    int32       rc;
    ULONG       actual = sizeof(struct mbr);
    ULONG       ParmLen = sizeof(TRACKLAYOUT);

    ptrklay->usFirstSector = 0;
    ptrklay->usHead = 0;
    ptrklay->usCylinder = 0;
    ptrklay->cSectors = 1;

    switch (mode)
    {
        case GET:
            rc = DosDevIOCtl(dev_ptr, IOCTL_DISK, DSK_READTRACK, (PVOID)ptrklay,
                             ParmLen, &ParmLen, (void *)ourmbr, actual, &actual);
            break;
        case PUT:
            rc = DosDevIOCtl(dev_ptr, IOCTL_DISK, DSK_WRITETRACK,
                             (PVOID)ptrklay, ParmLen, &ParmLen, (void *)ourmbr,
                             actual, &actual);
            break;
        default:
            DBG_ERROR(("Internal error: %s(%d): bad mode: %d\n", __FILE__,
                       __LINE__, mode))
            rc = ERROR_INVALID_HANDLE;
            break;
    }
    return rc;
}


/*
 * NAME: ujfs_update_mbr
 *
 * FUNCTION: Update master boot record's partition type to IFS_PART,
 *
 * PARAMETERS:
 *      dev_ptr - Handle for device
 *
 * NOTES: Read master boot record for the partition and set its partition type
 *      to indicate an IFS partition.
 *
 * RETURNS: 0 for success; Other indicates failure
 */
int32
ujfs_update_mbr(HFILE dev_ptr)
{
    int32       rc;
    struct mbr  ourmbr;
    int16       index;
    UCHAR       cur_systind;

    /* Get the master boot record */
    if (rc = ujfs_rw_mbr(dev_ptr, &ourmbr, GET))
        return rc;

    /*
     * Look for a DOS partition: I copied this straight out of the HPFS code.
     * It is looking for the different possible DOS partition types, or an IFS
     * partition type.  I don't understand why it uses the first one it sees.
     */
    /* ADDITIONAL NOTE: It doesn't use the first one it sees, it tries to match
     * the location (lsn/bpb_hidden_sectors).  If not equal, it moves on.
     */
    /* 201823 Begin */
    for ( index=0; index < 4; index++ )
    {
        if (ourmbr.ptbl[index].lsn == Datap.dev_bpb.hidden_sectors)
        {
            cur_systind = ourmbr.ptbl[index].systind;
            /* Found partition, need to change system indicator? */
            if ((cur_systind & ~HIDDEN_PART_MASK) != IFS_PART)
            {
                /* If system indicator is within known range, get hidden att. */
                if (cur_systind <= 0x1f)
                    cur_systind = cur_systind & HIDDEN_PART_MASK;
                else
                    cur_systind = 0;

                /* Set partition type to IFS, preserving hidden attribute */
                ourmbr.ptbl[index].systind = IFS_PART | cur_systind;

                rc = ujfs_rw_mbr(dev_ptr, &ourmbr, PUT);
            }

            break;
        }
    }
    /* 201823 End */


    return rc;
}


/*
 * NAME: ujfs_check_adapter
 *
 * FUNCTION: Checks to see if adapter supports memory above 16 Megabytes
 *
 * PARAMETERS: none
 *
 * NOTES: Device Parmeter Block has been read in by ujfs_open_device
 *
 * RETURNS: 0 for success; Other indicates failure
 */
int32
ujfs_check_adapter()
{
    if (Datap.device_attributes & DA_ABOVE16MB)
        return 0;
    else
        return ERROR_NOT_SUPPORTED;
}


/*
 * NAME: 	ujfs_verify_device_type
 *
 * FUNCTION: 	Query the LVM for the partition type. 
 *
 * PARAMETERS:	none
 *
 * NOTES:
 *
 * RETURNS:
 *      success: 0
 *      failure: something else
 */
int32 ujfs_verify_device_type( HFILE Dev_IOPort ) 
{
    int32 vdt_rc = 0;

    DDI_OS2LVM_param LVMpp;
    DDI_OS2LVM_param *pLVMpp = &LVMpp;
    DDI_OS2LVM_data LVMdp;
    DDI_OS2LVM_data *pLVMdp = &LVMdp;
    ULONG ppLen = 0;
    ULONG dpLen = 0;

    /* 
     * initialize the LVM DosDevIOCtl parm and data packets
     */
    pLVMpp->Command = 0;            /* Identify Volume Type */
    pLVMpp->DriveUnit = 0;
    pLVMpp->TableNumber = 0;
    pLVMpp->LSN = 0;
    pLVMdp->ReturnData = 0;
    pLVMdp->UserBuffer = NULL; 

    ppLen = sizeof(DDI_OS2LVM_param);
    dpLen = sizeof(DDI_OS2LVM_data);

    /* 
     * ask the LVM how many bad block lists it has for the filesystem
     */
    vdt_rc = DosDevIOCtl( Dev_IOPort,           IOC_DC, 
                          IODC_LV,            (void *) pLVMpp,    
                          sizeof(DDI_OS2LVM_param),
                          &ppLen,         (void *) pLVMdp,    
                          sizeof(DDI_OS2LVM_data),    &dpLen
                        );  

    if ( vdt_rc == 0 )
    {   /* DosDevIOCtl successful */
        if ( pLVMdp->ReturnData != 2 )
        {   /* It's not an LVM volume */
            vdt_rc = ERROR_NOT_SUPPORTED;
        }
    }  /* end else DosDevIOCtl successful */

    return( vdt_rc );
}               /* end ujfs_verify_device_type() */


/*  
 * NAME: ujfs_stdout_redirected
 *
 * FUNCTION: Calls the DosQueryHType API on the standard output
 *                handle to determine whether standard output is redirected.
 *
 * PRE CONDITIONS:
 *
 * POST CONDITIONS:
 *
 * PARAMETERS:	none
 *
 * NOTES:
 *
 * DATA STRUCTURES:
 *
 * RETURNS:   0  if standard out IS NOT redirected
 *               -1  if standard out IS redirected
 */

int32 ujfs_stdout_redirected( )
{
    ULONG handtype, devattr;

    DosQueryHType (1, &handtype, &devattr);
    if ( handtype != 1 )
        return( -1 );
    else
        return( 0 );
}

#endif /* _JFS_OS2 */
