/*
 *   MODULE_NAME:		jfs_mount.c
 *
 *   COMPONENT_NAME:		sysjfs
 *
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * Change History :
 *
 */

/*
 * Module: jfs_mount.c
 *
 * note: file system in transition to aggregate/fileset:
 *
 * file system mount is interpreted as the mount of aggregate, 
 * if not already mounted, and mount of the single/only fileset in 
 * the aggregate;
 *
 * a file system/aggregate is represented by an internal inode
 * (aka mount inode) initialized with aggregate superblock;
 * each vfs represents a fileset, and points to its "fileset inode 
 * allocation map inode" (aka fileset inode):
 * (an aggregate itself is structured recursively as a filset: 
 * an internal vfs is constructed and points to its "fileset inode 
 * allocation map inode" (aka aggregate inode) where each inode 
 * represents a fileset inode) so that inode number is mapped to 
 * on-disk inode in uniform way at both aggregate and fileset level;
 *
 * each vnode/inode of a fileset is linked to its vfs (to facilitate
 * per fileset inode operations, e.g., unmount of a fileset, etc.);
 * each inode points to the mount inode (to facilitate access to
 * per aggregate information, e.g., block size, etc.) as well as
 * its file set inode.
 *
 *   aggregate 
 *   ipmnt
 *   mntvfs -> fileset ipimap+ -> aggregate ipbmap -> aggregate ipaimap;
 *             fileset vfs     -> vp(1) <-> ... <-> vp(n) <->vproot;
 */

#include <linux/fs.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_superblock.h>
#include <linux/jfs/jfs_dmap.h>
#include <linux/jfs/jfs_imap.h>
#include <linux/jfs/jfs_metapage.h>
#include <linux/jfs/jfs_logmgr.h>
#include <linux/jfs/jfs_debug.h>

#include <linux/string.h>


/*
 * forward references
 */
int32 readSuper(struct super_block *ipmnt, metapage_t **bpp);
static int32 chkSuper(struct super_block *);
static int32 updateSuper(struct super_block *);
static int32 logMOUNT(struct super_block *sb);

/*
 * NAME:	jfs_mount(vfsp, crp)
 *
 * FUNCTION:	vfs_mount()
 *
 * PARAMETER:	vfsp	- virtual file system pointer
 *		crp	- credential
 *
 * RETURN:	EBUSY	- device already mounted or open for write
 *		EBUSY	- cvrdvp already mounted;
 *		EBUSY	- mount table full
 *		ENOTDIR	- cvrdvp not directory on a device mount
 *		ENXIO	- device open failure
 */
int32 jfs_mount(
#ifdef __linux__
	struct super_block	*sb,
	char			*options,
	int32			silent)
#else /* ! __linux__ */
	struct vfs	*vfsp,		/* vfs to be mounted */
	struct ucred	*crp)		/* credential */
#endif /* ! __linux__ */
{
	int32	rc = 0;			/* Return code		*/
	kdev_t	fsdev;			/* fs device number */
	struct inode	*ipaimap = NULL;
	struct inode	*ipaimap2 = NULL;
	struct inode	*ipimap = NULL;
	struct inode	*ipbmap = NULL;
	log_t *log;
#ifdef _STILL_TO_PORT
	int32	pbsize;
	union	mntvfs	dummyvfs, *mntvfsp;
	struct vnode	*cvrdvp;	/* covered vnode */
	struct vnode	*mntdvp = NULL;	/* mounted device object vnode */
	struct file *fp;		/* mounted fs device file ptr */
	dev_t	logdev = 0;		/* log device number	*/
	struct inode	*ipmnt = NULL;	/* mount inode */
	struct vnode	*vpmnt;		/* mount vnode */
	struct inode	*iplog = NULL;
	struct inode	*ipbmap = NULL;
	struct inode	*ipimap = NULL;
	struct inode	*iplist[8] = {0,0,0,0,0,0,0,0};
	struct inode	**ipp = &iplist[0];
	struct inode *iproot;		/* root inode */
	struct vnode *vproot;		/* root vnode */
	int32	flag;
#endif /* _STILL_TO_PORT */

jFYI(1,("\nMount JFS\n"));

	/*
	 * serialize mount/unmount with big bouncer lock
	 * (until Logical File System provides serialization
	 * between mount, unmount, name/file handle translation)
	 */
//	JFS_LOCK();

	/*
	 * get the file system device being mounted
	 */

	jEVENT(1,("ToDo: Parse mount options: \"%s\"\n", (char *)options));

	fsdev = sb->s_dev;
	sb->s_blocksize = PSIZE;
	sb->s_blocksize_bits = L2PSIZE;
	set_blocksize(fsdev, PSIZE);

	/*
	 *	mount device:
	 */
#ifdef _STILL_TO_PORT
mntDevice:
	/*
	 * get the file system inode (aka mount inode)
	 * (dev_t = file system device, fileset number = 0, i_number = 0)
	 *
	 * hand craft dummy/transient vfs to force iget()/iread() to 
	 * the special case of an in-memory inode allocation without 
	 * on-disk inode
	 */
	bzero(&dummyvfs, sizeof(struct vfs));
	/* dummyvfs.filesetvfs.vfs_data = NULL; */
	dummyvfs.dummyvfs.dev = fsdev;
	/* dummyvfs.dummyvfs.ipmnt = NULL; */
	ICACHE_LOCK();
	rc = iget((struct vfs *)&dummyvfs, 0, &ipmnt, 0);
	ICACHE_UNLOCK();
	if (rc)
		goto errout10;
jEVENT(0,("jfs_mount: ipmnt:0x%p\n", ipmnt));

	/* aggregate mounted ? */
	if (ipmnt->i_devfp)
		goto mntLog;

	/* decouple mount inode from dummy vfs */
	vPut(ipmnt);

	/* ipmnt point itself */
	ipmnt->i_ipmnt = ipmnt;


#endif /* _STILL_TO_PORT */

	/*
	 * read/validate superblock 
	 * (initialize mount inode from the superblock)
	 */
	if ((rc = chkSuper(sb)))
	{
#ifdef _STILL_TO_PORT
		if (rc == EROFS)
			vfsp->vfs_flag |= VFS_READONLY;
		else
#endif /* _STILL_TO_PORT */
			goto errout20;
	}
#ifdef _STILL_TO_PORT
	else
	{
		if (ipmnt->i_mntflag & JFS_DASD_ENABLED)
			vfsp->vfs_flag |= VFS_DASDLIM;

		if (ipmnt->i_mntflag & JFS_DASD_PRIME)
			vfsp->vfs_flag |= VFS_DASDPRIME;
	}
#endif /* _STILL_TO_PORT */

	ipaimap = diReadSpecial(sb, AGGREGATE_I);
	if (ipaimap == NULL) {
		jERROR(1,("jfs_mount: Faild to read AGGREGATE_I\n"));
		rc = EIO;
		goto errout20;
	}
	sb->s_jfs_ipaimap = ipaimap;

jEVENT(1,("jfs_mount: ipaimap:0x%p\n", ipaimap));

	/*
	 * initialize aggregate inode allocation map
	 */
	if ((rc = diMount(ipaimap))) {
		jERROR(1,("jfs_mount: diMount(ipaimap) failed w/rc = %d\n",
			  rc));
		goto errout21;
	}

	/*
	 * open aggregate block allocation map
	 */
	ipbmap = diReadSpecial(sb, BMAP_I);
	if (ipbmap == NULL)
	{
		rc = EIO; 
		goto errout22;
	}

jEVENT(1,("jfs_mount: ipbmap:0x%p\n", ipbmap));

	sb->s_jfs_ipbmap = ipbmap;

	/*
	 * initialize aggregate block allocation map
	 */
	if ((rc = dbMount(ipbmap)))
	{
		jERROR(1,("jfs_mount: dbMount failed w/rc = %d\n", rc));
		goto errout22;
	}

#ifdef _STILL_TO_PORT
#ifdef	_JFS_COMPRESSION
	/*
	 * initialize for compressed file system
	 *
	 * initialize decompression storage. 
	 * if not read-only create compression daemon.
	 */
#endif	/* _JFS_COMPRESSION */

	/*
	 *	mount log associated with the aggregate
	 */
mntLog:
	/* open log ? */
	if (ipmnt->i_iplog ||			/* log opened ? */
	    ipmnt->i_mntflag & JFS_TMPFS ||	/* tmp fs ? */
	    vfsp->vfs_flag & VFS_READONLY)	/* rofs ? */
		goto mntFileset;

	/*
	 * get log device associated with the fs being mounted;
	 */
#ifdef _JFS_OS2
	if (ipmnt->i_mntflag & JFS_INLINELOG)
	{
		vfsp->vfs_logVPB = vfsp->vfs_hVPB;
		vfsp->vfs_logvpfs = vfsp->vfs_vpfsi;
	}
	else if (vfsp->vfs_logvpfs == NULL)
	{
		/*
		 * XXX: there's only one external log per system;
		 */
jERROR(1,("jfs_mount: Mount Failure! No Log Device.\n"));
		goto errout30;
	}

	logdev = vfsp->vfs_logvpfs->vpi_unit;
	ipmnt->i_logdev = logdev;
#endif /* _JFS_OS2 */
#endif /* _STILL_TO_PORT */

	/*
	 * open/initialize log
	 */
	if ((rc = lmLogOpen(sb, &log)))
		goto errout30;

	sb->s_jfs_log = log;

#ifdef _STILL_TO_PORT
	/* validate file system/commit option */
	if ((ipmnt->i_mntflag & JFS_COMMIT) != 
	    (((log_t *)iplog)->flag & JFS_COMMIT))
		goto errout31;
#endif /* _STILL_TO_PORT */

	/*
	 * update file system superblock;
	 */
	if ((rc = updateSuper(sb)))
	{
		jERROR(1,("jfs_mount: updateSuper failed w/rc = %d\n", rc));
		goto errout31;
	}

	/*
	 * write MOUNT log record of the file system
	 */
	logMOUNT(sb);

#ifdef _STILL_TO_PORT
	/* initialize aggregate wide rename lock */
	MUTEXLOCK_ALLOC(&ipmnt->i_renamelock,LOCK_ALLOC_PAGED,
			JFS_RENAME_LOCK_CLASS, -1);
	MUTEXLOCK_INIT(&ipmnt->i_renamelock);

mntAggregate2:
#endif /* _STILL_TO_PORT */
	/*
	 * open the secondary aggregate inode allocation map
	 *
	 * This is a duplicate of the aggregate inode allocation map.
	 *
	 * hand craft a vfs in the same fashion as we did to read ipaimap.
	 * By adding INOSPEREXT (32) to the inode number, we are telling
	 * diReadSpecial that we are reading from the secondary aggregate
	 * inode table.  This also creates a unique entry in the inode hash
	 * table.
	 */
	if ((sb->s_jfs_mntflag & JFS_BAD_SAIT) == 0)
	{
		ipaimap2 = diReadSpecial(sb, AGGREGATE_I + INOSPEREXT);
		if (ipaimap2 == 0) {
			jERROR(1,("jfs_mount: Faild to read AGGREGATE_I\n"));
			goto errout35;
		}
		sb->s_jfs_ipaimap2 = ipaimap2;

		jEVENT(1,("jfs_mount: ipaimap2:0x%p\n", ipaimap2));

		/*
		 * initialize secondary aggregate inode allocation map
		 */
		if ((rc = diMount(ipaimap2))) {
			jERROR(1,("jfs_mount: diMount(ipaimap2) failed, rc = %d\n",
			  rc));
			goto errout35;
		}
	}
	else
		/* Secondary aggregate inode table is not valid */
		sb->s_jfs_ipaimap2 = 0;

	/*
	 *	mount (the only/single) fileset
	 */
//mntFileset:
	/*
	 * open fileset inode allocation map (aka fileset inode)
	 */
	ipimap = diReadSpecial(sb, FILESYSTEM_I);
	if (ipimap == NULL)
	{
		jERROR(1,("jfs_mount: Failed to read FILESYSTEM_I\n"));
		/* open fileset secondary inode allocation map */
		rc = EIO; 
		goto errout40;
	}
jEVENT(1,("jfs_mount: ipimap:0x%p\n", ipimap));

	/* map further access of per fileset inodes by the fileset inode */
	sb->s_jfs_ipimap = ipimap;

	/* initialize fileset inode allocation map */
	if ((rc = diMount(ipimap)))
	{
		jERROR(1,("jfs_mount: diMount failed w/rc = %d\n", rc));
		goto errout41;
	}

#ifdef _STILL_TO_PORT
	/*
	 * get root vnode/inode of mounted fileset
	 */
	rc = iget(vfsp, ROOT_I, &iproot, 0);
	if (rc)
		goto errout42;

        vproot = IP2VP(iproot);
	vproot->v_flag |= V_ROOT;
jEVENT(0,("jfs_mount: iproot:0x%p\n", iproot));

	/* establish mounted/root vnode of mounted fileset */
	vfsp->vfs_mntd = vproot;

#endif /* _STILL_TO_PORT */

jFYI(1,("Mount JFS Complete.\n"));
	goto out;

	/*
	 *	unwind on error
	 */
//errout42: /* close fileset inode allocation map */
	diUnmount(ipimap, 1);

errout41: /* close fileset inode allocation map inode */
#ifdef _STILL_TO_PORT
	bmInodeInvalidate(ipimap);
#endif /* _STILL_TO_PORT */
	diFreeSpecial(ipimap);

errout40: /* fileset closed */

	/* close secondary aggregate inode allocation map */
	if (ipaimap2) {
		diUnmount(ipaimap2, 1);
		diFreeSpecial(ipaimap2);
	}

errout35:

errout31: /* close log */
	if (log)
		lmLogClose(sb, log);

errout30: /* log closed */

	/* close aggregate block allocation map */
	dbUnmount(ipbmap, 1);
	diFreeSpecial(ipbmap);

errout22: /* close aggregate inode allocation map */

	diUnmount(ipaimap, 1);

errout21: /* close aggregate inodes */
	diFreeSpecial(ipaimap);
errout20: /* aggregate closed */
#ifdef _STILL_TO_PORT

errout11: /* close mount inode */
	if (ipmnt->i_count == 1)
	{
		ICACHE_LOCK();
		iunhash(ipmnt);
		ICACHE_UNLOCK();
	}
	else
		ipmnt->i_count--;

errout10: /* device unmounted */

#endif /* _STILL_TO_PORT */

out:
//	JFS_UNLOCK();

	if (rc)
	{
jERROR(1,("Mount JFS Failure: %d\n", rc));
	}
	return rc;
}


/*
 *	chkSuper()
 *
 * validate the superblock of the file system to be mounted and 
 * get the file system parameters.
 *
 * returns
 *	0 with fragsize set if check successful
 *	error code if not successful
 */
static int32
chkSuper(
	struct super_block *sb)
{
	int32	rc = 0;
	metapage_t	*mp;
	struct jfs_superblock *j_sb;
//	int32	niperblk;
	int32 AIM_bytesize, AIT_bytesize;
	int32 expected_AIM_bytesize, expected_AIT_bytesize;
	int64 AIM_byte_addr, AIT_byte_addr, fsckwsp_addr;
	int64 byte_addr_diff0, byte_addr_diff1;

	if ((rc = readSuper(sb, &mp)))
		return rc;
	j_sb = (struct jfs_superblock *)(mp->data);

	/*
	 * validate superblock
	 */
	/* validate fs signature */
	if (strncmp(j_sb->s_magic, JFS_MAGIC, 4) ||
	    j_sb->s_version != JFS_VERSION)
	{
		//rc = EFORMAT;
		rc = EINVAL;
		goto out;
	}

#ifdef _JFS_4K
	if (j_sb->s_bsize != PSIZE)
	{
		jERROR(1,("Currently only 4K block size supported!\n"));
		rc = EINVAL;
		goto out;
	}
#endif /* _JFS_4K */

jFYI(1,("superblock: flag:0x%08x state:0x%08x size:0x%llx\n", 
	j_sb->s_flag, j_sb->s_state, j_sb->s_size));

	/* validate the descriptors for Secondary AIM and AIT */
	if( (j_sb->s_flag & JFS_BAD_SAIT) != JFS_BAD_SAIT ) 
	{
		expected_AIM_bytesize = 2 * PSIZE;
		AIM_bytesize = lengthPXD(&(j_sb->s_aim2)) * j_sb->s_bsize;
		expected_AIT_bytesize = 4 * PSIZE;
		AIT_bytesize = lengthPXD(&(j_sb->s_ait2)) * j_sb->s_bsize;
		AIM_byte_addr = addressPXD(&(j_sb->s_aim2)) * j_sb->s_bsize;
		AIT_byte_addr = addressPXD(&(j_sb->s_ait2)) * j_sb->s_bsize;
		byte_addr_diff0 = AIT_byte_addr - AIM_byte_addr;
		fsckwsp_addr = addressPXD(&(j_sb->s_fsckpxd)) * j_sb->s_bsize;
		byte_addr_diff1 = fsckwsp_addr - AIT_byte_addr;
		if(	(AIM_bytesize != expected_AIM_bytesize) ||
			(AIT_bytesize != expected_AIT_bytesize) ||
			(byte_addr_diff0 != AIM_bytesize) ||
			(byte_addr_diff1 <= AIT_bytesize) ) 
			j_sb->s_flag |= JFS_BAD_SAIT;
	}

	/* in release 1, the flag MUST reflect OS2, inline log, and group commit */
	/* We'll have to differentiate here between an OS2-compatable volume
	 * and a well-behaved linux volume.  For now, lets only implement
	 * OS2-compatable.
	 */
	if( (j_sb->s_flag & JFS_INLINELOG) != JFS_INLINELOG )
		j_sb->s_flag |= JFS_INLINELOG;
//	if( (j_sb->s_flag & JFS_OS2) != JFS_OS2 )
//		j_sb->s_flag |= JFS_OS2;
	if( (j_sb->s_flag & JFS_GROUPCOMMIT) != JFS_GROUPCOMMIT )
		j_sb->s_flag |= JFS_GROUPCOMMIT;
jFYI(0,("superblock: flag:0x%08x state:0x%08x size:0x%08llx\n", 
	j_sb->s_flag, j_sb->s_state, j_sb->s_size));

	/* validate fs state */
	if (j_sb->s_state != FM_CLEAN)
	{
jERROR(1,("jfs_mount: Mount Failure: File System Dirty.\n"));
#if 1
		//rc = EFORMAT;
		rc = EINVAL;
		goto out;
#else
jERROR(1,("  Mounting anyway :^)\n"));
#endif
	}

	sb->s_jfs_mntflag = j_sb->s_flag;

	/*
	 * JFS always does I/O by 4K pages.  Don't tell the buffer cache
	 * that we use anything else (leave s_blocksize alone).
	 */
	sb->s_jfs_bsize = j_sb->s_bsize;
	sb->s_jfs_l2bsize = j_sb->s_l2bsize;

	/*
 	 * For now, ignore s_pbsize, l2bfactor.  All I/O going through buffer
	 * cache.
	 */

	sb->s_jfs_nbperpage = PSIZE >> sb->s_jfs_l2bsize;
	sb->s_jfs_l2nbperpage = L2PSIZE - sb->s_jfs_l2bsize;
	sb->s_jfs_l2niperblk = sb->s_jfs_l2bsize - L2DISIZE;
	if (sb->s_jfs_mntflag & JFS_INLINELOG)
		sb->s_jfs_logpxd = j_sb->s_logpxd;
	sb->s_jfs_ait2 = j_sb->s_ait2;

#ifdef _OLD_STUFF
	/*
	 * initialize mount inode from superblock
	 */
	ipmnt->i_mntflag = sb->s_flag;
	ipmnt->i_mntflag |= JFS_GROUPCOMMIT;
/*
	ipmnt->i_mntflag |= JFS_LAZYCOMMIT;
*/
	if (ipmnt->i_pbsize != sb->s_pbsize ||
	    ipmnt->i_l2pbsize != sb->s_l2pbsize)
	{
		rc = EFORMAT;
		goto out;
	}

	ipmnt->i_blocks = sb->s_size;
	ipmnt->i_bsize = sb->s_bsize;
	ipmnt->i_l2bsize = sb->s_l2bsize;
	ipmnt->i_l2bfactor = sb->s_l2bfactor;
	ipmnt->i_compress = sb->s_compress;

	ipmnt->i_nbperpage = PSIZE >> ipmnt->i_l2bsize;
	ipmnt->i_l2nbperpage = log2shift(ipmnt->i_nbperpage);
	niperblk = ipmnt->i_bsize / DISIZE;
	ipmnt->i_l2niperblk = log2shift(niperblk);
jFYI(0,("superblock: bsize:%d(%d) pbsize:%d(%d) l2factor:%d\n",
	ipmnt->i_bsize, ipmnt->i_l2bsize, ipmnt->i_pbsize, ipmnt->i_l2pbsize,
	ipmnt->i_l2bfactor)); 

	if (sb->s_flag & JFS_INLINELOG)
		ipmnt->i_logpxd = sb->s_logpxd;
#endif /* _OLD_STUFF */

out:
#ifdef _STILL_TO_PORT
	if(rc == 0)
	{
		if (rawWrite(ipmnt, bp, 1))
			rc = EROFS;
	}	
	else
		rawRelease(bp);
#endif /* _STILL_TO_PORT */
	release_metapage(mp);

	return rc;
}


/*
 *	updateSuper()
 *
 * update synchronously superblock if it is mounted read-write.
 */
static int32
updateSuper(
	struct super_block *sb)
{
	int32			rc;
	metapage_t		*mp;
	struct jfs_superblock	*j_sb;

	if ((rc = readSuper(sb, &mp)))
		return rc;
	j_sb = (struct jfs_superblock *)(mp->data);

	/*
	 * file system state transition: mounted-clean. 
	 */
#ifdef _JFS_NOJOURNAL
	j_sb->s_state = FM_DIRTY;
#else
	j_sb->s_state = FM_MOUNT;
#endif

#ifdef _JFS_FASTDASD
	/*
	 * When mounted and DASD limits enabled, we set the DASD_PRIME flag
	 * in the superblock.  If we don't unmount properly, DASD usage will
	 * be recaculated at the next reboot.
	 */
	if (j_sb->s_flag & JFS_DASD_ENABLED)			// D233382
		j_sb->s_flag |= JFS_DASD_PRIME;			// D233382
#endif /* _JFS_FASTDASD */

	/* record log's dev_t and mount serial number */
	j_sb->s_logdev = sb->s_jfs_log->dev;
	j_sb->s_logserial = sb->s_jfs_log->serial;

	write_metapage(mp);

	return 0;
}


/*
 *	readSuper()
 *
 * read superblock by raw sector address
 */
int32
readSuper(
	struct super_block *sb,
	metapage_t **mpp)
{
	/* read in primary superblock */ 
	*mpp = read_metapage(sb->s_jfs_direct_inode,
			     SUPER1_OFF >> sb->s_blocksize_bits,
			     PSIZE, META_ABSOLUTE);
	if (*mpp == NULL)
	{
		/* read in secondary/replicated superblock */ 
		*mpp = read_metapage(sb->s_jfs_direct_inode,
				     SUPER2_OFF >> sb->s_blocksize_bits,
				     PSIZE, META_ABSOLUTE);
	}
	return *mpp ? 0 : 1;
}


/*
 *	logMOUNT()
 *
 * function: write a MOUNT log record for file system.
 *
 * MOUNT record keeps logredo() from processing log records
 * for this file system past this point in log.
 * it is harmless if mount fails.
 *
 * note: MOUNT record is at aggregate level, not at fileset level, 
 * since log records of previous mounts of a fileset
 * (e.g., AFTER record of extent allocation) have to be processed 
 * to update block allocation map at aggregate level.
 */
static int32
logMOUNT(
	struct super_block *sb)
{
	log_t	*log = sb->s_jfs_log;
	lrd_t	lrd;

	lrd.logtid = 0;
	lrd.backchain = 0;
	lrd.type = LOG_MOUNT;
	lrd.length = 0;
	lrd.aggregate = sb->s_dev;
	lmLog(log, NULL, &lrd, NULL);

	return 0;
}
