/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: jfs/jfs_metapage.c
 *
 */

#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/pagemap.h>
#include <linux/smp_lock.h>
#include <linux/jfs/jfs_types.h>
#include <linux/jfs/jfs_filsys.h>
#include <linux/jfs/jfs_metapage.h>
#include <linux/jfs/jfs_txnmgr.h>
#include <linux/jfs/jfs_debug.h>
#ifdef kern22
#include <linux/jfs/jfs_xtree.h>
#endif

void jfs_logredrive(void);
void finish_async_io(metapage_t *);
extern struct task_struct *jfsIOtask;
extern struct semaphore jfsIOsem;
struct list_head async_list;
spinlock_t async_lock;

unsigned int	metapages = 1024;	/* ??? Need a better number */
metapage_t	*metapage_buf;
unsigned long	meta_order;
metapage_t	*meta_free_list = NULL;
spinlock_t	meta_lock = SPIN_LOCK_UNLOCKED;
wait_queue_head_t	meta_wait;

#define HASH_BITS 10			/* This makes hash_table 1 4K page */
#define HASH_SIZE (1 << HASH_BITS)
metapage_t	**hash_table = NULL;
unsigned long	hash_order;

/*
 * meta_lock must be held by caller
 */
static inline void wait_on_metapage(metapage_t *mp)
{
	if (mp->flag & META_LOCKED) {
		DECLARE_WAITQUEUE(wait, current);
		add_wait_queue(&mp->wait, &wait);
		while (mp->flag & META_LOCKED) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			spin_unlock(&meta_lock);
			schedule();
			current->state = TASK_RUNNING;
			spin_lock(&meta_lock);
		}
		remove_wait_queue(&mp->wait, &wait);
	}
}

int __init metapage_init(void)
{
	int		i;
	metapage_t	*last = NULL;
	metapage_t	*mp;

	/*
	 * Initialize wait queue
	 */
	init_waitqueue_head(&meta_wait);

	INIT_LIST_HEAD(&async_list);
	async_lock = SPIN_LOCK_UNLOCKED;

	/*
	 * Allocate the metapage structures
	 */
	for (meta_order = 0;
	     ((PAGE_SIZE << meta_order)/sizeof(metapage_t)) < metapages;
	     meta_order++);
	metapages = (PAGE_SIZE << meta_order)/sizeof(metapage_t);

	jFYI(1,("metapage_init: metapage size = %d, metapages = %d\n",
		sizeof(metapage_t), metapages));

	metapage_buf = (metapage_t *) __get_free_pages(GFP_ATOMIC, meta_order);
	assert(metapage_buf);
	memset(metapage_buf, 0, PAGE_SIZE << meta_order);

	mp = metapage_buf;
	for (i = 0; i < metapages; i++, mp++) {
		mp->flag = META_FREE;
		init_waitqueue_head(&mp->wait);
		INIT_LIST_HEAD(&mp->io_list);
		mp->hash_next = last;
		last = mp;
	}
	meta_free_list = last;

	/*
	 * Now the hash list
	 */
	for (hash_order = 0;
	     ((PAGE_SIZE << hash_order)/sizeof(void *)) < HASH_SIZE;
	     hash_order++);
	hash_table = (metapage_t **) __get_free_pages(GFP_ATOMIC, hash_order);
	assert(hash_table);
	memset(hash_table, 0, PAGE_SIZE << hash_order);

	return 0;
}

#ifdef kern22
void metapage_exit(void)
#else
void __exit metapage_exit(void)
#endif
{
	free_pages((unsigned long)metapage_buf, meta_order);
	free_pages((unsigned long)hash_table, hash_order);
	metapage_buf = 0;	/* This is a signal to the jfsIOwait thread */
}

/*
 * Get metapage structure from freelist
 * 
 * Caller holds meta_lock
 */
static metapage_t *alloc_metapage(
int *dropped_lock)
{
	metapage_t *new;

	*dropped_lock = FALSE;
	if (meta_free_list == NULL) {
		DECLARE_WAITQUEUE(wait, current);
		*dropped_lock = TRUE;
		add_wait_queue(&meta_wait, &wait);
		while (meta_free_list == NULL) {
			set_current_state(TASK_UNINTERRUPTIBLE);
			spin_unlock(&meta_lock);
			schedule();
			current->state = TASK_RUNNING;
			spin_lock(&meta_lock);
		}
		remove_wait_queue(&meta_wait, &wait);
	}

	new = meta_free_list;
	meta_free_list = new->hash_next;

	return new;
}

/*
 * Put metapage on freelist (holding meta_lock)
 */
static inline void __free_metapage(
metapage_t	*mp)
{
	mp->flag = META_FREE;
	mp->hash_next = meta_free_list;
	meta_free_list = mp;
	wake_up(&meta_wait);
}

/*
 * Put metapage on freelist (not holding meta_lock)
 */
static inline void free_metapage(
metapage_t	*mp)
{
	spin_lock(&meta_lock);
	__free_metapage(mp);
	spin_unlock(&meta_lock);
}

/*
 * Basically same hash as in pagemap.h, but using our hash table
 */
#ifdef kern22
static metapage_t **meta_hash(
struct inode	*ip,
unsigned long	index)
{
#define i (((unsigned long)ip)/ \
	   (sizeof(struct inode) & ~(sizeof(struct inode) -1 )))
#define o (index + (index << PAGE_SHIFT))
	return (hash_table + ((i+o) & (HASH_SIZE - 1)));
#undef i
#undef o
}
#else /* ! kern22 */
static metapage_t **meta_hash(
struct address_space	*mapping,
unsigned long		index)
{
#define i (((unsigned long)mapping)/ \
	   (sizeof(struct inode) & ~(sizeof(struct inode) -1 )))
#define s(x) ((x) + ((x) >> HASH_BITS))
	return hash_table + (s(i+index) & (HASH_SIZE - 1));
#undef i
#undef s
}
#endif /* kern22 */

static metapage_t *search_hash(
metapage_t		**hash_ptr,
#ifdef kern22
struct inode		*ip,
#else
struct address_space	*mapping,
#endif
unsigned long		index)
{
	metapage_t *ptr;

	for (ptr = *hash_ptr; ptr; ptr = ptr->hash_next) {
#ifdef kern22
		if ((ptr->mapping == ip) &&
#else
		if ((ptr->mapping == mapping) &&
#endif
		    (ptr->index == index))
			return ptr;
	}

	return NULL;
}

static void add_to_hash(
metapage_t	*mp,
metapage_t	**hash_ptr)
{
	if (*hash_ptr)
		(*hash_ptr)->hash_prev = mp;

	mp->hash_prev = NULL;
	mp->hash_next = *hash_ptr;
	*hash_ptr = mp;
}

static void remove_from_hash(
metapage_t	*mp,
metapage_t	**hash_ptr)
{
	if (mp->hash_prev)
		mp->hash_prev->hash_next = mp->hash_next;
	else {
		assert(*hash_ptr == mp);
		*hash_ptr = mp->hash_next;
	}

	if (mp->hash_next)
		mp->hash_next->hash_prev = mp->hash_prev;
}

#ifndef kern22
/*
 * Direct address space operations
 */

static int direct_get_block(
struct inode		*ip,
long			lblock,
struct buffer_head	*bh_result,
int			create)
{
	bh_result->b_dev = ip->i_dev;
	bh_result->b_blocknr = lblock;
	if (create)
		bh_result->b_state |= (1UL << BH_Mapped) | (1UL << BH_New);
	else
		bh_result->b_state |= (1UL << BH_Mapped);

	return 0;
}

static int direct_writepage(
struct page	*page)
{
	return block_write_full_page(page, direct_get_block);
}

static int direct_readpage(
struct file	*fp,
struct page	*page)
{
	return block_read_full_page(page, direct_get_block);
}

static int direct_prepare_write(
struct file	*file,
struct page	*page,
unsigned	from,
unsigned	to)
{
	return block_prepare_write(page, from, to, direct_get_block);
}

static int direct_bmap(
struct address_space	*mapping,
long			block)
{
	return generic_block_bmap(mapping, block, direct_get_block);
}

struct address_space_operations direct_aops = {
	readpage:	direct_readpage,
	writepage:	direct_writepage,
	sync_page:	block_sync_page,
	prepare_write:	direct_prepare_write,
	commit_write:	generic_commit_write,
	bmap:		direct_bmap,
};
#endif /* ! kern22 */

metapage_t *__get_metapage(
struct inode	*inode,
unsigned long	lblock,
unsigned int	size,
unsigned long	flag,
unsigned long	new)
{
	int		dropped_lock;
	metapage_t	**hash_ptr;
	int		l2BlocksPerPage;	
	int		l2bsize;
#ifdef kern22
	struct inode	*mapping;
	unsigned long	offset;
	size_t		page_cache = 0;
	struct page	**page_hash;
#else
	struct address_space	*mapping;
#endif
	metapage_t		*mp;
	unsigned long		page_index;
	unsigned long		page_offset;

	jFYI(1,("__get_metapage: inode = 0x%p, lblock = 0x%lx\n",
	       inode, lblock));

#ifdef kern22
	if (flag & META_ABSOLUTE)
		mapping = inode->i_sb->s_jfs_direct_inode;
	else
		mapping = inode;
#else
	if (flag & META_ABSOLUTE)
		mapping = inode->i_sb->s_jfs_direct_mapping;
	else
		mapping = inode->i_mapping;
#endif

	spin_lock(&meta_lock);

	hash_ptr = meta_hash(mapping, lblock);

	mp = search_hash(hash_ptr, mapping, lblock);
	if (mp) {
page_found:
		mp->count++;
		jFYI(1,("__get_metapage: found 0x%p, in hash\n", mp));
		wait_on_metapage(mp);
		assert(mp->logical_size == size);
		mp->flag |= META_LOCKED;
		spin_unlock(&meta_lock);
	} else {
		l2bsize = inode->i_sb->s_blocksize_bits;
		l2BlocksPerPage = PAGE_CACHE_SHIFT - l2bsize;
		page_index = lblock >> l2BlocksPerPage;
		page_offset = (lblock - (page_index << l2BlocksPerPage)) <<
			      l2bsize;
		if ((page_offset + size) > PAGE_SIZE) {
			spin_unlock(&meta_lock);
			jERROR(1,("MetaData crosses page boundary!!\n"));
			return NULL;
		}

		mp = alloc_metapage(&dropped_lock);
		if (dropped_lock) {
			/* alloc_metapage blocked, we need to search the hash
			 * again.  (The goto is ugly, maybe we'll clean this
			 * up in the future.)
			 */
			metapage_t *mp2;
			mp2 = search_hash(hash_ptr, mapping, lblock);
			if (mp2) {
				__free_metapage(mp);
				mp = mp2;
				goto page_found;
			}
		}
		mp->flag = flag | META_LOCKED;
		mp->xflag = COMMIT_PAGE;
		mp->count = 1;
		mp->nohomeok = 0;
		mp->mapping = mapping;
		mp->index = lblock;
		mp->page = 0;
		mp->logical_size = size;
		add_to_hash(mp, hash_ptr);
		spin_unlock(&meta_lock);

#ifdef kern22
		/* Offset is not true byte offset, but offset in logical
		 * block size.  Page cache is limited to 32 bits.
		 */
		offset = page_index << l2BlocksPerPage;
		page_hash = page_hash(mapping, offset);
retry:
		mp->page = __find_page(mapping, offset, *page_hash);
		if (mp->page) {
			if (page_cache)
				page_cache_free(page_cache);
			wait_on_page(mp->page);
		} else if (page_cache) {
			mp->page = page_cache_entry(page_cache);

			/* add_to_page_cache(mp->page, mapping, offset,
			 *		     page_hash);
			 */
			atomic_inc(&mp->page->count);
			clear_bit(PG_uptodate, &mp->page->flags);
			clear_bit(PG_error, &mp->page->flags);
			set_bit(PG_referenced, &mp->page->flags);
			mp->page->offset = offset;
			add_page_to_inode_queue(mapping, mp->page);
			__add_page_to_hash_queue(mp->page, page_hash);
		} else {
			page_cache = page_cache_alloc();
			if (page_cache)
				goto retry;
			jERROR(1,("page_cache_alloc failed!\n"));
			free_metapage(mp);
			return NULL;
		}
		if (! PageUptodate(mp->page)) {
			if (new && (size == PAGE_SIZE)) {
				set_bit(PG_uptodate, &mp->page->flags);
			} else {
				int b[PAGE_SIZE/512];
				int first = page_offset >> l2bsize;
				int last = (page_offset + size) >> l2bsize;
				int nbufs;
				int i;
	
				set_bit(PG_locked, &mp->page->flags);

				nbufs = 1 << l2BlocksPerPage;

				if (flag & META_ABSOLUTE) {
					for (i = 0; i < nbufs; i++)
						b[i] = offset + i;
				} else {
					int   rc;
					int64 xaddr;
					uint8 xflag;
					int32 xlen;

					i = 0;
					while (i < nbufs) {
	/***********  Too many nested loops, indenting backwards **********/
	rc = xtLookup(inode, offset + i, nbufs - i, &xflag, &xaddr, &xlen, 0);
	if (rc) {
		jERROR(1,("__get_metapage: xtLookup failed with rc = %d\n", rc));
		clear_bit(PG_locked, &mp->page->flags);
		wake_up(&mp->page->wait);
		assert(atomic_read(&mp->page->count) > 1);
		page_cache_release(mp->page);
		free_metapage(mp);
		return NULL;
	}
	if (xlen == 0) {
		/* File hole */
		int ii;
		for (ii = 0; ii < inode->i_sb->s_jfs_nbperpage; ii++)
			b[i++] = 0;
	} else {
		while (xlen--)
			b[i++] = xaddr++;
		/* In case extent defines less than 1 4K page */
		while (i & (inode->i_sb->s_jfs_nbperpage - 1))
			b[i++] = 0;
	}
	/***********  End of backwards indenting **************/
					}
				}

				if (new) {
					/* No need to read new buffers */
					for (i = first; i < last; i++)
						b[i] = 0;
				}

				atomic_inc(&mp->page->count);
				set_bit(PG_free_after, &mp->page->flags);
				assert(atomic_read(&mp->page->count) > 2);
				brw_page(READ, mp->page, mapping->i_dev, b,
					 inode->i_sb->s_blocksize, 1);
				wait_on_page(mp->page);
			}
		}
		mp->data = (void *)(page_address(mp->page) + page_offset);
#else
		if (new) {
			jFYI(1,("__get_metapage: Calling grab_cache_page\n"));
			mp->page = grab_cache_page(mapping, page_index);	
			if (!mp->page) {
				jERROR(1,("grab_cache_page failed!\n"));
				free_metapage(mp);
				return NULL;
			}
			UnlockPage(mp->page);
		}
		else {
			jFYI(1,("__get_metapage: Calling read_cache_page\n"));
			mp->page = read_cache_page(mapping, lblock,
					(filler_t *)mapping->a_ops->readpage,
					NULL);
			if (!mp->page) {
				jERROR(1,("read_cache_page failed!\n"));
				free_metapage(mp);
				return NULL;
			}
			wait_on_page(mp->page);
		}
		mp->data = (void *)(kmap(mp->page) + page_offset);
#endif
	}
	jFYI(1,("__get_metapage: returning = 0x%p\n", mp));
	return mp;
}

void hold_metapage(metapage_t *mp)
{
	spin_lock(&meta_lock);

	mp->count++;
	wait_on_metapage(mp);
	mp->flag |= META_LOCKED;

	spin_unlock(&meta_lock);
}

void __write_metapage(metapage_t *mp)
{
#ifdef kern22
	int		b[PAGE_SIZE/512];
	int		i;
	int		nbufs;
	struct inode	*ip = (struct inode *)mp->mapping;
#else
	struct inode	*ip = (struct inode *)mp->mapping->host;
	unsigned long	page_index;
	unsigned long	page_offset;
	int		rc;
#endif
	int		l2bsize = ip->i_sb->s_blocksize_bits;
	int		l2BlocksPerPage = PAGE_CACHE_SHIFT - l2bsize;

	jFYI(1,("__write_metapage: mp = 0x%p\n", mp));

	wait_on_page(mp->page);
#ifdef kern22
	set_bit(PG_locked, &mp->page->flags);

	nbufs = 1 << l2BlocksPerPage;

	if (mp->flag & META_ABSOLUTE) {
		for (i = 0; i < nbufs; i++)
			b[i] = mp->page->offset + i;
	} else {
		int   rc;
		int64 xaddr;
		uint8 xflag;
		int32 xlen;

		i = 0;
		while (i < nbufs) {
			rc = xtLookup(ip, mp->page->offset + i,
				      nbufs - i, &xflag, &xaddr, &xlen, 0);
			if (rc) {
				jERROR(1,("__write_metapage: xtLookup failed with rc = %d\n", rc));
				set_bit(PG_error, &mp->page->flags);
				clear_bit(PG_locked, &mp->page->flags);
				wake_up(&mp->page->wait);
				return;
			}
			if (xlen == 0) {
				/* File hole */
				int ii;
				for (ii = 0; ii < ip->i_sb->s_jfs_nbperpage;
				     ii++)
					b[i++] = 0;
			} else {
				while (xlen--)
					b[i++] = xaddr++;
				/* In case extent defines less than 1 4K page */
				while (i & (ip->i_sb->s_jfs_nbperpage - 1))
					b[i++] = 0;
			}
		}
	}

	atomic_inc(&mp->page->count);
	set_bit(PG_free_after, &mp->page->flags);
	assert(atomic_read(&mp->page->count) > 2);
	brw_page(WRITE, mp->page, ip->i_dev, b, ip->i_sb->s_blocksize, 0);
#else
	page_index = mp->page->index;
	page_offset = (mp->index - (page_index << l2BlocksPerPage)) << l2bsize;

	LockPage(mp->page);
	rc = mp->mapping->a_ops->prepare_write(NULL, mp->page, page_offset,
					page_offset + mp->logical_size);
	if (rc) {
		jERROR(1,("prepare_write return %d!\n", rc));
	}
	rc = mp->mapping->a_ops->commit_write(NULL, mp->page, page_offset,
					      page_offset + mp->logical_size);
	if (rc) {
		jERROR(1,("commit_write returned %d\n", rc));
	}
	UnlockPage(mp->page);
#endif

	mp->flag &= ~META_DIRTY;

	if (mp->flag & META_SYNC) {
		__sync_metapage(mp);
		mp->flag &= ~META_SYNC;
	}
	jFYI(1,("__write_metapage done\n"));
}

void release_metapage(
metapage_t	*mp)
{
	log_t *log;

	jFYI(1,("release_metapage: mp = 0x%p, flag = 0x%lx\n", mp, mp->flag));

	if (mp->flag & META_SYNC) {
		if (mp->flag & META_DIRTY)
			__write_metapage(mp);
		else {
			__sync_metapage(mp);
		}
	}
	spin_lock(&meta_lock);
	assert(mp->count);
	if (--mp->count || mp->nohomeok) {
		mp->flag &= ~META_LOCKED;
		wake_up(&mp->wait);
		spin_unlock(&meta_lock);
	} else {
		if (mp->lsn &&
		    ((mp->flag & META_DIRTY) || PageLocked(mp->page))) {
			mp->count++;
			spin_unlock(&meta_lock);
			if (mp->flag & META_DIRTY)
				__write_metapage(mp);
			mp->end_io = 0;
			finish_async_io(mp);
			return;
		}
#ifndef kern22
		kunmap(mp->page);
#endif
		remove_from_hash(mp, meta_hash(mp->mapping, mp->index));
		spin_unlock(&meta_lock);	
		if (mp->flag & META_DIRTY) {
			__write_metapage(mp);
		}

		if (mp->lsn) {
			/*
			 * There should be no outstanding I/O at this point
			 */
			log = mp->log;
			LOGSYNC_LOCK(log);
			mp->log = 0;
			mp->lsn = 0;
			mp->clsn = 0;
			log->count--;
			CDLL_REMOVE(&log->synclist, (logsyncblk_t *)mp,
				    synclist);
			LOGSYNC_UNLOCK(log);
		}

		if (mp->flag & META_DISCARD) {
			/* BUGBUG - What's the right way to do this?
			 *   Should this be exported by the kernel?
			 */
			//block_flushpage(mp->page, 0);
		}
		jFYI(1,("release_metapage: calling page_cache_release\n"));
		assert(atomic_read(&mp->page->count) > 1);
		page_cache_release(mp->page);
		free_metapage(mp);
	}
	jFYI(1,("release_metapage: done\n"));
}

void invalidate_metapage(
metapage_t	*mp)
{
	jFYI(1,("invalidate_metapage: mp = 0x%p\n", mp));

	spin_lock(&meta_lock);
	mp->flag &= ~META_LOCKED;
	assert(mp->count);
	if (--mp->count) {
		jERROR(1,("Invalidating referenced metapage\n"));
		wake_up(&mp->wait);
		spin_unlock(&meta_lock);
	}
	else {
		remove_from_hash(mp, meta_hash(mp->mapping, mp->index));
		spin_unlock(&meta_lock);	
		assert(atomic_read(&mp->page->count) > 1);
		/* BUGBUG - How to we invalidate page cache page? */
#ifndef kern22
		/* Should this be exported by the kernel?
		 */
		//block_flushpage(mp->page, 0);
		kunmap(mp->page);
#endif
		page_cache_release(mp->page);
		free_metapage(mp);
	}
}

int jfsIOWait(
void *arg)
{
	siginfo_t info;
	unsigned long signr;
	int recheck;
	struct list_head *lh, *next_lh;
	metapage_t *mp;

	jFYI(1,("jfsIOWait is here!\n"));

	lock_kernel();

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,2,17))
	daemonize();
#endif
	current->tty = NULL;
	strcpy(current->comm, "jfsIO");

	unlock_kernel();

	jfsIOtask = current;

	spin_lock_irq(&current->sigmask_lock);
	siginitsetinv (&current->blocked, sigmask(SIGHUP) | sigmask(SIGKILL) | sigmask(SIGSTOP) | sigmask(SIGCONT));
	spin_unlock_irq(&current->sigmask_lock);

	up(&jfsIOsem);

	while(1) {
		recheck = 0;
		jfs_logredrive();
		spin_lock(&async_lock);
		if (list_empty(&async_list)) {
			set_current_state(TASK_INTERRUPTIBLE);
			spin_unlock(&async_lock);
			schedule();
			spin_lock_irq(&current->sigmask_lock);
			signr = dequeue_signal(&current->blocked, &info);
			spin_unlock_irq(&current->sigmask_lock);
			switch(signr) {	
			case SIGKILL:
				if (metapage_buf)
					/* Not our SIGKILL signal */
					continue;

				if (! list_empty(&async_list)) {
					jERROR(1,("jfsIOWait being killed with pending I/O\n"));
				} else {
					jEVENT(1,("jfsIOWait being killed!\n"));
				}
				up(&jfsIOsem);
				return 0;
			}
			spin_lock(&async_lock);
		} else {
			for (lh = async_list.next;
			     lh != &async_list;
			     lh = next_lh) {
				next_lh = lh->next;
				mp = list_entry(lh, metapage_t, io_list);
				if (!PageLocked(mp->page)) {
					list_del(lh);
					INIT_LIST_HEAD(&mp->io_list); /* Not needed */
					spin_unlock(&async_lock);
					if (mp->end_io)
						mp->end_io(mp->endio_arg);
					else
						release_metapage(mp);
					recheck = 1;
					spin_lock(&async_lock);
				}
			}

			if (!recheck) {
				/* Wait for first page's I/O to complete */
				lh = async_list.next;
				ASSERT(lh != &async_list);
				mp = list_entry(lh, metapage_t, io_list);
				spin_unlock(&async_lock);
				wait_on_page(mp->page);
				spin_lock(&async_lock);
			}
		}
		spin_unlock(&async_lock);
	}
}

void finish_async_io(
metapage_t *mp)
{
	int wake;
	spin_lock(&async_lock);
	wake = list_empty(&async_list);
#ifdef kern22
	list_add(&mp->io_list, &async_list);
#else
	list_add_tail(&mp->io_list, &async_list);
#endif
	spin_unlock(&async_lock);
	if (wake)
		send_sig(SIGHUP, jfsIOtask, 1);
}
