/*
 * Bink-style outbound manager with extensions made especially for ifmail :)
 *
 * Copyright (C) 1997 Valery Shchedrin (valery@forthpick.kiev.ua)
 *
 * You can do with this program almost everything you want, except
 * you must not make changes in this original copyright notice and
 * if you change code, you should notify all users of modified copy
 * that they use not original version. Author is not responsible
 * for any damages or security violations caused by use of this program.
 *
 * VERSION 1.1.2
 */

#include <stdio.h>
#include <string.h>
#include <glob.h>
#include <ctype.h>
#include <getopt.h>
#include <sys/stat.h>
#include <fnmatch.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include "config.h"

#define MAXPATH 256

#ifdef DEBUG
# define DBG(a) (a)
#else
# define DBG(a)
#endif

typedef struct _faddr {
	int zn;
	int net;
	int nd;
	int pnt;
	char dom[50];
}faddr_t;

#define F_CRASH (1<<1)
#define F_NORM  (1<<2)
#define F_HOLD  (1<<3)
#define F_REQ   (1<<4)
#define F_MAIL  (1<<5)
#define F_ADD   (1<<6)
#define F_DEL	(1<<7)
#define F_DIRT  (1<<8)
#define F_KILLE	(1<<9)

int flags;	/* Processing flags */
faddr_t paddr   = {-2, -2, -2, -2}; /* Processing address */
faddr_t defaddr = {-2, -2, -2, -2}; /* Our primary address */
faddr_t fdr_lresv;
int sumsize;
int parseconfig_r; /* check for infinite includes */
char **del_list;

char tmpstr[MAXPATH];
char outb[MAXPATH];

typedef void (*walk_func)(const char *name);

void walk_outb(walk_func f);
int init_vars(void);          /* Initialize variables */
void parseconfig(const char *name);/*Extract _some_ info from ifmail configs */
int parse_faddr(const char *s, faddr_t *addr);
void catglobext(char *s);
void resolve_addr(const char *pn, faddr_t *ad);
void del_file(const char *file);
void print_file(const char *file);
void print_att(const char *file);
void print_req(const char *file);
void print_fsize(const char *p);
void print_size(long size);
void print_fin_line(void);
void add_lo(char **files);
void addreq(char **files);
void usage(void);

struct option longopts[] = {
	{"help",	0,	0,	'u'},
	{"delete",	0,	0,	'd'},
	{"kill-empty",	0,	0,	'k'},
	{"create",	0,	0,	's'},
	{"config",	1,	0,	'a'},
	{0,		0,	0,	0}
};

main(int ac, char **av)
{
	int c, cfr; char *p;

	setuid(geteuid());
	
	p  = strrchr(av[0], '/'); if (!p) p = av[0]; else p++;
	
	if (!strcmp(p, "ifatt")) flags |= F_ADD;
	else if (!strcmp(p, "ifreq")) flags |= F_REQ | F_ADD;
	
	cfr = 0;
	while ((c = getopt_long(ac, av, "cdfhkmrsua:", longopts, NULL)) != EOF) 
		switch (c) {
		case 'a': if (!cfr) { parseconfig(optarg); cfr = 1; } break;
		case 'c': flags |= F_CRASH; break;
		case 'f': flags |= F_NORM; break;
		case 'h': flags |= F_HOLD; break;
		case 'r': flags |= F_REQ; break;
		case 'm': flags |= F_MAIL; break;
		case 's': flags |= F_ADD; break;
		case 'd': flags |= F_DEL; break;
		case 'k': flags |= F_KILLE; break;
		case 'u': usage();
		default: exit(0); break;
		}
	
	if (!cfr) parseconfig(IFCONFIG);
	
	if (ac-optind <= 0) {
		fprintf(stderr, "Address must be specified\n");
		exit(-1);
	}
	
	if (parse_faddr(av[optind++], &paddr)) {
		fprintf(stderr, "Bad address\n");
		exit(-1);
	}
	
	if (init_vars()) {
		fprintf(stderr, "Invalid parameters\n");
		exit (-1);
	}
	
	if (flags & F_ADD) {
		if (flags & F_REQ) addreq(&av[optind]);
		else add_lo(&av[optind]);
	} else if (flags & F_DEL) {
		del_list = &av[optind];
		walk_outb(del_file);
	} else {
		walk_outb(print_file);
		print_fin_line();
	}
}

void addreq(char **av)
{
	char str[MAXPATH];
	FILE *fd;
	int i;
	
	sprintf(str, "%s/%.4x%.4x.", outb, paddr.net, paddr.nd);
	if (paddr.pnt != 0) sprintf(strchr(str, 0), "pnt/%.8x.",
					paddr.pnt);
	strcat(str, "req");
	
	umask(0133);
	fd = fopen(str, "a");
	if (!fd) { perror(str); exit(0); }
	for (i = 0; av[i]; i++) fprintf(fd, "%s\n", av[i]);
	fclose(fd);
}

void add_lo(char **av)
{
	char str[MAXPATH], *p;
	FILE *fd; 
	glob_t g; 
	int i, j;
	
	sprintf(str, "%s/%.4x%.4x.", outb, paddr.net, paddr.nd);
	if (paddr.pnt != 0) {
		strcat(str, "pnt"); umask(022);	mkdir(str, 0777);
		sprintf(strchr(str, 0), "/%.8x.", paddr.pnt);
	}
	strcat(str, (flags & F_CRASH)?"clo":(flags & F_HOLD)?"hlo":"flo");
	
	DBG(printf("add_lo() making %s\n", str));
	
	umask(0133);
	fd = fopen(str, "a");
	if (!fd) { perror(str); exit(0); }
	
	for (i = 0;av[i];i ++) { /* Adding files */
		p = av[i]; if (*p == '#' || *p == '^') p++;
		glob(p, 0, 0, &g);
		for (j = 0; j < g.gl_pathc; j++) {
			p = g.gl_pathv[j];
			if (p[0] != '/') {
				getcwd(str, MAXPATH);
				strcat(str, "/"); strcat(str, p);
				p = str;
			}
			if (av[i][0] == '#' || av[i][0] == '^')
				fprintf(fd, "%c%s\n", av[i][0], p);
			else fprintf(fd, "%s\n", p);
		}
		globfree(&g);
	}
	
	fclose(fd);
}

int init_vars(void)
{
	if (flags & F_ADD && flags & (F_DEL | F_KILLE | F_MAIL)) {
		fprintf(stderr, "-s can't be used with -d, -k or -m\n");
		return -1;
	}

	if (flags & F_ADD) {
		int i, f;
		
		for (f = 0, i = F_REQ; i != 0; i>>=1) 
			if (flags & i) { 
				if (f) {
					fprintf(stderr, "Only one of -[cfhr] is allowed with -s\n");
					return -1;
				}
				f = 1; 
			}
				
		if ((flags & (F_NORM | F_CRASH | F_HOLD | F_REQ)) == 0)
			flags |= F_NORM;
	} else if ((flags & (F_NORM|F_CRASH|F_HOLD|F_REQ|F_MAIL)) == 0)
		flags |= (F_NORM|F_CRASH|F_HOLD|F_REQ|F_MAIL);

	if (paddr.zn == -2 || paddr.nd == -2 || paddr.net == -2 ||
		paddr.pnt == -2) {
		fprintf(stderr, "Processing address is not complete\n");
		return -1;
	}
		
	if (!outb[0]) {
		fprintf(stderr, "No outbound specified\n");
		return -1;
	}

	if (flags & F_ADD && (paddr.zn < 0 || paddr.nd < 0 || 
		paddr.net < 0 || paddr.pnt < 0)) {
		fprintf(stderr, "No address globing is allowed with -s\n");
		return -1;
	}
	
	if (strcmp(paddr.dom, defaddr.dom)) 
		strcpy(strrchr(outb,'/')+1, paddr.dom);
	
	else if (paddr.zn == defaddr.zn) return 0;
	if (paddr.zn >= 0) sprintf(strchr(outb, 0), ".%.3x", paddr.zn);

	return 0; 
}

/* Nice parser, can eat both formats:
	2:463/74@fidonet and f74.n463.z2@fidonet, of course you can write
	360 if your def address is 2:463/74 then it will be 2:463/360 :),
   I was too drunk, when I wrote this down, so it also is very strict,
   just try to feed it with 'hello' - :))))
*/

#define s2i(s) ((*(s) == '*')?-1:atoi((s)))
int parse_faddr(const char *s, faddr_t *ad)
{
	const char *p, *pl;
	char *pw;
	int st;
	
	DBG(printf("parse_faddr(s = \"%s\", ad = $%x)\n", s, (int) ad));
	
	memcpy(ad, &defaddr, sizeof(faddr_t));
	
	ad->pnt = 0;

	for (st = 0, pl = p = s; ; p++) {
		DBG(printf("[%c %d]", *p, st));
		switch (st) {
st_0:		case 0:	switch (*p) {
			case 'p': st = 100; goto st_100;
			case 'f': st = 102; goto st_102;
			case ':': st = 1; ad->zn = s2i(pl); pl=p+1; goto done;
			case '/': st = 2; ad->net = s2i(pl); pl=p+1; goto done;
			case '.': st = 3; if (*s != '.') ad->nd = s2i(pl);
				  ad->pnt = s2i(p+1); goto done;
			}
			goto st_50;
			
		case 1: if (*p == '/') goto st_0;    break;
		case 2: if (*p == '.') goto st_0;
			if (*p == '@' || !*p) goto st_50;
			break;
				
st_3:		case 3: if (*p == '@') { pw = ad->dom; st = 200; goto done; }
			if (!*p) goto done;
			break;
			
st_50:		case 50:switch (*p) {
			case '@': st = 200; pw = ad->dom;
			case 0: if (*pl == 0) goto ret_err;
				ad->nd = s2i(pl); goto done;
			}
			break;
			
st_100:		case 100:if (*p == 'p'){ st = 101; ad->pnt=s2i(p+1);goto done;}
			 goto ret_err;
		case 101:if (*p == '.'){ st = 102; goto done; } goto st_3;
st_102:		case 102:if (*p == 'f'){ st = 103; ad->nd=s2i(p+1); goto done;}
			 goto ret_err;
		case 103:if (*p == '.'){ st = 104; goto done; } goto st_3;
		case 104:if (*p == 'n'){ st = 105; ad->net=s2i(p+1); goto done;}
			 goto ret_err;
		case 105:if (*p == '.'){ st = 106; goto done; } goto st_3;
		case 106:if (*p == 'z'){ st = 3; ad->zn=s2i(p+1); goto done;}
			 goto ret_err;
			 
		case 200:if (pw - ad->dom > sizeof(ad->dom)) goto ret_err;
			 *pw++ = *p;
			 goto done;
		}
		
		/* skip other characters */
		if (*p == '*') {
			switch(p[1]) {
			case '.': if (st != 2 && st != 101 && st != 103 &&
				      st != 105 && st != 0) goto ret_err;
				break;
			case ':': if (st != 0) goto ret_err;
				break;
			case '/': if (st != 0 && st != 1) goto ret_err;
				break;
			case 0:
			case '@': if (st != 50 && st != 3 && st != 2 &&
				      st != 0)
					  goto ret_err;
				break;
			default: goto ret_err;
			}
		} else if (!isdigit(*p)) goto ret_err;

	done:
		if (!*p) break;
		continue;
		
	ret_err:
		{
			int i;
			
			fprintf(stderr, "parse error: %s\n", s);
			for (i = 13+p-s; i > 0; i--) fputc(' ', stderr);
			fputc('^', stderr); fputc('\n', stderr);
			return -1;
		}
	}
	if (ad->zn == -2 || ad->nd == -2 || ad->pnt == -2 || ad->net == -2 ||
	    ad->dom[0] == 0) {
		fprintf(stderr, "Can't complete: %s\n", s);
		return -1;
	}
	DBG(printf(" resolved %d:%d/%d.%d@%s\n", ad->zn, ad->net, ad->nd, ad->pnt,
			ad->dom));
	return 0;
}


void parseconfig(const char *name)
{
	FILE *fd;
	char str[MAXPATH];
	char *p;
	
	if (parseconfig_r > 30) {
		fprintf(stderr, "%s: too deep includes", name);
		exit(0);
	}
	
	parseconfig_r ++;
	
	DBG(printf("parseconfig(name = \"%s\")\n", name));
	
	fd = fopen(name, "r");
	if (!fd) { fprintf(stderr, "Unable to open %s\n", name); exit(0); }
	while (fgets(str, MAXPATH, fd) != NULL) {
		strtok(str, "\n\r"); strtok(str, "\t ");
		if (defaddr.zn == -2 && !strcasecmp(str, "address")) 
			parse_faddr(strtok(NULL, "\t "), &defaddr);
		else if (!strcasecmp(str, "outbound")) {
			strcpy(outb, strtok(NULL, "\t "));
		} if (!strcasecmp(str, "include")) {
			p = strtok(NULL, "\t ");
			if (!strchr(p, '/')) {
				char *p1, *p2;
				
				strcpy(tmpstr, name);
				p1 = strrchr(tmpstr, '/');
				if (!p1) { printf("fatal error (p1 == NULL)\n"); exit(0); }
				*p1 = 0;
				p2 = p; p = strchr(p2, 0)+1;
				if (p == (void*)1) { printf("fatal error (p == 1)\n"); exit(0); }
				sprintf(p, "%s/%s", tmpstr, p2);
			}
			parseconfig(p);
		}
	}
	fclose(fd);
	DBG(printf(" outb = \"%s\"\n", outb));
	
	parseconfig_r --;
}

void walk_outb(walk_func f)
{
	glob_t g;
	char *p, *p1;
	char woutb[MAXPATH];
	int i;
	
	DBG(printf("walk_outb()\n"));
	
	strcpy(woutb, outb);
	
	if (paddr.zn == -1) {
		p = strrchr(woutb, '/');
		if (!p) { printf("fatal error (p == NULL)\n"); exit(0); }
		p = strchr(p+1, '.');
		if (!p) p = strchr(woutb, 0);
		strcpy(p, "*");
	}
	
	if (paddr.net == -1) strcat(woutb, "/????");
		else sprintf(strchr(woutb,0), "/%.4x", paddr.net);

	if (paddr.nd == -1) strcat(woutb, "????");
		else sprintf(strchr(woutb,0), "%.4x", paddr.nd);
	
	catglobext(woutb);
	
	DBG(printf(" glob(\"%s\", ...)\n", woutb));
		
	glob(woutb, 0, NULL, &g);
	
	for (i = 0; i < g.gl_pathc; i++) {
		p = strrchr(g.gl_pathv[i], '.')+1;
		if (p == (void*)1) { printf("fatal error (p == 1)\n", p); exit(0); }
		if (!strcmp(p, "pnt")) {
			glob_t gg; int ii;
			
			strcpy(woutb, g.gl_pathv[i]);
			if (paddr.pnt == -1) strcat(woutb, "/????????");
			else sprintf(strchr(woutb,0), "/%.8x", paddr.pnt);
			catglobext(woutb);
			
			DBG(printf(" pnt glob(\"%s\", ...)\n", woutb));
			
			glob(woutb, 0, NULL, &gg);
			
			for (ii = 0; ii < gg.gl_pathc; ii++)
				f(gg.gl_pathv[ii]);

			globfree(&gg);
		}else f(g.gl_pathv[i]);
	}
	
	globfree(&g);
}

void catglobext(char *s)
{
	/* Extension globbing (.flo/.hlo/.clo/.req/.pnt) */
	strcat(s, ".[");
	if (flags & F_CRASH) strcat(s, "c");
	if (flags & F_HOLD) strcat(s, "h");
	if (flags & F_NORM) strcat(s, "f");
	if (flags & F_REQ) strcat(s, "r");
	if (flags & F_MAIL) strcat(s, "o");
	if (paddr.pnt != 0) strcat(s, "p");
	strcat(s, "][");
	if (flags & (F_CRASH | F_HOLD | F_NORM)) strcat(s, "l");
	if (flags & F_REQ) strcat(s, "e");
	if (flags & F_MAIL) strcat(s, "u");
	if (paddr.pnt != 0) strcat(s, "n");
	strcat(s, "][");
	if (flags & (F_CRASH | F_HOLD | F_NORM)) strcat(s, "o");
	if (flags & F_REQ) strcat(s, "q");
	if (flags & F_MAIL || paddr.pnt != 0) strcat(s, "t");
	strcat(s, "]");
}

void print_file(const char *file)
{
	faddr_t ad;
	char *p;
	
	DBG(printf("print_file(file = \"%s\")\n",file));
	
	p = strrchr(file, '/');
	if (!p) {
		printf("fatal error, no slash in %s\n", file);
		exit(0);
	}
	
	if (p[-1] == 't' && p[-2] == 'n' && p[-3] == 'p' &&
		paddr.pnt != 0 && (paddr.nd != -1 || paddr.pnt != -1)) return;
	
	resolve_addr(file, &ad);
	p = strrchr(file, '.');
	if (!p++) {
		printf("fatal error, no dot in %s\n", file);
		exit(0);
	}
	if (!strcmp(p, "clo") || !strcmp(p, "flo") ||
	    !strcmp(p, "hlo")) {
		print_att(file);
	} else if (!strcmp(p, "req")) {
		print_req(file);
	} else if (!strcmp(p, "out")) {
		printf(" Pkt --- ");
		print_fsize(file);
		printf(" %s\n", file);
	}
}

void print_att(const char *file)
{
	FILE *fd;
	char *p;
	struct stat st;
	
	fd = fopen(file, "r");
	if (!fd) { perror(file); exit(0); }
	while (fgets(tmpstr, MAXPATH, fd)) {
		printf(" Att ");
		switch(strchr(file, 0)[-3]) {
			case 'c': printf("Cra "); break;
			case 'f': printf("--- "); break;
			case 'h': printf("Hld "); break;
		}

		p = strtok(tmpstr, " \n\t");
		while (*p == '^' || *p == '#') p++;
		print_fsize(p);
		printf(" %s\n", tmpstr);
	}
}

void print_fsize(const char *p)
{
	static struct stat st;
	
	if (stat(p, &st)) { /* Bad, bad :( */
		printf("NOENT");
	} else {
		print_size(st.st_size);
		sumsize += st.st_size;
	}
}

void print_size(long size)
{
	if (size < 10000L) {
		printf("%4ldb", size);
	} else if (size < 1000000L) {
		printf("%3ldkb", size/1024L);
	} else if (size < 10000000L) {
		printf("%01ld.%01ldmb", size/1048576L,
			(size/104857L)%10L);
	} else printf("%3ldmb", size/1048576L);
}

void print_req(const char *file)
{
	FILE *fd;
	char *p;
	
	fd = fopen(file, "r");
	if (!fd) { perror(file); exit(0); }
	while (fgets(tmpstr, MAXPATH, fd)) {
		strtok(tmpstr, " \n\t");
		printf(" Req --- ----- %s\n", tmpstr);
	}
	fclose(fd);
}

void print_fin_line(void)
{
	if (flags & F_DIRT) {
		printf(" Total : ");
		print_size(sumsize);
		sumsize = 0;
		putchar('\n');
	} else flags |= F_DIRT;
}

void resolve_addr(const char *pn, faddr_t *ad)
{
	char *p, *p1;
	
	DBG(printf("resolve_addr(pn = \"%s\", ad = $%x)\n", pn, (int) ad));
	
	p = strrchr(pn, '/');
	if (!p) {
		printf("fatal error, no slash in %s\n", pn);
		exit(0);
	}
	if (p[-3] == 'p' && p[-2] == 'n' && p[-1] == 't') {
		sscanf(p+1, "%8x", &ad->pnt);
		*p = 0;
	} else { ad->pnt = 0; p = NULL; }
		
	p1 = strrchr(pn, '/');
	if (!p1) {
		printf("fatal error, no slash in %s\n", pn);
		exit(0);
	}
	sscanf(p1+1, "%4x%4x", &ad->net, &ad->nd);
	
	if (isxdigit(p1[-1]) && isxdigit(p1[-2]) && isxdigit(p1[-3])) {
		sscanf(p1-3, "%3x", &ad->zn);
		p1[-4] = 0;
		strcpy(ad->dom, strrchr(pn, '/')+1);
		if (!strcmp(ad->dom, strrchr(outb, '/')+1))
			strcpy(ad->dom, defaddr.dom);
		p1[-4] = '.';
	} else {
		*p1 = 0; strcpy(ad->dom, strrchr(pn, '/')+1); *p1 = '/';
		if (!strcmp(ad->dom, strrchr(outb, '/')+1)) {
			ad->zn = defaddr.zn; strcpy(ad->dom, defaddr.dom);
		} else ad->zn = 0;
	}
	
	if (p) *p = '/';

	DBG(printf(" resolved %d:%d/%d.%d@%s\n", ad->zn, ad->net, ad->nd, 
		ad->pnt, ad->dom));
	
	if (memcmp(ad, &fdr_lresv, sizeof(faddr_t))) {
		print_fin_line();
		
		if (ad->pnt == 0) {
			printf("Node %d:%d/%d@%s:\n", ad->zn, ad->net,
				ad->nd, ad->dom);
		} else {
			printf("Point %d:%d/%d.%d@%s:\n", ad->zn, ad->net,
				ad->nd, ad->pnt, ad->dom);
		}
		
	} else memcpy(&fdr_lresv, ad, sizeof(faddr_t));
}

void del_file(const char *name)
{
	char *p;
	
	DBG(printf("del_file(name = \"%s\")\n", name));
	p = strrchr(name, '.') + 1;
	if (!strcmp(p, "flo") || !strcmp(p, "clo") ||
	    !strcmp(p, "hlo") || !strcmp(p, "req")) {
		int fd, i;
		char *mem, *pnext;
		long size;
		
		if ((fd = open(name, O_RDWR)) < 0) {
			fprintf(stderr, "Can't open %s\n", name);
			return;
		}
		size = lseek(fd, 0L, SEEK_END); lseek(fd, 0L, SEEK_SET);
		p = mem = malloc(size+1); read(fd, mem, size);
		
		while (1) {
			if (size <= p - mem) break;
			pnext = memchr(p, '\n', size - (p - mem));
			if (!pnext) pnext = mem + size;
			*pnext = 0;
			
			for (i = 0; del_list[i]; i++)
				if (!fnmatch(del_list[i], p, 0)) goto kill_line;

			if (flags & F_KILLE) {
				char *p1;
				struct stat st;
				
				p1 = p;
				while (*p1 == ' ') p1++;
				if (*p1 == '#' || *p1 == '^') p1++;
				if (stat(p1, &st)) goto kill_line;
			}

			*pnext = '\n'; p = pnext + 1; continue;
			kill_line:
				DBG(printf("detaching %s\n", p));
				memmove(p, pnext + 1, size - (pnext - mem + 1));
				size -= pnext - p + 1;
		}
		
		write(fd, mem, size); ftruncate(fd, size); free(mem);
		close(fd);
		if (flags & F_KILLE && size == 0) unlink(name);
	} else if (!strcmp(p, "out")) unlink(name); /* and kill mail */
}

void usage(void)
{
	puts("\
Create att/req     : ifqman [-s|--create] [-r|-c|-h|-f] address [file1] ...\n\
                     ifatt ... == ifqman -s ...\n\
                     ifreq ... == ifqman -r ...\n\
Delete att/req/mail: ifqman [-d|--delete] [-k|--kill-empty] [-chfrm]\n\
                            address [file1] ...\n\
List   att/req/mail: ifqman [-chfrm] addr\n\
You can use '-a' or '--config' to specify alternate config");
}

