/*:ts=8*/
/*****************************************************************************
 * FIDOGATE --- Gateway UNIX Mail/News <-> FTN NetMail/EchoMail
 *
 * $Id: areasbbs.c,v 4.14 1998/04/18 20:20:07 mj Exp $
 *
 * Function for processing AREAS.BBS EchoMail distribution file.
 *
 *****************************************************************************
 * Copyright (C) 1990-1998
 *  _____ _____
 * |     |___  |   Martin Junius             FIDO:      2:2452/110
 * | | | |   | |   Radiumstr. 18             Internet:  mj@fido.de
 * |_|_|_|@home|   D-51069 Koeln, Germany
 *
 * This file is part of FIDOGATE.
 *
 * FIDOGATE is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * FIDOGATE is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with FIDOGATE; see the file COPYING.  If not, write to the Free
 * Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *****************************************************************************/

#include "fidogate.h"
#include "annou.h"
#undef eAF



static char     *areasbbs_1stline = NULL;
static AreasBBS *areasbbs_list    = NULL;
static AreasBBS *areasbbs_last    = NULL;



/*
 * Remove area from areas.bbs
 */
void areasbbs_remove(AreasBBS *cur, AreasBBS *prev)
{
    if(!cur)
	return;
    
    if(prev)
	prev->next = cur->next;
    else
	areasbbs_list = cur->next;
    if(areasbbs_last == cur)
	areasbbs_last = prev;
}



/*
 * Alloc and init new AreasBBS struct
 */
AreasBBS *areasbbs_new(void)
{
    AreasBBS *p;
    
    p = (AreasBBS *)xmalloc(sizeof(AreasBBS));

    /* Init */
    p->next  = NULL;
    p->flags = 0;
    p->dir   = NULL;
    p->area  = NULL;
    p->zone  = -1;
    node_invalid(&p->addr);
    p->lvl   = -1;
    p->key   = NULL;
    p->desc  = NULL;
    p->state = NULL;
#ifdef eAF
    p->export = NULL;
#endif
    lon_init(&p->nodes);

    return p;
}



/*
 * Add nodes from string to list of nodes
 */
static int areasbbs_add_string(LON *lon, char *p)
{
    Node node, old;
    int ret;
    
    old.zone = cf_zone();
    old.net = old.node = old.point = -1;

    ret = OK;
    for(; p; p=xstrtok(NULL, " \t\r\n"))
    {
	if( asc_to_node_diff(p, &node, &old) == OK )
	{
	    old = node;
	    lon_add(lon, &node);
	}
	else
	{
	    ret = ERROR;
	    break;
	}
    }

    return ret;
}


/*
 * Create new AreasBBS struct for line from AREAS.BBS
 */
static AreasBBS *areasbbs_parse_line(char *line)
{
    AreasBBS *p;
    char *dir, *tag, *nl, *o2;
   
    dir = xstrtok(line, " \t\r\n");
    tag = xstrtok(NULL, " \t\r\n");
    if(!dir || !tag)
	return NULL;

    /* New areas.bbs entry */
    p = areasbbs_new();

    if(*dir == '#')
    {
	p->flags |= AREASBBS_PASSTHRU;
	dir++;
    }
    p->dir   = strsave(dir);
    p->area  = strsave(tag);
    
    /*
     * Options:
     *
     *     -a Z:N/F.P    alternate AKA for this area
     *     -z ZONE       alternate zone AKA for this area
     *     -l LVL        Areafix access level
     *     -k KEY        Areafix access key
     *     -d DESC       Area description text
     *     -#            Passthru
     +     -r            Read-only for new downlinks
     +	   -s		 state
#ifdef eAF
     +	   -e		 export
#endif
     */
    nl  = xstrtok(NULL, " \t\r\n");
    while(nl && *nl=='-')
    {
	if(streq(nl, "-a"))		/* -a Z:N/F.P */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    asc_to_node(o2, &p->addr, FALSE);
	}
	
	if(streq(nl, "-z"))		/* -z ZONE */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    p->zone = atoi(o2);
	}
	
	if(streq(nl, "-l"))		/* -l LVL */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    p->lvl = atoi(o2);
	}
	
	if(streq(nl, "-k"))		/* -k KEY */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    p->key = strsave(o2);
	}
	
	if(streq(nl, "-d"))		/* -d DESC */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    p->desc = strsave(o2);
	}
	
	if(streq(nl, "-s"))		/* -s STATE */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    p->state = strsave(o2);
	}
	
#ifdef eAF
	if(streq(nl, "-e"))		/* -e EXPORT */
	{
	    o2 = xstrtok(NULL, " \t\r\n");
	    p->export = strsave(o2);
	}
#endif
	
	if(streq(nl, "-#"))		/* -# */
	{
	    p->flags |= AREASBBS_PASSTHRU;
	}
	
	if(streq(nl, "-r"))		/* -r */
	{
	    p->flags |= AREASBBS_READONLY;
	}

	nl  = xstrtok(NULL, " \t\r\n");
    }	
    
    areasbbs_add_string(&p->nodes, nl);

    if(p->zone == -1)
	p->zone = p->nodes.first ? p->nodes.first->node.zone : 0;
    
    return p;
}



/*
 * Read area distribution list from AREAS.BBS file
 *
 * Format:
 *    [#$]DIR AREA [-options] Z:N/F.P Z:N/F.P ...
 */
int areasbbs_init(char *name)
{
    FILE *fp;
    AreasBBS *p;

    if(!name)
	return ERROR;
    
#ifdef ANNOU7
    if (areasbbs_list) {
	areasbbs_free();
    }
#endif

    debug(14, "Reading %s file" , name);
    
    fp = fopen_expand_name(name, R_MODE, FALSE);
    if(!fp)
	return ERROR;
    
    /*
     * 1st line is special
     */
    if(fgets(buffer, BUFFERSIZE, fp))
    {
	strip_crlf(buffer);
	areasbbs_1stline = strsave(buffer);
    }

    /*
     * The following lines are areas and linked nodes
     */
    while(fgets(buffer, BUFFERSIZE, fp))
    {
	strip_crlf(buffer);
	p = areasbbs_parse_line(buffer);
	if(!p)
	    continue;
	
	debug(15, "areas.bbs: %s %s Z%d", p->dir, p->area, p->zone);

	/*
	 * Put into linked list
	 */
	if(areasbbs_list)
	    areasbbs_last->next = p;
	else
	    areasbbs_list       = p;
	areasbbs_last       = p;
    }

    fclose(fp);

    return OK;
}



/*
 * Output AREAS.BBS, format short sorted list of downlink
 */
int areasbbs_print(FILE *fp)
{
    AreasBBS *p;
    
    fprintf(fp, "%s\r\n", areasbbs_1stline);
    
    for(p=areasbbs_list; p; p=p->next)
    {
	if(p->flags & AREASBBS_PASSTHRU)
	    fprintf(fp, "#");
	fprintf(fp, "%s %s ", p->dir, p->area);
	if(p->zone != -1)
	    fprintf(fp, "-z %d ", p->zone);
	if(p->addr.zone != -1)
	    fprintf(fp, "-a %s ", node_to_asc(&p->addr, TRUE));
	if(p->lvl != -1)
	    fprintf(fp, "-l %d ", p->lvl);
	if(p->key)
	    fprintf(fp, "-k %s ", p->key);
	if(p->desc)
	    fprintf(fp, "-d \"%s\" ", p->desc);
	if(p->state)
	    fprintf(fp, "-s %s ", p->state);
#ifdef eAF
	if(p->export)
	    fprintf(fp, "-e %s ", p->export);
#endif
	lon_print_sorted(&p->nodes, fp, 1);
	fprintf(fp, "\r\n");
    }
    
    return ferror(fp);
}



/*
 * Return areasbbs_list
 */
AreasBBS *areasbbs_first(void)
{
    return areasbbs_list;
}



/*
 * Lookup area
 */
AreasBBS *areasbbs_lookup(char *area)
{
    AreasBBS *p;
    
    /**FIXME: the search method should use hashing or similar**/
    for(p=areasbbs_list; p; p=p->next)
    {
	if(area  && !stricmp(area,  p->area ))
	    return p;
    }
    
    return NULL;
}



/*
 * Add areas.bbs entry
 */
void areasbbs_add(AreasBBS *p)
{
    /* Put into linked list */
    if(areasbbs_list)
	areasbbs_last->next = p;
    else
	areasbbs_list       = p;
    areasbbs_last       = p;
}
    
    

#ifdef AUTOCREATE
/*
 * Rewrite AREAS.BBS
 */
int rewrite_areas_bbs2(int n_history, char *areas_bbs)
{
    char old[MAXPATH], new[MAXPATH];
    int i, ovwr;
    FILE *fp;

    /*
     * Base name
     */
    str_expand_name(buffer, MAXPATH, areas_bbs);
    ovwr = strlen(buffer) - 3;		/* 3 = extension "bbs" */
    if(ovwr < 0)			/* Just to be sure */
	ovwr = 0;

    /*
     * Write new one as AREAS.NEW
     */
    strcpy(new, buffer);
    strcpy(new+ovwr, "new");
    debug(4, "Writing %s", new);
	if( (fp = fopen(new, W_MODE)) == NULL )
	{
	    log("$ERROR: can't open %s for writing AREAS.BBS", new);
	    return ERROR;
	}
	if( areasbbs_print(fp) == ERROR )
	{
	    log("$ERROR: writing to %s", new);
	    fclose(fp);
	    unlink(new);
	    return ERROR;
	}
	if( fclose(fp) == ERROR )
	{
	    log("$ERROR: closing %s", new);
	    unlink(new);
	    return ERROR;
	}

    /*
     * Renumber saved AREAS.Onn
     */
    strcpy(old, buffer);
    sprintf(old+ovwr, "o%02d", n_history);
    debug(4, "Removing %s", old);
    unlink(old);
    for(i=n_history-1; i>=1; i--)
    {
	strcpy(old, buffer);
	sprintf(old+ovwr, "o%02d", i);
	strcpy(new, buffer);
	sprintf(new+ovwr, "o%02d", i+1);
	debug(4, "Renaming %s -> %s", old, new);
        rename(old, new);
    }
    
    /*
     * Rename AREAS.BBS -> AREAS.O01
     */
    strcpy(old, buffer);
    strcpy(old+ovwr, "bbs");
    strcpy(new, buffer);
    strcpy(new+ovwr, "o01");
    debug(4, "Renaming %s -> %s", old, new);
    rename(old, new);
    
    /*
     * Rename AREAS.NEW -> AREAS.BBS
     */
    strcpy(old, buffer);
    strcpy(old+ovwr, "new");
    strcpy(new, buffer);
    strcpy(new+ovwr, "bbs");
    debug(4, "Renaming %s -> %s", old, new);
    rename(old, new);
    
    return OK;
}
#endif

/*
 * areasbbs_xfree
 */
void areasbbs_xfree	(AreasBBS *p)
{
    if (p->next) {
	areasbbs_xfree(p->next);
    }
    xfree(p);
}

/*
 * areasbbs_free
 */
void areasbbs_free	(void)
{

    if (areasbbs_list) {
	areasbbs_xfree(areasbbs_list);
    }
    areasbbs_list = NULL;
    areasbbs_last    = NULL;
}
