/*
    Any and all routines to worry about managing the screen using ncurses.
    Dickey rules.
*/

#include "cmp3funcs.h"
#include "keydef.h"

/****************************************************************************
 *  NCurses initialization
 *  Returns: nothing
 ****************************************************************************/
extern void curs_init()
{
    initscr();
    cbreak();
    nonl();
    keypad(stdscr,TRUE);
    curs_set(0);
    noecho();
    pad_left = newpad(MAX_LIST,MAX_WIDTH);
    pad_right = newpad(MAX_LIST,MAX_WIDTH);
    pad_list = newpad(MAX_LIST, MAX_FULL);
    win_right = newwin(LINES-9,COLS/2-1,3,COLS/2);
    win_left = newwin(LINES-9,COLS/2-2,3,1);
    win_list = newwin(LINES-9,COLS-2,3,1);
    if (has_colors() &&
        ini_getValueBool(cmp3rc, "cmp3_options", "show_colors", 1))
    {
        start_color();
        init_pair(CCOLOR_STD,COLOR_BLUE,COLOR_BLACK);        /* stdwin */
        init_pair(CCOLOR_LEFT,COLOR_GREEN,COLOR_BLACK);      /* win_left */
        init_pair(CCOLOR_RIGHT,COLOR_RED,COLOR_BLACK);       /* rghtshel */
        init_pair(CCOLOR_LIST,COLOR_CYAN,COLOR_BLACK);       /* listwin */
        init_pair(CCOLOR_HIGHLIGHT,COLOR_CYAN,COLOR_YELLOW); /* grabbed file */
        init_pair(CCOLOR_HELP,COLOR_MAGENTA,COLOR_BLACK);    /* help window */
        init_pair(CCOLOR_DIALOG,COLOR_CYAN,COLOR_BLACK);     /* dialog box */
        attron(COLOR_PAIR(CCOLOR_STD));
        wattron(pad_list, COLOR_PAIR(CCOLOR_LIST));
        wattron(win_list, COLOR_PAIR(CCOLOR_LIST));
        wattron(pad_left, COLOR_PAIR(CCOLOR_LEFT));
        wattron(win_left, COLOR_PAIR(CCOLOR_LEFT));
        wattron(pad_right, COLOR_PAIR(CCOLOR_RIGHT));
        wattron(win_right, COLOR_PAIR(CCOLOR_RIGHT));
    }

    box(stdscr, 0, 0);
    box(win_list, 0, 0);

    if (shmptr->repeat == 1) {
        mvprintw(LINES - 2, 3, "R");
    }

    return;
}

/****************************************************************************
 *  Make a dialog box with the message displayed, get yes or no
 *  Returns: PASS or FAIL
 ****************************************************************************/
extern int dialogbox(char *message)
{
    WINDOW *dialog;
    char answer;

    dialog=newwin(LINES/6,COLS/3,LINES/3,COLS/3);
    wattron(dialog,COLOR_PAIR(CCOLOR_DIALOG));
    mvwaddstr(dialog, 1, (COLS/6) - (strlen(message)/2), message);
    mvwaddstr(dialog, 2, (COLS/6) - 2, "(y/n)");
    box(dialog,0,0);
    wrefresh(dialog);
    answer=getch();
    while ((answer != 'y') && (answer != CMP3_KEY_ENTER) &&
           (answer != 'n') && (answer != CMP3_KEY_ESCAPE)) {
        wattron(dialog,A_BLINK);
        mvwaddstr(dialog,2,5,"No you WHORE! (y,n)");
        wattroff(dialog,A_BLINK);
        wrefresh(dialog);
        answer=getch();
    }
    delwin(dialog);
    if ((answer == 'n') || (answer == CMP3_KEY_ESCAPE))
        return(0);
    return(1);
}

/* XXX - try forms */
static int form_virtualize(FORM *f, WINDOW *w, int *thechar)
{
    static const struct {
        int code;
        int result;
    } lookup[] = {
        { CTRL('Z'),            REQ_PREV_WORD },
        { CTRL('J'),            REQ_CLR_EOL },
        { CTRL('G'),            REQ_DEL_WORD },
        { CTRL('H'),            REQ_DEL_PREV },
        { CTRL(' '),            REQ_INS_CHAR },
        { CTRL('K'),            REQ_CLR_EOF },
        { CTRL('A'),            REQ_BEG_LINE },
        { CTRL('D'),            REQ_DEL_CHAR },
        { CTRL('X'),            REQ_NEXT_WORD },
        { CTRL('Y'),            REQ_CLR_FIELD },
        { CTRL('Y'),            REQ_DEL_LINE },
        { CTRL('F'),            REQ_RIGHT_CHAR },
        { CTRL('B'),            REQ_LEFT_CHAR },
        { CTRL('E'),            REQ_END_LINE },
        { KEY_BACKSPACE,        REQ_DEL_PREV },
        { 127,                  REQ_DEL_PREV },
        { KEY_DC,               REQ_DEL_PREV },
/*         { KEY_DOWN,             REQ_DOWN_CHAR }, */
        { KEY_END,              REQ_LAST_FIELD },
        { KEY_HOME,             REQ_FIRST_FIELD },
        { KEY_LEFT,             REQ_LEFT_CHAR },
        { KEY_RIGHT,            REQ_RIGHT_CHAR },
/*         { KEY_UP,               REQ_UP_CHAR },        */
        { CMP3_KEY_ENTER,       MAX_FORM_COMMAND + 1 },
        { CMP3_KEY_ESCAPE,      MAX_FORM_COMMAND + 1 },
        { CTRL('Q'),            MAX_FORM_COMMAND + 1 },
        { 0, 0}
    };

    int             c;
    int             n;

    c = wgetch(w);
    *thechar = c;

    for (n = 0; (lookup[n].code != 0) || (lookup[n].result != 0); n++)
    {
        if (lookup[n].code == c) {
            c = lookup[n].result;
            if (lookup[n].result == MAX_FORM_COMMAND + 1)
            {
                if (*thechar != CMP3_KEY_ENTER)
                {
                    *thechar = CMP3_KEY_ESCAPE;
                }
            }
            break;
        }
    }

    return c;
}

extern int inputbox(char *message, char *buffer, int bufflen)
{
    WINDOW     *dialog;
    FORM       *form;
    FIELD      *field[3];
    int         c;
    int         finished;
    int         toreturn;

    dialog=newwin(LINES/6, COLS/3, LINES/3, COLS/3);
    wattron(dialog,COLOR_PAIR(CCOLOR_DIALOG));

    curs_set(1);

    field[0] = new_field(1, COLS/3 - 4, 2, 2, 3, 1);
    if (field[0] == NULL)
        return(0);
    set_field_back(field[0], A_REVERSE | COLOR_PAIR(CCOLOR_DIALOG));
    set_field_buffer(field[0], 0, buffer);
    field[1] = new_field(1, strlen(message), 1,
                        (COLS/6) - (strlen(message)/2), 0, 0);
    set_field_buffer(field[1], 0, message);
    set_field_fore(field[1], COLOR_PAIR(CCOLOR_DIALOG));
    field[2] = NULL;

    form = new_form(field);
    set_form_win(form, dialog);
    
    post_form(form);
/*     mvwaddstr(dialog, 1, (COLS/6) - (strlen(message)/2), message); */
    box(dialog, 0, 0);

    wrefresh(dialog);

    finished = 0;
    while (!finished)
    {
        switch(form_driver(form, form_virtualize(form, dialog, &c)))
        {
        case E_OK:
        break;
        case E_UNKNOWN_COMMAND:
            finished = 1;
            form_driver(form, REQ_VALIDATION);
            switch(c)
            {
                case CMP3_KEY_ENTER:
                    toreturn = 1;
                break;
                case CMP3_KEY_ESCAPE:
                    toreturn = 0;
                break;
                default:
                    toreturn = 0;
                break;
            }
        break;
        default:
            beep();
        break;
        }
    }
    
    strcpy(buffer, field_buffer(field[0], 0));
    for (finished = 80; (buffer[finished] == ' ') && (finished >= 0);
         finished--)
        ;
    if (finished >= 0)
        buffer[++finished] = '\0';
    else
        toreturn = 0;

/*
    fprintf(stderr, "return %d: %s", toreturn, field_buffer(field[0], 0));
    sleep(2);
*/

    free_form(form);
    free_field(field[0]);
    delwin(dialog);

    curs_set(0);

    return(toreturn);
}

#if 0
/****************************************************************************
 *  Make an input box with the message displayed, get response
 *  Returns: PASS or FAIL
 ****************************************************************************/
extern int inputbox(char *message, char *buffer, int bufflen)
{
    WINDOW     *dialog;
    int         answer;
    int         i;
    int         cleared = 0;

    dialog=newwin(LINES/6, COLS/3, LINES/3, COLS/3);
    wattron(dialog,COLOR_PAIR(CCOLOR_DIALOG));
    curs_set(1);
    box(dialog,0,0);
    mvwaddstr(dialog, 1, (COLS/6) - (strlen(message)/2), message);
    wattron(dialog, COLOR_PAIR(CCOLOR_HIGHLIGHT));
    mvwprintw(dialog, 2, 3, "%.*s", COLS/3 - 5, buffer);

    wmove(dialog, 2, 3);
    wrefresh(dialog);
    i = 0;
    answer=getch();
    while((answer != CMP3_KEY_ENTER) && (answer != CMP3_KEY_ESCAPE)) {
        switch(answer)
        {
        case KEY_BACKSPACE: case KEY_DC:
            if (i == 0)
                break;
            i--;
            wprintw(dialog, "\b \b");
        break;
        case KEY_LEFT:
            wprintw(dialog, "\b");
            i--;
        break;
        case KEY_RIGHT:
            wprintw(dialog, "%c", buffer[i]);
            i++;
        break;
        default:
            if (isprint(answer)) {
                if ((i == 0) && (cleared == 0)) {
                    memset(buffer, ' ', bufflen);
                    cleared = 1;
                }
                wprintw(dialog, "%c", answer);
                buffer[i] = answer;
                i++;
            }
        break;
        }
        wrefresh(dialog);
        answer=getch();
    }
    curs_set(0);
    wattron(dialog,COLOR_PAIR(CCOLOR_DIALOG));
    if (answer == CMP3_KEY_ESCAPE)
        return(0);
    buffer[i] = '\0';
    return(1);
}
#endif

/****************************************************************************
 *  Switch left and right windows
 *  Returns: pointer to current window
 ****************************************************************************/
extern int switchwin(int thewin)
{
    if (thewin == FOCUSED_LEFT) {
        curwin = FOCUSED_RIGHT;
        wattron(win_right,A_REVERSE);
        mvwprintw(win_right,1,2,".mp3 files:");
        wattroff(win_right,A_REVERSE);
        mvwprintw(win_left,1,2,"Directory:");
    } else {
        curwin = FOCUSED_LEFT;
        wattron(win_left,A_REVERSE);
        mvwprintw(win_left,1,2,"Directory:");
        wattroff(win_left,A_REVERSE);
        mvwprintw(win_right,1,2,".mp3 files:");
    }
    ll_prncur(list_left);
    ll_prncur(list_right);
    refreshpad(FOCUSED_LEFT);
    refreshpad(FOCUSED_RIGHT);
    wnoutrefresh(win_left);
    wrefresh(win_right);
    return(curwin);
}

/****************************************************************************
 *  Show help information
 *  Returns: nothing
 ****************************************************************************/
extern void showhelp()
{
    int curwinbackup;
    WINDOW *helpwin;

    curwinbackup = curwin;
    helpwin = newwin(LINES-9,COLS-2,3,1);
    wattron(helpwin, COLOR_PAIR(CCOLOR_HELP));
    wattron(helpwin, A_REVERSE);
    mvwprintw(helpwin, 1, ((COLS-2)/2)-3, " Help ");
    wattroff(helpwin, A_REVERSE);

    if (curwin == FOCUSED_LIST) {
        curwin = FOCUSED_HELP;
        mvwprintw(helpwin, 3, 0,
                         "   %c\t\t - Remove file from list\n", CMP3_KEY_LISTDEL);
        wprintw(helpwin, "   %c\t\t - grab/let go of a file\n", CMP3_KEY_GRABFILE); 
        wprintw(helpwin, "   %c\t\t - Randomize the playlist (starting at grab)\n",
                                                        CMP3_KEY_RANDOMIZE);
        wprintw(helpwin, "   %c\t\t - Quit List Mode\n", CMP3_KEY_LISTQUIT);

    } else {
        curwin = FOCUSED_HELP;
        mvwprintw(helpwin, 3, 0,
                         "   %c\t\t - Kill song\n", CMP3_KEY_KILL1);
        wprintw(helpwin, "   %c\t\t - Pause song\n", CMP3_KEY_PAUSE);
        wprintw(helpwin, "   F5\t\t - Restart song\n");
        wprintw(helpwin, "   %c\t\t - suspend display\n", CMP3_KEY_SUSPEND);
        wprintw(helpwin, "   %c\t\t - enter list mode\n", CMP3_KEY_LISTMODE);
        wprintw(helpwin, "   %c\t\t - add all mp3s in dir to playlist\n", 
                                                            CMP3_KEY_ADDDIR);
        wprintw(helpwin, "   %c\t\t - recursively add all mp3s to playlist\n",
                                                        CMP3_KEY_RECURADDIR);
        wprintw(helpwin, "   %c\t\t - clear playlist\n", CMP3_KEY_CLEARPL);
        wprintw(helpwin, "   %c\t\t - write playlist\n", CMP3_KEY_WRITEPL);
        wprintw(helpwin, "   %c/%c\t\t - volume control\n", CMP3_KEY_VOLUP1,
                                                            CMP3_KEY_VOLDWN1);
        wprintw(helpwin, "   %c\t\t - toggle repeat mode\n", CMP3_KEY_REPEAT);
        wprintw(helpwin, "   F2\t\t - change dir dialog\n");
        wprintw(helpwin, "   %c\t\t - change dir to home\n", CMP3_KEY_CHDIRHOME);
#ifdef ASS_LOC
        wprintw(helpwin, "   %c\t\t - change dir to ass home\n",
                                                            CMP3_KEY_CHDIRASS);
#endif /* ASS_LOC */
        wprintw(helpwin, "   %c\t\t - refresh screen\n", CMP3_KEY_REFRESH);
        wprintw(helpwin, "   delete\t - delete file\n");
    }
    mvwprintw(helpwin, LINES-11, (COLS-2)-20, "hit any key to exit");
    box(helpwin, 0, 0);
    wrefresh(helpwin);
    getch();
    curwin = curwinbackup;
    delwin(helpwin);
}

/****************************************************************************
 *  Do refreshing of a pad
 *  Returns: nothing
 ****************************************************************************/
extern void refreshpad(int curwin)
{
    void *infoptr;

    if (curwin == FOCUSED_LEFT) {
        infoptr = ll_info(list_left, NULL);
        prefresh(pad_left, ((info_left_t*) infoptr)->line, 0, 6, 3,
                LINES-8, COLS/2-4);
    } else {
        infoptr = ll_info(list_right, NULL);
        prefresh(pad_right, ((info_right_t*) infoptr)->line, 0, 6, COLS/2+2,
                LINES-8,COLS-4);
    }
}

/****************************************************************************
 *  Refresh both pads
 *  Returns: nothing
 ****************************************************************************/
void refreshpads()
{
    void *infoptr;

    infoptr = ll_info(list_left, NULL);
    pnoutrefresh(pad_left, ((info_left_t*) infoptr)->line, 0, 6, 3,
            LINES-8, COLS/2-4);

    infoptr = ll_info(list_right, NULL);
    prefresh(pad_right, ((info_right_t*) infoptr)->line, 0, 6, COLS/2+2,
            LINES-8,COLS-4);
}

/****************************************************************************
 *  Clean pads
 *  Returns: nothing
 ****************************************************************************/
extern void pads_clean()
{
    int i;

    if (ll_total(list_right) > 0) {
        mvwprintw(pad_right, 0, 0, "\n");
        for (i=0; i < ll_total(list_right) - 1; i++)
            wprintw(pad_right, "\n");
    }
    if (ll_total(list_left) > 0) {
        mvwprintw(pad_left, 0, 0, "\n");
        for (i=0; i < ll_total(list_left) - 1; i++)
            wprintw(pad_left, "\n");
    }
}

/****************************************************************************
 *  Refresh parts after a dialog box was drawn
 *  Returns: nothing
 ****************************************************************************/
void dialog_clean()
{
    volclean();
    wredrawln(win_left,LINES/3-3,LINES/6);
    wredrawln(win_right,LINES/3-3,LINES/6);

    wnoutrefresh(stdscr);
    wnoutrefresh(win_left);
    wnoutrefresh(win_right);

    refreshpad(FOCUSED_LEFT);
    refreshpad(FOCUSED_RIGHT);
}

/****************************************************************************
 *  Refresh left and right windows after the list box was drawn
 *  Returns: nothing
 ****************************************************************************/
void after_list()
{
    volclean();
    redrawwin(stdscr);
    redrawwin(win_left);
    redrawwin(win_right);

    wnoutrefresh(stdscr);
    wnoutrefresh(win_left);
    wnoutrefresh(win_right);

    refreshpad(FOCUSED_LEFT);
    refreshpad(FOCUSED_RIGHT);
}

/* EOF */
