/*
 *  apcreports.c --
 *
 *  apcupsd.c    -- Simple Daemon to catch power failure signals from a
 *                  BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *               -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <syslog.h>

#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#ifdef NEW_THREADS
static char largebuf[4096];
#endif /* NEW_THREADS */

#ifdef NEW_THREADS
/*********************************************************************/
/*
 * When do we get rid of this monster? -BSC
 *
 * When apcaccess will be working flawlessy. -RF
 */
int updating_procfs (FILE *file, UPSINFO *ups)
{
	get_upsstat(largebuf, ups);

	file = freopen(UPS_STAT, "w", file);
	fputs(largebuf, file);
	fflush(file);
	return(0);
}

#else /* NEW_THREADS */

/*********************************************************************/
int updating_procfs (FILE *file, UPSINFO *ups)
{
	char msg[100];
	time_t nowtime;

	file = freopen(UPS_STAT, "w", file);
	time(&nowtime);
	strftime(msg, 100, "%b %d %X", localtime(&nowtime));

	fprintf(file,"APC      : %s\n", msg);
	fprintf(file,"CABLE    : %s\n", ups->cable.long_name);
	fprintf(file,"UPSMODEL : %s\n", ups->mode.long_name);
	fprintf(file,"UPSMODE  : %s\n", ups->class.long_name);
	if (ups->sharenet.type != DISABLE)
		fprintf(file,"SHARE    : %s\n", ups->sharenet.long_name);

	switch(ups->mode.type) {
		case BK:
		case SHAREBASIC:
		case NETUPS:
			if (ups->LineUp == 0) {
				fprintf(file, "LINEFAIL : OK\n");
				fprintf(file, "BATTSTAT : OK\n");
			} else {
				fprintf(file, "LINEFAIL : DOWN\n");
				if ((ups->BattUp == 0) ||
				    ((ups->BattUp != 0) &&
				    ((flags & dtr_bit) == 0)))
					fprintf(file, "BATTSTAT : RUNNING\n");
				else
					fprintf(file, "BATTSTAT : FAILING\n");
			}
			break;
		case BKPRO:
		case VS:
		case NBKPRO:
		case SMART:
		case SHARESMART:
		case MATRIX:

			if (ups->UPSlink_flags == 0x0000800f) {
				if (ups->LineUp == 0) {
					fprintf(file, "LINEFAIL : OK\n");
					fprintf(file, "BATTSTAT : OK\n");
					if (ups->LineLevel == -1)
						fprintf(file, "LINEVOLT : LOW\n");
					else if (ups->LineLevel == 1)
						fprintf(file, "LINEVOLT : HIGH\n");
					else
						fprintf(file, "LINEVOLT : OK\n");
				} else {
					fprintf(file, "LINEFAIL : DOWN\n");
					if (ups->BattUp == 0)
						fprintf(file, "BATTSTAT : RUNNING\n");
					else
						fprintf(file, "BATTSTAT : FAILING\n");
				}
				break;
			}

			if (ups->UPSlink_flags & 0x01000000)
				fprintf(file, "UPSNAME  : %s\n", ups->name);

			if (ups->UPSlink_flags & 0x00000010)
				fprintf(file, "ULINE    : %05.1f Volts\n", ups->LineVoltage);

			if (ups->UPSlink_flags & 0x00000020)
				fprintf(file, "MLINE    : %05.1f Volts\n", ups->LineMax);

			if (ups->UPSlink_flags & 0x00000040)
				fprintf(file, "NLINE    : %05.1f Volts\n", ups->LineMin);

			if (ups->UPSlink_flags & 0x00000800)
				fprintf(file, "FLINE    : %03.1f Hz\n", ups->LineFreq);

			if (ups->UPSlink_flags & 0x00000080)
				fprintf(file, "VOUTP    : %05.1f Volts\n", ups->OutputVoltage);

			if (ups->UPSlink_flags & 0x00000400)
				fprintf(file, "LOUTP    : %05.1f Load Capacity\n", ups->UPSLoad);

			if (ups->UPSlink_flags & 0x00000200)
				fprintf(file, "BOUTP    : %04.1f Volts\n", ups->BattVoltage);

			if (ups->UPSlink_flags & 0x00000100)
				fprintf(file, "BCHAR    : %05.1f Batt. Charge\n", ups->BatLoad);

			if (ups->UPSlink_flags & 0x00001000)
				fprintf(file, "TIME     : %04.1f Minutes\n", ups->TimeLeft);

			if (ups->UPSlink_flags & 0x00010000) {
				switch((*ups).sensitivity[0]) {
					case 'L':
						fprintf(file, "SENSE    : LOW\n");
						break;
					case 'M':
						fprintf(file, "SENSE    : MEDIUM\n");
						break;
					case 'H':
					default :
						fprintf(file, "SENSE    : HIGH\n");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00020000)
				fprintf(file, "WAKEUP   : %03d Cycles\n", ups->wakeup);

			if (ups->UPSlink_flags & 0x00040000)
				fprintf(file, "SLEEP    : %03d Cycles\n", ups->sleep);

			if (ups->UPSlink_flags & 0x00080000)
				fprintf(file, "LOTRANS  : %03d.0 Volts\n", ups->lotrans);

			if (ups->UPSlink_flags & 0x00100000)
				fprintf(file, "HITRANS  : %03d.0 Volts\n", ups->hitrans);

			if (ups->UPSlink_flags & 0x00200000)
				fprintf(file, "CHARGE   : %03d.0 Percent\n", ups->minon);

			if (ups->UPSlink_flags & 0x00000001)
				fprintf(file, "BFAIL    : 0x%02X Status Flag\n", ups->Status);

			if (ups->UPSlink_flags & 0x00002000)
				fprintf(file, "UTEMP    : %04.1f C Internal\n", ups->UPSTemp);

			if (ups->UPSlink_flags & 0x00400000) {
				switch((*ups).beepstate[0]) {
					case 'T':
						fprintf(file, "ALARM    : Low Batt + 30\n");
						break;
					case 'L':
						fprintf(file, "ALARM    : Low Batt\n");
						break;
					case 'N':
						fprintf(file, "ALARM    : Never\n");
						break;
					case '0':
					default :
						fprintf(file, "ALARM    : Always\n");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00000004) {
				switch((*ups).G[0]) {
					case 'O':
						fprintf(file, "LASTEVNT : POWER UP\n");
						break;
					case 'S':
						fprintf(file, "LASTEVNT : SELF TEST\n");
						break;
					case 'L':
						fprintf(file, "LASTEVNT : LINE VOLTAGE DECREASE\n");
						break;
					case 'H':
						fprintf(file, "LASTEVNT : LINE VOLTAGE INCREASE\n");
						break;
					case 'T':
						fprintf(file, "LASTEVNT : POWER FAILURE\n");
						break;
					default :
						fprintf(file, "LASTEVNT : UNKNOWN EVENT %c\n", (*ups).G[1]);
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00800000)
				fprintf(file, "LOWBATT  : %02d Minutes\n", ups->lowbatt);

			if (ups->UPSlink_flags & 0x00004000) {
				if (ups->seven & 0x08)
					fprintf(file, "DIPSW    : 0x%d", TRUE);
				else
					fprintf(file, "DIPSW    : 0x%d", FALSE);
				if (ups->seven & 0x04)
					fprintf(file, "%d", TRUE);
				else
					fprintf(file, "%d", FALSE);
				if (ups->seven & 0x02)
					fprintf(file, "%d", TRUE);
				else
					fprintf(file, "%d", FALSE);
				if (ups->seven & 0x01)
					fprintf(file, "%d\n", TRUE);
				else
					fprintf(file, "%d\n", FALSE);
			}

			break;
		default:
			break;
	}
	fflush(file);
	time(&last_time_procfs);
	return (0);
}
#endif /* NEW_THREADS */

/*********************************************************************/
int updating_logs (FILE *file, UPSINFO *ups)
{
	char msg[100];
	time_t nowtime;

	time(&nowtime);
	strftime(msg, 100,"%b %d %X", localtime(&nowtime));

	fprintf(file, "APC: %s ", msg);

	switch(ups->mode.type) {
		case BK:
		case SHAREBASIC:
		case NETUPS:
			if (ups->LineUp == 0) {
				fprintf(file, "LINEFAIL:OK ");
				fprintf(file, "BATTSTAT:OK\n");
			} else {
				fprintf(file, "LINEFAIL:DOWN ");
				if ((ups->BattUp == 0) ||
				    ((ups->BattUp != 0) &&
				    ((flags & dtr_bit) == 0)))
					fprintf(file, "BATTSTAT:RUNNING\n");
				else
					fprintf(file, "BATTSTAT:FAILING\n");
			}
			break;
		case BKPRO:
		case VS:
		case NBKPRO:
		case SMART:
		case SHARESMART:
		case MATRIX:

			if (ups->UPSlink_flags == 0x0000800f) {
				if (ups->LineUp == 0) {
					fprintf(file, "LINEFAIL:OK ");
					fprintf(file, "BATTSTAT:OK ");
					if (ups->LineLevel == -1)
						fprintf(file, "LINEVOLT:LOW ");
					else if (ups->LineLevel == 1)
						fprintf(file, "LINEVOLT:HIGH ");
					else
						fprintf(file, "LINEVOLT:OK ");
				} else {
					fprintf(file, "LINEFAIL:DOWN ");
					if (ups->BattUp == 0)
						fprintf(file, "BATTSTAT:RUNNING ");
					else
						fprintf(file, "BATTSTAT:FAILING ");
				}

				switch ((*ups).G[0]) {
					case 'O':
						fprintf(file, "LASTEVNT:POWER UP\n");
						break;
					case 'S':
						fprintf(file, "LASTEVNT:SELF TEST\n");
						break;
					case 'L':
						fprintf(file, "LASTEVNT:LINE VOLTAGE DECREASE\n");
						break;
					case 'H':
						fprintf(file, "LASTEVNT:LINE VOLTAGE INCREASE\n");
						break;
					case 'T':
						fprintf(file, "LASTEVNT:POWER FAILURE\n");
						break;
					default :
						fprintf(file, "LASTEVNT:UNKNOWN EVENT %c\n", (*ups).G[1]);
						break;
				}
			} else {
				fprintf(file, "%05.1f ", ups->LineVoltage);
				fprintf(file, "%05.1f ", ups->LineMax);
				fprintf(file, "%05.1f ", ups->LineMin);
				fprintf(file, "%03.1f ", ups->LineFreq);
				fprintf(file, "%05.1f ", ups->OutputVoltage);
				fprintf(file, "%05.1f ", ups->UPSLoad);
				fprintf(file, "%04.1f ", ups->BattVoltage);
				fprintf(file, "%05.1f ", ups->BatLoad);
				if (ups->UPSlink_flags & 0x00002000)
					fprintf(file, "%04.1f C", ups->UPSTemp);
				fprintf(file, "\n");
			}
			break;
		default:
			break;
	}
	fflush(file);
	time(&last_time_logging);
	return (0);
}

/*********************************************************************/
int logprintf (char *fmt, ...)
{
	va_list ap;
	char timestr[32];
	time_t now;
	int ret;

	if (!logfile) return(0);                     /* No logging enabled */

	va_start(ap, fmt);

	/* All entries starts with a timestamp */
	time(&now);
	strftime(timestr, sizeof(timestr), "%b %d %X", localtime(&now));
	fprintf(logfile, "APC: %s ", timestr);

	/* And now for the real message .. */
	ret = vfprintf(logfile, fmt, ap);

	if (fflush(logfile))
		ret = -1;

	va_end(ap);
	return(ret);
}

#ifdef NEW_THREADS

/*********************************************************************/
void logopen(void)
{
	openlog("apcupsd", LOG_PID|LOG_CONS, LOG_DAEMON);
}

/*********************************************************************/
void logclose(void)
{
	closelog();
}

/*********************************************************************/
void logmess(char *message)
{
	syslog(LOG_NOTICE, message);
}

/*********************************************************************/
void logerr(char *message)
{
	syslog(LOG_ERR, message);
}

/*********************************************************************/
/*
 * Caution!
 *
 * Remember to size largebuf when used in 'updating_procfs'.
 * Remember to size buffer when used in apchttp.
 * Don't remove the '\r's - they are required by apchttp.
 *
 * -BSC
 */
int get_upsstat (char *buf, UPSINFO *ups)
{
	char msg[100];
	
	time_t nowtime;

	time(&nowtime);
	strftime(msg, 100, "%b %d %X", localtime(&nowtime));

	buf+=sprintf(buf, "\n");
	buf+=sprintf(buf-1, "APC      : %s\r\n", msg);
	buf+=sprintf(buf-1, "CABLE    : %s\r\n", ups->cable.long_name);
	buf+=sprintf(buf-1, "UPSMODEL : %s\r\n", ups->mode.long_name);
	buf+=sprintf(buf-1, "UPSMODE  : %s\r\n", ups->class.long_name);
	if (ups->sharenet.type != DISABLE)
		buf+=sprintf(buf-1, "SHARE    : %s\r\n", ups->sharenet.long_name);

	switch(ups->mode.type) {
		case BK:
		case SHAREBASIC:
		case NETUPS:
			if (ups->LineUp == 0) {
				buf+=sprintf(buf-1, "LINEFAIL : OK\r\n");
				buf+=sprintf(buf-1, "BATTSTAT : OK\r\n");
			} else {
				buf+=sprintf(buf-1, "LINEFAIL : DOWN\r\n");
				if ((ups->BattUp == 0) ||
				    ((ups->BattUp != 0) &&
				    ((flags & dtr_bit) == 0)))
					buf+=sprintf(buf-1, "BATTSTAT : RUNNING\r\n");
				else
					buf+=sprintf(buf-1, "BATTSTAT : FAILING\r\n");
			}
			break;
		case BKPRO:
		case VS:
		case NBKPRO:
		case SMART:
		case SHARESMART:
		case MATRIX:
			if (ups->UPSlink_flags == 0x0000800f) {
				if (ups->LineUp == 0) {
					buf+=sprintf(buf-1, "LINEFAIL : OK\r\n");
					buf+=sprintf(buf-1, "BATTSTAT : OK\r\n");
					if (ups->LineLevel == -1)
						buf+=sprintf(buf-1, "LINEVOLT : LOW\r\n");
					else if (ups->LineLevel == 1)
						buf+=sprintf(buf-1, "LINEVOLT : HIGH\r\n");
					else
						buf+=sprintf(buf-1, "LINEVOLT : OK\r\n");
				} else {
					buf+=sprintf(buf-1, "LINEFAIL : DOWN\r\n");
					if (ups->BattUp == 0)
						buf+=sprintf(buf-1, "BATTSTAT : RUNNING\r\n");
					else
						buf+=sprintf(buf-1, "BATTSTAT : FAILING\r\n");
				}
			}

			if (ups->UPSlink_flags & 0x01000000)
				buf+=sprintf(buf-1, "UPSNAME  : %s\r\n", ups->name);

			if (ups->UPSlink_flags & 0x00000010)
				buf+=sprintf(buf-1, "ULINE    : %05.1f Volts\r\n", ups->LineVoltage);

			if (ups->UPSlink_flags & 0x00000020)
				buf+=sprintf(buf-1, "MLINE    : %05.1f Volts\r\n", ups->LineMax);

			if (ups->UPSlink_flags & 0x00000040)
				buf+=sprintf(buf-1, "NLINE    : %05.1f Volts\r\n", ups->LineMin);

			if (ups->UPSlink_flags & 0x00000800)
				buf+=sprintf(buf-1, "FLINE    : %03.1f Hz\r\n", ups->LineFreq);

			if (ups->UPSlink_flags & 0x00000080)
				buf+=sprintf(buf-1, "VOUTP    : %05.1f Volts\r\n", ups->OutputVoltage);

			if (ups->UPSlink_flags & 0x00000400)
				buf+=sprintf(buf-1, "LOUTP    : %05.1f Load Capacity\r\n", ups->UPSLoad);

			if (ups->UPSlink_flags & 0x00000200)
				buf+=sprintf(buf-1, "BOUTP    : %04.1f Volts\r\n", ups->BattVoltage);

			if (ups->UPSlink_flags & 0x00000100)
				buf+=sprintf(buf-1, "BCHAR    : %05.1f Batt. Charge\r\n", ups->BatLoad);

			if (ups->UPSlink_flags & 0x00001000)
				buf+=sprintf(buf-1, "TIME     : %04.1f Minutes\r\n", ups->TimeLeft);

			if (ups->UPSlink_flags & 0x00010000) {
				switch((*ups).sensitivity[0]) {
					case 'L':
						buf+=sprintf(buf-1, "SENSE    : LOW\r\n");
						break;
					case 'M':
						buf+=sprintf(buf-1, "SENSE    : MEDIUM\r\n");
						break;
					case 'H':
					default :
						buf+=sprintf(buf-1, "SENSE    : HIGH\r\n");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00020000)
				buf+=sprintf(buf-1, "WAKEUP   : %03d Cycles\r\n", ups->wakeup);

			if (ups->UPSlink_flags & 0x00040000)
				buf+=sprintf(buf-1, "SLEEP    : %03d Cycles\r\n", ups->sleep);

			if (ups->UPSlink_flags & 0x00080000)
				buf+=sprintf(buf-1, "LOTRANS  : %03d.0 Volts\r\n", ups->lotrans);

			if (ups->UPSlink_flags & 0x00100000)
				buf+=sprintf(buf-1, "HITRANS  : %03d.0 Volts\r\n", ups->hitrans);

			if (ups->UPSlink_flags & 0x00200000)
				buf+=sprintf(buf-1, "CHARGE   : %03d.0 Percent\r\n", ups->minon);

			if (ups->UPSlink_flags & 0x00000001)
				buf+=sprintf(buf-1, "BFAIL    : 0x%02X Status Flag\r\n", ups->Status);

			if (ups->UPSlink_flags & 0x00002000)
				buf+=sprintf(buf-1, "UTEMP    : %04.1f C Internal\r\n", ups->UPSTemp);

			if (ups->UPSlink_flags & 0x00400000) {
				switch((*ups).beepstate[0]) {
					case 'T':
						buf+=sprintf(buf-1, "ALARM    : Low Batt + 30\r\n");
						break;
					case 'L':
						buf+=sprintf(buf-1, "ALARM    : Low Batt\r\n");
						break;
					case 'N':
						buf+=sprintf(buf-1, "ALARM    : Never\r\n");
						break;
					case '0':
					default :
						buf+=sprintf(buf-1, "ALARM    : Always\r\n");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00000004) {
				switch((*ups).G[0]) {
					case 'O':
						buf+=sprintf(buf-1, "LASTEVNT : POWER UP\r\n");
						break;
					case 'S':
						buf+=sprintf(buf-1, "LASTEVNT : SELF TEST\r\n");
						break;
					case 'L':
						buf+=sprintf(buf-1, "LASTEVNT : LINE VOLTAGE DECREASE\r\n");
						break;
					case 'H':
						buf+=sprintf(buf-1, "LASTEVNT : LINE VOLTAGE INCREASE\r\n");
						break;
					case 'T':
						buf+=sprintf(buf-1, "LASTEVNT : POWER FAILURE\r\n");
						break;
					default :
						buf+=sprintf(buf-1, "LASTEVNT : UNKNOWN EVENT %c\r\n", (*ups).G[1]);
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00800000)
				buf+=sprintf(buf-1, "LOWBATT  : %02d Minutes\r\n", ups->lowbatt);

			if (ups->UPSlink_flags & 0x00004000) {
				if (ups->seven & 0x08)
					buf+=sprintf(buf-1, "DIPSW    : 0x%d", TRUE);
				else
					buf+=sprintf(buf-1, "DIPSW    : 0x%d", FALSE);
				if (ups->seven & 0x04)
					buf+=sprintf(buf-1, "%d", TRUE);
				else
					buf+=sprintf(buf-1, "%d", FALSE);
				if (ups->seven & 0x02)
					buf+=sprintf(buf-1, "%d", TRUE);
				else
					buf+=sprintf(buf-1, "%d", FALSE);
				if (ups->seven & 0x01)
					buf+=sprintf(buf-1, "%d\r\n", TRUE);
				else
					buf+=sprintf(buf-1, "%d\r\n", FALSE);
			}

			break;
		default:
			break;
	}
	time(&last_time_procfs);
	return (0);
}

/*
 * This function write the reports into the local files that need to be
 * updated. Note that there is still the last_time_* control so that we can
 * have a fast path when time is not enought.
 *
 * -RF
 */

#define REPORTS_RW		0

void timer_reports (int sig)
{
	UPSINFO *ups = &myUPS;
	int alarm_val = 0;

#if REPORTS_RW
	if (read_andlock_shmarea(ups) != SUCCESS) {
#else /* !REPORTS_RW */
	if (read_shmarea(ups) != SUCCESS) {
#endif /* REPORTS_RW */
		fprintf(stderr, "Can not read shm data area.\n");
		logprintf("Can not read shm data area.\n");
		alarm(TIMER_REPORTS);
	}

	if (logfile != NULL) {
		if ((time(NULL) - last_time_logging) > ups->logtime) {
			updating_logs(logfile, ups);
		}
	}

	if (procfile != NULL) {
		if ((time(NULL) - last_time_procfs) > ups->proctime) {
			updating_procfs(procfile, ups);
		}
	}

#if REPORTS_RW
	write_andunlock_shmarea(ups);
#endif /* REPORTS_RW */

#if 0
	alarm_val = alarm(TIMER_REPORTS);
	logprintf("GOT SIGALRM for timer_reports %d.\n", alarm_val);
	fprintf(stderr, "GOT SIGALRM for timer_reports %d.\n", alarm_val);
#else
	alarm_val = alarm(TIMER_REPORTS);
#endif
}

/*********************************************************************/
void do_reports (void) {
	init_thread_signals();
	init_timer(TIMER_REPORTS, timer_reports);
	sleep_forever();
}
#endif /* NEW_THREADS */
