/*
 *  apchttp.c -- simple http routines for presenting UPS status on the web
 *
 *  Copyright (C) 1999 Brian Schau <bsc@fleggaard.dk>
 *
 *  apcupsd.c -- Simple Daemon to catch power failure signals from a
 *               BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *            -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*  Written and maintained by Brian Schau (BSC).
 *
 *  Threaded (Brian, sorry. I promise this is just a temporary solution until
 *  we get the pool of network FDs over which select() in one-net-thread-only).
 *  -RF
 */

/* system wide includes */

#include <ctype.h>
#include <errno.h>
#include <netinet/in.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

/* apc includes */
#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#if defined(NEW_THREADS) && defined(NEW_HTTP)

typedef struct {
	char *data;
	int maxsize;
	int size;
} BUFFER;

static int httpinitbd (BUFFER *, int);
static void httpparse (char *, int);
static void httpbuf (BUFFER *, char *);
static void httpheader (BUFFER*, int, int, int);
static void httphtmlin (BUFFER *);
static void httphtmlout (BUFFER *);
static void httpemit (BUFFER*, int);
static void httpstatus (int, int);
static char *httpgetasciiconfig (void);
static void httpgethtmlconfig (char *);
static void httpconfig (int, int);

extern int errno;

#define HTTPERROR 0
#define HTTPOK 1
#define EVER ;;

static char host[MAXSTRING+2];
static char dontcalltwice=0;

/*
 * Global buffers. Is used by all http responses. Will be filled with
 * headers, page data etc. etc.
 */
#define HTTPBUFFER 4096
#define HTTPHEADER 1024
#define HTTPCNFHTML 2048
#define HTTPCNFTEXT 1024

static BUFFER buffer;
static BUFFER header;
static BUFFER cnfhtml;
static BUFFER cnftext;

/******************************************************************************/
void httpcleanup (void)
{
	if (buffer.data) {
		free(buffer.data);
		buffer.data=(char *)NULL;
	}

	if (header.data) {
		free(header.data);
		header.data=(char *)NULL;
	}

	if (cnfhtml.data) {
		free(cnfhtml.data);
		cnfhtml.data=(char *)NULL;
	}

	if (cnftext.data) {
		free(cnftext.data);
		cnftext.data=(char *)NULL;
	}
}

/******************************************************************************/
int httpinitbd (BUFFER *dst, int size)
{
	if (!(dst->data)) {
		if (!((dst->data)=(char *)calloc(size, 1))) {
			logmess("couldn't allocate memory!");
			return 1;
		}
		dst->maxsize=size;
	}
	dst->size=0;
	return 0;
}

/******************************************************************************/
void http (void)
{
	int clntfd=-1;
	int srvfd, clntlen, bytes, toread;
	struct sockaddr_in srvadr, clntadr;
	UPSINFO *ups;
	char *ptr;

	if (dontcalltwice) {
		logmess("yuck!  http() called twice!");
		return;
	}

	if ((gethostname(host, MAXSTRING))) strcpy(host, "localhost");

	buffer.data=(char *)NULL;
	header.data=(char *)NULL;
	cnfhtml.data=(char *)NULL;
	cnftext.data=(char *)NULL;
	if (httpinitbd(&buffer, HTTPBUFFER)) return;
	if (httpinitbd(&header, HTTPHEADER)) return;
	if (httpinitbd(&cnfhtml, HTTPCNFHTML)) return;
	if (httpinitbd(&cnftext, HTTPCNFTEXT)) return;

	httpgethtmlconfig(httpgetasciiconfig());

	if ((srvfd=socket(AF_INET, SOCK_STREAM, 0))==-1) {
		logerr("couldn't create network socket.");
		return;
	}

	srvadr.sin_family=AF_INET;
	srvadr.sin_addr.s_addr=htonl(INADDR_ANY);
	
	ups=lock_shmups();
	srvadr.sin_port=htons(ups->httpport);
	unlock_shmups();

	if (bind(srvfd, (struct sockaddr *)&srvadr, sizeof(srvadr))) {
		switch (errno) {
			case EINVAL:
				logerr("bind: port in use.");
				break;
			case EACCES:
				logerr("bind: lacking permissions.");
				break;
			default:
				logerr("bind: couldn't.");
				break;
		}
		close(srvfd);
		return;
	}

	signal(SIGCHLD, SIG_IGN);

	if (listen(srvfd, 5)) {
		logerr("couldn't create listen-queue.");
		close(srvfd);
		return;
	}

	/*
	 * Ok, let the game begin ...
	 *
	 * 1) accept connection - one at a time.  Would it be overkill to fork
	 *    and accept more connections?  This is _not_ a webserver!
	 * 2) read the first line (max MAXSTRING bytes).
	 * 3) httpparse()
	 * 4) loop
	 */

	logmess("web-service started.");
	for(EVER) {
		clntlen=0;
		if ((clntfd=accept(srvfd, (struct sockaddr *)&clntadr, &clntlen))>=0) {
			ptr=buffer.data;
			*ptr='\x00';
			clntlen=MAXSTRING;
			while (clntlen>0) {
				ioctl(clntfd, FIONREAD, &toread);
				if (toread) {
					bytes=read(clntfd, ptr, 1);
					if (*ptr=='\n') break;
					if (!bytes) break;
					if (bytes==-1) {
						logmess("unsatisfired read.");
						break;
					}
					
					ptr++;
					clntlen--;
				} else sleep(1);
			}

			httpparse(buffer.data, clntfd);
			shutdown(clntfd, 2);
			close(clntfd);
			clntfd=-1;
		} else logmess("accept-error");
	}

	if (clntfd!=-1) close(clntfd);
	close(srvfd);
}

/******************************************************************************/
static void httpparse (char *request, int clntfd)
{
	int web=1;

	httpinitbd(&buffer, HTTPBUFFER);
/*
 * Allowed formats:
 *
 * 1) GET {page} HTTP/{version}
 *    Response will be sent as html.
 * 2) {page}
 *    Response will be sent as plain text.
 */
	if (!strncmp(request, "GET ", 4)) request+=4;
	else web=0;

	while (*request && (*request==' ' || *request=='\t')) request++;
/*
 * Allowed pages:
 *
 * 1) /, /index.htm, /index.html
 *    Response is a dump of shmUPS.
 * 2) /config.htm, /config.html
 *    Response is current config.
 *
 * Wishlist:
 *
 * 3) /slaves.htm, /slaves.html
 *    Response contains information about slaves (which, connected, etc.)
 */
	if ((*request=='/' && *(request+1)<' ') ||
	    !(strncmp(request, "/ ", 2)) ||
	    !(strncmp(request, "/index.htm", 10))) {
		httpstatus(clntfd, web);
		return;
	}

	if (!(strncmp(request, "/config.htm", 11))) {
		httpconfig(clntfd, web);
		return;
	}
/*
 * Page not found.
 */
	if (web) {
		httphtmlin(&buffer);
		httpbuf(&buffer, "The requested document was not found on this server!");
		httpbuf(&buffer, "</TD></TR><TR><TD><A HREF=\"/index.htm\">View Status</A>\r\n");
		httpbuf(&buffer, "</TD></TR><TR><TD><A HREF=\"/config.htm\">View Configuration</A>\r\n");
		httphtmlout(&buffer);
		httpheader(&buffer, clntfd, web, HTTPERROR);
	} else httpbuf(&buffer, "The requested document was not found on this server!\r\n");
	httpemit(&buffer, clntfd);
}

/******************************************************************************/
static void httpbuf (BUFFER *dst, char *str)
{
	int strsize=strlen(str);

	if (!strsize || (!(dst->data))) return;

	if (((dst->size)+strsize)>((dst->maxsize)-1)) {
		logmess("buffer would overflow.");
		return;
	}

	dst->size+=sprintf(dst->data+dst->size, "%s", str);
}

/******************************************************************************/
static void httpheader (BUFFER *src, int clntfd, int web, int rescode)
{
	char *httperror="HTTP/1.0 404 Document Not Found\r\n";
	char *httpok="HTTP/1.0 200 OK\r\n";
	time_t now;
	char timebuf[MAXSTRING];

	if (web) {
		httpinitbd(&header, HTTPHEADER);
		time(&now);
		strftime(timebuf, MAXSTRING, "%a, %d %b %Y %H:%M:%S GMT\r\n", gmtime(&now));

		if (!rescode) httpbuf(&header, httperror);
		else httpbuf(&header, httpok);

		httpbuf(&header, "Date: ");
		httpbuf(&header, timebuf);
		httpbuf(&header, "Server: apchttp\r\n");
		httpbuf(&header, "Last-Modified: ");
		httpbuf(&header, timebuf);
/*
 * FIXME: 'Pragma' is soon to be declared obsolete.  Use Cache-Control instead.
 * Bump http/1.0 to http/1.1.
 */
		httpbuf(&header, "Pragma: no-cache\r\n");
		httpbuf(&header, "Accept-Ranges: bytes\r\n");
		httpbuf(&header, "Content-Length: ");
		sprintf(timebuf, "%i\r\n", src->size);
		httpbuf(&header, timebuf);
		httpbuf(&header, "Connection: close\r\n");
		httpbuf(&header, "Content-Type: text/html\r\n\r\n");
	}
}

/******************************************************************************/
static void httphtmlin (BUFFER *dst)
{
	httpbuf(dst, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\" \"http://www.w3c.org/TR/REC-html40/loose.dtd\">\r\n");
	httpbuf(dst, "<HTML><HEAD><TITLE>apcupsd on ");
	httpbuf(dst, host);
	httpbuf(dst, "</TITLE></HEAD><BODY BGCOLOR=\"#FFFFFF\">\r\n");
	httpbuf(dst, "<CENTER><H1>apcupsd on ");
	httpbuf(dst, host);
	httpbuf(dst, "</H1><HR WIDTH=\"70%%\"><TABLE BORDER=\"0\"><TR><TD>\r\n");
}

/******************************************************************************/
static void httphtmlout (BUFFER *dst)
{
	httpbuf(dst, "</TD></TR></TABLE><HR WIDTH=\"70%%\"></CENTER></BODY></HTML>\r\n");
}

/******************************************************************************/
static void httpemit (BUFFER *src, int clntfd)
{
	char *ptr=src->data;
	int cnt=src->size;
	int bytes;

	while (cnt>0) {

		bytes=write(clntfd, ptr, cnt);

		sleep(1);

		switch(bytes) {
		    case -1:
			logmess("unsatisfied write.");
		    case 0:
			logmess("zero write.");
			return;
		    default:
			ptr+=bytes;
			cnt-=bytes;
			break;
		}
	}
}

/******************************************************************************/
static void httpstatus (int clntfd, int web)
{
	if (web) {
		httphtmlin(&buffer);
		httpbuf(&buffer, "<PRE>\r\n");
		read_shmarea(&myUPS);
		get_upsstat(buffer.data+buffer.size, &myUPS);
		buffer.size=strlen(buffer.data);
		httpbuf(&buffer, "</PRE></TD></TR><TR><TD><A HREF=\"/config.htm\">View Configuration</A>\r\n");
		httphtmlout(&buffer);
		httpheader(&buffer, clntfd, web, HTTPOK);
		httpemit(&header, clntfd);
	} else {
		httpbuf(&buffer, "apcupsd on ");
		httpbuf(&buffer, host);
		httpbuf(&buffer, "\r\n\r\n");
		read_shmarea(&myUPS);
		get_upsstat(buffer.data, &myUPS);
		buffer.size=strlen(buffer.data);
		httpbuf(&buffer, "\r\n");
	}
	httpemit(&buffer, clntfd);
}

/******************************************************************************/
static char * httpgetasciiconfig (void)
{
	FILE *apcconf;
	char line[MAXTOKENLEN];
	char *ptr, *tptr, *ref;

	httpbuf(&cnftext, "apcupsd on ");
	httpbuf(&cnftext, host);
	httpbuf(&cnftext, "\r\n\r\n");
	
	ref=cnftext.data+cnftext.size;

	if (!(apcconf=fopen(APCCONF, "r"))) {
		httpbuf(&cnftext, "(broken)\r\n");
		return ref;
	}

	if (fgets(line, sizeof(line), apcconf)) {
		if (strncmp(line, APC_CONFIG_MAGIC, sizeof(APC_CONFIG_MAGIC)-1)) {
			httpbuf(&cnftext, "Magic Mismatch. Please check new configuration file format\r\n");
			httpbuf(&cnftext, "and modify accordingly the first line of the configuration file!\r\n\r\n");
		}
	}
	rewind(apcconf);

	/* the parsing below follows the work done in apcconfig.c
	   Thanks Jonathan - I don't have to think about what I do ... I can
	   have an easy ride by copying your work  :-)    -BSC */

	while (fgets(line, sizeof(line), apcconf)) {
		/* skip whitespaces */
		for (ptr=line; *ptr && isspace(*ptr); ptr++);
		/* skip blank lines/lines with comments */
		if (!*ptr || *ptr=='#') continue;
		/* uppercase key */
		for (tptr=ptr; *tptr && !isspace(*tptr); tptr++) *tptr=toupper(*tptr);
		/* I'm on my own now ... */
		for (; *tptr && isspace(*tptr); tptr++) *tptr='\x00';
		httpbuf(&cnftext, ptr);
		
		for(EVER) {
			if (!*tptr) break;
			ptr=tptr;
			for (; *tptr && !isspace(*tptr); tptr++);
			for (; *tptr && isspace(*tptr); tptr++) *tptr='\x00';
			httpbuf(&cnftext, " ");
			httpbuf(&cnftext, ptr);
		}

		httpbuf(&cnftext, "\r\n");
	}
	fclose(apcconf);
	return ref;
}

/******************************************************************************/
static void httpgethtmlconfig (char *ptr)
{
	char chrstr[2];
	int mark;

	httphtmlin(&cnfhtml);
	httpbuf(&cnfhtml, "<TABLE><TR><TD>");
	
	chrstr[1]='\x00';
	mark=1;
	for(EVER) {
		if (!*ptr) break;

		switch (*ptr) {
			case ' ':
				if (mark) {
					httpbuf(&cnfhtml, "</TD><TD>&nbsp; &nbsp;</TD><TD>");
					mark=0;
					*chrstr='\x00';
				} else *chrstr=' ';
				break;
			case '<':
				httpbuf(&cnfhtml, "&lt;");
				break;
			case '>':
				httpbuf(&cnfhtml, "&gt;");
				break;
			case '&':
				httpbuf(&cnfhtml, "&amp;");
				break;
			case '\n':
				httpbuf(&cnfhtml, "</TD></TR><TR><TD>");
				*chrstr='\x00';
				mark=1;
				break;
			default:
				*chrstr=*ptr;
				break;
		}

		if (*chrstr) httpbuf(&cnfhtml, chrstr);
		ptr++;
		
	}

	httpbuf(&cnfhtml, "</TD></TR></TABLE>");
	httpbuf(&cnfhtml, "</TD></TR><TR><TD>&nbsp;</TD></TR><TR><TD><A HREF=\"/index.htm\">View Status</A>\r\n");
	httphtmlout(&cnfhtml);
}

/******************************************************************************/
static void httpconfig (int clntfd, int web)
{
	if (web) {
		httpheader(&cnfhtml, clntfd, web, HTTPOK);
		httpemit(&header, clntfd);
		httpemit(&cnfhtml, clntfd);
	} else httpemit(&cnftext, clntfd);
}

/******************************************************************************/
void do_http (void) {
	init_thread_signals();
	/*
	 * No temporization is needed for http thread so no init_timer is
	 * needed.
	 */
	http();
}
#endif /* defined(NEW_THREADS) && defined(NEW_HTTP) */
