/*

Name:
DRV_OS2.C

Description:
Mikmod driver for output on OS/2 with MMPM/2 and DART installed

    Copyright (c) Tom Stokes, 1995

Portability:
IBM C/Set++ with MMPM/2 libs, headers

*/

#define INCL_DOSMEMMGR
#define INCL_DOSSEMAPHORES
#define INCL_DOSERRORS
#define INCL_MCIOS2
#define INCL_MMIOOS2

#include <os2.h>
#include <os2me.h>
#include <mmio.h>

#include <stdio.h>
#include "mikmod.h"

#define BUFFLEN                 8*1024
#define NUM_PLAY_BUFFS          8

#define PRODUCT_SIZE          128

typedef enum KNOB_ {VOLUME, BALANCE, BASS, TREBLE} KNOB;

extern DRIVER drv_os2;

MCI_WAVE_GETDEVCAPS_PARMS     AudioCaps;
MCI_AMP_OPEN_PARMS            mciAmpOpenParms;   // the amp_mixer device
MCI_OPEN_PARMS                mciOpenParms;      // the wave device
MCI_WAVE_SET_PARMS            WaveForm;
MCI_BUFFER_PARMS              BufferParms;
MCI_GENERIC_PARMS             genParms;
MCI_STATUS_PARMS              statParms;
MCI_INFO_PARMS                infoParms;
MCI_MIX_BUFFER                mciBuffers[NUM_PLAY_BUFFS];
MCI_MIXSETUP_PARMS            MixSetupParms;


USHORT                        usWaveDeviceID,
                              usAmpMixDeviceID;

ULONG                         ulNumBuffers,
                              ulBufferSize;

BOOL                          fTerminate,
                              fSaidOnce;
ULONG                         lCnt;

HMTX                          hmtxBufferSem;
HEV                           hevDirectDone;

CHAR                          Product[PRODUCT_SIZE];

LONG APIENTRY EventProc(         /* the callback routine */
                ULONG           ulStatus,
                PMCI_MIX_BUFFER pBuff,
                ULONG           ulFlags) {

       ULONG   SemRslt;
       LONG    lBufferNum;
       LONG    lBytes;

               mciSendCommand( usAmpMixDeviceID,
                  MCI_MIXSETUP,
                  MCI_WAIT | MCI_MIXSETUP_INIT,
                  ( PVOID ) &MixSetupParms,
                  0 );

       switch(ulFlags) {
          case MIX_STREAM_ERROR:
               if (( ulStatus == ERROR_DEVICE_UNDERRUN) ||
                  ( ulStatus == ERROR_DEVICE_OVERRUN))  {
                  /*  I think the stream can be restarted somehow.
                      However, this condition usually occurs because
                      of a heavily loaded system. In this case, you
                      probably don't want automatic restart. Right?
                      So just die. */
               }
               myerr = "Cannot stream audio data";
               DosPostEventSem(hevDirectDone);
               break;

          case MIX_WRITE_COMPLETE:

               if (pBuff->ulFlags & MIX_BUFFER_EOS) {
                  lBufferNum = pBuff->ulUserParm;  // if needed
                  if (!fSaidOnce) {
                     printf("** end of stream **\n");
                     fSaidOnce++;
                  }
                  DosPostEventSem(hevDirectDone);
                  break;
               }
               SemRslt = DosRequestMutexSem(hmtxBufferSem, SEM_INDEFINITE_WAIT);
               if (pBuff->pBuffer != NULL) {
                  lBytes = VC_WriteBytes(pBuff->pBuffer, ulBufferSize);
                  if ((lBytes < ulBufferSize) || (fTerminate)) {
                      pBuff->ulFlags = MIX_BUFFER_EOS;
                      pBuff->ulBufferLength = lBytes;
                  }
                  MixSetupParms.pmixWrite( MixSetupParms.ulMixHandle,
                                           pBuff,
                                           1 );
               }
               SemRslt = DosReleaseMutexSem(hmtxBufferSem);
               break;
          default:
               break;
    } // switch

    return;
}

VOID AcquireDevice(USHORT usDeviceID)
{
    mciSendCommand(
               usDeviceID,
               MCI_ACQUIREDEVICE,
               MCI_WAIT,
               (PVOID) NULL,
               (USHORT)NULL);
}

VOID Set_Amp_Knob(KNOB eKnob, ULONG ulLevel)
{
      MCI_AMP_SET_PARMS     mciAmpParms;
      ULONG                 mciRC,
                            Set_Which;

      switch (eKnob) {
          case VOLUME :
             Set_Which = MCI_AMP_SET_VOLUME;
             break;
          case BALANCE :
             Set_Which = MCI_AMP_SET_BALANCE;
             break;
          case BASS :
             Set_Which = MCI_AMP_SET_BASS;
             break;
          case TREBLE :
             Set_Which = MCI_AMP_SET_TREBLE;
             break;
          default :
             Set_Which = 0;
             break;
      }

      AcquireDevice(usAmpMixDeviceID);
      mciAmpParms.hwndCallback = NULLHANDLE;
      mciAmpParms.ulAudio      = MCI_SET_AUDIO_ALL;    // set all channels
      mciAmpParms.ulLevel      = ulLevel;        // percentage, 0-100

      mciRC = mciSendCommand(
                 usAmpMixDeviceID,
                 MCI_SET,
                 MCI_WAIT | MCI_SET_AUDIO | Set_Which,
                 (PVOID) &mciAmpParms,
                  0);
      return;
}

VOID CloseDevice(USHORT usDeviceID)
{
        if (usDeviceID > 0)  {
           mciSendCommand(
                usDeviceID,
                MCI_CLOSE,
                MCI_WAIT,
                (ULONG) NULL,
                (USHORT) NULL);
        }
}

BOOL Init_Dart(VOID)
{
       ULONG   mciRC;

       AcquireDevice(usAmpMixDeviceID);
       memset( &genParms, '\0', sizeof(MCI_GENERIC_PARMS));
       memset( &MixSetupParms, '\0', sizeof(MCI_MIXSETUP_PARMS));
       MixSetupParms.ulBitsPerSample = AudioCaps.ulBitsPerSample;
       MixSetupParms.ulFormatTag = DATATYPE_WAVEFORM;
       MixSetupParms.ulSamplesPerSec = AudioCaps.ulSamplesPerSec;
       MixSetupParms.ulChannels = AudioCaps.ulChannels;
       MixSetupParms.ulFormatMode = MCI_PLAY;
       MixSetupParms.ulDeviceType = MCI_DEVTYPE_WAVEFORM_AUDIO;
       MixSetupParms.hwndCallback = NULLHANDLE;

       /*-- these 3 "who ya gonna call" will be filled in -----
       MixSetupParms.ulMixHandle
       MixSetupParms.pmixWrite
       MixSetupParms.pmixRead
       ------------------------------------*/

       MixSetupParms.pmixEvent = EventProc;   // the callback routine

       mciRC = mciSendCommand( usAmpMixDeviceID,
                  MCI_MIXSETUP,
                  MCI_WAIT | MCI_MIXSETUP_INIT,
                  ( PVOID ) &MixSetupParms,
                  0 );

       if (ULONG_LOWD(mciRC) != MCIERR_SUCCESS) {
            myerr = "DART not installed (mixer setup failed).";
            return FALSE;
       }

       /*--- the MCI_MIX_SETUP call fills in these as "suggested vals"
       MixSetupParms.ulBufferSize
       MixSetupParms.ulNumBuffers
       ---------------------------------------------------------------*/

       memset( &BufferParms, '\0', sizeof(MCI_BUFFER_PARMS));
       ulNumBuffers = NUM_PLAY_BUFFS;
       ulBufferSize = BUFFLEN;
       BufferParms.ulNumBuffers = ulNumBuffers;
       BufferParms.ulBufferSize = ulBufferSize;
       BufferParms.pBufList = mciBuffers;
       BufferParms.hwndCallback = NULLHANDLE;
       BufferParms.ulStructLength = sizeof(BufferParms);

       mciRC = mciSendCommand(
                     usAmpMixDeviceID,
                     MCI_BUFFER,
                     MCI_WAIT | MCI_ALLOCATE_MEMORY,
                     (PVOID) &BufferParms,
                     0 );

       if (ULONG_LOWD( mciRC) != MCIERR_SUCCESS) {
             myerr = "Allocate memory errror in DART";
             return FALSE;
       }

       return TRUE;
}

BOOL OS2_IsThere(void)
{
       /* We need to have the hardware, MMPM/2, and DART installed.
          If we don't, the device open (hardware and mmpm2) or the
          mixer setup (dart) will fail in short order.
          So - */

       return TRUE;
}


BOOL OS2_Init(void)
{
       ULONG    ulOpenFlags = MCI_WAIT           |
                              MCI_OPEN_TYPE_ID   |
                              MCI_OPEN_SHAREABLE;

       ULONG    mciRC,
                SemRslt;

       mciOpenParms.pszDeviceType = (PSZ)MAKEULONG(MCI_DEVTYPE_WAVEFORM_AUDIO,1);
       mciOpenParms.pszElementName = (PSZ)NULL;
       mciOpenParms.hwndCallback = NULLHANDLE;
       mciOpenParms.pszAlias = (CHAR)NULL;

       mciRC = mciSendCommand(
                  0,                    // device id (not yet known)
                  MCI_OPEN,             // message
                  ulOpenFlags,          // flags
                  (PVOID)&mciOpenParms,
                  0);                   // param for notify msg

       if (ULONG_LOWD(mciRC) != MCIERR_SUCCESS) {
           myerr="Can't open wave audio device";
           usWaveDeviceID = 0;
           return FALSE;
       }

       usWaveDeviceID = mciOpenParms.usDeviceID;

       mciAmpOpenParms.pszDeviceType = (PSZ)MAKEULONG(MCI_DEVTYPE_AUDIO_AMPMIX,
1);
       mciAmpOpenParms.hwndCallback = NULLHANDLE;
       mciAmpOpenParms.pszAlias = (CHAR)NULL;

       mciRC = mciSendCommand(
                  0,                    // device id (not yet known)
                  MCI_OPEN,             // message
                  ulOpenFlags,          // flags
                  (PVOID)&mciAmpOpenParms,
                  0);                   // param for notify msg

       if (ULONG_LOWD(mciRC) != MCIERR_SUCCESS) {
           myerr="Can't open amp/mixer device";
           usAmpMixDeviceID = 0;
           CloseDevice(usWaveDeviceID);
           return FALSE;
       }
       usAmpMixDeviceID = mciAmpOpenParms.usDeviceID;

       memset(&AudioCaps, 0, sizeof(AudioCaps));
       AudioCaps.ulSamplesPerSec = md_mixfreq;
       AudioCaps.ulBitsPerSample = (md_mode & DMODE_16BITS) ? 16 : 8;
       AudioCaps.ulChannels = (md_mode & DMODE_STEREO) ? 2 : 1;
       AudioCaps.ulFormatMode = MCI_PLAY;
       AudioCaps.ulFormatTag = DATATYPE_WAVEFORM;
       AudioCaps.ulItem = MCI_GETDEVCAPS_WAVE_FORMAT;
       AcquireDevice(usWaveDeviceID);
       mciRC = mciSendCommand(
                 usWaveDeviceID,
                 MCI_GETDEVCAPS,
                 MCI_WAIT | MCI_GETDEVCAPS_EXTENDED | MCI_GETDEVCAPS_ITEM,
                 (PVOID)&AudioCaps,
                 0);
       /*
             Unfortunately, OS/2 does not provide a good way of
             determining device capabilities. For a player, the 3
             important variables are samps/sec, stereo/mono, and
             sample size; but it is really a guessing game. The
             following code gives an idea of how/what to try,
             but it's probably better to just rely on command line
             args. In the future, it might be a good idea to provide
             a "md_mode_bit_mask" to indicate what params the user
             specified.
       */
//#if 0
       if (ULONG_LOWD(mciRC) != MCIERR_SUCCESS) {
           if (md_mode & DMODE_16BITS) {
               md_mode &= ~DMODE_16BITS;
               AudioCaps.ulBitsPerSample = 8;
               if (md_mode & DMODE_STEREO) {
                  md_mixfreq >>= 1;
                  AudioCaps.ulSamplesPerSec = md_mixfreq;
               }
               mciRC = mciSendCommand(
                         usWaveDeviceID,
                         MCI_GETDEVCAPS,
                         MCI_WAIT | MCI_GETDEVCAPS_EXTENDED |
MCI_GETDEVCAPS_ITEM,
                         (PVOID)&AudioCaps,
                         0);
               if (ULONG_LOWD(mciRC) != MCIERR_SUCCESS) {
                   md_mode &= ~DMODE_STEREO;
                   AudioCaps.ulChannels = 1;
                   mciRC = mciSendCommand(
                             usWaveDeviceID,
                             MCI_GETDEVCAPS,
                             MCI_WAIT | MCI_GETDEVCAPS_EXTENDED |
MCI_GETDEVCAPS_ITEM,
                             (PVOID)&AudioCaps,
                             0);
               }
           }
       }
//#endif
       if (ULONG_LOWD(mciRC) != MCIERR_SUCCESS) {
           myerr="Given settings are beyond capability of audio device";
           CloseDevice(usWaveDeviceID);
           CloseDevice(usAmpMixDeviceID);
           return FALSE;
       }

       if (!Init_Dart()) {
            CloseDevice(usWaveDeviceID);
            CloseDevice(usAmpMixDeviceID);
            return FALSE;
       }

       // set amp/mix default levels (percent)
       Set_Amp_Knob(BASS, 50);
       Set_Amp_Knob(TREBLE, 50);
       Set_Amp_Knob(BALANCE, 50);
       Set_Amp_Knob(VOLUME, 100);

       // get product info into a string
       infoParms.pszReturn = (PSZ)&Product;
       infoParms.ulRetSize = PRODUCT_SIZE;
       mciRC = mciSendCommand(
                 usWaveDeviceID,
                 MCI_INFO,
                 MCI_WAIT | MCI_INFO_PRODUCT,
                 (PVOID)&infoParms,
                 0);

       if (ULONG_LOWD(mciRC) == MCIERR_SUCCESS)
           drv_os2.Name = infoParms.pszReturn;

       if(!VC_Init()){
           CloseDevice(usWaveDeviceID);
           CloseDevice(usAmpMixDeviceID);
           return FALSE;
       }

       fTerminate = FALSE;    // event thread killer
       SemRslt = DosCreateEventSem(NULL, &hevDirectDone, 0, 0);
       SemRslt = DosCreateMutexSem(NULL, &hmtxBufferSem, 0, FALSE);
       return TRUE;
}


void OS2_Exit(void)
{
       ULONG mciRC, SemRslt, i;
       fTerminate = TRUE;
       mciRC = mciSendCommand( usAmpMixDeviceID,
                       MCI_STOP,
                       MCI_WAIT,
                       (PVOID) &genParms,
                       0 );

       /* tell the mixer that we are done with the buffers
          NOTE: access to the buffers is serialized
          with hmtxBufferSem. */

       SemRslt = DosRequestMutexSem(hmtxBufferSem, SEM_INDEFINITE_WAIT);

       mciRC = mciSendCommand(
                       usAmpMixDeviceID,
                       MCI_BUFFER,
                       MCI_WAIT | MCI_DEALLOCATE_MEMORY,
                       (PVOID) &BufferParms,
                       0);

       for (i=0; i<ulNumBuffers; i++)
          mciBuffers[i].pBuffer = NULL;

       SemRslt = DosReleaseMutexSem(hmtxBufferSem);

       mciRC = mciSendCommand(
                       usAmpMixDeviceID,
                       MCI_CLOSE,
                       MCI_WAIT ,
                       (PVOID) &genParms,
                       0 );
        CloseDevice(usWaveDeviceID);
        CloseDevice(usAmpMixDeviceID);
        DosCloseEventSem(hevDirectDone);
        DosCloseMutexSem(hmtxBufferSem);
        VC_Exit();
}


ULONG GetPos(void)
{
        memset( &statParms, '\0', sizeof( MCI_STATUS_PARMS ) );
        statParms.ulItem = MCI_STATUS_POSITION;   // always milliseconds

        mciSendCommand(
                  usAmpMixDeviceID,
                  MCI_STATUS,
                  MCI_WAIT | MCI_STATUS_ITEM,
                  (PVOID) &statParms,
                  0 );
        return 3*statParms.ulValue;   // *3 for MMTIME
}



void OS2_PlayStart(void)
{
       ULONG i, SemRslt, mciRC;
       fTerminate = FALSE;
       fSaidOnce = FALSE;
       mciRC = mciSendCommand(
                     usAmpMixDeviceID,
                     MCI_STOP,
                     MCI_WAIT,
                     (PVOID) &genParms,
                     0 );
       VC_PlayStart();
       for (i=0; i<ulNumBuffers; i++) {
          mciBuffers[i].ulUserParm = i;
          mciBuffers[i].ulFlags = 0;
          VC_WriteBytes(mciBuffers[i].pBuffer, ulBufferSize);
       }

       /* Note: can write multiple buffers at once */
       MixSetupParms.pmixWrite( MixSetupParms.ulMixHandle,
                                mciBuffers,
                                ulNumBuffers);
       mciRC = mciSendCommand(
                     usAmpMixDeviceID,
                     MCI_PLAY,
                     MCI_WAIT,
                     (PVOID) &genParms,
                     0 );
}


void OS2_PlayStop(void)
{
   ULONG SemRslt;
   if (usAmpMixDeviceID > 0)  {
         DosResetEventSem (hevDirectDone, &lCnt);
         fTerminate = TRUE;
         SemRslt = DosWaitEventSem (hevDirectDone, 3000);  // 3 seconds
         if (SemRslt == ERROR_TIMEOUT)
            printf(" **** event semaphore timeout ****\n");
         mciSendCommand(
                  usAmpMixDeviceID,
                  MCI_STOP,
                  MCI_WAIT,
                  (PVOID) &genParms,
                  0 );
   }
   VC_PlayStop();
}


void OS2_Update(void)
{
     /* does nothing, buffers are updated in the background */
}


DRIVER drv_os2={
       NULL,
       "OS/2",
       "MikMod OS/2 Driver v1.0 - Warp + DART",
       OS2_IsThere,
       VC_SampleLoad,
       VC_SampleUnload,
       OS2_Init,
       OS2_Exit,
       OS2_PlayStart,
       OS2_PlayStop,
       OS2_Update,
       VC_VoiceSetVolume,
       VC_VoiceSetFrequency,
       VC_VoiceSetPanning,
       VC_VoicePlay
};
