/*
 *
 *   qrash: the second portable demo in the world
 *
 *   Copyright (C) 1997  Queue Members Group Art Division
 *   Coded by Mad Max / Queue Members Group (Mike Shirobokov)
 *   <mad_max@qmg.rising.ru>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 */
#ifndef __3D__
#define __3D__

#include <stdio.h>
#include <math.h>
#include "video.h"
#include "misc.h"
#include "image.h"

//#define PURE_C

#define MAX_Z	32767
#define MIN_Z	(-MAX_Z)

#define COS_STEPS_3D 4096
#define ACOS_STEPS   256
#define PHONG_STEPS  256
#define TEXTURE_SIZE 256

extern "C" {
  extern uchar phongTable[PHONG_STEPS][PHONG_STEPS];
}

struct Vector {
  bool processed;
  int x,y,z;
  Vector( int _x=0, int _y=0, int _z=0 ) {
    x=_x; y=_y; z=_z; processed = false;
  }
  void Normalize( int value ) {
    float n = sqrt(sqr((float)x) + sqr((float)y) + sqr((float)z))
	       / (value-1);
    if(n>0) {
      x /= n; y /= n; z /= n;
    }
  }
  void Move( int dx, int dy, int dz )
  {
    x += dx; y+=dy; z+=dz;
  }
  void Rotate( int cx, int cy, int cz, int rx, int ry, int rz );
};

struct Point: public Vector {
  bool clipped;
  Vector normal;
  shorts l, t;
  Point(): Vector(), normal() {}
  Point( int _x, int _y, int _z, shorts _t ) :
  Vector( _x, _y, _z ), normal(), t(_t) {}
};

struct Face {
  Point* vert[3];
  Vector normal;
  uchar* texture;
  int du, dv, size_u, size_v;
  Face( Point* p1, Point* p2, Point* p3, uchar* texture=0 );

  Face( Point* p1, Point* p2, Point* p3, uchar* _texture, Vector _normal,
	int _du, int _dv )
  {
    vert[0]=p1; vert[1]=p2; vert[2]=p3;
    normal = _normal;
    texture = _texture;
    du = _du; dv = _dv;
  }
  void Draw( int flags );
};

struct Object3D {
  int dx, dy, dz, rx, ry, rz, lx, ly, lz;
  float lv, lo;
  float sx, sy, sz;
  void RotateTo( int _rx, int _ry, int _rz ) {
    rx = _rx; ry = _ry; rz = _rz;
  }
  void SetLightTo( int _lx, int _ly, int _lz, float _lv, float _lo ) {
    lx = _lx; ly = _ly; lz = _lz; lv = _lv; lo=_lo;
  }
  void MoveTo( int _dx, int _dy, int _dz ) {
    dx = _dx; dy = _dy; dz = _dz;
  }
  void ScaleTo( float _sx, float _sy, float _sz ) {
    sx = _sx; sy = _sy; sz = _sz;
  }
  Object3D( int _dx=0,int _dy=0,int _dz=0,
	    int _rx=0,int _ry=0,int _rz=0,
	    int _lx=0,int _ly=0,int _lz=0, float _lv=1, float _lo=0,
	    float _sx=1,float _sy=1,float _sz=1 ) {
    dx=_dx,dy=_dy,dz=_dz,rx=_rx,ry=_ry,rz=_rz,
    lx=_lx,ly=_ly,lz=_lz,lv=_lv,lo=_lo,sx=_sx,sy=_sy,sz=_sz;
  };
  static void ClearZBuffer( short* zbuffer, int value = MIN_Z );
};

#define MAP_XY	     (1<<0)
#define MAP_YZ	     (1<<1)
#define MAP_XZ	     (1<<2)
#define MAP_SPHERE   (1<<3)
#define MAP_NORMAL   (1<<4)
#define MAP_ASC      (1<<5)

#define MAP_SCALE    0
#define MAP_NOSCALE  (1<<6)

#define MAP_TILE     (1<<7)

#define TEXTURE
#define NO_TEXTURE   (1<<0)

#define SHADOW	     (1<<1)
#define NO_SHADOW    0

#define PERSPECT     (1<<2)
#define NO_PERSPECT  0

#define ONE_SIDE     0
#define TWO_SIDES    (1<<3)

#define ZBUFFER      0
#define SORT	     (1<<4)

#define GOURAUD      0
#define PHONG	     (1<<5)

typedef bool FaceCallback( Face* );
typedef bool PointCallback( Point* );

struct FaceCollection: public TCollection<Face> {
  void Insert( Face* f );
};

struct FacedObject: public Object3D {

  int maxx,maxy,maxz,minx,miny,minz;
  TCollection<Point> points, tmp_points;
  FaceCollection faces;
  uint flags;
  float perspect;

  FacedObject( uint flags_ ) :
  Object3D(), points(), faces(), tmp_points(), flags(flags_),
  perspect(1), faceCallback(0), pointCallback(0) {}

  FacedObject( FacedObject& );
  void Load( char* filename, Image* texture );
  void Store( char* filename );
  void ImportASC( char* filename, char* name, float Scale,
		  Image* texture, int mapping );
  void Prepare();
  void Draw( short* zbuffer, PAGE color, PAGE bw );
  void Morph( FacedObject&, FacedObject&, int step, int steps );
  void MoveAbs( int dx, int dy, int dz );
  void RotateAbs( int rx, int ry, int rz );

  void setFaceCallback( FaceCallback* callback ) { faceCallback = callback; }
  void setPointCallback( PointCallback* callback ) { pointCallback = callback; }
  void Attach( FacedObject* );

  Image* texture_image;
  FaceCallback *faceCallback;
  PointCallback *pointCallback;

  void setDrawLineZGTS( PAGE color, PAGE bw, short* zbuffer,
		       int shadow_offset );
  void setDrawLineZGT( PAGE color, PAGE bw, short* zbuffer );
  void setDrawLineZPT( PAGE color, PAGE bw, short* zbuffer );
  void setDrawLineZG( PAGE bw, short* zbuffer );
  void setDrawLineZP( PAGE bw, short* zbuffer );
  void setDrawLineSPT( PAGE color, PAGE bw );
  void setDrawLineSGT( PAGE color, PAGE bw );
  void setDrawLineSP( PAGE bw );
  void SplitFace( Face* f, bool vertical );
  void SplitFaces();
  void SplitTexture();
};

void Init3D();

struct Cluster: public TCollection<Face> {
  int du, dv, size_u, size_v;
  void UpdateBlock();
  void Insert( Face* f );
  Cluster* Split();
};

extern TCollection<Cluster> clusters;  // to instantiate it in misc.cc

#endif
