;zblast SD - port/rewrite of a simple shoot-em-up.
;public domain by Russell Marks, 2003
;
;based on my C zblast for Linux, originally written in 1993 (eek).
;
;CPC conversion by Nicholas Campbell
;Started 21st February 2005; finished 26th April 2005

org &8000

;The tables containing the addresses of the sprites for each type of baddie and
;the ink data for writing text characters; these are stored at other locations
;initially, but they must be page-aligned
sprite_addr_table_hb equ &3b
char_ink_table_hb equ &3c

;The mask table, used for drawing sprites transparently; this is 256 bytes long
;and must be page-aligned 
mask_table_hb equ &3d


;size of the grid used internally for positioning
;(currently twice attribute res)
num_grid_x	equ 48
num_grid_y	equ 48
num_grid_y_big	equ 32	;effective y max for `big' enemies


;each entry in baddies array is page-aligned 128 bytes:
;offset	size	desc
;0	1	type of baddie (0=inactive)
;1	1	x pos (in grid terms, higher-res than screen pos)
;2	1	y pos (ditto)
;3	1	dx (signed)
;4	1	dy (signed)
;5	1	hits left till death (unused for type 2 = enemy shot)
;6  2   screen address on screen at &4000
;8  2   screen address on screen at &c000
;10	120	unused

baddie_x	equ 1
baddie_y	equ 2
baddie_dx	equ 3
baddie_dy	equ 4
baddie_hits	equ 5
baddie_screen_addr_40 equ 6
baddie_screen_addr_c0 equ 8

;each entry in (player) shots array is page-aligned (+128) 64 bytes:
;offset	size	desc
;128	1	active (non-zero if true)
;129	1	x (in grid terms)
;130	1	y (ditto)
;131	2   screen address on screen at &4000
;133	2   screen address on screen at &c000
;134	57	unused

shot_active	equ 128
shot_x		equ 129
shot_y		equ 130
shot_screen_addr_40 equ 131
shot_screen_addr_c0 equ 133

;each entry in debris array is page-aligned (+192) 64 bytes:
;offset	size	desc
;0	1	frames left before it disappears (or zero if inactive)
;1	1	x pos (in grid terms)
;2	1	y pos (ditto)
;3	1	dx (signed)
;4	1	dy (signed)
;5	1	'hits' (useless, but the addnewbaddie routine is used when adding
;		debris, and the byte following dy is corrupted - which would otherwise
;       corrupt the screen address below)
;6	2   screen address on screen at &4000
;8	2   screen address on screen at &c000
;10	54	unused

debris_timeout	equ 192
debris_x	equ 193
debris_y	equ 194
debris_dx	equ 195
debris_dy	equ 196
debris_screen_addr_40 equ 198
debris_screen_addr_c0 equ 200


max_baddies	equ 40		;as used by proper zblast
max_shots	equ max_baddies	;less than proper zblast, but should be fine
max_debris	equ max_baddies
baddies_hb  equ &02
baddies     equ baddies_hb*256   ;The baddies array must be page-aligned
shots		equ baddies+128	;interleaved with above
debris		equ baddies+192	;interleaved with above


;the stars array is just a series of y/dy pairs (in pixels)
stars		equ &3e00
num_stars	equ 24

;The CPC version also requires an array to store the screen addresses of the
;stars for both of the screens (&4000 and &c000), so that they can be cleared
;and the stars printed in their new positions
stars_old   equ &3f00
num_stars_old equ num_stars*4

tmpsprite	equ &a500	;temp. sprite used for drawing text
nbspace		equ 46		;char used as printable space

shipxy	equ &a600
shipy	equ shipxy
shipx	equ shipy+1
ship_screen_addr_40 equ shipx+1
ship_screen_addr_c0 equ ship_screen_addr_40+2
shield	equ ship_screen_addr_c0+2
wasfire	equ shield+1
wavenum	equ wasfire+1
dead	equ wavenum+1
gotbaddies equ dead+1
cyclecol equ gotbaddies+1
tweenwave equ cyclecol+1
score	equ tweenwave+1		;2 bytes
lives	equ score+2
lostlife equ lives+1
ship_collision equ lostlife+1
musicvars equ ship_collision+1	;the rest used by music.asm

;The CPC version of the game is a lot easier than the Spectrum version, because
;of the slow speed compared to the Spectrum version. To increase the difficulty
;level, the length of time that the shield lasts for after being hit has been
;decreased (it is 25 on the Spectrum)
shield_life	equ 10

delay_auto_fire	equ 3

start_lives	equ 30


queuesnd_plyr equ queuesnd_sfx
queuesnd_enemy equ queuesnd_sfx


start:
di
ld sp,&c000
ld hl,&0802
ld (tmpsprite),hl   ;Set the size of an ASCII character, which is stored in the
                    ;temporary sprite area

call initstuff

;reset AY (and sound queues)
ld de,0
ld b,14
ayresetlp:
call aywrite
inc d
djnz ayresetlp

ld a,255-snd_music_tone_mask-snd_sfx_tone_mask
call mixerwrite

ld a,1   ;On a Spectrum, this is set to bright ink 1 (65)
ld (cyclecol),a

;Clear both screens

mainloop:
call fastclear
call doflip
call fastclear

;start title music
ld hl,lil_blocklist
ld bc,lil_blocks-64
xor a
call startmusic

mainloop2:
call doflip

call drawstars

ld hl,cyclecol
ld a,(hl)
inc a
cp 8
jr c,mlskip
ld a,1
mlskip:
ld (hl),a
ld bc,&0806
call ilprint
defb 'z b l a s t  S ','D'+128

ld bc,&0e1c
ld a,4
call ilprint
defb 'high',':'+128

ld de,(high)
ld bc,&181c
;don't actually need all that this does, it's just the smallest way
ld hl,&2020
call drawnumtmp2_noclear

ld bc,&0c24
ld a,3
call ilprint
defb '1 - keyboar','d'+128

ld bc,&0c28
ld a,3
call ilprint
defb '2 - joystic','k'+128

call read_keyboard
ld hl,joystick
ld (hl),0   ;Set the default control method to be the keyboard
ld a,(keyboard_info+8)   ;Get information about the keys pressed on matrix line
                         ;8 (which includes the 1 and 2 keys)
bit 0,a   ;The state of the 1 key is stored in bit 0
jr z,playgame   ;If it is reset, then the 1 key has been pressed

ld (hl),a   ;A is non-zero, and the user has not selected keyboard control, so
            ;set the control method to be the joystick now
bit 1,a   ;The state of the 2 key is stored in bit 1
jr nz,mainloop2   ;If it is set, then the 2 key has not been pressed, so stay on
                  ;the menu screen

playgame:
;ok, start up a game.

;clear the whole baddies/shots array.
;this gives a brief but noticeable pause, but OTOH we do want a short
;pause to push the two tunes apart slightly. :-)
call musicoff
halt
halt
halt
halt
halt
halt

ld hl,baddies
ld de,baddies+1
ld bc,max_baddies*256-1
ld (hl),l	;zero, as we know it's page-aligned
ldir

;Set up screen addresses for both screens for the player's ship

ld hl,ship_screen_addr_40-1
call addnewsomething_set_screen_addrs

;a zero noise period is equivalent to one so I don't actually need
;this. But some emulators don't get that right, and it sounds bad on
;those without this. :-( The same is true of the sawtooth envelope,
;though some emulators will do badly on that one either way...
;Is this necessary on the CPC?
;ld de,&0601
;call aywrite
;ld d,11
;call aywrite

ld de,&0d08	;sawtooth envelope type
call aywrite

call doflip

ld hl,&162c
ld (shipxy),hl
ld a,0
ld (wavenum),a
xor a
ld (wasfire),a
ld (dead),a
ld (tweenwave),a
ld (shield),a

ld hl,mac_blocklist
ld bc,mac_blocks-64
;a still zero
call startmusic

ld h,a
ld l,a
ld (score),hl

ld a,start_lives
ld (lives),a
call drawscore
call drawlives

;Clear both screens
newwave:
call fastclear
call doflip
call fastclear
call createwave

gameloop:
call doflip
call doinput

ld hl,tweenwave
push hl
ld a,(hl)
and a
jr z,gameloop2

;Clear the 'wave xx' message from the screen once the tweenwave counter is
;sufficiently low, but don't do this if the counter is zero, as the message
;would be cleared while the game is playing as well!

push af   ;Store the result of the comparison of the tweenwave counter
cp 11
call c,cleartweenwave
pop af   ;Get the result of the comparison of the tweenwave counter

gameloop2:
push af   ;Store it again

call z,clear_baddies
call clear_shots
call clear_ship
call clear_debris

ld a,(tweenwave)
and a
call z,seeifshothitbaddie

call drawstars
call drawshots

pop af   ;Get the result of the comparison of the tweenwave counter
call z,drawbaddies
call drawship
call drawdebris

pop hl
ld a,(dead)
and a
jr nz,gameover

ld a,(hl)	;tweenwave
and a
jr z,skip1

dec a
jr nz,skip2
ld (lostlife),a		;zero that
skip2:
ld (hl),a
cp 10
call nc,showtweenwave
jr gameloop		;skip baddie-count check

skip1:
ld a,(gotbaddies)
and a
jr nz,gameloop

jr newwave


;entry: a=0, hl=blocklist, bc=blocks-64
startmusic:
di	;changes must be atomic
ld (note_timeout),a
ld (notes_left),a

ld (blocklist),hl
ld (blockpos),hl
ld (blocks_minus64),bc
ei
ret


gameover:
ld bc,8*256+20
ld a,8
call ilprint
defb nbspace,nbspace,nbspace
defb 'GAME',nbspace,nbspace,'OVER'
defb nbspace,nbspace,nbspace+128

call doflip   ;Show the 'GAME OVER' message on the screen

call musicoff   ;Turn off the music

ld a,snd_plyr_died
call queuesnd_plyr

;Wait for three seconds before returning to the menu (300 = 3 seconds)
ld bc,150*6
gowaitlp:
halt
dec bc
ld a,b
or c
jr nz,gowaitlp

ld hl,(score)
push hl
ld de,(high)
and a
sbc hl,de
pop hl
jr c,nothigh	;well, thighs are pretty unlikely

ld (high),hl

nothigh:
jp mainloop


musicoff:
xor a
ld (blockpos+1),a
ld de,chan_music_1*256+8*256
call aywrite
ld d,chan_music_2+8
jp aywrite


;ret 8-bit rand in A, other regs ok.
rand8:
push bc
push de
push hl

ld hl,(seed+2)
ld d,l
add hl,hl
add hl,hl
ld c,h
ld hl,(seed)
ld b,h
rl b
ld e,h
rl e
rl d
add hl,bc
ld (seed),hl
ld a,l
ld hl,(seed+2)
adc hl,de
ld (seed+2),hl

pop hl
pop de
pop bc
ret

;Table containing the addresses of the sprites for each type; this is moved to
;another location when the program is initialised, as it must be page-aligned
sprite_addr_table_temp:
defw 0
defw type134_dat
defw enemyshot_dat
defw type134_dat
defw type134_dat
defw type5_dat
defw 0
defw 0
defw 0
defw 0
defw type10_dat
defw type11_dat
sprite_addr_table_temp_end:


include 'sprites.dat'

include 'font.dat'

include 'levels.asm'

include 'sound.asm'

include 'music.asm'

include 'tune.dat'

;initialise various different things.
initstuff:

;Clear both screens
ld hl,&4000
ld (hl),l
ld de,&4001
ld bc,&3fff
ldir

ld hl,&c000
ld (hl),l
ld de,&c001
ld bc,&3fff
ldir

;Set the screen mode and the border colour
ld bc,&7f8c
out (c),c   ;Select MODE 0 (16 colours, 160x200 resolution)
ld c,16
ld a,&54
out (c),c   ;Select the border (ink 16)
out (c),a   ;Set the border to black (hardware colour &54)

;Set the inks using hardware colours
;The inks have been set up to resemble normal and bright Spectrum colours. Inks
;0-7 represent normal colours; inks 8-15 represent bright colours. Furthermore,
;ink 0 is the transparent black ink, but ink 8 (which is also black) is opaque

xor a   ;A = ink number
ld hl,ink_colour_data   ;HL = address of ink table

set_inks_loop:
push af
ld e,(hl)   ;Get the hardware colour from the table
out (c),a   ;Select the ink number (0-15)
out (c),e   ;Set the ink to this hardware colour
pop af
inc hl   ;Go to the next entry in the table
inc a   ;Go to the next ink number
cp 16
jr nz,set_inks_loop
jp (hl)
ink_colour_data:
defb &54   ;Ink 0 = black (&54)
defb &44   ;Ink 1 = dark blue (&44)
defb &5c   ;Ink 2 = brown (&5c)
defb &58   ;Ink 3 = purple (&58)
defb &56   ;Ink 4 = dark green (&56)
defb &46   ;Ink 5 = grey/blue (&46)
defb &5e   ;Ink 6 = yellow/brown (&5e)
defb &40   ;Ink 7 = grey (&40)
defb &54   ;Ink 8 = black (&54)
defb &55   ;Ink 9 = blue (&55)
defb &4c   ;Ink 10 = red (&4c)
defb &4d   ;Ink 11 = magenta (&4d)
defb &52   ;Ink 12 = green (&52)
defb &53   ;Ink 13 = cyan (&53)
defb &4a   ;Ink 14 = yellow (&4a)
defb &4b   ;Ink 15 = white (&4b)

xor a
ld b,&bc
ld hl,crtc_data

;Change CRTC settings
set_crtc_loop:
ld c,(hl)   ;Get the CRTC register to change
inc hl
ld e,(hl)   ;Get the value to change it to
inc hl
out (c),c   ;Select the CRTC register
inc b   ;Now B = &bd
out (c),e   ;Change it to the new value
dec b   ;Now B = &bc
inc a   ;Set the number of the next entry in the CRTC data table
cp 5   ;There is a total of five registers to set
jr nz,set_crtc_loop
jp (hl)
crtc_data:
defb 1,40,2,46,6,25,7,30,13,0

ld bc,&bc0d
out (c),c   ;Select CRTC register 13
inc b
xor a
out (c),a   ;Set the low byte of the screen address to 0

;generate zxyaddr table

zxyaddrgen:
ld de,zxyaddr
ld hl,&0008   ;the screen starts at an offset of 8 (i.e. &4008 or &c008)
ld b,200   ;there are 200 lines on the screen
gzxylp1:
push bc   ;Store the number of lines remaining
ld a,l
ld (de),a
inc de
ld a,h
ld (de),a
inc de

add a,8   ;Calculate the screen address for the line below the current one, by
ld h,a    ;adding &800 to it
bit 6,a
call nz,nextline   ;The screen address needs to be adjusted after every eight
                  ;lines, by resetting the high byte and adding &c050
pop bc   ;Get the number of lines remaining

djnz gzxylp1


;Set up the mask table, so that sprites can be displayed transparently, using
;ink 0 as a transparent ink

xor a
ld hl,mask_table_hb*256

setup_mask_table_loop:
push af   ;Store the current byte

set_mask_for_left_pixel:
and &aa   ;Get the value of the left pixel only by resetting the even bits
          ;(&aa = &x10101010)
or a
jr nz,left_pixel_is_not_0
ld b,&aa
jr set_mask_for_right_pixel
left_pixel_is_not_0:
ld b,0

set_mask_for_right_pixel:
pop af   ;Get the current byte
push af   ;Store it again
and &55   ;Get the value of the right pixel only, by resetting the odd bits
          ;(&55 = &x01010101)
or a
jr nz,right_pixel_is_not_0
ld a,&55
jr store_mask_byte_in_table
right_pixel_is_not_0:
xor a

store_mask_byte_in_table:
or b
ld (hl),a

pop af   ;Get the current byte
inc hl   ;Go to the next entry in the mask table
add a,1   ;Go to the next byte; unlike the inc a instruction, this affects the
          ;carry flag, which we need to test for
jr nc,setup_mask_table_loop


;Set up the random number seed, based on the hexadecimal digits of pi
;(which in hex is 3.243f6a88...). Almost any set of pseudo-random numbers can
;be used, though
ld hl,&3f24
ld (seed),hl
ld hl,&886a
ld (seed+2),hl


;make the dot char into a printable space (since true spaces
;will get skipped to make the title screen look a bit nicer :-)).
xor a	;zero
ld (char_table+&76),a


;Set up interrupts
;Unlike the Spectrum, the RST &38 interrupt can be altered, so there is no
;need to use interrupt mode (IM) 2 on the CPC
ld a,&c3
ld hl,inthndl		;see sound.asm
ld (&38),a
ld (&39),hl
ei

;initialise stars array
ld hl,stars
ld b,num_stars
isloop:
call rand8
and 127
call initstar
djnz isloop

;Copy the sprite address table to another location so that it is page-aligned
ld hl,sprite_addr_table_temp
ld de,sprite_addr_table_hb*256
ld bc,sprite_addr_table_temp_end-sprite_addr_table_temp
ldir

;Copy the character ink table to another location so that it is page-aligned
ld hl,char_ink_table_temp
ld de,char_ink_table_hb*256
ld bc,char_ink_table_temp_end-char_ink_table_temp
ldir

;start up double-buffering
ld hl,bankm
ld a,&40   ;set the screen to draw
ld hl,banks
ld a,&10   ;set the screen to display
call flipend	;writes byte and re-enables ints

;and finally, do the initial sidebar stuff
call drawsidebar
;falls through

;draw the basic elements of the sidebar in an `empty' state.
;this only needs to be done once for each screen, as we never
;clear this area anywhere else.
drawsidebar:
;first do a *full* cls
ld a,(bankm)  ;get current screen address
ld h,a
ld l,0
ld d,h
ld e,l
inc e
ld bc,&3fff
ld (hl),l
ldir

ld bc,52*256+8
ld a,4
call ilprint
defb 'zblas','t'+128

ld bc,54*256+26
ld a,5
call ilprint
defb 'scor','e'+128

ld bc,52*256+36
ld a,4
call ilprint
defb 'energ','y'+128

;flip to other screen's mem
;falls through

;flip memory bank *only*, without waiting
;no need to disable ints for this one
memflip:
ld a,(bankm)
xor &80
ld (bankm),a
ret


;these draw the score/lives on both screens at once,
;so despite the double-buffering we only need to update them
;when they change.
drawscore:
ld de,(score)
ld bc,54*256+28
jr drawnumtmp

drawlives:
ld de,(lives)
ld d,0
ld bc,54*256+38

drawnumtmp:
ld hl,nbspace*256+nbspace
drawnumtmp2:
push bc
ld (numtmp),hl
ld (numtmp+2),hl
call gendec
pop bc
ld hl,numtmp
push bc
push hl
exx
call memflip
exx
call clear_number
ld a,7
call drawnum
call memflip
pop hl
pop bc
call clear_number
ld a,7
jr drawnum


;This is an altered version of the drawnumtmp2 routine that does not clear the
;area where the number is to be displayed. It is used on the main menu when
;printing the high score every time the screen is redrawn, to avoid flicker
drawnumtmp2_noclear:
push bc
ld (numtmp),hl
ld (numtmp+2),hl
call gendec
pop bc
ld hl,numtmp
push bc
push hl
exx
call memflip
exx
ld a,7
call drawnum
call memflip
pop hl
pop bc
ld a,7
jr drawnum


;show info preceding waves
showtweenwave:
ld bc,18*256+20
ld a,5
call ilprint
defb 'wav','e'+128

ld de,(wavenum)   ;Get the number of the current wave
ld d,0
ld hl,&2020
ld (numtmp+2),hl
call gendec   ;Convert the wave number to a decimal number
ld bc,28*256+20
ex de,hl
ld a,5
;falls through

drawnum:
ex (sp),hl	;now hl=ret addr, (sp)=addr in numtmp
ld (jpop+1),hl
jr ilprint
numtmp: defb '00000'
;it'll return here
jpop: jp 0	;modified


;Clear an area from the screen to allow a number to be displayed

clear_number:
push bc   ;Store the x- and y-coordinates
push de   ;Store the number to display
push hl
ld a,5   ;All the numbers that are displayed are five digits long
call clear_string   ;Clear the number from the screen
pop hl
pop de   ;Get the number to display
pop bc   ;Get the x- and y-coordinates
ret


;Clear the 'wave xx' message from the screen

cleartweenwave:
ld bc,18*256+20
ld a,7
jr clear_string


;Print a string
;
;Entry conditions - A = ink to write string in (0-7), B = x-coordinate, C =
; y-coordinate; the string must immediately follow the CALL to this routine, and
; the last character in the string must have bit 7 set
;Exit conditions - B = 0, C = 0, DE = screen address immediately after last
; character in the string, HL = address to jump to after exiting from this
; routine
; A corrupted

get_char_data_ink equ get_char_data_inkop+1
ilprint:
rlca   ;Multiply the ink by 4, as each entry in the ink table consists of 4
rlca   ;bytes
ld (get_char_data_ink),a

call calc_screen_addr   ;Calculate the screen address to write the string to
                        ;(stored in HL)
ex de,hl   ;Now DE = screen address

pop hl   ;Get the address of the string immediately after the CALL to this
         ;routine

ilprint_loop:
ld a,(hl)   ;Get the character from the string
and 127   ;Ensure that it is between 0 and 127
cp 32
jr z,ilprint_next_char

push hl   ;Store the address of the current character in the string
push de   ;Store the screen address
call get_char_data   ;Convert the character to a MODE 0 sprite and store it in
                     ;the temporary sprite
pop hl   ;Get the screen address (now stored in HL)
push hl   ;Store it again
ld de,tmpsprite   ;The temporary sprite contains the character to print
call drawsprgen2
pop de   ;Get the screen address (now stored in DE)

pop hl   ;Get the address of the current character in the string

ilprint_next_char:
inc de
inc de   ;Set the screen address for the next character
bit 7,(hl)   ;The last character in the string must have bit 7 set
inc hl   ;Go to the next character in the string
jr z,ilprint_loop   ;Repeat for the remaining characters in the string

jp (hl)   ;HL = address immediately after end of string, so return from this
	      ;routine

	      
;Create a temporary sprite for an ASCII character, to be used when printing the
;character itself
;
;Entry conditions - A = ASCII number of character
;Exit conditions - B = 0, C = 0, DE = byte immediately following end of temporary
; sprite, HL = byte immediately following end of character data
; A corrupted
;
;The ink to use should already be set at get_char_data_inkop; the routine to
; print a string does this

get_char_data:
ld de,char_table-256   ;The character table starts at character 32, but it is
                       ;necessary to start at the address of the non-existent
                       ;character 0
ld h,0
ld l,a
add hl,hl   ;HL = character number * 8 (each entry in the character table uses
add hl,hl   ;8 bytes)
add hl,hl
				  
ex de,hl   ;Now DE = character number * 8, HL = address of entry in character
	   ;table for character 0
add hl,de   ;HL = address of entry in character table

ld de,tmpsprite+2
ld b,8   ;Each character consists of 8 lines

get_char_data_loopy:
ld c,2
ld a,(hl)   ;Get the bit data for the current line of the current character

get_char_data_loopx:
rlca   ;Rotate the bits left twice so that the next set of 2 pixels can be read
rlca
push af   ;Store the bit data

and 3   ;A is now a number between 0 and 3, representing the bit data for 2
        ;pixels
push de   ;Store the temporary sprite address
ld e,a   ;E = bit data for 2 pixels

;Compare this data with values in the ink table

push hl   ;Store the character table address
ld h,char_ink_table_hb   ;Set high byte of ink table
get_char_data_inkop: ld a,0   ;modified (A = ink * 4)
add a,e
ld l,a   ;DE = ink * 4 + bit data for 2 pixels
ld a,(hl)   ;A = ink data corresponding to this style and the 2 pixels
            ;specified

pop hl   ;Get the character table address
pop de   ;Get the temporary sprite address
ld (de),a   ;Write the result to the temporary sprite
inc de   ;Go to the next byte in the temporary sprite
pop af   ;Get the bit data

dec c
jr nz,get_char_data_loopx

inc hl   ;Go to the next line for this character in the character table

djnz get_char_data_loopy   ;Repeat for the remaining lines
ret


;Clear a string by printing blank areas instead of letters, numbers and
;punctuation
;
;Entry conditions - A = length of string, B = x-coordinate, C = y-coordinate
;Exit conditions - B = 0, HL = screen address immediately after last character
; in the string
; A, DE corrupted

clear_string:
push af   ;Store the length of the string
call calc_screen_addr   ;Calculate the screen address to write the string to
                        ;(stored in HL)
pop bc   ;Now B = length of string
                        
clear_string_loop:
push bc
push hl   ;Store the screen address
ld de,tmpsprite   ;The temporary sprite contains the character to print
call clear_sprite2
pop hl   ;Get the screen address

clear_string_next_char:
inc hl
inc hl   ;Set the screen address for the next character
pop bc
djnz clear_string_loop   ;Repeat for the remaining characters in the string
ret


doinput:
call read_keyboard   ;Read both the keyboard and joystick
ld a,(joystick)   ;Is the joystick selected?
and a
jr z,doinput_keyboard   ;If not, read from the keyboard

doinput_joystick:
ld e,0
ld a,(keyboard_info+9)   ;Get information about the keys pressed on matrix line
                         ;9 (which includes the joystick)
rla
rla
rla
rla   ;Read bit 4 (joystick fire 0) and store it in the carry flag
rl e   ;Add the bit to the E register
ld a,(keyboard_info+9)   ;Get the information about the joystick again
rra   ;Read bit 0 (joystick up)
rl e
rra   ;Read bit 1 (joystick down)
rl e
rra   ;Read bit 2 (joystick left)
rl e
rra   ;Read bit 3 (joystick right)
rl e
ld a,e
cpl   ;Invert the bits, so that 1 means the key has been pressed and 0 means it
      ;has not been pressed
jr doinput2   ;E now contains the joystick information in the form 000FUDLR

;nope, read keyboard instead
doinput_keyboard:
ld e,0
ld a,(keyboard_info+5)   ;Get information about the keys pressed on matrix line
                         ;5 (which includes the SPACE bar)
rla   ;Read bit 7 (SPACE) and store it in the carry flag
rl e   ;Add the bit to the E register
ld a,(keyboard_info+8)   ;Get information about the Q and A keys
rra
rra
rra
rra   ;Read bit 3 (Q)
rl e
rra
rra   ;Read bit 5 (A)
rl e
ld a,(keyboard_info+4)   ;Get information about the O key
rra
rra
rra
rl e   ;Read bit 2 (O)
ld a,(keyboard_info+3)   ;Get information about the P key
rra
rra
rra
rra
rl e   ;Read bit 3 (P)
       ;E now contains the keyboard information in the form 000FUDLR

ld a,e
cpl   ;Invert the bits, so that 1 means the key has been pressed and 0 means it
      ;has not been pressed

doinput2:
;A is input, in the form 000FUDLR.

;player movement
ld de,(shipxy)
ld hl,shipx
rra
jr nc,keyrd1
inc (hl)
inc (hl)
keyrd1:
rra
jr nc,keyrd2
dec (hl)
dec (hl)
keyrd2:
dec hl	;shipy
rra
jr nc,keyrd3
inc (hl)
inc (hl)
keyrd3:
rra
jr nc,keyrd4
dec (hl)
dec (hl)
keyrd4:
rra
jr nc,keyrd5

;space - fire
ld hl,wasfire
ld a,(hl)
and a
jr z,keyrd5a
dec (hl)
jr keyrd6

keyrd5a:
ld (hl),delay_auto_fire

;ok then, actually shoot :-)
ld a,snd_plyr_shot
ld a,3
call queuesnd_plyr
push de
exx
pop de
call addnewshot_auto
inc d
inc d
inc d
inc d
call addnewshot_auto
exx
jr keyrd6

keyrd5:
xor a
ld (wasfire),a

keyrd6:
ld a,(shipy)
cp num_grid_y_big
jr c,keymove_y_undo
cp num_grid_y-3
jr c,keyrd7
keymove_y_undo:
ld a,e
ld (shipy),a
keyrd7:
ld a,(shipx)
cp num_grid_x-4
ret c

ld a,d
ld (shipx),a
ret


doflip:
;On the Spectrum, frame control is used when flipping screens, in order to slow
;down the game a bit. On the CPC, there is no frame control as it takes several
;frames to draw everything on the screen

;Wait for frame flyback
ld b,&f5
frame_flyback_loop:
in a,(c)   ;Get the frame flyback status (stored in bit 0)
rra   ;Bit 0 is stored in the carry flag
jr nc,frame_flyback_loop

di
ld hl,banks
ld a,(hl)
xor &20   ;switches between &10 and &30, which correspond to &40 and &c0
flipend:
ld (hl),a
push af
call memflip   ;change the screen to draw to as well (either &4000 or &c000)
pop af
ld bc,&bc0c
out (c),c  ;select CRTC register 12
inc b
out (c),a  ;output the high byte of the screen address
ei
ret


;Clear the playing area of the current screen

;On the Spectrum, this routine sets the colour attributes of each cell in the
;playing area to black, effectively clearing it. This requires changing 576
;bytes of memory. On a CPC, this method is not possible, so the playing area is
;cleared properly, although this requires changing 9216 bytes of memory!
fastclear:
di   ;Disable interrupts to prevent interference with the stack
ld (fcspop+1),sp  ;store stack pointer

ld a,(bankm)  ;get current screen address
ld h,a
ld l,56  ;start at right-hand side of first line in first row

ld a,8  ;each row of the game screen consists of 8 lines
ld de,80  ;when a line has been cleared, add 80 to go to the right-hand side of
          ;the next line
ld bc,0  ;bytes to fill the game screen

fclp:
ld (fclpa),a  ;stack cannot be used here, so store A in memory instead
ld a,24  ;there are 24 rows in the game screen
fclp2:
ld sp,hl

push bc
push bc
push bc
push bc

push bc
push bc
push bc
push bc

push bc
push bc
push bc
push bc

push bc
push bc
push bc
push bc

push bc
push bc
push bc
push bc

push bc
push bc
push bc
push bc

add hl,de  ;go to same line in next row

dec a
jr nz,fclp2  ;repeat for all 24 rows

inc h  ;calculate screen address for right-hand side of next line in first row
ld a,l
add a,&80
ld l,a

ld a,(fclpa)
dec a
jr nz,fclp

fcspop: ld sp,0	;modified
ei
ret
fclpa: defb 0


last_wave equ 20

createwave:
;Change one byte of the random number seed using the refresh (R) register.
;Eventually the seed will reset to zero, so the baddies fire a continuous stream
;of shots! Changing the seed after each frame should prevent this from occurring
ld a,r
or &40
ld (seed+3),a

ld a,(wavenum)
inc a
cp last_wave+1
jr nz,cwskip
;On the Spectrum, the game is speeded up if the player finishes the last wave
;(as if it wasn't fast enough already!). On the CPC, the game cannot be speeded
;up any faster!
ld a,1
cwskip:
ld (wavenum),a

dec a
jr z,cwskip3

;end wave bonus
ld a,(lostlife)
and a
jr nz,cwskip3
ld a,(lives)
add a,10
jr nc,cwskip5
ld a,255
cwskip5:
ld (lives),a
call drawlives

cwskip3:
;get it to show the between-waves screen
ld a,60
ld (tweenwave),a

ld a,(wavenum)
dec a
ld l,a
ld h,0
ld c,h		;c=0, handy for some of the level routines
ld b,h		;...and b=0 for that matter
add hl,hl
ld de,wavetable	;-)
add hl,de
ld e,(hl)
inc hl
ld d,(hl)
push de
ld hl,baddies
ret

;this are all in levels.z
wavetable:
defw wave1,wave2,wave3,wave4,wave5
defw wave6,wave7,wave8,wave9,wave10
defw wave11,wave12,wave13,wave14,wave15
defw wave16,wave17,wave18,wave19,wave20


;add new baddie at hl (which should be in baddies array)
;does everything other than type (which is at hl anyway, so not too hard)
;a=hits, b=x, c=y, d=dx, e=dy.
;exits with hl pointing to the next entry in the array.
addnewbaddie:
inc l
ld (hl),b
inc l
ld (hl),c
inc l
ld (hl),d
inc l
ld (hl),e
inc l
ld (hl),a
;falls through


;Set the screen addresses of a sprite when adding a new entry into an array

;On the CPC, it is also necessary to reset the screen addresses of a sprite for
;both screens, otherwise areas outside the playing area will be cleared

addnewsomething_set_screen_addrs:
inc l
ld (hl),8
inc l
ld (hl),&40   ;&4008 = top-left corner of playing area of screen at &4000
inc l
ld (hl),8
inc l
ld (hl),&c0   ;&c008 = top-left corner of playing area of screen at &c000
ld l,0
inc h
ret


;as above, but searches for empty baddie slot first.
;be sure to decr h afterwards and write the type.
addnewbaddie_auto:
ld hl,baddies
addnewsomething_auto:
push bc
ld b,max_baddies
ld c,a
anbalp:
ld a,(hl)
and a
jr nz,anbaskip

;it's always a shot of some kind, so...
ld a,snd_enemy_shot
call queuesnd_enemy

ld a,c
pop bc
jr addnewbaddie

anbaskip:
inc h
djnz anbalp

pop bc

;failed, so make sure any write afterwards won't hurt. :-)
ld h,1
ret


;similar, but for player shots (only requires shot position, in *de*)
addnewshot_auto:
ld hl,shots
ld b,max_shots
ansalp:
ld a,(hl)
and a
jr nz,ansaskip

ld (hl),1
inc l
ld (hl),d
inc l
ld (hl),e
jr addnewsomething_set_screen_addrs

ansaskip:
inc h
djnz ansalp
ret


;move/draw background stars
store_old_star equ store_old_starop+1
drawstars:
ld hl,stars   ;HL = address of stars table
ld b,0   ;B = number and x-coordinate of the current star
dstarlp:
ld a,(hl)   ;Get the y-coordinate (0-191) of the current star

push af   ;Store the y-coordinate of the current star
push hl   ;Store the address of the entry for the current star

;Clear the old star from the currently selected screen

ld hl,stars_old   ;HL = start address of old stars table for screen at &4000
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 0 and is also stored in the carry flag; A is now 0
       ;or 1

or a
jr z,dstarlp1
ld a,num_stars*2   ;If A = 1 (i.e. the current screen address is &c000), set it
                   ;to the first entry in the table for that screen
dstarlp1:
add a,l   ;Now A = high byte of screen address + number of stars * 2
add a,b   ;Add the number of the current star * 2
add a,b
ld l,a   ;HL = address in table containing the old screen address of the current
         ;star

ld (store_old_star),hl   ;Store the table address so that the new screen
                         ;address can be stored later on

;Read the old screen address from the table, where the old star is, and clear
;the star from the screen

ld a,(hl)
inc hl
ld h,(hl)
ld l,a
ld a,(bankm)
or h
ld h,a   ;Now HL = old screen address of current star on the currently selected
         ;screen
ld (hl),0   ;Clear the star from the screen

pop hl   ;Store the address of the entry for the current star
pop af   ;Store the y-coordinate of the current star

inc l
ld c,(hl)
add a,c   ;Move the star down by a set amount
dec l
cp 192   ;Check if it has gone off the bottom of the screen
jr nc,dstar1   ;If so, reset the y-coordinate

ld (hl),a   ;Store the new y-coordinate
inc l   ;Go to the next entry in the table
inc l

push hl   ;Store the address of the next entry in the table

;Draw the star on the screen

ld de,zxyaddr   ;DE = start of screen address table
ld h,0
ld l,a   ;HL = y-coordinate of star
add hl,hl   ;Multiply the y-coordinate by 2, since each entry in the screen address
            ;table consists of 2 bytes
add hl,de   ;HL = entry in screen address table for y-coordinate

ld e,(hl)
inc hl
ld d,(hl)   ;DE = screen address for row (from 0 to &3fff)
ld a,(bankm)   ;get current screen address
or d
ld h,a   ;Set the screen address so that it is between &4000 and &7fff, or &c000
         ;and &ffff
ld l,e   ;Now HL = screen address for first column of y-coordinate

ld d,0
ld a,b
add a,a   ;Multiply the x-coordinate of the star (stored in B) by 2
ld e,a   ;DE = x-coordinate of star * 2, since each x-coordinate on the screen
         ;consists of 2 bytes
add hl,de   ;Add the x-coordinate to the screen address, now stored in HL

;On the Spectrum, all the stars are the same colour. On the CPC, slow stars are
;grey, while fast stars are white
ld a,c
cp 5
jr nc,draw_fast_star
draw_slow_star:
ld (hl),84   ;84 = black (left pixel), grey (right pixel)
jr store_old_starop
draw_fast_star:
ld (hl),85   ;85 = black (left pixel), white (right pixel)
store_old_starop:
ld (&0000),hl   ;Store the screen address of this star in the table for the old
                ;stars - the address is modified

pop hl   ;get the address of the next entry in the table
jr dstar2

dstar1:
xor a
call initstar

dstar2:
inc b
ld a,l
cp num_stars*2   ;Check if the end of the stars table has been reached
jr nz,dstarlp   ;If not, display the next star
ret

initstar:
ld (hl),a
inc l
call rand8
and 7   ;Set the speed of the star from 0-7
inc a
inc a   ;Now set it from 2-9
ld (hl),a   ;Store it in the stars array
inc l
ret


seeifshothitbaddie:
ld hl,baddies
ld b,max_baddies
sishblp:
ld a,(hl)
and a
jp z,sishblpend
cp 2	;can't hit enemy shots
jr z,sishblpend

push bc
push hl

call sishb_get_coords

sishbchk:
;see if any shot hit it
;first fit left edge of box to screen pos so it doesn't seem to be
;unfairly missing it ;-)
ld hl,shots
sishblp2:
ld a,(hl)
and a
jr z,sishblp2end

inc l
ld a,(hl)   ;A = x-coordinate of left boundary of shot
cp d   ;D = x-coordinate just to right of right boundary of baddie
jr nc,sishblp2altend

inc a   ;Now A = x-coordinate of right boundary of shot
cp b   ;B = x-coordinate of left boundary of baddie
jr c,sishblp2altend

inc l
ld a,(hl)   ;A = y-coordinate of top boundary of shot
cp e   ;E = y-coordinate just below bottom boundary of baddie
jr nc,sishblp2altend

inc a   ;Now A = y-coordinate of bottom boundary of shot
cp c   ;C = y-coordinate of top boundary of baddie
jr c,sishblp2altend

;yes, shot hit it.
;remove shot
;On a CPC, when a shot is removed, it must not be drawn to the screen, and
;it must also be cleared from the other screen. Unlike the Spectrum version, it
;isn't possible to set the 'shot active' flag to 0, so bit 7 is set instead.
;This lets both the drawshots and clear_shots routines know what to do
ld l,shot_active
ld (hl),&80

ld a,snd_enemy_hit
call queuesnd_enemy

;decr hits
pop hl
push hl
ld l,baddie_hits
ld a,(hl)
dec a
ld (hl),a
jr nz,sishblpaltend

;baddie destroyed
;On a CPC, when a baddie is destroyed, it must not be drawn to the screen, and
;it must also be cleared from the other screen. Unlike the Spectrum version, it
;isn't possible to set the baddie type to 0, so bit 7 is set instead. This lets
;both the drawbaddies and clear_baddies routines know what to do
ld l,0
ld a,(hl)   ;Get the baddie type from the array
set 7,(hl)   ;Set bit 7 of the baddie type

push af   ;Store the baddie type
push hl

;add score for this type
dec a
ld d,l
ld e,a
ld hl,scoretbl
add hl,de
ld d,0
ld e,(hl)
ld hl,(score)
add hl,de
ld (score),hl

call drawscore

;add debris
pop hl
inc l
ld b,(hl)
inc l
ld c,(hl)
;bc is at top-left, so move down/across to make big enemy explosions
;look a bit less crap.
inc b
inc b
inc c
inc c
call add_debris

pop af

cp 10
jr c,debrisskip

inc b
inc b
call add_debris_extra

debrisskip:
;make sound for it
ld a,snd_enemy_died
call queuesnd_enemy

;don't check this one against further shots
jr sishblpaltend

scoretbl:
defb 1,0,2,5,2

;we don't have types 6-9 so five bytes in the table are unused.
;other data is stuffed here as follows...

joystick: defb 0
mixerdat: defb 0

high:	defw 250

;number of frames left before current note ends (see music.asm)
note_timeout: defb 0

;back to the score table for the last two types
defb 20,50


sishblp2altend:
ld l,shot_active

sishblp2end:
inc h
ld a,h
cp shots/256+max_shots
jr nz,sishblp2

sishblpaltend:
pop hl
pop bc

sishblpend:
inc h
dec b
jp nz,sishblp
ret


sishb_get_coords:
;done with bounding boxes, different for each type.
;well, we actually do it with top-left corner and
;the pos just to the bottom-right of the bottom-right corner.
inc l
ld b,(hl)
inc l
ld c,(hl)

cp 2
jr nz,sishb1

;types 1, 3, and 4
ld a,2
add a,(hl)
ld e,a
dec l
ld a,2
add a,(hl)
ld d,a
ret

sishb1:
cp 5
jr nc,sishb2

;types 1, 3, and 4
ld a,4
add a,(hl)
ld e,a
dec l
ld a,4
add a,(hl)
ld d,a
ret

sishb2:
cp 10
jr nz,sishb3

ld a,6
add a,(hl)
ld e,a
dec l
ld a,8
add a,(hl)
ld d,a
ret

sishb3:
cp 11
jr nz,sishb4

;type 11
ld a,8
add a,(hl)
ld e,a
dec l
ld a,12
add a,(hl)
ld d,a
ret

sishb4:
;type 5
ld a,6
add a,(hl)
ld e,a
dec l
ld a,6
add a,(hl)
ld d,a
ret


;Clear the player's ship from the screen

clear_ship_screen_addr equ clear_ship_screen_addrop+1

clear_ship:
ld hl,ship_screen_addr_40
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array for screen address of baddie

;Read the screen address and store it in HL
ld a,(hl)
inc hl
ld h,(hl)
ld l,a   ;HL = screen address of baddie, as stored in the array
ld (clear_ship_screen_addr),hl

ld de,player_dat
clear_ship_screen_addrop: ld hl,&0000   ;modified
jp clear_sprite2   ;Clear the sprite, without calculating the screen address,
                   ;as it is already known


drawship:
ld bc,(shipxy)
push bc   ;Store the x-coordinate

ld a,(tweenwave)
and a
jr nz,dship2   ;Do not check if the ship has hit a baddie if the tweenwave
               ;counter is not zero, since there are no baddies on the screen

call seeifshiphitbaddie
ld a,(ship_collision)
and a
jr z,dship2

dship1a:
;we have a collision
ld a,(shield)
and a
jr nz,dship2

ld a,1
ld (lostlife),a
ld a,(lives)
and a
jr z,lostall
dec a
ld (lives),a
ld a,shield_life
ld (shield),a
call drawlives

;player-hit sound
ld a,snd_plyr_lostlife
call queuesnd_plyr
jr dship2

lostall:
dec a
ld (dead),a

;*don't* draw the ship when we're dead :-)
pop bc
ret

dship2:
pop bc   ;Get the x- and y-coordinates

ld de,player_dat
ld hl,shield
ld a,(hl)
and a
push af   ;Store the results of the shield check
jr z,dship2a

drawshiphit:
ld de,player_dat+96+2   ;If the ship has been hit, display the blue ship sprite
                        ;instead

dship2a:
call drawsprgen   ;Print the player's ship

;Store the screen address of the ship for the current screen
ex de,hl   ;Now DE = screen address
ld hl,ship_screen_addr_40
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array to store screen address

ld (hl),e
inc l
ld (hl),d


pop af   ;Get the results of the shield check
ld de,shield
ld a,(de)
and a   ;Check if the shield has run out
ret z

dec a   ;If the shield has not yet run out, decrease the amount remaining
ld (de),a
ret


;Check if the ship has hit a baddie

;On the Spectrum, the colour attributes are checked before the ship is drawn
;to determine if there is a collision. This could be done on the CPC as well,
;but the bounding box method is used instead

;Furthermore, because of the slow speed compared to the Spectrum version, the
;game is too easy. Therefore, the collision routine has been changed so that if
;any part of the ship hits a baddie, the player loses energy. This makes the
;game more difficult. On the Spectrum, the player only loses energy if a
;baddie hits the centre section of the ship

seeifshiphitbaddie:
xor a
ld (ship_collision),a   ;Reset the ship collision flag

ld hl,baddies
ld b,max_baddies

seeifshiphitbaddie_loop:
ld a,(hl)   ;Get the baddie type
and a
jr z,seeifshiphitbaddie_loop_next
bit 7,a
jr nz,seeifshiphitbaddie_loop_next

push bc   ;Store the number of baddies remaining
push hl   ;Store the high byte of the array address

call sishb_get_coords

ld hl,shipy
ld a,(hl)   ;A = y-coordinate of top boundary of ship
cp e   ;E = y-coordinate just below bottom boundary of baddie
jr nc,seeifshiphitbaddie2

add a,3   ;Now A = y-coordinate of bottom boundary of ship
cp c   ;C = y-coordinate of top boundary of baddie
jr c,seeifshiphitbaddie2

inc l
ld a,(hl)   ;A = x-coordinate of left boundary of ship
cp d   ;D = x-coordinate just to right of right boundary of baddie
jr nc,seeifshiphitbaddie2

add a,5   ;Now A = x-coordinate of right boundary of ship
cp b   ;B = x-coordinate of left boundary of baddie
jr c,seeifshiphitbaddie2

;Set the collision flag if the ship hits a baddie or enemy shot
ld a,1
ld (ship_collision),a

seeifshiphitbaddie2:
pop hl   ;Get the high byte of the array address
pop bc   ;Get the number of baddies remaining

seeifshiphitbaddie_loop_next:
inc h   ;Go to the next entry in the array
dec b
jr nz,seeifshiphitbaddie_loop
ret


;Clear the player's shots from the screen

clear_shots_screen_addr equ clear_shots_screen_addrop+1

clear_shots:
ld h,baddies_hb   ;Start at the array for the first shot (which is in the same
                  ;area as the baddies array)
ld b,max_shots

clear_shots_loop:
ld l,shot_active   ;HL = address containing 'shot active' flag
ld a,(hl)   ;Find out if the flag is reset
and a   ;If it is, go to the next shot
jr z,clear_shots_next

clear_shots_loop2:
ld l,shot_screen_addr_40
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array for screen address of shot

push bc   ;Store the number of shots remaining
push hl   ;Store the high byte of the array address

;Read the screen address from the array and store it in HL
ld a,(hl)
inc hl
ld h,(hl)
ld l,a   ;HL = screen address of shot, as stored in the array
ld (clear_shots_screen_addr),hl

ld de,plyrshot_dat
clear_shots_screen_addrop: ld hl,&0000   ;modified
call clear_sprite2   ;Clear the sprite, without calculating the screen address,
                     ;as it is already known
pop hl   ;Get the high byte of the array address
pop bc   ;Get the number of baddies remaining

ld l,shot_active
ld a,(hl)
and &80   ;Check if bit 7 of the 'shot active' flag has been set (i.e. has the
          ;shot just been removed?)
jr z,clear_shots_checky   ;If bit 7 is reset, check the y-coordinates of the
                          ;shot

;If the shot has been removed, it has now been cleared from both screens, so it
;can be deleted from the array
ld (hl),0

clear_shots_checky:
ld l,shot_y
ld a,(hl)   ;Get the y-coordinate of the shot
bit 7,a
jr z,clear_shots_next   ;If it is outside the playing area, reset the
ld l,shot_active        ;'shot active' flag
ld (hl),0

clear_shots_next:
inc h   ;Set the high byte of the array address for the next shot
djnz clear_shots_loop   ;Clear the remaining shots
ret


;move and draw player shots
shot_screen_addr_hb equ shot_screen_addr_hbop+1
drawshots:
ld hl,shots
ld b,max_shots
dsloop:
ld a,h
ld (shot_screen_addr_hb),a   ;Store the high byte of the array of the current
                             ;shot, so that the routine to store the screen
                             ;address knows where the array is
push bc
ld a,(hl)
and a
jr z,dsloopend
bit 7,a           ;Do not draw the shot to the screen if bit 7 of the 'shot
jr nz,dsloopend   ;active' flag is set (i.e. the shot has just been removed)

ld l,shot_y
ld a,(hl)
dec a
dec a
ld (hl),a
bit 7,a
;On the Spectrum, the 'shot active' flag is reset, so the shot effectively no
;longer exists. On a CPC, the shot needs to be cleared from the other screen, so
;the resetting must occur during the routine to clear the player's shots from
;the screen
jr nz,dsloopend

dsskip:
ld c,a
dec l
ld a,(hl)
ld b,a
ld de,plyrshot_dat
push hl
call drawsprgen

;Store the screen address of a player's shot in the table, so that next time the
;shot sprite is written to the screen, the program knows where the old one is so
;that it can erase it from the screen

ex de,hl   ;Now DE = screen address
shot_screen_addr_hbop: ld h,0   ;High byte of shot array - this is modified
ld l,shot_screen_addr_40

ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array to store screen address

ld (hl),e
inc l
ld (hl),d

pop hl

dsloopend:
pop bc
ld l,shot_active
inc h
djnz dsloop
ret


;Clear the debris from the screen

clear_debris_screen_addr equ clear_debris_screen_addrop+1

clear_debris:
ld h,baddies_hb   ;Start at the array for the first bit of debris (which is in
                  ;the same area as the baddies array)
ld b,max_debris

clear_debris_loop:
ld l,debris_timeout   ;HL = address containing colour of debris
ld a,(hl)   ;Find out if it is 0 (i.e. the debris has disappeared)
and a   ;If it is, go to the next bit of debris
jr z,clear_debris_next
dec (hl)   ;Change the colour of the debris for the next time it is displayed

clear_debris_loop2:
ld l,debris_screen_addr_40
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array for screen address of debris

push bc   ;Store the number of bits of debris remaining
push hl   ;Store the high byte of the array address

;Read the screen address from the array and store it in HL
ld a,(hl)
inc hl
ld h,(hl)
ld l,a   ;HL = screen address of debris, as stored in the array
ld a,h
ld (clear_debris_screen_addr),hl

ld de,debris_dat
clear_debris_screen_addrop: ld hl,&0000   ;modified
call clear_sprite2   ;Clear the sprite, without calculating the screen address,
                     ;as it is already known
pop hl   ;Get the high byte of the array address
pop bc   ;Get the number of bits of debris remaining

clear_debris_next:
inc h   ;Set the high byte of the array address for the next bit of debris
djnz clear_debris_loop   ;Clear the remaining debris
ret


;move and draw debris

debris_screen_addr_hb equ debris_screen_addr_hbop+1
draw_debris_set_colour equ draw_debris_set_colourop+1

drawdebris:
ld hl,debris
ld b,max_debris
ddloop:
ld a,h
ld (debris_screen_addr_hb),a   ;Store the high byte of the array of the current
                               ;bit of debris, so that the routine to store the
                               ;screen address knows where the array is
ld a,(hl)
cp 2              ;Do not draw debris if the colour is set to 0 or 1; this is to
jr c,ddloopend    ;allow an extra frame for all of the debris to be cleared
                  ;from the screen

dec a   ;Decrease the colour
ld (draw_debris_set_colour),a   ;Set the actual colour of the debris when
                                ;displaying it, from 1-7 instead of 2-8
push bc

inc l		;x
ld a,(hl)
ld l,debris_dx
add a,(hl)
ld l,debris_x
ld (hl),a
ld b,a
inc l		;y
ld a,(hl)
ld l,debris_dy
add a,(hl)
ld l,debris_y
ld (hl),a
ld c,a

ld a,b
cp num_grid_x-1
jr nc,ddskip
ld a,c
cp num_grid_y-1
jr nc,ddskip

;On a Spectrum, the colour of the debris can be altered simply by changing the
;sprite colour attributes. On a CPC, seven separate sprites are required for
;each possible colour (blue, red, magenta, green, cyan, yellow, white)

draw_debris_set_colourop: ld a,0   ;The colour of the debris (1-7); this is
                                   ;modified
push bc   ;Store the x- and y-coordinates
push hl   ;Store the current address in the debris array
ld b,a
ld hl,debris_dat-18   ;Each debris sprite consists of 18 bytes (2 for the width
ld de,18              ;and height, 16 for the sprite data)
select_debris_type_loop:
add hl,de
djnz select_debris_type_loop
ex de,hl
pop hl   ;Get the current address in the debris array
pop bc   ;Get the x- and y-coordinates

push hl
call drawsprgen

;Store the screen address of a bit of debris in the table, so that next time the
;debris sprite is written to the screen, the program knows where the old one is
;so that it can erase it from the screen

ex de,hl   ;Now DE = screen address
debris_screen_addr_hbop: ld h,0   ;High byte of debris array - this is modified
ld l,debris_screen_addr_40

ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array to store screen address

ld (hl),e
inc l
ld (hl),d

pop hl

ddskip:
pop bc

ddloopend:
ld l,debris_timeout
inc h
djnz ddloop
ret


add_debris:
;up/down/left/right
ld de,&fe00
call addnewdebris_auto
ld d,2
call addnewdebris_auto
ld de,&00fe
call addnewdebris_auto
ld e,2
call addnewdebris_auto

;diagonals
ld de,&ffff
call addnewdebris_auto
ld d,1
call addnewdebris_auto
ld e,1
call addnewdebris_auto
ld d,&ff
;falls through

addnewdebris_auto:
ld hl,debris
call addnewsomething_auto
dec h
ld l,debris_timeout
ld (hl),8   ;On the CPC, an extra frame is required to clear all the debris from
            ;the screen; this is set to 7 on the Spectrum
ret


add_debris_extra:
;up/down/left/right
ld de,&fc00
call addnewdebris_auto
ld d,4
call addnewdebris_auto
ld de,&00fc
call addnewdebris_auto
ld e,4
call addnewdebris_auto

;diagonals
ld de,&fefe
call addnewdebris_auto
ld d,2
call addnewdebris_auto
ld e,2
call addnewdebris_auto
ld d,&fe
jr addnewdebris_auto


;Clear the baddies from the screen

clear_baddies_baddie_type equ clear_baddies_baddie_typeop+1
clear_baddies_screen_addr equ clear_baddies_screen_addrop+1

clear_baddies:
ld h,baddies_hb   ;Start at the array for the first baddie
ld b,max_baddies

clear_baddies_loop:
ld l,0   ;HL = address containing type of baddie
ld a,(hl)   ;Get the type of baddie
and &7f   ;Reset bit 7 of the baddie type
ld (clear_baddies_baddie_type),a   ;Store it for use later on
and a   ;If the baddie type is 0, go to the next one
jr z,clear_baddies_next

clear_baddies_loop2:
ld l,baddie_screen_addr_40
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array for screen address of baddie

push bc   ;Store the number of baddies remaining
push hl   ;Store the high byte of the array address

;Read the screen address from the array and store it in HL
ld a,(hl)
inc hl
ld h,(hl)
ld l,a   ;HL = screen address of baddie, as stored in the array
ld (clear_baddies_screen_addr),hl

ld h,sprite_addr_table_hb
clear_baddies_baddie_typeop: ld a,0   ;The baddie type (modified)
rlca   ;Multiply the baddie type by 2, since each entry in the sprite address
       ;table consists of 2 bytes
ld l,a   ;HL = address of entry in sprite address table for this baddie type
ld e,(hl)
inc hl
ld d,(hl)   ;DE = sprite address for this baddie type

clear_baddies_screen_addrop: ld hl,&0000   ;modified
call clear_sprite2   ;Clear the sprite, without calculating the screen address,
                     ;as it is already known
pop hl   ;Get the high byte of the array address
pop bc   ;Get the number of baddies remaining

ld l,0
ld a,(hl)   ;Get the baddie type
cp 2   ;Check if the current baddie type is a shot
jr z,clear_baddie_shot
and &80   ;Check if bit 7 of the baddie type has been set (i.e. has the baddie
          ;just been destroyed?)
jr z,clear_baddies_next   ;If bit 7 is reset, go to the next baddie

;If the baddie has been destroyed, it has now been cleared from both screens,
;so it can be deleted from the array
ld (hl),l
jr clear_baddies_next

clear_baddie_shot:
ld l,baddie_y
ld a,(hl)   ;Get the y-coordinate of the shot
cp num_grid_y-1
jr c,clear_baddies_next   ;If it is outside the playing area, reset the
ld l,0                    ;baddie type to zero so that it is effectively
ld (hl),l                 ;erased from the baddies array

clear_baddies_next:
inc h   ;Set the high byte of the array address for the next baddie
djnz clear_baddies_loop   ;Clear the remaining baddies
ret


;draw *and move* the baddies
;(also updates the gotbaddies end-of-level check flag)
;
;I messed about with the various probabilities here (relative to
;the original), so the C equivalents in comments are no more than a
;rough guide as far as the randomness goes. :-)
baddie_screen_addr_hb equ baddie_screen_addr_hbop+1

drawbaddies:
xor a
ld (gotbaddies),a
ld hl,baddies
ld b,max_baddies
dbadlp:
ld a,h
ld (baddie_screen_addr_hb),a   ;Store the high byte of the array of the current
                               ;baddie, so that the routine to store the screen
                               ;address knows where the array is
push bc
push hl

ld a,(hl)
and a
jp z,dbadlpend
bit 7,a           ;Do not draw the baddie to the screen if bit 7 of the baddie
jp nz,dbadlpend   ;type is set (i.e. the baddie has just been destroyed)

ld (gotbaddies),a
cp 2
jr nz,dbad0

;type 2 (enemy shot)
ld l,baddie_x
ld a,(hl)
ld l,baddie_dx
add a,(hl)
cp num_grid_x-1
call nc,reverse_dir
ld l,baddie_x
ld (hl),a
ld b,a
ld l,baddie_y
ld a,(hl)
ld l,baddie_dy
add a,(hl)
ld l,baddie_y
ld (hl),a
cp num_grid_y-1
jr c,dbad0b
;On the Spectrum, the baddie type is reset to zero, so the shot effectively no
;longer exists. On a CPC, the shot needs to be cleared from the other screen,
;so the resetting must occur during the routine to clear the baddies from the
;screen
jp dbadlpend
dbad0b:
ld c,a

;draw shot (no need to save hl)
;On a Spectrum, the colour of the bullet can be altered simply by changing the
;sprite colour attributes. On a CPC, four separate sprites are required for
;each possible colour (red, magenta, green, cyan)
ld a,(hl)   ;baddie_y
and 3
inc a   ;Convert to a number between 1 and 4
push bc   ;Store the x- and y-coordinates
ld b,a
ld hl,enemyshot_dat-18   ;Each enemy bullet sprite consists of 18 bytes (2 for
ld de,18                 ;the width and height, 16 for the sprite data)
select_enemyshot_type_loop:
add hl,de
djnz select_enemyshot_type_loop
ex de,hl
pop bc   ;Get the x- and y-coordinates
call drawsprgen
call baddie_store_screen_addr
jp dbadlpend


dbad0:
cp 1
jr z,dbad1
cp 3
jr z,dbad1
cp 4
jr nz,dbad2
dbad1:
;types 1, 3, and 4
ld l,baddie_x
ld a,(hl)
ld l,baddie_dx
add a,(hl)
cp num_grid_x-3
call nc,reverse_dir
ld l,baddie_x
ld (hl),a
ld b,a
ld l,baddie_y
ld a,(hl)
ld l,baddie_dy
add a,(hl)
cp num_grid_y-3
call nc,reverse_dir
ld l,baddie_y
ld (hl),a
ld c,a

;draw baddie
;On a Spectrum, there is only one sprite which is used for baddie types 1, 3 and
;4, and each type can be selected simply by altering the sprite colour
;attributes. On a CPC, three separate sprites are required for each possible
;baddie (type 1 is yellow and cyan, type 3 is magenta and red, type 4 is cyan
;and green)
ld l,0
ld a,(hl)
ld de,type134_dat
cp 4
jr nz,dbad1a
ld de,type134_dat+66*2
dbad1a:
cp 3
jr nz,dbad1b
ld de,type134_dat+66
dbad1b:
push hl
call drawsprgen
call baddie_store_screen_addr
pop hl

;type 4 shot
;          if(baddie[f].which==4)
;            {
;            if(lowrand()<800)                                /* shoot? */
;              addnewbaddie(baddie[f].x,baddie[f].y+2,2,1+(lowrand()&1),1,2);
;            }
ld l,0
ld a,(hl)
cp 4
jr nz,nottype4
call rand8
cp 6
jp nc,dbadlpend

ld d,2		;dx (only other difference!)
jr dbad1shot

;types 1 and 3 do shots like this
;            if(lowrand()<100+500*(baddie[f].which-1))         /* shoot? */
;              addnewbaddie(baddie[f].x,baddie[f].y+2,0,1+(lowrand()&1),1,2);
nottype4:
cp 1
ld b,6
jr z,dbad1c
ld b,12
dbad1c:
call rand8
cp b
jp nc,dbadlpend

ld d,0		;dx
dbad1shot:
ld l,baddie_x
ld b,(hl)	;x
inc l
ld c,(hl)	;y
inc b
inc c
inc c
ld a,(seed+1)	;use the other byte of last rand num
and 1
inc a
ld e,a		;dy
call addnewbaddie_auto
dec h
ld (hl),2	;type
jp dbadlpend


dbad2:
cp 10
jr nz,dbad3

ld l,baddie_x
ld a,(hl)
ld l,baddie_dx
add a,(hl)
cp num_grid_x-7
call nc,reverse_dir
ld l,baddie_x
ld (hl),a
ld b,a
ld l,baddie_y
ld a,(hl)
ld l,baddie_dy
add a,(hl)
cp num_grid_y_big-9
call nc,reverse_dir
ld l,baddie_y
ld (hl),a
ld c,a

;draw baddie
push hl
ld de,type10_dat
call drawsprgen
call baddie_store_screen_addr
pop hl

;do shot
;          if(lowrand()<2000)                /* shoot? */
;            {
;            addnewbaddie(baddie[f].x-6,baddie[f].y+8,0,2,1,2);
;            addnewbaddie(baddie[f].x+6,baddie[f].y+8,0,2,1,2);
;            }
call rand8
cp 12
jr nc,dbad2a

push hl

ld l,baddie_x
ld b,(hl)	;x
inc l
ld a,(hl)
add a,9
ld c,a		;y
ld de,2		;dx/dy
call addnewbaddie_auto
dec h
ld (hl),2	;type

;again for the other side
ld a,6
add a,b
ld b,a
call addnewbaddie_auto
dec h
ld (hl),2	;type

pop hl

dbad2a:
;do new enemy
;          if(lowrand()<500)                /* shoot a type 3? */
;            addnewbaddie(baddie[f].x,baddie[f].y+2,(lowrand()&1)?-1:1,
;            				1+(lowrand()&1),3,3);   /* 3 hits */
call rand8
cp 3
jp nc,dbadlpend

ld l,baddie_x
ld a,(hl)
add a,2
ld b,a		;x
inc l
ld a,(hl)
add a,4
ld c,a		;y
ld a,(seed+1)	;use the other byte of last rand num
rra
ld d,1		;dx
jr c,dbad2b
ld d,-1
dbad2b:
and 1
inc a
ld e,a		;dy
ld a,3		;hits
call addnewbaddie_auto
dec h
ld (hl),3	;type
jp dbadlpend


dbad3:
cp 11
jr nz,dbad4

ld l,baddie_x
ld a,(hl)
ld l,baddie_dx
add a,(hl)
cp num_grid_x-11
call nc,reverse_dir
ld l,baddie_x
ld (hl),a
ld b,a
ld l,baddie_y
ld a,(hl)
ld l,baddie_dy
add a,(hl)
cp num_grid_y_big-11
call nc,reverse_dir
ld l,baddie_y
ld (hl),a
ld c,a

;draw baddie
push hl
ld de,type11_dat
call drawsprgen
call baddie_store_screen_addr
pop hl

;do shot
;          if(lowrand()<2000)                /* shoot? */
;            {
;            addnewbaddie(baddie[f].x-24,baddie[f].y+12,0,2,1,2);
;            addnewbaddie(baddie[f].x+24,baddie[f].y+12,0,2,1,2);
;            }
call rand8
cp 8
jr nc,dbad3a

push hl

ld l,baddie_x
ld b,(hl)	;x
inc l
ld a,(hl)
add a,11
ld c,a		;y
ld de,2		;dx/dy
call addnewbaddie_auto
dec h
ld (hl),2	;type

;again for the other side
ld a,10
add a,b
ld b,a
call addnewbaddie_auto
dec h
ld (hl),2	;type

pop hl

dbad3a:
;do new enemy
;          if(lowrand()<200)                /* shoot a type 10!? */
;            addnewbaddie(baddie[f].x-((lowrand()&1)?-15:15),baddie[f].y+8,
;                         (lowrand()&1)?-1:1,1,10,10);   /* 10 hits */
;
call rand8
cp 2
jp nc,dbadlpend

;simplified slightly from the above
ld l,baddie_x
ld a,(hl)
add a,2
ld b,a		;x
inc l
ld a,(hl)
add a,4
ld c,a		;y
ld a,(seed+1)	;use the other byte of last rand num
rra
ld de,0101h	;dx/dy
jr c,dbad3b
ld d,-1
dbad3b:
ld a,10		;hits
call addnewbaddie_auto
dec h
ld (hl),10	;type
jp dbadlpend


dbad4:
;must be type 5 then
ld l,baddie_x
ld a,(hl)
ld l,baddie_dx
add a,(hl)
ld l,baddie_x
ld (hl),a
ld l,baddie_y
ld a,(hl)
ld l,baddie_dy
add a,(hl)
ld l,baddie_y
ld (hl),a

ld l,baddie_x
ld a,(hl)
bit 7,a
jr z,dbad4a

ld (hl),0
ld l,baddie_dx
ld a,(hl)
ld (hl),0
ld l,baddie_dy
ld (hl),a
ld l,baddie_x
ld a,(hl)

dbad4a:
cp num_grid_x-6+1
jr c,dbad4b

ld (hl),num_grid_x-6
ld l,baddie_dx
ld a,(hl)
ld (hl),0
ld l,baddie_dy
ld (hl),a

dbad4b:
ld l,baddie_y
ld a,(hl)
bit 7,a
jr z,dbad4c

ld (hl),0
ld l,baddie_dy
ld a,(hl)
neg
ld (hl),0
ld l,baddie_dx
ld (hl),a
ld l,baddie_y
ld a,(hl)

dbad4c:
cp num_grid_y-6+1
jr c,dbad4d

ld (hl),num_grid_y-6
ld l,baddie_dy
ld a,(hl)
neg
ld (hl),0
ld l,baddie_dx
ld (hl),a

dbad4d:
;draw baddie
ld l,baddie_x
ld b,(hl)
inc l
ld c,(hl)
push hl
ld de,type5_dat
call drawsprgen
call baddie_store_screen_addr
pop hl

;do shot
;          if(lowrand()<500)                /* shoot? */
;            addnewbaddie(baddie[f].x,baddie[f].y+4,0,2,1,2);
call rand8
cp 6
jr nc,dbadlpend

ld l,baddie_x
ld b,(hl)
inc b
inc b		;x
inc l
ld a,(hl)
add a,6
ld c,a		;y
ld de,2		;dx/dy
call addnewbaddie_auto
dec h
ld (hl),2	;type

dbadlpend:
pop hl
pop bc
inc h
dec b
jp nz,dbadlp
ret


;negate dx or dy and fix x or y after addition gave overrun.
;entry: a=x or y, hl points to dx or dy.
reverse_dir:
ex af,af'
ld a,(hl)
neg
ld (hl),a
ex af,af'
add a,(hl)
add a,(hl)
ret


;Store the screen address of a baddie in the table, so that next time the baddie
;sprite is written to the screen, the program knows where the old one is so that
;it can erase it from the screen
baddie_store_screen_addr:
ex de,hl   ;Now DE = screen address
baddie_screen_addr_hbop: ld h,0   ;High byte of baddie array - this is modified
ld l,baddie_screen_addr_40

ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
and &80   ;Reset all bits except bit 7, so A = 0 or &80
rlca   ;Bit 7 now becomes bit 1, so A = 0 (for the entry at &4000) or 2 (for the
rlca   ;entry at &c000)
add a,l
ld l,a   ;HL = address in array to store screen address

ld (hl),e
inc l
ld (hl),d
ret


;Clear a sprite from the screen
;
;Entry conditions - B = x-coordinate (0-31), C = y-coordinate (0-23), DE =
; sprite address containing width and height of area to clear
;Exit conditions - B = 0, DE = byte immediately following height, HL = screen
; address immediately below bottom left of cleared area
; A corrupted
;
;The screen address is calculated using the x- and y-coordinates stored in BC. If
; it is already known, you can call drawsprgen2 instead and skip the screen
; address calculation (this is exploited in the routine to print a string)

clear_sprite_width equ clear_sprite_widthop+1

clear_sprite:
push de   ;Store the sprite address
call calc_screen_addr   ;Calculate the screen address to print the sprite
                        ;(from 0 to &3fff)
pop de   ;Get the sprite address

clear_sprite2:
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
or h   ;Add this to the screen address to print the sprite
ld h,a   ;Now HL = screen address from &4000 to &7fff or &c000 to &ffff
                                   
ld a,(de)   ;Get the width of the sprite in bytes
ld (clear_sprite_width),a   ;Set the width of the sprite in bytes
inc de
ld a,(de)   ;Get the height of the sprite in lines
ld b,a   ;B = height of sprite in lines
inc de

clear_sprite_loopy:
push bc   ;Store the number of lines remaining

clear_sprite_widthop: ld b,0   ;B = width of sprite (modified)
push hl   ;Store the screen address of the start of the current line

clear_sprite_loopx:
ld (hl),0
inc hl   ;Go to the next line for this character

djnz clear_sprite_loopx

pop hl   ;Get the screen address of the start of this line
ld bc,&0800
add hl,bc   ;Set the screen address to the line below
bit 6,h   ;Check if bit 14 of the screen address is reset (which means that the
          ;result of the addition is off the screen and must be re-adjusted)
call z,nextline   ;Calculate the screen address of the next line if necessary

pop bc
djnz clear_sprite_loopy
ret


;Print a sprite using transparent mode (i.e. without erasing the background)
;
;Entry conditions - B = x-coordinate (0-31), C = y-coordinate (0-23), DE =
; sprite address
;Exit conditions - B = 0, DE = byte immediately following end of sprite, HL =
; screen address of top left of sprite
; A corrupted
;
;This routine requires a mask table of 256 bytes (1 for each combination of 2
; pixels) which must be initialised before calling this routine
;The screen address is calculated using the x- and y-coordinates stored in BC. If
; it is already known, you can call drawsprgen2 instead and skip the screen
; address calculation (this is exploited in the routine to print a string)
;Ink 0 is used as the transparent ink

print_sprite_width equ print_sprite_widthop+1
drawsprgen_screen_addr equ drawsprgen_screen_addrop+1

drawsprgen:
push de   ;Store the sprite address
call calc_screen_addr   ;Calculate the screen address to print the sprite
                        ;(from 0 to &3fff)
pop de   ;Get the sprite address

drawsprgen2:
ld a,(bankm)   ;Get the current screen address (either &40 or &c0)
or h   ;Add this to the screen address to print the sprite
ld h,a   ;Now HL = screen address from &4000 to &7fff or &c000 to &ffff
ld (drawsprgen_screen_addr),hl   ;Store the screen address so that it is
                                 ;returned in HL on exiting

ld a,(de)   ;Get the width of the sprite in bytes
ld (print_sprite_width),a   ;Set the width of the sprite in bytes
inc de
ld a,(de)   ;Get the height of the sprite in lines
ld b,a   ;B = height of sprite in lines
inc de

print_sprite_loopy:
push bc   ;Store the number of lines remaining

print_sprite_widthop: ld b,0   ;B = width of sprite (modified)
push hl   ;Store the screen address of the start of the current line

print_sprite_loopx:
ex de,hl   ;Now DE = screen address, HL = sprite address
ld a,(hl)   ;Get the sprite data

push hl   ;Store the sprite address
ld h,mask_table_hb
ld l,a   ;HL = address of entry in mask table
ld c,(hl)   ;C = entry in mask table corresponding to pixel data
pop hl   ;Get the sprite address

ld a,(de)   ;Get the byte that is currently on the screen
and c   ;Apply the mask, so that any pixel that does not contain ink 0 will be
        ;erased, allowing the sprite pixels to be merged with the screen pixels
or (hl)   ;Merge the sprite pixels with the screen pixels

ld (de),a   ;Write this to the screen
ex de,hl   ;Now DE = sprite address, HL = screen address
inc de   ;Go to the next byte of the sprite
inc hl   ;Go to the next byte on the screen

djnz print_sprite_loopx

pop hl   ;Get the screen address of the start of this line
ld bc,&0800
add hl,bc   ;Set the screen address to the line below
bit 6,h   ;Check if bit 14 of the screen address is reset (which means that the
          ;result of the addition is off the screen and must be re-adjusted)
call z,nextline   ;Calculate the screen address of the next line if necessary

pop bc
djnz print_sprite_loopy
drawsprgen_screen_addrop: ld hl,&0000   ;The screen address of the top left of
                                        ;the sprite - this is modified
ret


;Calculate the screen address of a specified x- and y-coordinate
;Entry conditions - B = x-coordinate, C = y-coordinate
;Exit conditions - A = x-coordinate * 2, HL = screen address
; DE corrupted

;This routine requires a screen address table, which stores the addresses of each
;of the lines on the screen, to have been initialised beforehand

calc_screen_addr:
ld de,zxyaddr   ;DE = address of screen address table
ld a,c
rlca   ;Multiply the y-coordinate by 4 (each y-coordinate is separated by 4
rlca   ;lines)

ld h,0
ld l,a   ;HL = y-coordinate * 4
add hl,hl   ;Multiply HL by 2, since each entry in the screen address table uses
            ;2 bytes
add hl,de   ;Now HL = location of screen address of current line
ex de,hl    ;Now DE = location of screen address of current line

;Read the screen address of the current line from the screen address table
;(stored in DE), and store the address in HL

calc_screen_addr_x:
ld a,(de)
ld l,a
inc de
ld a,(de)
ld h,a
ld d,0
ld e,b   ;DE = x-coordinate
add hl,de   ;Now HL = the screen address with the x-coordinate set
ret


;Go to the first line of the next row of the screen
;Entry conditions - HL = screen address
;Exit conditions - BC = &c050, HL = screen address of line below

nextline:
ld bc,&c050
add hl,bc
ret


;Convert a 16-bit decimal number to a string and store it at numtmp:
;
;Entry conditions - DE = number
;Exit conditions - A, BC, DE, HL corrupted
;
;This is the routine used in the Spectrum version, and I have no idea how it
;works; it just does!

gendec:
ld hl,numtmp+4
push hl

gendlp:
push hl

;divide de by 10
ld a,d
ld c,e

ld de,0
ld h,d	;zero
ld l,d
ld b,16

dvlp:
add hl,hl
sla e
rl d

bit 7,a
jr z,dvs1

set 0,l

dvs1:
push hl
push de
ld de,10
and a
sbc hl,de
pop de
jp m,dvs2

;nasty! fiddle the stack
ex (sp),hl

inc de

dvs2:
pop hl

sla c
rla

djnz dvlp

;now de=result, hl=remainder

ld a,l
or 48

pop hl
ld (hl),a
dec hl
ld a,d
or e
jr nz,gendlp

inc hl
ex de,hl
pop hl
set 7,(hl)
ret


;Read the keyboard
;Entry conditions: None
;Exit conditions: HL = address of first byte after keyboard information table
; A, BC corrupted

read_keyboard:
di
ld hl,keyboard_info   ;HL = address of table to store information read from
                      ;the keyboard
ld bc,&f40e
out (c),c   ;Select PSG register 14
ld bc,&f6c0   ;Set the PPI to 'select PSG register' mode (bit 7 = 1, bit 6 = 1)
out (c),c
xor a   ;Set the PPI to inactive mode (bits 7 = 0, bit 6 = 0)
out (c),a

inc b
ld c,&92
out (c),c   ;Set port A to input mode

;At this point, A is still 0
ld c,a
read_keyboard_loop:
set 6,c   ;Set the PPI to 'read PSG register' mode (bit 7 = 0, bit 6 = 1), as
          ;well as setting the matrix line to read
ld b,&f6
out (c),c   ;Send the matrix line to read, and the mode, to port C

ld b,&f4
in a,(c)   ;Read the line in the keyboard matrix
ld (hl),a   ;Store it in the keyboard information table
inc hl   ;Go to the next entry in the table

inc c   ;Go to the next line in the keyboard matrix
ld a,c
and &0f
cp 10   ;There are 10 lines to scan
jr nz,read_keyboard_loop

ld bc,&f600   ;Set the PPI to inactive mode (bits 7 = 0, bit 6 = 0)
out (c),c

inc b
ld c,&82   ;Set port A to output mode (the default)
out (c),c

ei
ret

keyboard_info: defs 10   ;The keyboard consists of an 8 x 10 matrix, so 80 bits
                         ;(i.e. 10 bytes) are required


;high byte of address to draw screen (&4000 or &c000)
bankm defb &40
;high byte of CRTC register 12, which determines the screen to display (&10 =
;&4000; &30 = &c000)
banks defb &10

;The random number seed, which is 4 bytes long
seed:

;lookup table for pixel line addresses is generated here, which is 400 bytes
;long on the CPC
zxyaddr equ seed+4

;The character table containing the bit data for each ASCII character, which is
;converted to MODE 0 in order to write the characters to the screen; each entry
;uses 8 bytes, so with 96 characters, the table is 768 bytes long
;char_table equ zxyaddr+400
