#ifndef bbcSOUND_H_
#define bbcSOUND_H_

#include <vector>
#include <deque>

//#define bbcSOUND_SOUND_CYCLES() (BBCMicro::cycles&(~7))
#define bbcSOUND_CYCLES() (bbcComputer::cycles/8)
#define bbcFIXED_OF(N) (fixedpoint((N)*256))
#define bbcFLOAT_OF(N) ((N)/256.f)

struct bbcSound {
public:
	typedef int fixedpoint;

	static void Init();
	
	static void Write(t65::byte v);

	static void SetFrequency(int hz);

	static void Render8BitMono(void **buffers,unsigned *buffer_sizes,unsigned num_buffers);
	static void Render16BitMono(void **buffers,unsigned *buffer_sizes,unsigned num_buffers);
	static void Render8BitStereo(void **buffers,unsigned *buffer_sizes,unsigned num_buffers);
	static void Render16BitStereo(void **buffers,unsigned *buffer_sizes,unsigned num_buffers);

	static void SetChannelStereoLeftRight(int channel,bool is_left,bool is_right);

	static bool IsRecording();
	static void StartRecording();
	static void StopRecording();
	static void GetRecordedVgmData(std::vector<t65::byte> *vgm_data);
private:
	struct SoundWriteEntry {
		t65::byte val;
		int cycles;
	};
	
	static bool is_recording_;
	static std::deque<SoundWriteEntry> recorded_writes_;

	//Sound queue -- built up while emulator runs, flushed when data is rendered.
	//
	//maximum output to sound chip is 250000Hz
	enum {
		max_num_writes=20000,
	};

	static unsigned write_idx_;
	static SoundWriteEntry writes_[max_num_writes+1];
	static int last_render_at_;

	//Current sound state
	//Not externally visible -- just used by the renderer

	//Latch
	static t65::byte latch_;
	struct SoundRegister {
		t65::word vol;
		t65::word pitch;
		fixedpoint counter;
		int mul;
	};
	static SoundRegister cur_regs_[4];

	//Whether this channel is stereo left
	static bool is_stereo_left_[4];

	//Whether this channel is stereo right
	static bool is_stereo_right_[4];
	
	//Noise generator
	static t65::word noise_seed_;
	static int noise_bit_;//The bit currently being output.

	//Updates current settings as if byte 'v' were just written
	static void DoByte(t65::byte v);

	//value in Hz given tone register setting
	static int hz_from_tone_[1024];

	//static fixedpoint freq_mul_;

	static fixedpoint samples_per_switch_[1024];

	//Gets next noise value, updates seed & all the rest.
	static int NextNoiseBit();

	//Current frequency
	static int frequency_;

	struct Render8BitTraits {
		typedef t65::byte sample_type;
		enum {
			nchan=1,
			silence=0x80,
			sample_min=0,
			sample_max=255,
		};
		static sample_type volumes[16];
	};
	struct Render16BitTraits {
		typedef t65::int16 sample_type;
		enum {
			nchan=1,
			silence=0x0000,
			sample_min=-32767,
			sample_max=32767,
		};
		static sample_type volumes[16];
	};
	
	template<class T>
	static void RenderMono(void **buffers,unsigned *buffer_sizes_bytes,
		unsigned num_buffers,const T *)
	{
		typedef t65TYPENAME T::sample_type sample_type;
		const unsigned bytes_per_sample=sizeof(sample_type);
		unsigned i;
		unsigned widx=0;
		unsigned cur_sample=0;
		unsigned cur_buffer=0;
		unsigned total_size_samples=0;
		int soundcycles=bbcSOUND_CYCLES()-last_render_at_;
		if(soundcycles==0) {
			return;
		}

		//Get total number of samples needed
		for(i=0;i<num_buffers;++i) {
			BASSERT(buffer_sizes_bytes[i]%bytes_per_sample==0);
			total_size_samples+=buffer_sizes_bytes[i]/bytes_per_sample;
		}
		float soundcycles_per_sample=soundcycles/float(total_size_samples);
#ifdef CRAPPY_DEBUG
	//	printf("soundcycles_per_byte=%f\n",soundcycles_per_byte);
#endif
		float this_c=float(last_render_at_);
		for(i=0;i<total_size_samples;++i) {
			while(widx<write_idx_&&writes_[widx].cycles<int(this_c)) {
				DoByte(writes_[widx].val);
				++widx;
			}

			//Mix tone channels
			int samp=0;
			for(unsigned j=0;j<3;++j) {
				SoundRegister *reg=&cur_regs_[j];
				fixedpoint sps=samples_per_switch_[reg->pitch&1023];
				if(sps==bbcFIXED_OF(0)) {
					samp+=T::volumes[reg->vol&0xf];
				} else {
					if(is_stereo_left_[j]||is_stereo_right_[j]) {
						samp+=T::silence+reg->mul*T::volumes[reg->vol&0xf];
					}
					reg->counter+=bbcFIXED_OF(1);
					if(reg->counter>=sps) {
						reg->mul=-reg->mul;
						reg->counter-=sps;
					}
				}
			}

			//Mix noise channel
			{
				SoundRegister *reg=&cur_regs_[3];
				fixedpoint stop;
				switch(reg->pitch&3) {
				case 0:
					//High freq
					stop=0x10;
					break;
				case 1:
					//Mid freq
					stop=0x20;
					break;
				case 2:
					//Low freq
					stop=0x40;
					break;
				case 3:
					//Tone 2 freq
					stop=cur_regs_[2].pitch&1023;
					break;
				}
				stop=samples_per_switch_[stop];
				
				reg->counter+=bbcFIXED_OF(1);
				if(reg->counter>=stop) {
					reg->mul=-reg->mul;
					if(reg->mul>0) {
						//A new bit is output just once for every 2 transitions.
						if(reg->pitch&4) {
							//White noise
							noise_bit_=NextNoiseBit();
						} else {
							//Periodic noise -- outputs bottom seed bit, and
							//the seed is rotated. (15bit according to Kortink, 16bit according
							//to Maxim, I'm using 15bit here.)
							//
							noise_bit_=noise_seed_&1;
							noise_seed_=((noise_seed_>>1)|(noise_seed_<<14))&0x7FFF;
						}
					}
					reg->counter-=stop;//bbcFIXED_OF(0);
				}
				//Apparently, the noise output is either 0 or +1 rather than -1/+1.
				if(is_stereo_left_[3]||is_stereo_right_[3]) {
					samp+=T::silence+noise_bit_*T::volumes[reg->vol&0xf];
				}
			}

			//Put in buffer
			BASSERT(samp/4>=T::sample_min&&samp/4<=T::sample_max);
			static_cast<sample_type *>(buffers[cur_buffer])[cur_sample]=sample_type(samp/4);
			this_c+=soundcycles_per_sample;
			++cur_sample;
			if(cur_sample*bytes_per_sample>=buffer_sizes_bytes[cur_buffer]) {
				BASSERT(cur_buffer<num_buffers);
				++cur_buffer;
				cur_sample=0;
			}
		}

		//Flush remaining writes.
		//There shouldn't be any, but one or two might hang around due to inaccuracies
		//or something.
		while(widx<write_idx_) {
			DoByte(writes_[widx].val);
			++widx;
		}
		last_render_at_=bbcSOUND_CYCLES();
		write_idx_=0;
	}

	template<class T>
	static void RenderStereo(void **buffers,unsigned *buffer_sizes_bytes,
		unsigned num_buffers,const T *)
	{
		typedef t65TYPENAME T::sample_type sample_type;
		const unsigned bytes_per_sample=sizeof(sample_type);
		unsigned i;
		unsigned widx=0;
		unsigned cur_sample=0;
		unsigned cur_buffer=0;
		unsigned total_size_samples=0;
		int soundcycles=bbcSOUND_CYCLES()-last_render_at_;
		if(soundcycles==0) {
			return;
		}

		//Get total number of samples needed
		for(i=0;i<num_buffers;++i) {
			BASSERT(buffer_sizes_bytes[i]%bytes_per_sample==0);
			total_size_samples+=buffer_sizes_bytes[i]/bytes_per_sample/2;
		}
		float soundcycles_per_sample=soundcycles/float(total_size_samples);
#ifdef CRAPPY_DEBUG
	//	printf("soundcycles_per_byte=%f\n",soundcycles_per_byte);
#endif
		float this_c=float(last_render_at_);
		for(i=0;i<total_size_samples;++i) {
			while(widx<write_idx_&&writes_[widx].cycles<int(this_c)) {
				DoByte(writes_[widx].val);
				++widx;
			}

			//Mix tone channels
			int left=0,right=0;
			for(unsigned j=0;j<3;++j) {
				int samp;
				SoundRegister *reg=&cur_regs_[j];
				fixedpoint sps=samples_per_switch_[reg->pitch&1023];
				if(sps==bbcFIXED_OF(0)) {
					samp=T::volumes[reg->vol&0xf];
				} else {
					samp=T::silence+reg->mul*T::volumes[reg->vol&0xf];
					reg->counter+=bbcFIXED_OF(1);
					if(reg->counter>=sps) {
						reg->mul=-reg->mul;
						reg->counter-=sps;
					}
				}
				if(is_stereo_left_[j]) {
					left+=samp;
				}
				if(is_stereo_right_[j]) {
					right+=samp;
				}
			}

			//Mix noise channel
			{
				SoundRegister *reg=&cur_regs_[3];
				fixedpoint stop;
				switch(reg->pitch&3) {
				case 0:
					//High freq
					stop=0x10;
					break;
				case 1:
					//Mid freq
					stop=0x20;
					break;
				case 2:
					//Low freq
					stop=0x40;
					break;
				case 3:
					//Tone 2 freq
					stop=cur_regs_[2].pitch&1023;
					break;
				}
				stop=samples_per_switch_[stop];
				
				reg->counter+=bbcFIXED_OF(1);
				if(reg->counter>=stop) {
					reg->mul=-reg->mul;
					if(reg->mul>0) {
						//A new bit is output just once for every 2 transitions.
						if(reg->pitch&4) {
							//White noise
							noise_bit_=NextNoiseBit();
						} else {
							//Periodic noise -- outputs bottom seed bit, and
							//the seed is rotated. (15bit according to Kortink, 16bit according
							//to Maxim, I'm using 15bit here.)
							//
							noise_bit_=noise_seed_&1;
							noise_seed_=((noise_seed_>>1)|(noise_seed_<<14))&0x7FFF;
						}
					}
					reg->counter-=stop;//bbcFIXED_OF(0);
				}
				//Apparently, the noise output is either 0 or +1 rather than -1/+1.
				int samp=T::silence+noise_bit_*T::volumes[reg->vol&0xf];
				if(is_stereo_left_[3]) {
					left+=samp;
				}
				if(is_stereo_right_[3]) {
					right+=samp;
				}
			}

			//Put in buffer
			BASSERT(left/4>=T::sample_min&&left/4<=T::sample_max);
			BASSERT(right/4>=T::sample_min&&right/4<=T::sample_max);
			sample_type *buffer=static_cast<sample_type *>(buffers[cur_buffer]);
			buffer[cur_sample++]=sample_type(left/4);
			buffer[cur_sample++]=sample_type(right/4);
			if(cur_sample*bytes_per_sample>=buffer_sizes_bytes[cur_buffer]) {
				BASSERT(cur_buffer<num_buffers);
				++cur_buffer;
				cur_sample=0;
			}
		}

		//Flush remaining writes.
		//There shouldn't be any, but one or two might hang around due to inaccuracies
		//or something.
		while(widx<write_idx_) {
			DoByte(writes_[widx].val);
			++widx;
		}
		last_render_at_=bbcSOUND_CYCLES();
		write_idx_=0;
	}
};

inline void bbcSound::Render8BitMono(void **buffers,unsigned *buffer_sizes,
	unsigned num_buffers)
{
	RenderMono(buffers,buffer_sizes,num_buffers,static_cast<Render8BitTraits *>(0));
}

inline void bbcSound::Render8BitStereo(void **buffers,unsigned *buffer_sizes,
	unsigned num_buffers)
{
	RenderStereo(buffers,buffer_sizes,num_buffers,static_cast<Render8BitTraits *>(0));
}

inline void bbcSound::Render16BitMono(void **buffers,unsigned *buffer_sizes,
	unsigned num_buffers)
{
	RenderMono(buffers,buffer_sizes,num_buffers,static_cast<Render16BitTraits *>(0));
}

inline void bbcSound::Render16BitStereo(void **buffers,unsigned *buffer_sizes,
										unsigned num_buffers)
{
	RenderStereo(buffers,buffer_sizes,num_buffers,static_cast<Render16BitTraits *>(0));
}

#endif
