/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

	bitmap.cpp
	==========

	This file makes up my 'bitmap' class.
	
	Contains :

		routines for creating and destroying a bitmap
		routines for pixel access, and saving out to a PCX file

	Bibliography :

		the DirectDraw and Allegro readme docs, and the SVGALib man page

*/
#include "gfx.h"
#include <sys/stat.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

C_bitmap::C_bitmap(void)
{
	surface = NULL;
}

C_bitmap::~C_bitmap(void)
{
	Destroy();
}

bool C_bitmap::Destroy(void)
{
	if(surface)
	{
		#ifdef TARGET_WIN32
		surface->Release();
		#endif

		#ifdef TARGET_SVGALIB
		free(surface);
		#endif

		#ifdef TARGET_ALLEGRO
		//crashes here, don't know why (yet)
		destroy_bitmap(surface);
		#endif

		surface = NULL;
	}

	return false;
}

bool C_bitmap::Create(C_gfx_api &apiref, int w, int h)
{
	#ifdef TARGET_WIN32
	DDRAW_INIT_STRUCT(sdesc);

	sdesc.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH;
	sdesc.dwHeight = h;
	sdesc.dwWidth = w;
	sdesc.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;

	if(FAILED(apiref.lpdd4->CreateSurface(&sdesc, &surface, NULL)))
		return true;

	sdesc.dwFlags = 0;
	#endif

	#ifdef TARGET_ALLEGRO
	if(surface = create_system_bitmap(w, h))
		clear(surface);
	#endif

	#ifdef TARGET_SVGALIB
	surface = (unsigned __int8 *)malloc(w*h);
	pitch = w;
	#endif

	api = &apiref;

	width = w;
	height = h;

	return false;
}

void C_bitmap::Blit(void) // blits to the 'centre' of the screen, or stretches as appropriate
{
	#ifdef TARGET_WIN32
	HRESULT res;

	if(api->slevel >= api->rslevel)
	{
		res = api->front4->Blt(&api->StretchArea, surface, NULL, DDBLT_WAIT, NULL);

		if(res == DDERR_SURFACELOST)
		{
			api->front4->Restore();
			api->front4->Blt(&api->StretchArea, surface, NULL, DDBLT_WAIT, NULL);
		}
	}
	else
	{
		res = api->front4->BltFast(api->tx, api->ty, surface, NULL, DDBLTFAST_NOCOLORKEY | DDBLTFAST_WAIT);

		if(res == DDERR_SURFACELOST)
		{
			api->front4->Restore();
			api->front4->BltFast(api->tx, api->ty, surface, NULL, DDBLTFAST_NOCOLORKEY | DDBLTFAST_WAIT);
		}
	}
	#endif

	#ifdef TARGET_ALLEGRO
	if(api->slevel >= api->rslevel)
		stretch_blit(surface, screen, 0, 0, surface->w, surface->h, api->StretchArea.left, api->StretchArea.top, api->StretchArea.right-api->StretchArea.left, api->StretchArea.bottom-api->StretchArea.top);
	else
		blit(surface, screen, 0, 0, api->tx, api->ty, surface->w, surface->h);
	#endif

	#ifdef TARGET_SVGALIB
	int c, pos, scy, addy, y;

	if(api->slevel >= api->rslevel)
	{
		c = api->StretchArea.bottom-api->StretchArea.top;
		addy = (height << 16) / c;
		scy = api->StretchArea.top;

		pos = 0;
		y = 0;
		while(c--)
		{
			vga_drawscansegment(&surface[pos], api->tx, scy, width);
			scy++;

			y += addy;
			if(y>> 16)
			{
				pos += pitch;
				y -= 65536;
			}
		}
	}
	else
	{
		scy = api->ty;
		c = height;

		pos = 0;
		y = 0;
		while(c--)
		{
			vga_drawscansegment(&surface[pos], api->tx, scy, width);
			scy++;
			pos += pitch;
		}
	}

	#endif
}

bool C_bitmap::BeginDraw(void)
{
	#ifdef TARGET_ALLEGRO
	acquire_bitmap(surface);
	bmp_select(surface);
	pitch = surface->w;
	return true;
	#endif

	#ifdef TARGET_SVGALIB
	return true;
	#endif

	#ifdef TARGET_WIN32
	HRESULT res;

	res = surface->Lock(NULL, &sdesc, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR | DDLOCK_WRITEONLY | DDLOCK_NOSYSLOCK, NULL);

	if(res == DDERR_SURFACELOST)
	{
		surface->Restore();
		res = surface->Lock(NULL, &sdesc, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR | DDLOCK_WRITEONLY | DDLOCK_NOSYSLOCK, NULL);
	}

	if(res != DD_OK)
		return false;
	else
	{
		pitch = sdesc.lPitch;
		base = (unsigned __int8 *)sdesc.lpSurface;
		return true;
	}
	#endif
}

bool C_bitmap::BeginRead(void)
{
	#ifdef TARGET_ALLEGRO
	acquire_bitmap(surface);
	bmp_select(surface);
	pitch = surface->w;
	return true;
	#endif

	#ifdef TARGET_SVGALIB
	return true;
	#endif

	#ifdef TARGET_WIN32
	HRESULT res;

	res = surface->Lock(NULL, &sdesc, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR | DDLOCK_READONLY | DDLOCK_NOSYSLOCK, NULL);

	if(res == DDERR_SURFACELOST)
	{
		surface->Restore();
		res = surface->Lock(NULL, &sdesc, DDLOCK_WAIT | DDLOCK_SURFACEMEMORYPTR | DDLOCK_READONLY | DDLOCK_NOSYSLOCK, NULL);
	}

	if(res != DD_OK)
		return false;
	else
	{
		pitch = sdesc.lPitch;
		base = (unsigned __int8 *)sdesc.lpSurface;
		return true;
	}
	#endif
}

void C_bitmap::EndOperation(void)
{
	#ifdef TARGET_ALLEGRO
	release_bitmap(surface);
	#endif

	#ifdef TARGET_WIN32
	surface->Unlock(NULL);
	#endif
}

unsigned __int8 *C_bitmap::GetLinePtr(int y)
{
	#ifdef TARGET_ALLEGRO
	return (unsigned __int8 *)bmp_write_line(surface, y);
	#endif

	#ifdef TARGET_WIN32
	return base + (y*pitch);
	#endif

	#ifdef TARGET_SVGALIB
	return surface + (y*pitch);
	#endif
}

void C_bitmap::ReleaseLinePtr(void)
{
	#ifdef TARGET_ALLEGRO
	bmp_unwrite_line(surface);
	#endif
}

void C_bitmap::DumpToFile(char *name)
{
	bool mallocd = false, found = false;
	struct stat r;
	int c;
	FILE *outpcx;
	unsigned char col, count;
	unsigned short pitch, wval;
	unsigned char rubbish[54], *vptr;
	int x, y, tc;

	if(!name)
	{
		mallocd = true;
		name = (char *)malloc(13);
		c = 0;

		do
		{
			sprintf(name, "snap%04d.pcx", c);
			c++;
		} while(!stat(name, &r));
	}

	memset(rubbish, 0, 54);

	if(outpcx = fopen(name, "wb"))
	{
		fputc(10, outpcx);
		fputc(5, outpcx);
		fputc(1, outpcx);
		fputc(8, outpcx);

		pitch = 0; fwrite(&pitch, 2, 1, outpcx);
		pitch = 0; fwrite(&pitch, 2, 1, outpcx);
		pitch = width-1; fwrite(&pitch, 2, 1, outpcx);
		pitch = height-1; fwrite(&pitch, 2, 1, outpcx);

		///assume a 15" screen	=> 15^2 = x^2 + y^2, and x = 4*y/3
		//						=> 15^2 = (4*y/3)^2 + y^2
		//						=> 225 = (25/9)y^2
		//						=> y = 9.581231654 => x = 11.54123044
		pitch = (int)((float)width / 11.54123044f);
		fwrite(&pitch, 2, 1, outpcx);
		pitch = (int)((float)height / 9.581231654f);
		fwrite(&pitch, 2, 1, outpcx);

		fwrite(rubbish, 1, 48, outpcx); //16 colour palette
		fputc(0, outpcx); //reserved

		//65 to here

		fputc(1, outpcx);
		wval = width; //should check width is a power of two really
		fwrite(&wval, 2, 1, outpcx);

		wval = 1;
		fwrite(&wval, 2, 1, outpcx);

		wval = width;
		fwrite(&wval, 2, 1, outpcx);
		wval = height;
		fwrite(&wval, 2, 1, outpcx);
		
		fwrite(rubbish, 1, 54, outpcx); //header padding

		BeginRead();
		for(y = 0; y < height; y++)
		{
			tc = 0;
			vptr = GetLinePtr(y);
			x = width;
			while(tc < width)
			{
				count = 0;
				col = *vptr;

				while((count < 63) && x && (*vptr == col))
				{
					x--;
					count++;
					vptr++;
				}

				tc += count;

				if((count > 1) || (col&192))
				{
					count |= 192;

					fputc(count, outpcx);
					fputc(col, outpcx);
				}
				else
					fputc(col, outpcx);
			}
		}
		EndOperation();

		//palette now
		putc(12, outpcx);
		for(x = 0; x < 256; x++)
		{
			#ifdef TARGET_WIN32
			putc(api->cols[x].peRed, outpcx);
			putc(api->cols[x].peGreen, outpcx);
			putc(api->cols[x].peBlue, outpcx);
			#endif

			#ifdef TARGET_ALLEGRO
			putc(api->cols[x].r << 2, outpcx);
			putc(api->cols[x].g << 2, outpcx);
			putc(api->cols[x].b << 2, outpcx);
			#endif
		}

		fclose(outpcx);
	}

	if(mallocd)
		free(name);
}
