#ifndef GENERIC6502_H_
#define GENERIC6502_H_

#include "t65.h"
//#include <typeinfo>

//Define to use table of NZ settings rather than calculating by hand.
//(Probably bad because it exercises the data cache.)
//#define t65_USE_NZTABLE

//Set V flag like BeebEm in ADC/SBC
#define t65_BEEBEM_VFLAG

//Define to use table of function pointers rather than switch() case for each opcode.
#define t65_USE_FUNCTION_TABLE

#ifdef t65_USE_FUNCTION_TABLE
#include "t65OpcodeTable.h"
#endif

namespace t65 {
	template<class ConfigType>
	struct Generic6502 {
	public://Was previously protected, seems vc.net doesn't like that though? no idea why, am sure it should work that way.
		// These are used as the last parameter of the WriteByte/ReadByte
		// functions.
		// TODO	make these typedefs, change WriteByte/ReadByte to take unnamed
		//		defaulted const &, sod visual C++ (not a moment too soon).
		static const t65TYPENAME ConfigType::ReadZP *catReadZP;
		static const t65TYPENAME ConfigType::WriteZP *catWriteZP;
		static const t65TYPENAME ConfigType::ReadStack *catReadStack;
		static const t65TYPENAME ConfigType::WriteStack *catWriteStack;
		static const t65TYPENAME ConfigType::FetchInstr *catFetchInstr;
		static const t65TYPENAME ConfigType::FetchAddress *catFetchAddress;
		static const t65TYPENAME ConfigType::ReadOperand *catReadOperand;
		static const t65TYPENAME ConfigType::WriteOperand *catWriteOperand;
		static const t65TYPENAME ConfigType::ReadDebug *catReadDebug;
		static const t65TYPENAME ConfigType::WriteDebug *catWriteDebug;
	protected:
		//////////////////////////////////////////////////////////////////////////
		// weird null execute class
		//
		// TODO work out what this is for :-/
		template<class T>
		struct NullExecute {
			static FINLINE void Execute(T &) {
			}

			static FINLINE void Execute(const T &) {
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Disassembly helpers
		static char *AddHexByteChars(char *buf,byte val) {
			static const char hex_digits[]="0123456789ABCDEF";

			*buf++=hex_digits[val>>4];
			*buf++=hex_digits[val&0xf];
			*buf=0;
			return buf;
		}

		static char *AddHexByte(char *buf,byte val) {
			*buf++='&';
			return AddHexByteChars(buf,val);
		}

		static char *AddHexWord(char *buf,Word val) {
			*buf++='&';
			buf=AddHexByteChars(buf,val.h);
			buf=AddHexByteChars(buf,val.l);
			return buf;
		}

		static char *AddIndex(char *buf,char reg) {
			*buf++=',';
			*buf++=reg;
			*buf=0;
			return buf;
		}

		//////////////////////////////////////////////////////////////////////////
		// oh my poor fingers.
		typedef ConfigType C;
	public:
		//////////////////////////////////////////////////////////////////////////
		// useful(ish) user accesible typedef
		typedef ConfigType Config;
		typedef t65TYPENAME ConfigType::MachineType MachineType;
		
		//////////////////////////////////////////////////////////////////////////
		// pointer to an execute function for an instruction
		typedef void (*InstrExecuteFn)(MachineType &state);
		
		//////////////////////////////////////////////////////////////////////////
		//
		static void EmulatorBreak() {
			WriteHistory();
			SoftBreak();
		}
		//////////////////////////////////////////////////////////////////////////
		//
		static byte DebugRead(MachineType &state,word ptr) {
			Word tmp;
			tmp.w=ptr;
			return state.ReadByte(tmp,catReadDebug);
		}

		static void DebugWrite(MachineType &state,word ptr,byte val) {
			Word tmp;
			tmp.w=ptr;
			state.WriteByte(tmp,val,catWriteDebug);
		}

		//////////////////////////////////////////////////////////////////////////
		// Does the common stages of an interrupt -- the last 5 cycles
		//
		// suitable for brk, nmi, irq, reset.
		static FINLINE void Interrupt(MachineType &state,Word ptr) {
			Push(state,state.cpustate.pc.h);
			++state.cycles;
			Push(state,state.cpustate.pc.l);
			++state.cycles;
			Push(state,state.cpustate.p);
			++state.cycles;
			state.cpustate.pc.l=state.ReadByte(ptr,catFetchAddress);
			++state.cycles;
			++ptr.w;
			state.cpustate.pc.h=state.ReadByte(ptr,catFetchAddress);
			++state.cycles;
		}

		//////////////////////////////////////////////////////////////////////////
		// nztable[i] has the Z bit set if i is zero, and the N bit set if i
		// is negative.
		//
#ifdef t65_USE_NZTABLE
		static byte nztable[256];
#endif

		//////////////////////////////////////////////////////////////////////////
		// The vectors
		static const Word nmivector;
		static const Word resetvector;
		static const Word irqvector;
		
		//////////////////////////////////////////////////////////////////////////
		// IRQ
		static FINLINE unsigned IRQ(MachineType &state) {
			if(!(state.cpustate.p&I_MASK)) {
				state.cycles+=2;//I don't know what happens in these 2 cycles
				state.cpustate.p&=~B_MASK;//clear B bit (it's an IRQ)
				Interrupt(state,irqvector);
	//			state.cpustate.p|=B_MASK;//reinstate B bit (it's always set)
				//I'm pretty sure this is required, otherwise the 6502 will keep getting interrupted.
				state.cpustate.p|=I_MASK;
				return 7;
			}
			return 0;
		}

		//////////////////////////////////////////////////////////////////////////
		// NMI
		static FINLINE void NMI(MachineType &state) {
			//state.cpustate.pc.w+=2;//I don't know what happens in these 2 cycles
			state.cycles+=2;//I think this is what I meant :-) [1/3/2003]
			Interrupt(state,nmivector);

			//64doc says this isn't the case, but VICE does it -- Opus DDOS at least
			//doesn't disable interrupts in its NMI handler, and I don't see how that
			//can be safe unless NMI sets I, particularly as interrupt timings are
			//somewhat quantized under emulation
			state.cpustate.p|=I_MASK;
		}
		
		//////////////////////////////////////////////////////////////////////////
		// Pushes a value on to the stack
		static FINLINE void Push(MachineType &state,byte val) {
			state.WriteByte(state.cpustate.s,val,catWriteStack);
			--state.cpustate.s.l;
		}

		//////////////////////////////////////////////////////////////////////////
		// Pops the value from the stack
		static FINLINE byte Pop(MachineType &state) {
			++state.cpustate.s.l;
			return state.ReadByte(state.cpustate.s,catReadStack);
		}	

		//////////////////////////////////////////////////////////////////////////
		// ORs in the N+Z settings for the given value. Doesn't clear old values.
		// This way can call this from other code and change from nztable to
		// something else as desired.
		static FINLINE void DoOrNZFlags(MachineType &state,byte val) {
#ifdef t65_USE_NZTABLE
			state.cpustate.p|=nztable[val];
#else
			state.cpustate.p|=(val==0)<<1;//note true/false thingy
			state.cpustate.p|=val&0x80;
#endif
		}
		
		//////////////////////////////////////////////////////////////////////////
		// set state's n+z based on the value 'val'
		static FINLINE void SetNZ(MachineType &state,byte val) {
			state.cpustate.p&=~(N_MASK|Z_MASK);
			DoOrNZFlags(state,val);
		}

		//////////////////////////////////////////////////////////////////////////
		// Loads a value into a register, setting the flags as it does so
		static FINLINE void LoadRegister(MachineType &state,byte &dest,byte src) {
			dest=src;
			SetNZ(state,dest);
		}

		//////////////////////////////////////////////////////////////////////////
		//
		static void InitialiseCPUState(MachineType &state);

		//////////////////////////////////////////////////////////////////////////
		// Reset
		static FINLINE void Reset(MachineType &state) {
			//state.cpustate.pc.w+=2;//I don't know what happens in these 2 cycles
			state.cycles+=2;//I think this is what I meant :-) [15/9/2002]
			Interrupt(state,resetvector);
			state.cpustate.p|=I_MASK;
		}

		//TODO vc.net fixes, take these lines out.
		//////////////////////////////////////////////////////////////////////////
		// resetvector
//		template<class InstructionSet>
//		static int Run6502(MachineType &state,int maxcycles,const InstructionSet &);
		
//		template<class InstructionSet>
//		static void RunSingleInstruction(MachineType &state,const InstructionSet &);

		// er... can't remember what this was for.
	//	static int Run6502New(MachineType &state,int maxcycles);

		//////////////////////////////////////////////////////////////////////////
		// Set up the nztable, and anything else required.
		static void Init();

		//////////////////////////////////////////////////////////////////////////
		// Addressing modes
		//
		struct ModeIMM {
			enum {operand_size=1,};
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				*buf++='#';
				return AddHexByte(buf,bytes[0]);
			}

			//My guess is it takes no time to calculate the EA for an
			//immediate instruction...
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word old=state.cpustate.pc;
				++state.cpustate.pc.w;
				return old;
			}

			static FINLINE byte ReadOperand(MachineType &state,Word addr) {
				byte b=state.ReadByte(addr,catFetchInstr);
				++state.cycles;
				return b;
			}
		};

		struct ModeZPG {
			enum {operand_size=1,};
			static char *Disassemble(char *buf,const byte *bytes,Word=Word()) {
				return AddHexByte(buf,bytes[0]);
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word addr;

				addr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				return addr;
			}

			static FINLINE Word WriteModeEA(MachineType &state) {
				return ReadModeEA(state);
			}

			static FINLINE byte ReadOperand(MachineType &state,Word addr) {
				byte b=state.ReadByte(addr,catReadZP);
				++state.cycles;
				return b;
			}

			static FINLINE void WriteOperand(MachineType &state,Word addr,byte val) {
				state.WriteByte(addr,val,catWriteZP);
				++state.cycles;
			}
		};

		struct ModeZPX:
		public ModeZPG
		{
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				return AddIndex(ModeZPG::Disassemble(buf,bytes),'X');
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word addr;
				
				addr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				state.ReadByte(addr,catReadZP);
				addr.l+=state.cpustate.x;
				++state.cycles;
				return addr;
			}
			
			static FINLINE Word WriteModeEA(MachineType &state) {
				return ReadModeEA(state);
			}
		};

		struct ModeZPY:
		public ModeZPG
		{
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				return AddIndex(ModeZPG::Disassemble(buf,bytes),'Y');
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word addr;
				
				addr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				state.ReadByte(addr,catReadZP);
				addr.l+=state.cpustate.y;
				++state.cycles;
				return addr;
			}
			
			static FINLINE Word WriteModeEA(MachineType &state) {
				return ReadModeEA(state);
			}
		};
		
		struct ModeABS {
			enum {operand_size=2,};
			static char *Disassemble(char *buf,const byte *bytes,Word=Word()) {
				return AddHexWord(buf,MakeWord(bytes[0],bytes[1]));
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word addr;
				
				addr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				return addr;
			}

			static FINLINE Word WriteModeEA(MachineType &state) {
				return ReadModeEA(state);
			}

			static FINLINE byte ReadOperand(MachineType &state,Word addr) {
				byte b=state.ReadByte(addr,catReadOperand);
				++state.cycles;
				return b;
			}

			static FINLINE void WriteOperand(MachineType &state,Word addr,byte val) {
				state.WriteByte(addr,val,catWriteOperand);
				++state.cycles;
			}
		};

		struct ModeABX:
		public ModeABS
		{
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				return AddIndex(ModeABS::Disassemble(buf,bytes),'X');
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word addr,newaddr;
				addr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				newaddr.w=addr.l+state.cpustate.x;
				addr.l=newaddr.l;
				++state.cpustate.pc.w;
				++state.cycles;
				if(newaddr.h) {
					state.ReadByte(addr,catReadOperand);
					++addr.h;
					++state.cycles;
				}
				return addr;
			}
			static FINLINE Word WriteModeEA(MachineType &state) {
				Word addr,newaddr;
				
				addr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				newaddr.w=addr.w;
				newaddr.l+=state.cpustate.x;
				++state.cpustate.pc.w;
				++state.cycles;
				state.ReadByte(newaddr,catReadOperand);
				newaddr.w=addr.w+state.cpustate.x;
				++state.cycles;
				return newaddr;
			}
		};
		
		struct ModeABY:
		public ModeABS
		{
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				return AddIndex(ModeABS::Disassemble(buf,bytes),'Y');
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word addr,lowbyteresult;
				addr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				lowbyteresult.w=addr.l+state.cpustate.y;//NOTE 9/4/2003 eliminate C4244
				addr.l=lowbyteresult.l;
				++state.cpustate.pc.w;
				++state.cycles;
				if(lowbyteresult.h) {
					state.ReadByte(addr,catReadOperand);
					++addr.h;
					++state.cycles;
				}
				return addr;
			}

			static FINLINE Word WriteModeEA(MachineType &state) {
				Word addr,newaddr;
				
				addr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				newaddr.w=addr.w;
				newaddr.l+=state.cpustate.y;
				++state.cpustate.pc.w;
				++state.cycles;
				state.ReadByte(newaddr,catReadOperand);
				newaddr.w=addr.w+state.cpustate.y;
				++state.cycles;
				return newaddr;
			}
		};
		  
		struct ModeINX {
			enum {operand_size=1,};
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				*buf++='(';
				buf=AddHexByte(buf,bytes[0]);
				buf=AddIndex(buf,'X');
				*buf++=')';
				*buf=0;
				return buf;
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word ptr,addr;

				ptr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				state.ReadByte(ptr,catReadZP);
				ptr.l+=state.cpustate.x;
				++state.cycles;
				addr.l=state.ReadByte(ptr,catReadZP);
				++state.cycles;
				++ptr.l;
				addr.h=state.ReadByte(ptr,catReadZP);
				++state.cycles;
				return addr;
			}
			
			static FINLINE Word WriteModeEA(MachineType &state) {
				Word ptr,addr;
				
				ptr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				state.ReadByte(ptr,catReadZP);
				ptr.l+=state.cpustate.x;
				++state.cycles;
				addr.l=state.ReadByte(ptr,catReadZP);
				++ptr.l;
				++state.cycles;
				addr.h=state.ReadByte(ptr,catReadZP);
				++state.cycles;
				return addr;
			}
			
			static FINLINE byte ReadOperand(MachineType &state,Word addr) {
				byte b=state.ReadByte(addr,catReadOperand);
				++state.cycles;
				return b;
			}
			
			static FINLINE void WriteOperand(MachineType &state,Word addr,byte val) {
				state.WriteByte(addr,val,catWriteOperand);
				++state.cycles;
			}
		};

		struct ModeINY {
			enum {operand_size=1,};
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				*buf++='(';
				buf=AddHexByte(buf,bytes[0]);
				*buf++=')';
				buf=AddIndex(buf,'Y');
				return buf;
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word ptr,addr,lsbresult;

				ptr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.l=state.ReadByte(ptr,catReadZP);
				++state.cycles;
				++ptr.l;
				addr.h=state.ReadByte(ptr,catReadZP);
				lsbresult.w=addr.l+state.cpustate.y;
				addr.l=lsbresult.l;//9/4/2003 eliminate C4244
				++state.cycles;
				if(lsbresult.h) {
					state.ReadByte(addr,catReadOperand);
					++addr.h;
					++state.cycles;
				}
				return addr;
			}
			
			static FINLINE Word WriteModeEA(MachineType &state) {
				Word ptr,addr,lsbresult;

				ptr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				addr.l=state.ReadByte(ptr,catReadZP);
				++ptr.l;
				++state.cycles;
				addr.h=state.ReadByte(ptr,catReadZP);
				lsbresult.w=addr.l+state.cpustate.y;
				addr.l=lsbresult.l;//9/4/2003 eliminate C4244
				++state.cycles;
				state.ReadByte(addr,catReadOperand);
				addr.h+=lsbresult.h;
				++state.cycles;
				return addr;
			}
			
			static FINLINE byte ReadOperand(MachineType &state,Word addr) {
				byte b=state.ReadByte(addr,catReadOperand);
				++state.cycles;
				return b;
			}
			
			static FINLINE void WriteOperand(MachineType &state,Word addr,byte val) {
				state.WriteByte(addr,val,catWriteOperand);
				++state.cycles;
			}
		};

		struct ModeIND {
			enum {operand_size=2,};
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				*buf++='(';
				buf=AddHexWord(buf,MakeWord(bytes[0],bytes[1]));
				*buf++=')';
				*buf=0;
				return buf;
			}
			static FINLINE Word ReadModeEA(MachineType &state) {
				Word ptr,addr;
				
				ptr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;//2
				++state.cycles;
				ptr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;//3
				++state.cycles;
				addr.l=state.ReadByte(ptr,catFetchAddress);
				++ptr.l;
				++state.cycles;//4
				addr.h=state.ReadByte(ptr,catFetchAddress);
				++state.cycles;//5
				return addr;
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// other addressing modes -- these are needed for the templates, but
		// do not actually do anything useful. sometimes other templates are
		// specialized using these types.
		struct ModeIMP {
			// Implied
			enum {operand_size=0,};
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				*buf=0;
				return buf;
			}
		};
		
		struct ModeREL {
			enum {operand_size=1,};
			// Relative
			static char *Disassemble(char *buf,const byte *bytes,Word bytes_base) {
//				return AddHexWord(buf,MakeWord(bytes_base.w+MakeWord(bytes[0],bytes[1]).w));
				return AddHexWord(buf,MakeWord(bytes_base.w+2+int8(bytes[0])));
			}
		};

		struct ModeACC {
			enum {operand_size=0,};
			// Accumulator
			static char *Disassemble(char *buf,const byte *bytes,Word) {
				*buf++='A';
				*buf=0;
				return buf;
			}
/*
#if t65_COMPILER!=t65_icl
			//TODO see comment for RMW specialization
			static FINLINE Word WriteModeEA(MachineType &) {
				static const t65::Word arse={0};
				//Does nothing.
				return arse;
			}

			static FINLINE byte ReadOperand(MachineType &state,Word) {
				//Accumulator operand -- should be clear enough :)
				//we increment the cycles here too, as this bit only happens 1x in a RMW ACC instr
				//and it's got to happen somewhere.
				++state.cycles;
				return state.cpustate.a;
			}

			//Write value back to A
			static FINLINE void WriteOperand(MachineType &state,Word,byte val) {
				state.cpustate.a=val;
			}
#endif
*/
		};

		// This is an artefact of the automatic code generation stuff.
		// TODO sort out maketable.bas etc. so this goes away.
		typedef ModeIMP ModeILL;

		//////////////////////////////////////////////////////////////////////////
		// Instruction types
		template<class Instr,class AddrMode>
		struct Write {
			typedef Instr InstrType;
			typedef AddrMode AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				Word addr=AddrMode::WriteModeEA(state);
				byte writeval=Instr::Execute(state);
				AddrMode::WriteOperand(state,addr,writeval);
			}
		};
		
		template<class Instr,class AddrMode>
		struct Read {
			typedef Instr InstrType;
			typedef AddrMode AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				Word addr=AddrMode::ReadModeEA(state);
				byte operand=AddrMode::ReadOperand(state,addr);
				Instr::Execute(state,operand);
			}
		};

		template<class Instr,class Addrmode>
		struct RMW;
		
		//#if t65_COMPILER==t65_icl
		//gcc and msvc don't like this.
		//i think i've got something wrong.
		//TODO fix this.
		template<class Instr>
		struct RMW<Instr,ModeACC> {
			typedef Instr InstrType;
			typedef ModeACC AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				state.cpustate.a=Instr::Execute(state,state.cpustate.a);
				++state.cycles;
			}
		};
		//#endif
		
		template<class Instr,class AddrMode>
		struct RMW {
			typedef Instr InstrType;
			typedef AddrMode AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				Word addr=AddrMode::WriteModeEA(state);
				byte val=AddrMode::ReadOperand(state,addr);
				AddrMode::WriteOperand(state,addr,val);
				val=Instr::Execute(state,val);
				AddrMode::WriteOperand(state,addr,val);
			}
		};

		//Instead of being overly clever these days I just ignore the AddrMode type
		//and assume it's ModeIMP as it should be. (The above was only for
		//making sure I got it right, anyway.)
		template<class Instr,class AddrMode>
		struct Implied {
			typedef Instr InstrType;
			typedef ModeIMP AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				Instr::Execute(state);
				++state.cycles;
			}
		};

		template<class Instr,class AddrMode>
		struct Relative {
			typedef Instr InstrType;
			typedef AddrMode AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				Instr::ExecuteRelative(state);
			}
		};

		template<class Instr,class AddrMode>
		struct Special {
			typedef Instr InstrType;
			typedef AddrMode AddrModeType;
			static FINLINE void Execute(MachineType &state) {
				Instr::Execute(state,AddrMode());
			}
		};

		//////////////////////////////////////////////////////////////////////////
		//
		struct InstrILL {
			static const char *Mnemonic() {
				return "ILL";
			}
			static FINLINE void Execute(MachineType &state) {
//				logErr.logf("ILL instruction at &%04X\n",state.cpustate.pc.w-1);
//				WriteHistory();
				SoftBreak();
			}
		};
		
		//////////////////////////////////////////////////////////////////////////
		// HLT
		struct InstrHLT {
			static const char *Mnemonic() {
				return "HLT";
			}
			static FINLINE void Execute(MachineType &state) {
				//This instruction locks the CPU.
				state.cpustate.p|=I_MASK;
				//Instruction fetch has incremented the PC.
				//Decrement it so the CPU keeps fetching this opcode.
				--state.cpustate.pc.w;
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// BRK
		struct InstrBRK {
			static const char *Mnemonic() {
				return "BRK";
			}
			static FINLINE void Execute(MachineType &state) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;
				state.cpustate.p|=B_MASK;
				Interrupt(state,irqvector);
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Store instructions
		struct InstrSTA {
			static const char *Mnemonic() {
				return "STA";
			}
			static FINLINE byte Execute(MachineType &state) {
				return state.cpustate.a;
			}
		};
		
		struct InstrSTX {
			static const char *Mnemonic() {
				return "STX";
			}
			static FINLINE byte Execute(MachineType &state) {
				return state.cpustate.x;
			}
		};
		
		struct InstrSTY {
			static const char *Mnemonic() {
				return "STY";
			}
			static FINLINE byte Execute(MachineType &state) {
				return state.cpustate.y;
			}
		};
		
		//////////////////////////////////////////////////////////////////////////
		// Load instructions
		struct InstrLDA {
			static const char *Mnemonic() {
				return "LDA";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				LoadRegister(state,state.cpustate.a,val);
			}
		};

		struct InstrLDX {
			static const char *Mnemonic() {
				return "LDX";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				LoadRegister(state,state.cpustate.x,val);
			}
		};

		struct InstrLDY {
			static const char *Mnemonic() {
				return "LDY";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				LoadRegister(state,state.cpustate.y,val);
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Index register 'arithmetic' instructions
		// TODO	this is really an implied form of the INC/DEC stuff: sort it out!
		//		Can use for INA/DEA as well.
		struct InstrINX {
			static const char *Mnemonic() {
				return "INX";
			}
			static FINLINE void Execute(MachineType &state) {
				++state.cpustate.x;
				SetNZ(state,state.cpustate.x);
				//++state.cycles;
			}
		};
		
		struct InstrINY {
			static const char *Mnemonic() {
				return "INY";
			}
			static FINLINE void Execute(MachineType &state) {
				++state.cpustate.y;
				SetNZ(state,state.cpustate.y);
				//++state.cycles;
			}
		};
		
		struct InstrDEX {
			static const char *Mnemonic() {
				return "DEX";
			}
			static FINLINE void Execute(MachineType &state) {
				--state.cpustate.x;
				SetNZ(state,state.cpustate.x);
				//++state.cycles;
			}
		};
		
		struct InstrDEY {
			static const char *Mnemonic() {
				return "DEY";
			}
			static FINLINE void Execute(MachineType &state) {
				--state.cpustate.y;
				SetNZ(state,state.cpustate.y);
				//++state.cycles;
			}
		};
		
		//////////////////////////////////////////////////////////////////////////
		// Transfer instructions
		struct InstrTAX {
			static const char *Mnemonic() {
				return "TAX";
			}
			static FINLINE void Execute(MachineType &state) {
				LoadRegister(state,state.cpustate.x,state.cpustate.a);
			}
		};

		struct InstrTXA {
			static const char *Mnemonic() {
				return "TXA";
			}
			static FINLINE void Execute(MachineType &state) {
				LoadRegister(state,state.cpustate.a,state.cpustate.x);
			}
		};

		struct InstrTAY {
			static const char *Mnemonic() {
				return "TAY";
			}
			static FINLINE void Execute(MachineType &state) {
				LoadRegister(state,state.cpustate.y,state.cpustate.a);
			}
		};

		struct InstrTYA {
			static const char *Mnemonic() {
				return "TYA";
			}
			static FINLINE void Execute(MachineType &state) {
				LoadRegister(state,state.cpustate.a,state.cpustate.y);
			}
		};

		struct InstrTSX {
			static const char *Mnemonic() {
				return "TSX";
			}
			static FINLINE void Execute(MachineType &state) {
				LoadRegister(state,state.cpustate.x,state.cpustate.s.l);
			}
		};
		
		struct InstrTXS {
			static const char *Mnemonic() {
				return "TXS";
			}
			static FINLINE void Execute(MachineType &state) {
				state.cpustate.s.l=state.cpustate.x;
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Stack ops


		static FINLINE void PushRegContentsInstr(MachineType &state,byte value) {
			state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cycles;
			Push(state,value);
			++state.cycles;
		}

		//Does everything except assign result
		static FINLINE byte PopRegContentsInstr(MachineType &state) {
			state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cycles;
			state.ReadByte(state.cpustate.s,catReadStack);
			++state.cycles;
			byte val=Pop(state);
			SetNZ(state,val);
			++state.cycles;
			return val;
		}

		struct InstrPHP {
			static const char *Mnemonic() {
				return "PHP";
			}
			static FINLINE void Execute(MachineType &state) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;
				Push(state,state.cpustate.p|B_MASK|U_MASK);
				++state.cycles;
			}
		};

		struct InstrPHA {
			static const char *Mnemonic() {
				return "PHA";
			}
			static FINLINE void Execute(MachineType &state) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;
				Push(state,state.cpustate.a);
				++state.cycles;
			}
		};
		
		struct InstrPLP {
			static const char *Mnemonic() {
				return "PLP";
			}
			static FINLINE void Execute(MachineType &state) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;
				state.cpustate.p=Pop(state)|U_MASK|B_MASK;
				++state.cycles;
			}
		};
		
		struct InstrPLA {
			static const char *Mnemonic() {
				return "PLA";
			}
			static FINLINE void Execute(MachineType &state) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;
				state.cpustate.a=Pop(state);
				SetNZ(state,state.cpustate.a);
				++state.cycles;
			}
		};
		
		//////////////////////////////////////////////////////////////////////////
		// INC and DEC
		struct InstrINC {
			static const char *Mnemonic() {
				return "INC";
			}
			static FINLINE byte Execute(MachineType &state,byte val) {
				++val;
				SetNZ(state,val);
				return val;
			}
		};

		struct InstrDEC {
			static const char *Mnemonic() {
				return "DEC";
			}
			static FINLINE byte Execute(MachineType &state,byte val) {
				--val;
				SetNZ(state,val);
				return val;
			}
		};
		
		//////////////////////////////////////////////////////////////////////////
		// Shifts/rotates
		struct InstrASL {
			static const char *Mnemonic() {
				return "ASL";
			}
			static FINLINE byte Execute(MachineType &state,byte val) {
				state.cpustate.p&=~C_MASK;
				state.cpustate.p|=val>>7;
				val<<=1;
				SetNZ(state,val);
				return val;
			}
		};

		struct InstrLSR {
			static const char *Mnemonic() {
				return "LSR";
			}
			static FINLINE byte Execute(MachineType &state,byte val) {
				state.cpustate.p&=~(N_MASK|Z_MASK|C_MASK);
				state.cpustate.p|=val&1;
				val>>=1;
				state.cpustate.p|=(!val)<<1;//Z_MASK=2
				return val;
			}
		};

		struct InstrROR {
			static const char *Mnemonic() {
				return "ROR";
			}
			static FINLINE byte Execute(MachineType &state,byte val) {
				Word rotate;
				
				rotate.h=state.cpustate.p&1;
				state.cpustate.p&=~1;
				state.cpustate.p|=val&1;
				rotate.l=val;
				rotate.w>>=1;
				SetNZ(state,rotate.l);
				return rotate.l;
			}
		};
		
		struct InstrROL {
			static const char *Mnemonic() {
				return "ROL";
			}
			static FINLINE byte Execute(MachineType &state,byte val) {
				Word rotate;

				rotate.w=val<<1;
				rotate.l|=state.cpustate.p&1;
				state.cpustate.p&=~1;
				state.cpustate.p|=rotate.h;
				SetNZ(state,rotate.l);
				return rotate.l;
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Comparison
		static FINLINE void Compare(MachineType &state,byte reg,byte val) {
			state.cpustate.p&=~(N_MASK|Z_MASK|C_MASK);
			state.cpustate.p|=reg>=val;//C_MASK
			state.cpustate.p|=(reg==val)<<1;//Z_MASK
			state.cpustate.p|=(reg-val)&0x80;//N_MASK
		}

		struct InstrCMP {
			static const char *Mnemonic() {
				return "CMP";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				return Compare(state,state.cpustate.a,val);
			}
		};
		
		struct InstrCPX {
			static const char *Mnemonic() {
				return "CPX";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				return Compare(state,state.cpustate.x,val);
			}
		};

		struct InstrCPY {
			static const char *Mnemonic() {
				return "CPY";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				return Compare(state,state.cpustate.y,val);
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Logical
		struct InstrAND {
			static const char *Mnemonic() {
				return "AND";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				state.cpustate.a&=val;
				SetNZ(state,state.cpustate.a);
			}
		};

		struct InstrEOR {
			static const char *Mnemonic() {
				return "EOR";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				state.cpustate.a^=val;
				SetNZ(state,state.cpustate.a);
			}
		};
		
		struct InstrORA {
			static const char *Mnemonic() {
				return "ORA";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				state.cpustate.a|=val;
				SetNZ(state,state.cpustate.a);
			}
		};
		
		//////////////////////////////////////////////////////////////////////////
		// Set/clear P flags
		template<byte mask>
		struct InstrSetP {
			static const char *Mnemonic() {
				return "SetP";
			}
			static FINLINE void Execute(MachineType &state) {
				state.cpustate.p|=mask;
			}
		};

		template<byte mask>
		struct InstrClearP {
			static const char *Mnemonic() {
				return "ClearP";
			}
			static FINLINE void Execute(MachineType &state) {
				state.cpustate.p&=~mask;
			}
		};

		struct InstrSED:
		public InstrSetP<D_MASK>
		{
			static const char *Mnemonic() {
				return "SED";
			}
		};
		struct InstrSEI:
		public InstrSetP<I_MASK>
		{
			static const char *Mnemonic() {
				return "SEI";
			}
		};
		struct InstrSEC:
		public InstrSetP<C_MASK>
		{
			static const char *Mnemonic() {
				return "SEC";
			}
		};
		struct InstrCLV:
		public InstrClearP<V_MASK>
		{
			static const char *Mnemonic() {
				return "CLV";
			}
		};
		struct InstrCLD:
		public InstrClearP<D_MASK>
		{
			static const char *Mnemonic() {
				return "CLD";
			}
		};
		struct InstrCLI:
		public InstrClearP<I_MASK>
		{
			static const char *Mnemonic() {
				return "CLI";
			}
		};
		struct InstrCLC:
		public InstrClearP<C_MASK>
		{
			static const char *Mnemonic() {
				return "CLC";
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// Branch & jump instrutcions
		//
		// Branch is weird, it looks like the 6502 keeps fetching the next
		// instruction whilst doing the final bits of the instruction. Bizarro.
		// TODO Take care when putting in catReadInternal, because of that.
		template<byte mask,byte check>
		struct InstrBranch {
			static const char *Mnemonic() {
				return "Branch";
			}
			static FINLINE void ExecuteRelative(MachineType &state) {
				Word lsbresult;

				// 2nd cycle -- fetch operand, increment PC
				int8 operand=int8(state.ReadByte(state.cpustate.pc,catFetchInstr));
				++state.cpustate.pc.w;
				++state.cycles;

				// If branch not taken, bail out right here. 3rd cycle PC read
				// happens in main loop (see 64doc). Otherwise, 3rd cycle PC
				// read happens here.
				if((state.cpustate.p&mask)==check) {
					// 3rd cycle -- read PC, add operand to PCL
					state.ReadByte(state.cpustate.pc,catFetchInstr);
					lsbresult.w=state.cpustate.pc.w+operand;
					state.cpustate.pc.l=lsbresult.l;
					++state.cycles;
					
					// If PC needs a fixup, read from (incorrect) PC whilst fixing
					// it up. If PC doesn't need fixing, we drop out here and let
					// the main loop do the PC read.
					if(lsbresult.h!=state.cpustate.pc.h) {
						// 4th cycle -- read PC, fix PCH, ++PC.
						state.ReadByte(state.cpustate.pc,catFetchInstr);
						state.cpustate.pc.h=lsbresult.h;
						++state.cycles;
						//++state.cpustate.pc.w;
					}
				}
			}
		};

		template<byte mask>
		struct InstrBranchIfSet:
		public InstrBranch<mask,mask>
		{
		};

		template<byte mask>
		struct InstrBranchIfClear:
		public InstrBranch<mask,0>
		{
		};
		
		struct InstrBMI:
		public InstrBranchIfSet<N_MASK>
		{
			static const char *Mnemonic() {
				return "BMI";
			}
		};
		struct InstrBVS:
		public InstrBranchIfSet<V_MASK>
		{
			static const char *Mnemonic() {
				return "BVS";
			}
		};
		struct InstrBEQ:
		public InstrBranchIfSet<Z_MASK>
		{
			static const char *Mnemonic() {
				return "BEQ";
			}
		};
		struct InstrBCS:
		public InstrBranchIfSet<C_MASK>
		{
			static const char *Mnemonic() {
				return "BCS";
			}
		};
		
		struct InstrBPL:
		public InstrBranchIfClear<N_MASK>
		{
			static const char *Mnemonic() {
				return "BPL";
			}
		};
		struct InstrBVC:
		public InstrBranchIfClear<V_MASK>
		{
			static const char *Mnemonic() {
				return "BVC";
			}
		};
		struct InstrBNE:
		public InstrBranchIfClear<Z_MASK>
		{
			static const char *Mnemonic() {
				return "BNE";
			}
		};
		struct InstrBCC:
		public InstrBranchIfClear<C_MASK>
		{
			static const char *Mnemonic() {
				return "BCC";
			}
		};

		struct InstrJMP {
			static const char *Mnemonic() {
				return "JMP";
			}
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				state.cpustate.pc=AddrMode::ReadModeEA(state);
			}
		};

		struct InstrJSR {
			static const char *Mnemonic() {
				return "JSR";
			}
			// no more specialization [4/18/2003]
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				byte lsb;
				
				lsb=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;//2
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;//3
				Push(state,state.cpustate.pc.h);
				++state.cycles;//4
				Push(state,state.cpustate.pc.l);
				++state.cycles;//5
				state.cpustate.pc.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				state.cpustate.pc.l=lsb;
				++state.cycles;//6
			}
			/*
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				SoftBreak();
			}

			template<>
			TSP_STATIC void Execute(MachineType &state,const ModeABS &) {
				byte lsb;
				
				lsb=state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;//2
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;//3
				Push(state,state.cpustate.pc.h);
				++state.cycles;//4
				Push(state,state.cpustate.pc.l);
				++state.cycles;//5
				state.cpustate.pc.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
				state.cpustate.pc.l=lsb;
				++state.cycles;//6
			}
*/
		};

		//////////////////////////////////////////////////////////////////////////
		//
		struct InstrRTI {
			static const char *Mnemonic() {
				return "RTI";
			}
			// no more specialization [4/18/2003]
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;//2
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;//3
				state.cpustate.p=Pop(state);
				++state.cycles;//4
				state.cpustate.pc.l=Pop(state);
				++state.cycles;//5
				state.cpustate.pc.h=Pop(state);
				++state.cycles;//6
			}
			/*			
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				SoftBreak();
			}
			
			template<>
			TSP_STATIC TSP_FINLINE void Execute<ModeIMP>(MachineType &state,const ModeIMP &) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;//2
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;//3
				state.cpustate.p=Pop(state);
				++state.cycles;//4
				state.cpustate.pc.l=Pop(state);
				++state.cycles;//5
				state.cpustate.pc.h=Pop(state);
				++state.cycles;//6
			}
*/
		};
				
		//////////////////////////////////////////////////////////////////////////
		// RTS
		struct InstrRTS {
			static const char *Mnemonic() {
				return "RTS";
			}
			// no more specialization [4/18/2003]
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;//2
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;//3
				state.cpustate.pc.l=Pop(state);
				++state.cycles;//4
				state.cpustate.pc.h=Pop(state);
				++state.cycles;//5
				// Don't know if this is the right category...
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;//6
			}
			
			/*
			template<class AddrMode>
			static FINLINE void Execute(MachineType &state,const AddrMode &) {
				SoftBreak();
			}

			template<>
			TSP_STATIC TSP_FINLINE void Execute<ModeIMP>(MachineType &state,const ModeIMP &) {
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cycles;//2
				state.ReadByte(state.cpustate.s,catReadStack);
				++state.cycles;//3
				state.cpustate.pc.l=Pop(state);
				++state.cycles;//4
				state.cpustate.pc.h=Pop(state);
				++state.cycles;//5
				// Don't know if this is the right category...
				state.ReadByte(state.cpustate.pc,catFetchInstr);
				++state.cpustate.pc.w;
				++state.cycles;//6
			}
			*/
		};

		//////////////////////////////////////////////////////////////////////////
		// NOP
		struct InstrNOP {
			static const char *Mnemonic() {
				return "NOP";
			}
			//NOP is a read instruction. There's a defaulted val parameter so
			//it may be used as one as well as an implied instruction.
			static FINLINE void Execute(MachineType &state,byte=0) {
				//do nothing... it's a nop...
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// BIT
		struct InstrBIT {
			static const char *Mnemonic() {
				return "BIT";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				state.cpustate.p&=~(N_MASK|Z_MASK|V_MASK);
				state.cpustate.p|=val&0xC0;//set N+V
				state.cpustate.p|=((state.cpustate.a&val)==0)<<1;//set Z
			}
		};

		//////////////////////////////////////////////////////////////////////////
		// ADC/SBC
		// See 64doc for more about these two!
		struct InstrADC {
			static const char *Mnemonic() {
				return "ADC";
			}
			static FINLINE void Execute(MachineType &state,byte val) {
				if(!(state.cpustate.p&D_MASK)) {
					Word result;

					result.w=state.cpustate.a+val+(state.cpustate.p&1);//&1=C
					state.cpustate.p&=~(N_MASK|Z_MASK|V_MASK|C_MASK);
					state.cpustate.p|=result.h;//do carry
					//state.cpustate.p|=nztable[result.l];//do n+z
					DoOrNZFlags(state,result.l);
#ifdef t65_BEEBEM_VFLAG
					int16 sresult=int8(state.cpustate.a)+int8(val)+(state.cpustate.p&1);
					if(((result.l&128)>0)^(sresult<0)) {
						state.cpustate.p|=V_MASK;
					}
#else
					//retro 1996 code from model-b: (think it works -- this was
					//what fixed cholo and os1.20 sound system.)
					//V=(~(A^(j)) & (A^ADCResult) & 0x80)>>1;
					state.cpustate.p|=(~(state.cpustate.a^val)&(state.cpustate.a^result.l)&0x80)>>1;
#endif
					state.cpustate.a=result.l;
				} else {
					//BCD code from BeebEm
					int ZFlag=0,NFlag=0,CFlag=0,VFlag=0;
					int TmpResult,TmpCarry=0;
					int ln,hn;
					
					// Z flag determined from 2's compl result, not BCD result!
					TmpResult=(state.cpustate.a)+(val)+(state.cpustate.p&1);
					ZFlag=((TmpResult & 0xff)==0);
					
					ln=((state.cpustate.a) & 0xf)+((val) & 0xf)+(state.cpustate.p&1);
					if (ln>9) {
						ln += 6;
						ln &= 0xf;
						TmpCarry=0x10;
					}
					hn=((state.cpustate.a) & 0xf0)+((val) & 0xf0)+TmpCarry;
					// N and V flags are determined before high nibble is adjusted.
					// NOTE: V is not always correct
					NFlag=hn & 128;
					VFlag=(hn ^ (state.cpustate.a)) & 128 && !(((state.cpustate.a) ^ (val)) & 128);
					if (hn>0x90) {
						hn += 0x60;
						hn &= 0xf0;
						CFlag=1;
					}
					(state.cpustate.a)=hn|ln;
					ZFlag=((state.cpustate.a)==0);
					NFlag=((state.cpustate.a)&128);
					//SetPSR(FlagC | FlagZ | FlagV | FlagN,CFlag,ZFlag,0,0,0,VFlag,NFlag);
					state.cpustate.p&=~(N_MASK|V_MASK|Z_MASK|C_MASK);
					state.cpustate.p|=CFlag?C_MASK:0;
					state.cpustate.p|=ZFlag?Z_MASK:0;
					state.cpustate.p|=VFlag?V_MASK:0;
					state.cpustate.p|=NFlag?N_MASK:0;
/*
					//code from 64doc, I think -- doesn't work. Maybe I typed it in
					//wrong?
					// binary coded decimal mode
					byte ha,la;//accumulator nybbles
					la=(state.cpustate.a&0xF)+(val&0xF)+(state.cpustate.p&1);//&1=C
					ha=(state.cpustate.a>>4)+(val>>4)+(la>15);
					if(la>9) {
						la+=6;// bcd fixup for lower nybble.
					}

					//now set nzc using same logic as used in non-bcd mode.
					//first z -- carefuly, note use of bool 1/0 value.
					state.cpustate.p&=~(N_MASK|Z_MASK|V_MASK|C_MASK);
					state.cpustate.p|=(!(state.cpustate.a+val+(state.cpustate.p&1)))<<1;
					state.cpustate.p|=(ha&8)<<4;//set n
					state.cpustate.p|=((((ha<<4)^state.cpustate.a)&128)&
						(~((state.cpustate.a^val)&128)))>>1;//set v, bit 6

					if(ha>9) {
						ha+=6;//bcd fixyup for upper nybble
					}
					
					//set c after all fixups -- note use of bool 1/0 value.
					state.cpustate.p|=ha>15;
					state.cpustate.a=(ha<<4)|(la&0xF);
*/
				}
			}
		};

		struct InstrSBC {
			static const char *Mnemonic() {
				return "SBC";
			}
			//like ADC but add ~val.
			static FINLINE void Execute(MachineType &state,byte val) {
				Word result;
				result.w=state.cpustate.a+byte(~val)+(state.cpustate.p&1);//&1=C
				if(!(state.cpustate.p&D_MASK)) {
					state.cpustate.p&=~(N_MASK|Z_MASK|V_MASK|C_MASK);
					state.cpustate.p|=result.h;//do carry (think this is right?)
					//state.cpustate.p|=nztable[result.l];//do n+z
					DoOrNZFlags(state,result.l);
#ifdef t65_BEEBEM_VFLAG
					//((Accumulator & 128)>0) ^ ((TmpResultV & 256)!=0)
					int16 sresult=int8(state.cpustate.a)-int8(val)-((~state.cpustate.p)&1);
					if(((result.l&128)>0)^((sresult&256)!=0)) {
						state.cpustate.p|=V_MASK;
					}
#else
					//retro 1996 code from model-b: (think it works -- this was
					//what fixed cholo and os1.20 sound system.)
					//V=(~(A^(j)) & (A^ADCResult) & 0x80)>>1;
					state.cpustate.p|=((state.cpustate.a^val)&(state.cpustate.a^result.l)&0x80)>>1;
#endif
					state.cpustate.a=result.l;
				} else {
					//BCD code from BeebEm
					//(state.cpustate.a)=Accumulator
					//(state.cpustate.p&1)=GETCFLAG
					//(val)=operand
					int TmpResultV;
					unsigned char nhn,nln;
					int ZFlag=0,NFlag=0,CFlag=1,VFlag=0;
					int TmpResult,TmpCarry=0;
					int ln,hn,oln,ohn;
					nhn=((state.cpustate.a)>>4)&15; nln=(state.cpustate.a) & 15;
					
					/* Z flag determined from 2's compl result, not BCD result! */
					TmpResult=(state.cpustate.a)-(val)-(1-(state.cpustate.p&1));
					ZFlag=((TmpResult & 0xff)==0);
					
					ohn=(val) & 0xf0; oln = (val) & 0xf;
					if ((oln>9) && (((state.cpustate.a)&15)<10)) { oln-=10; ohn+=0x10; } 
					// promote the lower nibble to the next ten, and increase the higher nibble
					ln=((state.cpustate.a) & 0xf)-oln-(1-(state.cpustate.p&1));
					if (ln<0) {
						if (((state.cpustate.a) & 15)<10) ln-=6;
						ln&=0xf;
						TmpCarry=0x10;
					}
					hn=((state.cpustate.a) & 0xf0)-ohn-TmpCarry;
					/* N and V flags are determined before high nibble is adjusted.
					NOTE: V is not always correct */
					NFlag=hn & 128;
					TmpResultV=(signed char)(state.cpustate.a)-(signed char)(val)-(1-(state.cpustate.p&1));
					if ((TmpResultV<-128)||(TmpResultV>127)) VFlag=1; else VFlag=0;
					if (hn<0) {
						hn-=0x60;
						hn&=0xf0;
						CFlag=0;
					}
					(state.cpustate.a)=hn|ln;
					if ((state.cpustate.a)==0) ZFlag=1;
					NFlag=(hn &128);
					CFlag=(TmpResult&256)==0;
					//SetPSR(FlagC | FlagZ | FlagV | FlagN,CFlag,ZFlag,0,0,0,VFlag,NFlag);
					state.cpustate.p&=~(N_MASK|V_MASK|Z_MASK|C_MASK);
					state.cpustate.p|=CFlag?C_MASK:0;
					state.cpustate.p|=ZFlag?Z_MASK:0;
					state.cpustate.p|=VFlag?V_MASK:0;
					state.cpustate.p|=NFlag?N_MASK:0;
					/*
					//64doc code, doesn't seem to work though I might have transcribed
					//it wrongly.
					byte ha,la;
					
					la=(state.cpustate.a&0xF)-(val&0xF)-((state.cpustate.p&1)^1);
					if(la&0x10) {
						la-=6;//bcd fixup for lower nybble
					}
					ha=(state.cpustate.a>>4)-(val>>4)-(la&16);
					if(ha&0x10) {
						ha-=6;
					}

					bool old_i=!!(state.cpustate.p&I_MASK);

					//set em like in 2s complement mode.
					state.cpustate.p&=~(N_MASK|Z_MASK|V_MASK|C_MASK);
					state.cpustate.p|=(result.h&1)^1;//do inverted carry
					state.cpustate.p|=nztable[result.l];//do n+z
					//retro 1996 code from model-b: (think it works -- this was
					//what fixed cholo and os1.20 sound system.)
					//V=(~(A^(j)) & (A^ADCResult) & 0x80)>>1;
					state.cpustate.p|=((state.cpustate.a^val)&(state.cpustate.a^result.l)&0x80)>>1;

					//reconstruct accumulator.
					state.cpustate.a=(ha<<4)|(la&0xF);
					*/
				}
			}
		};

		template<class InstructionSet>
		static void RunSingleInstruction(MachineType &state,const InstructionSet &) {
			byte op=state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cpustate.pc.w;
			++state.cycles;
#ifdef t65_USE_FUNCTION_TABLE
			(*OpcodeTable<InstrExecuteFn,InstructionSet>::opcode_table[op])(state);
#else
			switch(op&0xFF) {
			case 0x00:
				InstructionSet::Opcode00::Execute(state);
				break;
			case 0x01:
				InstructionSet::Opcode01::Execute(state);
				break;
			case 0x02:
				InstructionSet::Opcode02::Execute(state);
				break;
			case 0x03:
				InstructionSet::Opcode03::Execute(state);
				break;
			case 0x04:
				InstructionSet::Opcode04::Execute(state);
				break;
			case 0x05:
				InstructionSet::Opcode05::Execute(state);
				break;
			case 0x06:
				InstructionSet::Opcode06::Execute(state);
				break;
			case 0x07:
				InstructionSet::Opcode07::Execute(state);
				break;
			case 0x08:
				InstructionSet::Opcode08::Execute(state);
				break;
			case 0x09:
				InstructionSet::Opcode09::Execute(state);
				break;
			case 0x0A:
				InstructionSet::Opcode0A::Execute(state);
				break;
			case 0x0B:
				InstructionSet::Opcode0B::Execute(state);
				break;
			case 0x0C:
				InstructionSet::Opcode0C::Execute(state);
				break;
			case 0x0D:
				InstructionSet::Opcode0D::Execute(state);
				break;
			case 0x0E:
				InstructionSet::Opcode0E::Execute(state);
				break;
			case 0x0F:
				InstructionSet::Opcode0F::Execute(state);
				break;
			case 0x10:
				InstructionSet::Opcode10::Execute(state);
				break;
			case 0x11:
				InstructionSet::Opcode11::Execute(state);
				break;
			case 0x12:
				InstructionSet::Opcode12::Execute(state);
				break;
			case 0x13:
				InstructionSet::Opcode13::Execute(state);
				break;
			case 0x14:
				InstructionSet::Opcode14::Execute(state);
				break;
			case 0x15:
				InstructionSet::Opcode15::Execute(state);
				break;
			case 0x16:
				InstructionSet::Opcode16::Execute(state);
				break;
			case 0x17:
				InstructionSet::Opcode17::Execute(state);
				break;
			case 0x18:
				InstructionSet::Opcode18::Execute(state);
				break;
			case 0x19:
				InstructionSet::Opcode19::Execute(state);
				break;
			case 0x1A:
				InstructionSet::Opcode1A::Execute(state);
				break;
			case 0x1B:
				InstructionSet::Opcode1B::Execute(state);
				break;
			case 0x1C:
				InstructionSet::Opcode1C::Execute(state);
				break;
			case 0x1D:
				InstructionSet::Opcode1D::Execute(state);
				break;
			case 0x1E:
				InstructionSet::Opcode1E::Execute(state);
				break;
			case 0x1F:
				InstructionSet::Opcode1F::Execute(state);
				break;
			case 0x20:
				InstructionSet::Opcode20::Execute(state);
				break;
			case 0x21:
				InstructionSet::Opcode21::Execute(state);
				break;
			case 0x22:
				InstructionSet::Opcode22::Execute(state);
				break;
			case 0x23:
				InstructionSet::Opcode23::Execute(state);
				break;
			case 0x24:
				InstructionSet::Opcode24::Execute(state);
				break;
			case 0x25:
				InstructionSet::Opcode25::Execute(state);
				break;
			case 0x26:
				InstructionSet::Opcode26::Execute(state);
				break;
			case 0x27:
				InstructionSet::Opcode27::Execute(state);
				break;
			case 0x28:
				InstructionSet::Opcode28::Execute(state);
				break;
			case 0x29:
				InstructionSet::Opcode29::Execute(state);
				break;
			case 0x2A:
				InstructionSet::Opcode2A::Execute(state);
				break;
			case 0x2B:
				InstructionSet::Opcode2B::Execute(state);
				break;
			case 0x2C:
				InstructionSet::Opcode2C::Execute(state);
				break;
			case 0x2D:
				InstructionSet::Opcode2D::Execute(state);
				break;
			case 0x2E:
				InstructionSet::Opcode2E::Execute(state);
				break;
			case 0x2F:
				InstructionSet::Opcode2F::Execute(state);
				break;
			case 0x30:
				InstructionSet::Opcode30::Execute(state);
				break;
			case 0x31:
				InstructionSet::Opcode31::Execute(state);
				break;
			case 0x32:
				InstructionSet::Opcode32::Execute(state);
				break;
			case 0x33:
				InstructionSet::Opcode33::Execute(state);
				break;
			case 0x34:
				InstructionSet::Opcode34::Execute(state);
				break;
			case 0x35:
				InstructionSet::Opcode35::Execute(state);
				break;
			case 0x36:
				InstructionSet::Opcode36::Execute(state);
				break;
			case 0x37:
				InstructionSet::Opcode37::Execute(state);
				break;
			case 0x38:
				InstructionSet::Opcode38::Execute(state);
				break;
			case 0x39:
				InstructionSet::Opcode39::Execute(state);
				break;
			case 0x3A:
				InstructionSet::Opcode3A::Execute(state);
				break;
			case 0x3B:
				InstructionSet::Opcode3B::Execute(state);
				break;
			case 0x3C:
				InstructionSet::Opcode3C::Execute(state);
				break;
			case 0x3D:
				InstructionSet::Opcode3D::Execute(state);
				break;
			case 0x3E:
				InstructionSet::Opcode3E::Execute(state);
				break;
			case 0x3F:
				InstructionSet::Opcode3F::Execute(state);
				break;
			case 0x40:
				InstructionSet::Opcode40::Execute(state);
				break;
			case 0x41:
				InstructionSet::Opcode41::Execute(state);
				break;
			case 0x42:
				InstructionSet::Opcode42::Execute(state);
				break;
			case 0x43:
				InstructionSet::Opcode43::Execute(state);
				break;
			case 0x44:
				InstructionSet::Opcode44::Execute(state);
				break;
			case 0x45:
				InstructionSet::Opcode45::Execute(state);
				break;
			case 0x46:
				InstructionSet::Opcode46::Execute(state);
				break;
			case 0x47:
				InstructionSet::Opcode47::Execute(state);
				break;
			case 0x48:
				InstructionSet::Opcode48::Execute(state);
				break;
			case 0x49:
				InstructionSet::Opcode49::Execute(state);
				break;
			case 0x4A:
				InstructionSet::Opcode4A::Execute(state);
				break;
			case 0x4B:
				InstructionSet::Opcode4B::Execute(state);
				break;
			case 0x4C:
				InstructionSet::Opcode4C::Execute(state);
				break;
			case 0x4D:
				InstructionSet::Opcode4D::Execute(state);
				break;
			case 0x4E:
				InstructionSet::Opcode4E::Execute(state);
				break;
			case 0x4F:
				InstructionSet::Opcode4F::Execute(state);
				break;
			case 0x50:
				InstructionSet::Opcode50::Execute(state);
				break;
			case 0x51:
				InstructionSet::Opcode51::Execute(state);
				break;
			case 0x52:
				InstructionSet::Opcode52::Execute(state);
				break;
			case 0x53:
				InstructionSet::Opcode53::Execute(state);
				break;
			case 0x54:
				InstructionSet::Opcode54::Execute(state);
				break;
			case 0x55:
				InstructionSet::Opcode55::Execute(state);
				break;
			case 0x56:
				InstructionSet::Opcode56::Execute(state);
				break;
			case 0x57:
				InstructionSet::Opcode57::Execute(state);
				break;
			case 0x58:
				InstructionSet::Opcode58::Execute(state);
				break;
			case 0x59:
				InstructionSet::Opcode59::Execute(state);
				break;
			case 0x5A:
				InstructionSet::Opcode5A::Execute(state);
				break;
			case 0x5B:
				InstructionSet::Opcode5B::Execute(state);
				break;
			case 0x5C:
				InstructionSet::Opcode5C::Execute(state);
				break;
			case 0x5D:
				InstructionSet::Opcode5D::Execute(state);
				break;
			case 0x5E:
				InstructionSet::Opcode5E::Execute(state);
				break;
			case 0x5F:
				InstructionSet::Opcode5F::Execute(state);
				break;
			case 0x60:
				InstructionSet::Opcode60::Execute(state);
				break;
			case 0x61:
				InstructionSet::Opcode61::Execute(state);
				break;
			case 0x62:
				InstructionSet::Opcode62::Execute(state);
				break;
			case 0x63:
				InstructionSet::Opcode63::Execute(state);
				break;
			case 0x64:
				InstructionSet::Opcode64::Execute(state);
				break;
			case 0x65:
				InstructionSet::Opcode65::Execute(state);
				break;
			case 0x66:
				InstructionSet::Opcode66::Execute(state);
				break;
			case 0x67:
				InstructionSet::Opcode67::Execute(state);
				break;
			case 0x68:
				InstructionSet::Opcode68::Execute(state);
				break;
			case 0x69:
				InstructionSet::Opcode69::Execute(state);
				break;
			case 0x6A:
				InstructionSet::Opcode6A::Execute(state);
				break;
			case 0x6B:
				InstructionSet::Opcode6B::Execute(state);
				break;
			case 0x6C:
				InstructionSet::Opcode6C::Execute(state);
				break;
			case 0x6D:
				InstructionSet::Opcode6D::Execute(state);
				break;
			case 0x6E:
				InstructionSet::Opcode6E::Execute(state);
				break;
			case 0x6F:
				InstructionSet::Opcode6F::Execute(state);
				break;
			case 0x70:
				InstructionSet::Opcode70::Execute(state);
				break;
			case 0x71:
				InstructionSet::Opcode71::Execute(state);
				break;
			case 0x72:
				InstructionSet::Opcode72::Execute(state);
				break;
			case 0x73:
				InstructionSet::Opcode73::Execute(state);
				break;
			case 0x74:
				InstructionSet::Opcode74::Execute(state);
				break;
			case 0x75:
				InstructionSet::Opcode75::Execute(state);
				break;
			case 0x76:
				InstructionSet::Opcode76::Execute(state);
				break;
			case 0x77:
				InstructionSet::Opcode77::Execute(state);
				break;
			case 0x78:
				InstructionSet::Opcode78::Execute(state);
				break;
			case 0x79:
				InstructionSet::Opcode79::Execute(state);
				break;
			case 0x7A:
				InstructionSet::Opcode7A::Execute(state);
				break;
			case 0x7B:
				InstructionSet::Opcode7B::Execute(state);
				break;
			case 0x7C:
				InstructionSet::Opcode7C::Execute(state);
				break;
			case 0x7D:
				InstructionSet::Opcode7D::Execute(state);
				break;
			case 0x7E:
				InstructionSet::Opcode7E::Execute(state);
				break;
			case 0x7F:
				InstructionSet::Opcode7F::Execute(state);
				break;
			case 0x80:
				InstructionSet::Opcode80::Execute(state);
				break;
			case 0x81:
				InstructionSet::Opcode81::Execute(state);
				break;
			case 0x82:
				InstructionSet::Opcode82::Execute(state);
				break;
			case 0x83:
				InstructionSet::Opcode83::Execute(state);
				break;
			case 0x84:
				InstructionSet::Opcode84::Execute(state);
				break;
			case 0x85:
				InstructionSet::Opcode85::Execute(state);
				break;
			case 0x86:
				InstructionSet::Opcode86::Execute(state);
				break;
			case 0x87:
				InstructionSet::Opcode87::Execute(state);
				break;
			case 0x88:
				InstructionSet::Opcode88::Execute(state);
				break;
			case 0x89:
				InstructionSet::Opcode89::Execute(state);
				break;
			case 0x8A:
				InstructionSet::Opcode8A::Execute(state);
				break;
			case 0x8B:
				InstructionSet::Opcode8B::Execute(state);
				break;
			case 0x8C:
				InstructionSet::Opcode8C::Execute(state);
				break;
			case 0x8D:
				InstructionSet::Opcode8D::Execute(state);
				break;
			case 0x8E:
				InstructionSet::Opcode8E::Execute(state);
				break;
			case 0x8F:
				InstructionSet::Opcode8F::Execute(state);
				break;
			case 0x90:
				InstructionSet::Opcode90::Execute(state);
				break;
			case 0x91:
				InstructionSet::Opcode91::Execute(state);
				break;
			case 0x92:
				InstructionSet::Opcode92::Execute(state);
				break;
			case 0x93:
				InstructionSet::Opcode93::Execute(state);
				break;
			case 0x94:
				InstructionSet::Opcode94::Execute(state);
				break;
			case 0x95:
				InstructionSet::Opcode95::Execute(state);
				break;
			case 0x96:
				InstructionSet::Opcode96::Execute(state);
				break;
			case 0x97:
				InstructionSet::Opcode97::Execute(state);
				break;
			case 0x98:
				InstructionSet::Opcode98::Execute(state);
				break;
			case 0x99:
				InstructionSet::Opcode99::Execute(state);
				break;
			case 0x9A:
				InstructionSet::Opcode9A::Execute(state);
				break;
			case 0x9B:
				InstructionSet::Opcode9B::Execute(state);
				break;
			case 0x9C:
				InstructionSet::Opcode9C::Execute(state);
				break;
			case 0x9D:
				InstructionSet::Opcode9D::Execute(state);
				break;
			case 0x9E:
				InstructionSet::Opcode9E::Execute(state);
				break;
			case 0x9F:
				InstructionSet::Opcode9F::Execute(state);
				break;
			case 0xA0:
				InstructionSet::OpcodeA0::Execute(state);
				break;
			case 0xA1:
				InstructionSet::OpcodeA1::Execute(state);
				break;
			case 0xA2:
				InstructionSet::OpcodeA2::Execute(state);
				break;
			case 0xA3:
				InstructionSet::OpcodeA3::Execute(state);
				break;
			case 0xA4:
				InstructionSet::OpcodeA4::Execute(state);
				break;
			case 0xA5:
				InstructionSet::OpcodeA5::Execute(state);
				break;
			case 0xA6:
				InstructionSet::OpcodeA6::Execute(state);
				break;
			case 0xA7:
				InstructionSet::OpcodeA7::Execute(state);
				break;
			case 0xA8:
				InstructionSet::OpcodeA8::Execute(state);
				break;
			case 0xA9:
				InstructionSet::OpcodeA9::Execute(state);
				break;
			case 0xAA:
				InstructionSet::OpcodeAA::Execute(state);
				break;
			case 0xAB:
				InstructionSet::OpcodeAB::Execute(state);
				break;
			case 0xAC:
				InstructionSet::OpcodeAC::Execute(state);
				break;
			case 0xAD:
				InstructionSet::OpcodeAD::Execute(state);
				break;
			case 0xAE:
				InstructionSet::OpcodeAE::Execute(state);
				break;
			case 0xAF:
				InstructionSet::OpcodeAF::Execute(state);
				break;
			case 0xB0:
				InstructionSet::OpcodeB0::Execute(state);
				break;
			case 0xB1:
				InstructionSet::OpcodeB1::Execute(state);
				break;
			case 0xB2:
				InstructionSet::OpcodeB2::Execute(state);
				break;
			case 0xB3:
				InstructionSet::OpcodeB3::Execute(state);
				break;
			case 0xB4:
				InstructionSet::OpcodeB4::Execute(state);
				break;
			case 0xB5:
				InstructionSet::OpcodeB5::Execute(state);
				break;
			case 0xB6:
				InstructionSet::OpcodeB6::Execute(state);
				break;
			case 0xB7:
				InstructionSet::OpcodeB7::Execute(state);
				break;
			case 0xB8:
				InstructionSet::OpcodeB8::Execute(state);
				break;
			case 0xB9:
				InstructionSet::OpcodeB9::Execute(state);
				break;
			case 0xBA:
				InstructionSet::OpcodeBA::Execute(state);
				break;
			case 0xBB:
				InstructionSet::OpcodeBB::Execute(state);
				break;
			case 0xBC:
				InstructionSet::OpcodeBC::Execute(state);
				break;
			case 0xBD:
				InstructionSet::OpcodeBD::Execute(state);
				break;
			case 0xBE:
				InstructionSet::OpcodeBE::Execute(state);
				break;
			case 0xBF:
				InstructionSet::OpcodeBF::Execute(state);
				break;
			case 0xC0:
				InstructionSet::OpcodeC0::Execute(state);
				break;
			case 0xC1:
				InstructionSet::OpcodeC1::Execute(state);
				break;
			case 0xC2:
				InstructionSet::OpcodeC2::Execute(state);
				break;
			case 0xC3:
				InstructionSet::OpcodeC3::Execute(state);
				break;
			case 0xC4:
				InstructionSet::OpcodeC4::Execute(state);
				break;
			case 0xC5:
				InstructionSet::OpcodeC5::Execute(state);
				break;
			case 0xC6:
				InstructionSet::OpcodeC6::Execute(state);
				break;
			case 0xC7:
				InstructionSet::OpcodeC7::Execute(state);
				break;
			case 0xC8:
				InstructionSet::OpcodeC8::Execute(state);
				break;
			case 0xC9:
				InstructionSet::OpcodeC9::Execute(state);
				break;
			case 0xCA:
				InstructionSet::OpcodeCA::Execute(state);
				break;
			case 0xCB:
				InstructionSet::OpcodeCB::Execute(state);
				break;
			case 0xCC:
				InstructionSet::OpcodeCC::Execute(state);
				break;
			case 0xCD:
				InstructionSet::OpcodeCD::Execute(state);
				break;
			case 0xCE:
				InstructionSet::OpcodeCE::Execute(state);
				break;
			case 0xCF:
				InstructionSet::OpcodeCF::Execute(state);
				break;
			case 0xD0:
				InstructionSet::OpcodeD0::Execute(state);
				break;
			case 0xD1:
				InstructionSet::OpcodeD1::Execute(state);
				break;
			case 0xD2:
				InstructionSet::OpcodeD2::Execute(state);
				break;
			case 0xD3:
				InstructionSet::OpcodeD3::Execute(state);
				break;
			case 0xD4:
				InstructionSet::OpcodeD4::Execute(state);
				break;
			case 0xD5:
				InstructionSet::OpcodeD5::Execute(state);
				break;
			case 0xD6:
				InstructionSet::OpcodeD6::Execute(state);
				break;
			case 0xD7:
				InstructionSet::OpcodeD7::Execute(state);
				break;
			case 0xD8:
				InstructionSet::OpcodeD8::Execute(state);
				break;
			case 0xD9:
				InstructionSet::OpcodeD9::Execute(state);
				break;
			case 0xDA:
				InstructionSet::OpcodeDA::Execute(state);
				break;
			case 0xDB:
				InstructionSet::OpcodeDB::Execute(state);
				break;
			case 0xDC:
				InstructionSet::OpcodeDC::Execute(state);
				break;
			case 0xDD:
				InstructionSet::OpcodeDD::Execute(state);
				break;
			case 0xDE:
				InstructionSet::OpcodeDE::Execute(state);
				break;
			case 0xDF:
				InstructionSet::OpcodeDF::Execute(state);
				break;
			case 0xE0:
				InstructionSet::OpcodeE0::Execute(state);
				break;
			case 0xE1:
				InstructionSet::OpcodeE1::Execute(state);
				break;
			case 0xE2:
				InstructionSet::OpcodeE2::Execute(state);
				break;
			case 0xE3:
				InstructionSet::OpcodeE3::Execute(state);
				break;
			case 0xE4:
				InstructionSet::OpcodeE4::Execute(state);
				break;
			case 0xE5:
				InstructionSet::OpcodeE5::Execute(state);
				break;
			case 0xE6:
				InstructionSet::OpcodeE6::Execute(state);
				break;
			case 0xE7:
				InstructionSet::OpcodeE7::Execute(state);
				break;
			case 0xE8:
				InstructionSet::OpcodeE8::Execute(state);
				break;
			case 0xE9:
				InstructionSet::OpcodeE9::Execute(state);
				break;
			case 0xEA:
				InstructionSet::OpcodeEA::Execute(state);
				break;
			case 0xEB:
				InstructionSet::OpcodeEB::Execute(state);
				break;
			case 0xEC:
				InstructionSet::OpcodeEC::Execute(state);
				break;
			case 0xED:
				InstructionSet::OpcodeED::Execute(state);
				break;
			case 0xEE:
				InstructionSet::OpcodeEE::Execute(state);
				break;
			case 0xEF:
				InstructionSet::OpcodeEF::Execute(state);
				break;
			case 0xF0:
				InstructionSet::OpcodeF0::Execute(state);
				break;
			case 0xF1:
				InstructionSet::OpcodeF1::Execute(state);
				break;
			case 0xF2:
				InstructionSet::OpcodeF2::Execute(state);
				break;
			case 0xF3:
				InstructionSet::OpcodeF3::Execute(state);
				break;
			case 0xF4:
				InstructionSet::OpcodeF4::Execute(state);
				break;
			case 0xF5:
				InstructionSet::OpcodeF5::Execute(state);
				break;
			case 0xF6:
				InstructionSet::OpcodeF6::Execute(state);
				break;
			case 0xF7:
				InstructionSet::OpcodeF7::Execute(state);
				break;
			case 0xF8:
				InstructionSet::OpcodeF8::Execute(state);
				break;
			case 0xF9:
				InstructionSet::OpcodeF9::Execute(state);
				break;
			case 0xFA:
				InstructionSet::OpcodeFA::Execute(state);
				break;
			case 0xFB:
				InstructionSet::OpcodeFB::Execute(state);
				break;
			case 0xFC:
				InstructionSet::OpcodeFC::Execute(state);
				break;
			case 0xFD:
				InstructionSet::OpcodeFD::Execute(state);
				break;
			case 0xFE:
				InstructionSet::OpcodeFE::Execute(state);
				break;
			case 0xFF:
				InstructionSet::OpcodeFF::Execute(state);
				break;
			}
#endif
		}

		template<class InstructionSet>
		static FINLINE int Run6502(MachineType &state,int maxcycles,const InstructionSet &) {
			int startcycles=state.cycles;
			int endcycles=startcycles+maxcycles;
			while(state.cycles<endcycles) {
				RunSingleInstruction(state,InstructionSet());
			}
			return state.cycles-startcycles;
		}
	};

	template<class C>
	const Word Generic6502<C>::nmivector={0xFFFA};
	template<class C>
	const Word Generic6502<C>::resetvector={0xFFFC};
	template<class C>
	const Word Generic6502<C>::irqvector={0xFFFE};

	template<class C>
	void Generic6502<C>::InitialiseCPUState(MachineType &state) {
		state.cpustate.a=0;
		state.cpustate.x=0;
		state.cpustate.y=0;
		state.cpustate.s.h=1;
		state.cpustate.s.l=0xFF;
		state.cpustate.pc.w=0;
		state.cycles=0;
		state.cpustate.p=B_MASK|U_MASK;
	}

	template<class C>
	const t65TYPENAME C::ReadZP *Generic6502<C>::catReadZP=0;

	template<class C>
	const t65TYPENAME C::WriteZP *Generic6502<C>::catWriteZP=0;

	template<class C>
	const t65TYPENAME C::ReadStack *Generic6502<C>::catReadStack=0;

	template<class C>
	const t65TYPENAME C::WriteStack *Generic6502<C>::catWriteStack=0;

	template<class C>
	const t65TYPENAME C::FetchInstr *Generic6502<C>::catFetchInstr=0;

	template<class C>
	const t65TYPENAME C::FetchAddress *Generic6502<C>::catFetchAddress=0;

	template<class C>
	const t65TYPENAME C::ReadOperand *Generic6502<C>::catReadOperand=0;

	template<class C>
	const t65TYPENAME C::WriteOperand *Generic6502<C>::catWriteOperand=0;

	template<class C>
	const t65TYPENAME C::ReadDebug *Generic6502<C>::catReadDebug=0;

	template<class C>
	const t65TYPENAME C::WriteDebug *Generic6502<C>::catWriteDebug=0;

#ifdef t65_USE_NZTABLE
	template<class C>
	byte Generic6502<C>::nztable[256];
#endif

	template<class C>
	void Generic6502<C>::Init() {
#ifdef t65_USE_NZTABLE
		unsigned i;

		for(i=0;i<256;++i) {
			byte p=0;

			if(i==0) {
				p|=Z_MASK;
			}
			if(i&0x80) {
				p|=N_MASK;
			}
			Generic6502<C>::nztable[i]=p;
		}
//		printf("Generic6502<%s>::Init() done\n",
//			typeid(C).name());
#endif
	}
}

#endif
