#include "pch.h"
#include "mbMainFrame.h"
#include <bbcSY6502A.h>
#include <bbcModelB.h>
#include <bbcModelBPlus.h>
#include <bbcModelMaster128.h>
#include <bbcComputer.h>
#include <bbcKeyboardMatrix.h>
#include <bbcSound.h>
#include <bbcVideo.h>
#include <bbcTeletextFont.h>
#include <bbcSystemVIA.h>
#include <bbcUserVIA.h>
#include <bbc1770.h>
#include <bbc1770OpusChallenger.h>
#include <bbc1770Acorn.h>
#include <bbc1770Opus.h>
#include <bbc1770Watford.h>
#include "mbMisc.h"
#include "mbKeys.h"

//////////////////////////////////////////////////////////////////////////
// Determines how often things happen, in emulated terms

static bbcComputer the_beeb;

void mbMainFrame::DoBeebTick() {
	this->SetAutoPaint(false);
	for(;;) {
		bbcComputer::Run6502();

		//This is to cater for overflow, which takes about 15 mins to happen at 2MHz.
		//No update will cause the cycles counter to jump by 2,000,000.
		bbcComputer::next_stop+=2000000;//=INT_MAX;
		bbcComputer::Update();
		//Do this first, so it can change the screen buffer to indicate stuff
		if(int(bbcComputer::cycles-next_update_sound_)>0) {//bbcComputer::cycles>next_update_sound_) {
			this->DoUpdateSound();
			next_update_sound_=bbcComputer::cycles+update_sound_cycles;
		}
		if(bbcVideo::buffer_dirty) {
			if(screenshot_countdown_==1) {
				this->DoScreenshot();
			}
			int freq=beeb_fastforward_?50:cfg_.video.frame_skip;
			if(num_frames_%freq==0) {
				if(HostGfx_IsFullScreen(hostgfx_)) {
					bbcVideo::SetPixelFixedBits(HostGfx_FULL_SCREEN_PALETTE_BASE);
				} else {
					bbcVideo::SetPixelFixedBits(0);
				}
				HostGfx_Present(hostgfx_,bbcVideo::buffer,bbcVideo::buffer_width,
					bbcVideo::buffer_height,bbcVideo::buffer_width);
			}
			bbcVideo::buffer_dirty=false;
			if(screenshot_countdown_>0||num_frames_%cfg_.video.cleanup_frequency==0) {
				memset(bbcVideo::buffer,0,bbcVideo::buffer_size);
			}
			if(screenshot_countdown_>0) {
				--screenshot_countdown_;
			}
			++num_frames_;
		}
		if(int(bbcComputer::cycles-next_update_keys_)>0) {//bbcComputer::cycles>next_update_keys_) {
			this->DoUpdateKeys();
			next_update_keys_=bbcComputer::cycles+update_keys_cycles;
//			if(bbcKeyboardMatrix::KeyState(mbKEY_QUICKQUIT)) {
//				break;
//			}
		}
		if(int(bbcComputer::cycles-next_update_sys_)>0) {//bbcComputer::cycles>next_update_sys_) {
			break;
		}
	}
	this->SetAutoPaint(true);
}

//TODO some of this should go into RefreshWindow, possibly rename it to
//ApplySettings or something.
bool mbMainFrame::ApplyConfig() {
	unsigned i;
	std::vector<wxByte> rom;

	wxLogDebug("ApplyConfig...\n");

	//If the OS ROM is empty, fill with dummy code.
	//This allows the Beeb to boot and the user can select an OS filename there.
	wxLogVerbose("OS ROM: \"%s\"\n",cfg_.roms.os_filename.c_str());
	if(cfg_.roms.os_filename.empty()||!mbLoadFile(cfg_.roms.os_filename,rom)) {
		wxLogMessage("OS ROM: failed to load \"%s\"\n",cfg_.roms.os_filename.c_str());
		wxLogMessage("OS ROM: Using default dummy ROM\n");
		wxLogMessage("OS ROM: BBC will hang, use ROMs dialog to set OS ROM.\n");
	}
	bbcComputer::SetOsRomContents(&rom[0],rom.size());
//	bbcComputer::SetHasRomBoard(true);
	for(i=0;i<16;++i) {
		const mbRomsConfig::Rom *r=&cfg_.roms.roms[i];
		if(r->filename.empty()) {
			rom.clear();
		} else {
			wxLogVerbose("ROM %d: \"%s\"",i,r->filename.c_str());
			if(!mbLoadFile(r->filename,rom)) {
				wxLogError("ROM %d: failed to load \"%s\"\n",i,r->filename.c_str());
				return false;
			}
		}
		bbcComputer::SetRomContents(i,&rom[0],rom.size());
		bbcComputer::SetRomWriteable(i,r->ram);
	}

	this->SetSoundEnabled(cfg_.sound.enabled);
	for(i=0;i<4;++i) {
		bbcSound::SetChannelStereoLeftRight(i,cfg_.sound.is_stereo_left[i],
			cfg_.sound.is_stereo_right[i]);
	}
	if(hostsnd_) {
		HostSnd_SetVolume(hostsnd_,cfg_.sound.volume);
	}

	wxLogDebug("ApplyConfig: Keymap: keymap \"%s\"\n",cfg_.keyboard.selected_keymap.c_str());
	this->SetKeymap(cfg_.keyboard.selected_keymap);

	//If there is no window size stored in the INI file, resize the BBC screen
	//to the size specified in the INI file.
	this->Show();
	if(cfg_.misc.window_rect.GetWidth()<0) {
		wxLogDebug("ApplyConfig: no saved window rect: BBC screen is %dx%d\n",
			cfg_.video.width,cfg_.video.height);
		beeb_screen_->SetSize(cfg_.video.width,cfg_.video.height);
		this->GetSizer()->Layout();
	} else {
		wxLogDebug("ApplyConfig: saved window rect is @(%d,%d) +(%d,%d)\n",
			cfg_.misc.window_rect.GetLeft(),cfg_.misc.window_rect.GetTop(),
			cfg_.misc.window_rect.GetWidth(),cfg_.misc.window_rect.GetHeight());
		this->SetSize(cfg_.misc.window_rect);
	}
	
	if(cfg_.misc.window_is_maximized) {
		wxLogDebug("ApplyConfig: Maximizing window\n");
		this->Maximize();
	}

	this->RefreshWindow();
	
	this->PowerOnBeeb();

	beeb_paused_=false;

	wxLogDebug("ApplyConfig: done.\n");
	
	return true;
}

void mbMainFrame::PowerOnBeeb() {
	HostTmr_GetHfFrequency(&hf_freq_);
	
	//Beeb setup
	{
		const bbc1770Interface *disc_interface=0;
		wxString disc_lc=cfg_.misc.disc_interface.Lower();
		if(disc_lc=="acorn1770") {
			disc_interface=&bbc1770Acorn::disc_interface;
		} else if(disc_lc=="opus1770") {
			disc_interface=&bbc1770Opus::disc_interface;
		} else if(disc_lc=="chal256") {
			disc_interface=&bbc1770OpusChallenger::disc_interface_256k;
		} else if(disc_lc=="chal512") {
			disc_interface=&bbc1770OpusChallenger::disc_interface_512k;
		} else if(disc_lc=="watford1770") {
			disc_interface=&bbc1770Watford::disc_interface;
		}
		if(!disc_interface) {
			wxLogWarning("Unknown disc interface \"%s\"",cfg_.misc.disc_interface);
			cfg_.misc.disc_interface="acorn1770";
			disc_interface=&bbc1770Acorn::disc_interface;
			wxLogWarning("Disc interface set to \"%s\"",cfg_.misc.disc_interface);
			wxLogWarning("(\"%s\")",disc_interface->name);
		}

		bbcModel *model=0;
		wxString model_lc=cfg_.misc.model.Lower();
		if(model_lc=="b") {
			model=&bbcModelB::model;//new bbcModelB;
		} else if(model_lc=="b+") {
			model=&bbcModelBPlus::model;//new bbcModelBPlus;
		}
		if(!model) {
			wxLogWarning("BBC model set to 'B'");
			wxLogWarning("Unknown BBC model '%s'",cfg_.misc.model);
			cfg_.misc.model="b";
			model=&bbcModelB::model;//new bbcModelB;
		}

		bbcComputer::Init(disc_interface,cfg_.video.width,cfg_.video.height,model);
	}

	bbcSound::SetFrequency(cfg_.sound.hz);
	
	for(unsigned i=0;i<8;++i) {
		bbcKeyboardMatrix::ForceSetKeyState(bbcKEY_KEYLINKS+i,cfg_.keyboard.keylinks[i]);
	}
	
	num_frames_=0;
	
	HostTmr_ReadMilliseconds(&last_update_sys_time_);
	HostTmr_ReadHf(&last_update_sound_hf_time_);
	next_update_keys_=bbcComputer::cycles;
	next_update_sys_=bbcComputer::cycles;
	next_update_sound_=bbcComputer::cycles;
	last_update_sys_time_=bbcComputer::cycles;

	bbcComputer::ResetHard();
}
/*
void mbMainFrame::ResetBreak() {
	bbcComputer::SoftReset();
//	bbcModelBSim::Reset6502(the_beeb);
}

void mbMainFrame::ResetHard() {
}
*/
//////////////////////////////////////////////////////////////////////////
// Running disassembly
static const char hex_digits[]="0123456789ABCDEF";
/*
#ifdef BEEB_DEBUG_DISASSEMBLY
void mbMainFrame::DisassemblyAddLine(t65::Word addr,int when) {
	static t65::byte bytes_buf[10];
	static char opcode_buf[10],operand_buf[10];
	
	HistoryEntry *ent=&*history_.insert(history_.end(),HistoryEntry());

	ent->type=HIST_INSTR;
	ent->cycles=when;
	ent->next_stop=bbcComputer::next_stop;
	HistoryInstr *ins=&ent->instr;
	ins->pc=addr;
	ins->a=bbcComputer::cpustate.a;
	ins->x=bbcComputer::cpustate.x;
	ins->y=bbcComputer::cpustate.y;
	ins->p=bbcComputer::cpustate.p;
	ins->s=bbcComputer::cpustate.s.l;
	bbcModelBSim::ByteReader br;
	ins->bytes[0]=br(addr);
	++addr.w;
	ins->bytes[1]=br(addr);
	++addr.w;
	ins->bytes[2]=br(addr);
}

void mbMainFrame::DisassemblyAddIrq() {
	HistoryEntry *ent=&*history_.insert(history_.end(),HistoryEntry());
	ent->type=HIST_IRQ;
	ent->cycles=bbcComputer::cycles;
	ent->next_stop=bbcComputer::next_stop;
	HistoryIrq *irq=&ent->irq;
	irq->irqflags=bbcComputer::irq_flags_;
	irq->sys.ier=bbc_system_via.Ier();
	irq->sys.ifr=bbc_system_via.Ifr();
	irq->sys.acr=bbc_system_via.Acr();
	irq->usr.ier=bbc_user_via.Ier();
	irq->usr.ifr=bbc_user_via.Ifr();
	irq->usr.acr=bbc_user_via.Acr();
	irq->fdc_status=bbc1770::Status();
}
*/

#ifdef BEEB_DEBUG_DISASSEMBLE_RAM
void mbMainFrame::OnDisassembleRam(wxCommandEvent &event) {
	char opcode_buf[10],operand_buf[15];
	t65::byte bytes_buf[3];

	FILE *h=fopen("ram_disassembly.txt","wt");
	if(!h) {
		return;
	}
	int address=0;
	while(address<=0xFFFF) {
		t65::Word begin;
		begin.w=t65::word(address);
		t65::Word end=bbcModelBSim::Disassemble(the_beeb,begin,opcode_buf,operand_buf,
			bytes_buf);
		int len=end.w-begin.w;
		if(len<0) {
			//Went off the end of memory.
			//Never mind.
			break;
		}
		const char *nullbytes="  ";
		char bytes[3][3];
		for(int i=0;i<3;++i) {
			bytes[i][0]=hex_digits[bytes_buf[i]>>4];
			bytes[i][1]=hex_digits[bytes_buf[i]&0xf];
			bytes[i][2]=0;
		}
		
		fprintf(h,"%04X: %-5s %-10s %s %s %s\n",
			address,opcode_buf,operand_buf,
			len>0?bytes[0]:nullbytes,
			len>1?bytes[1]:nullbytes,
			len>2?bytes[2]:nullbytes);

		address+=len;
	}
	fclose(h);
}
#endif

#ifdef bbcDEBUG_TRACE
#endif
