#include "pch.h"
#include "mbKeyboardConfigDialog.h"
#include "HostWin32.h"
#include <bbcKeys.h>
#include "mbKeyboardLayout.inl"
#include "mbMisc.h"
#include <algorithm>
#include <set>

static const int border_size=6;

//////////////////////////////////////////////////////////////////////////
//
class SaveStringDialog:
public wxDialog
{
public:
	SaveStringDialog(wxWindow *parent,const wxArrayString &strings,
		const wxString &title,const wxString &caption,const wxString &def="");

	const wxString &Result() const;
protected:
private:
	DECLARE_EVENT_TABLE();
	enum {
		id_list=1,
		id_ok,
		id_input,
	};
	wxArrayString strings_;
	wxListBox *list_;
	wxTextCtrl *text_;
	wxString result_;

	void RefreshDialog();
	void OnOk(wxCommandEvent &);
	void OnList(wxListEvent &);
};

SaveStringDialog::SaveStringDialog(wxWindow *parent,const wxArrayString &strings,
	const wxString &title,const wxString &caption,const wxString &def):
wxDialog(parent,-1,title,wxDefaultPosition,wxDefaultSize,
	wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER),
strings_(strings)
{
	//Controls
	wxStaticText *caption_st=0;
	if(!caption.empty()) {
		caption_st=new wxStaticText(this,-1,caption);
	}
	list_=new wxListBox(this,id_list,wxDefaultPosition,wxDefaultSize,0,0,
		wxLB_SINGLE|wxLB_SORT);
	list_->Append(strings_);
	wxButton *ok=new wxButton(this,id_ok,"OK");
	ok->SetDefault();
	wxButton *cancel=new wxButton(this,wxID_CANCEL,"Cancel");
	text_=new wxTextCtrl(this,id_input);//,wxEmptyString,wxDefaultPosition,wxDefaultSize,)

	//Layout
	wxBoxSizer *btns_sz=new wxBoxSizer(wxHORIZONTAL);
	btns_sz->Add(ok,0,wxALL|wxALIGN_RIGHT,border_size);
	btns_sz->Add(cancel,0,wxALL|wxALIGN_RIGHT,border_size);
	wxBoxSizer *all_sz=new wxBoxSizer(wxVERTICAL);
	if(caption_st) {
		all_sz->Add(caption_st,0,wxALL,border_size);
	}
	all_sz->Add(text_,1,wxALL,border_size);
	all_sz->Add(list_,1,wxALL|wxGROW,border_size);
	all_sz->Add(btns_sz,0,0);

	this->SetSizer(all_sz);
	all_sz->SetSizeHints(this);

	text_->Clear();
	text_->AppendText(def);
	this->RefreshDialog();
}

void SaveStringDialog::RefreshDialog() {
	wxString cur=text_->GetLineText(0);
	int idx=list_->FindString(cur);
	if(idx>=0) {
		list_->SetSelection(idx);
	}
}

void SaveStringDialog::OnList(wxListEvent &event) {
	bool double_click=false;

	//For some reason wxEVT_COMMAND_LISTBOX_DOUBLECLICKED and
	//wxEVT_COMMAND_LISTBOX_SELECTED _aren't_ constants, even though it looks
	//like they should be (because they are enums)...
	WXTYPE event_type=event.GetEventType();
	if(event_type==wxEVT_COMMAND_LISTBOX_DOUBLECLICKED) {
		double_click=true;
	} else if(event_type!=wxEVT_COMMAND_LISTBOX_SELECTED) {
		//Not interested in anything else!
		return;
	}
/*
	switch(event.GetEventType()) {
	case wxEVT_COMMAND_LISTBOX_DOUBLECLICKED:
		double_click=true;
		//Fall through
	case wxEVT_COMMAND_LISTBOX_SELECTED:
		break;
	default:
		//Not interested in anything else!
		return;
	}
*/
	int idx=list_->GetSelection();
	wxString selection=list_->GetString(idx);
	text_->Clear();
	text_->AppendText(selection);
	if(double_click) {
		this->OnOk(wxCommandEvent());
	}
}

void SaveStringDialog::OnOk(wxCommandEvent &) {
	wxString cur=text_->GetLineText(0);
	bool do_save=true;
	if(list_->FindString(cur)>=0) {
		wxString msg=wxString::Format("Are you sure you want to overwrite\n\"%s\"?",
			cur.c_str());
		do_save=wxMessageBox(msg,"Overwrite?",wxYES_NO,this)==wxYES;
	}
	if(do_save) {
		result_=cur;
	}
	this->EndModal(wxID_OK);
}

const wxString &SaveStringDialog::Result() const {
	return result_;
}

BEGIN_EVENT_TABLE(SaveStringDialog,wxDialog)
	EVT_BUTTON(id_ok,SaveStringDialog::OnOk)
	EVT_LISTBOX(id_list,SaveStringDialog::OnList)
	EVT_LISTBOX_DCLICK(id_list,SaveStringDialog::OnList)
END_EVENT_TABLE()

//////////////////////////////////////////////////////////////////////////
//
class SelectHostKeysDialog:
public wxDialog
{
public:
	SelectHostKeysDialog(wxWindow *parent,const mbKeymap::HostKeys *host_keys);
	~SelectHostKeysDialog();
	void GetHostKeys(mbKeymap::HostKeys *host_keys);
protected:
private:
	DECLARE_EVENT_TABLE();
	mbKeymap::HostKeys host_keys_;
	HostInpState *hostinp_;
	wxListBox *host_keys_lb_;

	void ToggleKeyState(unsigned key);
	void OnIdle(wxIdleEvent &event);
	void FillListBoxWithHostKeys();
};

SelectHostKeysDialog::SelectHostKeysDialog(wxWindow *parent,
	const mbKeymap::HostKeys *host_keys):
wxDialog(parent,-1,"Select keys",wxDefaultPosition,wxDefaultSize,
	wxDEFAULT_DIALOG_STYLE|wxRESIZE_BORDER),
host_keys_(*host_keys),
hostinp_(0),
host_keys_lb_(0)
{
	hostinp_=HostInp_Start(this);
	wxBoxSizer *btns_sz=new wxBoxSizer(wxVERTICAL);
	wxButton *ok=new wxButton(this,wxID_OK,"OK");
	wxButton *cancel=new wxButton(this,wxID_CANCEL,"Cancel");
	btns_sz->Add(ok,1,wxALL,border_size);
	btns_sz->Add(cancel,1,wxALL,border_size);
	host_keys_lb_=new wxListBox(this,-1,wxDefaultPosition,wxDefaultSize,0,0,
		wxLB_SORT|wxLB_SINGLE);
	wxBoxSizer *all_sz=new wxBoxSizer(wxHORIZONTAL);
	all_sz->Add(host_keys_lb_,1,wxGROW|wxALL,border_size);
	all_sz->Add(btns_sz,0,wxALL|wxALIGN_RIGHT,border_size);
	this->SetSizer(all_sz);
	all_sz->SetSizeHints(this);
	this->FillListBoxWithHostKeys();
}

SelectHostKeysDialog::~SelectHostKeysDialog() {
	HostInp_Stop(hostinp_);
}

void SelectHostKeysDialog::ToggleKeyState(unsigned key) {
	mbKeymap::HostKeys::iterator k;

	k=std::find(host_keys_.begin(),host_keys_.end(),key);
	if(k==host_keys_.end()) {
		host_keys_.push_back(key);
	} else {
		host_keys_.erase(k);
	}
}

void SelectHostKeysDialog::OnIdle(wxIdleEvent &event) {
	unsigned char flags[256];
	HostInp_Poll(hostinp_);
	HostInp_GetKeyboardDownFlags(hostinp_,flags);
	bool changed=false;
	for(unsigned i=0;i<256;++i) {
		if(flags[i]) {
			this->ToggleKeyState(i);
			changed=true;
		}
	}
	if(changed) {
		this->FillListBoxWithHostKeys();
	}
	event.RequestMore();
}

void SelectHostKeysDialog::FillListBoxWithHostKeys() {
	if(!host_keys_lb_) {
		return;
	}
	host_keys_lb_->Clear();
	for(mbKeymap::HostKeys::const_iterator hk=host_keys_.begin();
		hk!=host_keys_.end();++hk)
	{
		host_keys_lb_->Append(HostInp_KeyNameFromCode(*hk));
	}
}

void SelectHostKeysDialog::GetHostKeys(mbKeymap::HostKeys *host_keys) {
	wxASSERT(host_keys);
	*host_keys=host_keys_;
}

BEGIN_EVENT_TABLE(SelectHostKeysDialog,wxDialog)
	EVT_IDLE(SelectHostKeysDialog::OnIdle)
END_EVENT_TABLE()

//////////////////////////////////////////////////////////////////////////
//
enum {
	id_base=1,
	id_beebkey,
	id_load,
	id_save,
	id_close,
};

mbKeyboardConfigDialog::Key::Key():
keycap(0),
button(0),
row(-1),
x(-1)
{
}

static const wxString default_title="Configure Keymap";

mbKeyboardConfigDialog::mbKeyboardConfigDialog(wxWindow *parent,
	const mbKeyboardConfig &cfg):
wxDialog(parent,-1,"Configure Keymap",wxDefaultPosition,wxDefaultSize,
	wxDEFAULT_DIALOG_STYLE),//|wxRESIZE_BORDER),
cfg_(cfg),
row_width_(0)
{
	this->SetExtraStyle(this->GetExtraStyle()|wxWS_EX_VALIDATE_RECURSIVELY);

	wxBoxSizer *all_sz=new wxBoxSizer(wxVERTICAL);

	//////////////////////////////////////////////////////////////////////////
	//Beeb kb box
	wxStaticBox *kb_box=new wxStaticBox(this,-1,"Beeb Keys");
	wxStaticBoxSizer *kb_box_sz=new wxStaticBoxSizer(kb_box,wxVERTICAL);
	wxScrolledWindow *scrollw=new wxScrolledWindow(this,-1);
	wxBoxSizer *kb_rows_sz=new wxBoxSizer(wxVERTICAL);

	unsigned row_idx;
	row_width_=0;
	for(row_idx=0;row_idx<num_keyboard_rows;++row_idx) {
		row_width_=wxMax(keyboard_rows[row_idx].TotalWidthInHalves(),row_width_);
	}
	for(row_idx=0;row_idx<num_keyboard_rows;++row_idx) {
		wxBoxSizer *row_sz=new wxBoxSizer(wxHORIZONTAL);
		const KeyboardRow *row=&keyboard_rows[row_idx];
		int x=0;
		for(unsigned key_idx=0;key_idx<row->num_keys;++key_idx) {
			Key *k=&*keys_.insert(keys_.end(),Key());
			k->keycap=&row->keys[key_idx];
			k->row=row_idx;
			k->x=x;
			k->button=new wxButton(scrollw,id_beebkey,k->keycap->Caption(),
				wxDefaultPosition,wxDefaultSize);//,wxBU_EXACTFIT);
			if(!k->keycap->shifted&&!k->keycap->unshifted) {
				k->button->Show(false);
			}
			x+=k->keycap->WidthInHalves();
			k->button->SetSize(wxSize(k->keycap->WidthInHalves()*16,32));
			row_sz->Add(k->button,0);
		}
		kb_rows_sz->Add(row_sz);//,1,wxGROW);
	}

	//scrollw->SetScrollbars(1,1,50,50);
	scrollw->SetSizer(kb_rows_sz);
	kb_rows_sz->SetSizeHints(scrollw);
	kb_box_sz->Add(scrollw,1,wxALL|wxGROW);

	//////////////////////////////////////////////////////////////////////////
	// Buttons area
	wxBoxSizer *btns_sz=new wxBoxSizer(wxHORIZONTAL);
	wxButton *load_bt=new wxButton(this,id_load,"&Load...");
	wxButton *save_bt=new wxButton(this,id_save,"&Save...");
	wxButton *close_bt=new wxButton(this,id_close,"&Close");
	close_bt->SetDefault();

	btns_sz->Add(load_bt,1,wxALL|wxALIGN_RIGHT|wxALIGN_BOTTOM,border_size);
	btns_sz->Add(save_bt,1,wxALL|wxALIGN_RIGHT|wxALIGN_BOTTOM,border_size);
	btns_sz->Add(close_bt,1,wxALL|wxALIGN_RIGHT|wxALIGN_BOTTOM,border_size);
	
	//////////////////////////////////////////////////////////////////////////
	// Join them all together!
	all_sz->Add(kb_box_sz,1,wxGROW);
	all_sz->Add(btns_sz,0,wxALIGN_RIGHT);//wxALIGN_BOTTOM);

	this->SetSizer(all_sz);
	all_sz->SetSizeHints(this);

	//
	this->LoadKeymap(cfg_.selected_keymap);

	this->Move(cfg_.dlg_rect.GetLeft(),cfg_.dlg_rect.GetTop());
}

// fixed so OK to pass 'current_keymap_.name' as keymap_name [11/5/2003]
bool mbKeyboardConfigDialog::LoadKeymap(const wxString &keymap_name) {
	mbKeymap::C::iterator km;
	for(km=cfg_.keymaps.begin();km!=cfg_.keymaps.end();++km) {
		if(km->name==keymap_name) {
			break;
		}
	}
	wxASSERT(km!=cfg_.keymaps.end());
	current_keymap_=*km;
	current_keymap_changed_=false;
	this->SetTitle(default_title+" "+current_keymap_.name);
	return true;
}

void mbKeyboardConfigDialog::Changed() {
	current_keymap_changed_=true;
}

void mbKeyboardConfigDialog::OnBeebKey(wxCommandEvent &event) {
	wxObject *src=event.GetEventObject();
	unsigned i;
	for(i=0;i<keys_.size();++i) {
		if(keys_[i].button==src) {
			break;
		}
	}
	if(i<keys_.size()) {
		const Key *key=&keys_[i];
		mbKeymap::HostKeys *oldkeys=&current_keymap_.keymap[key->keycap->beeb_key];
		SelectHostKeysDialog dlg(this,oldkeys);
		if(dlg.ShowModal()==wxID_OK) {
			mbKeymap::HostKeys newkeys;
			dlg.GetHostKeys(&newkeys);
			if(newkeys.size()!=oldkeys->size()) {
				this->Changed();
			} else {
				//super lame :)
				std::sort(oldkeys->begin(),oldkeys->end());
				std::sort(newkeys.begin(),newkeys.end());
				for(i=0;i<oldkeys->size();++i) {
					if(oldkeys->at(i)!=newkeys[i]) {
						this->Changed();
						break;
					}
				}
			}
			*oldkeys=newkeys;
		}
	}
}

bool mbKeyboardConfigDialog::DestructiveActionConfirmed() {
	static const char *caption=
		"You will lose your changes if you do this.\n"
		"\n"
		"Are you sure?";
	int r=wxYES;
	if(current_keymap_changed_) {
		r=wxMessageBox(caption,"Are you sure?",wxYES_NO,this);
	}
	return r==wxYES;
}

void mbKeyboardConfigDialog::OnLoad(wxCommandEvent &event) {
	if(this->DestructiveActionConfirmed()) {
		wxArrayString names;
		for(mbKeymap::C::iterator km=cfg_.keymaps.begin();km!=cfg_.keymaps.end();++km) {
			names.Add(km->name);
		}
		wxString selected=wxGetSingleChoice("Select keymap to load","Load keymap",
			names,this);
		if(!selected.empty()) {
			this->LoadKeymap(selected);
		}
	}
}

void mbKeyboardConfigDialog::OnSave(wxCommandEvent &event) {
	wxArrayString names;
	mbKeymap::C::iterator km;
	for(km=cfg_.keymaps.begin();km!=cfg_.keymaps.end();++km) {
		names.Add(km->name);
	}
	SaveStringDialog dlg(this,names,"Save keymap","",current_keymap_.name);
	int r=dlg.ShowModal();
	if(r==wxID_OK) {
		wxString name=dlg.Result();
		if(!name.empty()) {
			//If there's one with the same name, get rid of it
			for(km=cfg_.keymaps.begin();km!=cfg_.keymaps.end();++km) {
				if(km->name==name) {
					break;
				}
			}
			current_keymap_.name=name;
			if(km==cfg_.keymaps.end()) {
				cfg_.keymaps.insert(cfg_.keymaps.end(),current_keymap_);
			} else {
				*km=current_keymap_;
			}
			this->LoadKeymap(current_keymap_.name);
		}
	}
}

void mbKeyboardConfigDialog::OnCloseButton(wxCommandEvent &event) {
	this->Close();
}

void mbKeyboardConfigDialog::OnClose(wxCloseEvent &event) {
	if(event.CanVeto()&&!this->DestructiveActionConfirmed()) {
		event.Veto();
	} else {
		cfg_.dlg_rect=this->GetRect();
		this->EndModal(wxID_OK);
	}
}

void mbKeyboardConfigDialog::GetResult(mbKeyboardConfig *cfg) {
	wxASSERT(cfg);
	*cfg=cfg_;
}

BEGIN_EVENT_TABLE(mbKeyboardConfigDialog,wxDialog)
	EVT_BUTTON(id_beebkey,mbKeyboardConfigDialog::OnBeebKey)
	EVT_BUTTON(id_load,mbKeyboardConfigDialog::OnLoad)
	EVT_BUTTON(id_save,mbKeyboardConfigDialog::OnSave)
	EVT_BUTTON(id_close,mbKeyboardConfigDialog::OnCloseButton)
	EVT_CLOSE(mbKeyboardConfigDialog::OnClose)
END_EVENT_TABLE()
