#include "pch.h"
#include "HostWin32.h"
#include <math.h>
#include <bbcVideo.h>

//Right now, this only affects this file...
#define mbENABLE_FULL_SCREEN

//This is too slow for general use.
//#define CHECK_PIXEL_VALUES

//If defined, and DX3_ONLY not set, use emulated keyboard in debug build.
//DirectX3 does not support emulated keyboards it seems.
#define EMULATED_KEYBOARD

#define DIRECTDRAW_VERSION 0x300
#ifdef DX3_ONLY
#define DIRECTINPUT_VERSION 0x300
#define DIRECTSOUND_VERSION 0x300
#pragma comment(lib,"dinput.lib")
#else
#define DIRECTINPUT_VERSION 0x800
#define DIRECTSOUND_VERSION 0x800
#pragma comment(lib,"dinput8.lib")
#endif

#include <windows.h>
#include <ddraw.h>
#include <dsound.h>
#include <dinput.h>
#include <dxerr9.h>
#include <string.h>

//if you did N=offsetof(T,M), where M is of type R, then with T *p
//you can get M back like *offsetwith<R>(p,N)
template<class T>
static inline T *offsetwith(void *p,unsigned offset,const T *dummy=0) {
	return reinterpret_cast<T *>(reinterpret_cast<char *>(p)+offset);
}

//a ridiculous hack -- i'm hoping it will stop some of the funny
//full screen -> windowed mode behaviour!
static void DoSomeEvents() {
	//5 is pretty random.
	for(int i=0;i<5;++i) {
		//wxSafeYield();
	}
}

#ifdef _DEBUG
#pragma comment(lib,"dxerr9.lib")

#define DXCHK(X) RealDXCHK(X,#X,__FILE__,__LINE__)

static HRESULT RealDXCHK(HRESULT hr,const char *what,const char *file,unsigned line) {
	if(FAILED(hr)) {
		static char tmp[200];
		_snprintf(tmp,sizeof tmp,"%s(%u): ",file,line);
		OutputDebugStringA(tmp);
		OutputDebugStringA(what);
		OutputDebugStringA("\n");
		OutputDebugStringA(tmp);
		OutputDebugStringA("failed with error ");
		OutputDebugStringA(DXGetErrorString9(hr));
		OutputDebugStringA("\n");
	}
	return hr;
}
#else
#define DXCHK(X) X
#endif

template<class T>
inline void SafeRelease(T *&ptr) {
	if(ptr) {
		IUnknown_Release(ptr);
		ptr=0;
	}
}

static void dprintf(const char *fmt,...) {
	char tmpbuf[1024];
	va_list v;
	va_start(v,fmt);
	_vsnprintf(tmpbuf,sizeof tmpbuf,fmt,v);
	va_end(v);
	OutputDebugString(tmpbuf);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
// HostGfx
//
static const unsigned num_back_buffers=1;

struct HostGfxState {
	//palettes
	t65::byte lookup_8bit[8];
	t65::word lookup_16bit[8];
	//Actually, this is slightly dodgy -- if the reader should find a wrong endian
	//Windows/DirectX system let me know.
	t65::byte lookup_24bit[8][4];
	t65::dword lookup_32bit[8];

	//Owner. Used by DirectDraw. Must be toplevel window.
	HWND owner;
	//Target. Display is shown in this window, so could have
	//child window here.
	HWND target;
	IDirectDraw2 *ddraw;
	IDirectDrawSurface2 *primary;
	DDSURFACEDESC primary_desc;
	IDirectDrawSurface2 *back_buffers[num_back_buffers];
	DDSURFACEDESC back_buffer_desc;
	IDirectDrawClipper *primary_clipper;
	IDirectDrawPalette *primary_palette;
	int cpu_buffer,gpu_buffer;
	bool buffers_ok;
	HostGfx_BeebScreenType beeb_screen_type;
	HostGfx_ScanlinesMode scanlines_mode;//bool vertical_2x;

	bool full_screen;
	int full_screen_refresh_rate;

	DDCAPS halcaps;
};

static inline int HostGfx_Shift(HostGfxState *state) {
	wxASSERT(!state->full_screen);
	return state->scanlines_mode==HostGfx_DOUBLE?1:0;
}

static int HostGfx_HeightMultiplier(HostGfxState *state) {
	wxASSERT(!state->full_screen);
	return state->scanlines_mode==HostGfx_DOUBLE?2:1;
}

static void HostGfx_Copy8(HostGfxState *state,t65::byte *dest_bitmap,
	unsigned dest_pitch,const t65::byte *src_bitmap,int src_width,
	unsigned src_pitch,int src_height,int src_shift)
{
	int y;
	//const int src_shift=HostGfx_Shift(state);//state->vertical_2x?1:0;

	for(y=0;y<(src_height<<src_shift);++y) {
		memcpy(dest_bitmap+y*dest_pitch,src_bitmap+(y>>src_shift)*src_pitch,src_width);
	}
}
						  
static void HostGfx_Convert8(HostGfxState *state,t65::byte *dest_bitmap,
	unsigned dest_pitch,const t65::byte *src_bitmap,int src_width,
	unsigned src_pitch,int src_height)
{
	int y,x;
	const t65::byte *src;
	t65::byte *dest;
	const int src_shift=HostGfx_Shift(state);//state->vertical_2x?1:0;
	
	for(y=0;y<(src_height<<src_shift);++y) {
		src=src_bitmap+(y>>src_shift)*src_pitch;
		dest=dest_bitmap+y*dest_pitch;
		for(x=0;x<src_width;++x) {
#ifdef CHECK_PIXEL_VALUES
			wxASSERT((src[x]&~7)==0);
#endif
			dest[x]=state->lookup_32bit[src[x]];
		}
	}
}

static void HostGfx_Convert16(HostGfxState *state,t65::word *dest_bitmap,
	unsigned dest_pitch,const unsigned char *src_bitmap,int src_width,
	unsigned src_pitch,int src_height)
{
	int y,x;
	const t65::byte *src;
	t65::word *dest;
	const int src_shift=HostGfx_Shift(state);//state->vertical_2x?1:0;
	
	for(y=0;y<(src_height<<src_shift);++y) {
		src=src_bitmap+(y>>src_shift)*src_pitch;
		dest=dest_bitmap+y*(dest_pitch/2);
		for(x=0;x<src_width;++x) {
#ifdef CHECK_PIXEL_VALUES
			wxASSERT((src[x]&~7)==0);
#endif
			dest[x]=state->lookup_32bit[src[x]];
		}
	}
}

static void HostGfx_Convert24(HostGfxState *state,t65::byte *dest_bitmap,
	unsigned dest_pitch,const unsigned char *src_bitmap,int src_width,
	unsigned src_pitch,int src_height)
{
	int y,x;
	const t65::byte *src;
	t65::byte *dest;
	const int src_shift=HostGfx_Shift(state);//state->vertical_2x?1:0;
	
	for(y=0;y<(src_height<<src_shift);++y) {
		src=src_bitmap+(y>>src_shift)*src_pitch;
		dest=dest_bitmap+y*dest_pitch;
		for(x=0;x<src_width;++x) {
#ifdef CHECK_PIXEL_VALUES
			wxASSERT((src[x]&~7)==0);
#endif
			dest[0]=state->lookup_24bit[src[x]][0];
			dest[1]=state->lookup_24bit[src[x]][1];
			dest[2]=state->lookup_24bit[src[x]][2];
			dest+=3;
		}
	}
}

static void HostGfx_Convert32(HostGfxState *state,t65::dword *dest_bitmap,
	unsigned dest_pitch,const unsigned char *src_bitmap,int src_width,
	unsigned src_pitch,int src_height)
{
	int y,x;
	const t65::byte *src;
	t65::dword *dest;
	const int src_shift=HostGfx_Shift(state);//state->vertical_2x?1:0;
	
	for(y=0;y<(src_height<<src_shift);++y) {
		src=src_bitmap+(y>>src_shift)*src_pitch;
		dest=dest_bitmap+y*(dest_pitch/4);
		for(x=0;x<src_width;++x) {
#ifdef CHECK_PIXEL_VALUES
			wxASSERT((src[x]&~7)==0);
#endif
			dest[x]=state->lookup_32bit[src[x]];
		}
	}
}


static IDirectDrawSurface2 *NewSurface2(HostGfxState *state,DDSURFACEDESC *desc) {
	HRESULT hr;
	IDirectDrawSurface *surface;
	IDirectDrawSurface2 *surface2;
	DDBLTFX fx;
	
	hr=DXCHK(IDirectDraw2_CreateSurface(state->ddraw,desc,&surface,0));
	if(FAILED(hr)) {
		return 0;
	}
	hr=DXCHK(IDirectDrawSurface_QueryInterface(surface,IID_IDirectDrawSurface2,
		reinterpret_cast<void **>(&surface2)));
	surface->Release();
	if(FAILED(hr)) {
		return 0;
	}
	
	if(!(desc->ddsCaps.dwCaps&DDSCAPS_PRIMARYSURFACE)) {
		memset(&fx,0,sizeof fx);
		fx.dwSize=sizeof fx;
		fx.dwFillColor=0;
		DXCHK(IDirectDrawSurface2_Blt(surface2,0,0,0,DDBLT_COLORFILL|DDBLT_WAIT,&fx));
	}

	return surface2;
}

static void HostGfx_StopInternal(HostGfxState *state) {
	wxASSERT(state);
	if(state->ddraw) {
		if(state->full_screen) {
			IDirectDraw_RestoreDisplayMode(state->ddraw);
			DoSomeEvents();
		}
		IDirectDraw_SetCooperativeLevel(state->ddraw,0/*state->owner*/,DDSCL_NORMAL);
		DoSomeEvents();
	}
	wxLogDebug("HostGfx_StopInternal: Release: primary_clipper=0x%p\n",
		state->primary_clipper);
	SafeRelease(state->primary_clipper);
	wxLogDebug("HostGfx_StopInternal: Release: primary_palette=0x%p",
		state->primary_palette);
	SafeRelease(state->primary_palette);
	for(unsigned i=0;i<num_back_buffers;++i) {
		wxLogDebug("HostGfx_StopInternal: Release: back_buffers[%u]=0x%p",
			i,state->back_buffers[i]);
		SafeRelease(state->back_buffers[i]);
	}
	wxLogDebug("HostGfx_StopInternal: Release: primary=0x%p",state->primary);
	SafeRelease(state->primary);
	wxLogDebug("HostGfx_StopInternal: Release: ddraw=0x%p",state->ddraw);
	SafeRelease(state->ddraw);
	state->buffers_ok=false;
}

static HRESULT HostGfx_StartWindowedInternal(HostGfxState *state,int w,int h) {
	DDSURFACEDESC sd;
	HRESULT hr;
	unsigned i;

	wxLogDebug("HostGfx_StartWindowedInternal: SetCooperativeLevel.");
	hr=IDirectDraw2_SetCooperativeLevel(state->ddraw,state->owner,DDSCL_NORMAL);
	if(FAILED(hr)) {
		return hr;
	}
	
	//Primary surface and its description
	memset(&sd,0,sizeof sd);
	sd.dwSize=sizeof sd;
	sd.dwFlags=DDSD_CAPS;
	sd.ddsCaps.dwCaps=DDSCAPS_PRIMARYSURFACE;
	wxLogDebug("HostGfx_StartInternal: Create primary surface.");
	state->primary=NewSurface2(state,&sd);
	if(!state->primary) {
		return E_FAIL;
	}
	memset(&state->primary_desc,0,sizeof state->primary_desc);
	state->primary_desc.dwSize=sizeof state->primary_desc;
	DXCHK(IDirectDrawSurface2_GetSurfaceDesc(state->primary,&state->primary_desc));

	//Back surfaces, if wanted.
	if(w<0||h<0) {
		wxLogDebug("HostGfx_StartWindowedInternal: invalid dimensions, not creating back buffers.");
	} else {
		for(i=0;i<num_back_buffers;++i) {
			memset(&sd,0,sizeof sd);
			sd.dwSize=sizeof sd;
			sd.dwFlags=DDSD_CAPS|DDSD_WIDTH|DDSD_HEIGHT;
			sd.ddsCaps.dwCaps=DDSCAPS_OFFSCREENPLAIN;
			if(state->halcaps.dwCaps&DDCAPS_BLT) {
				sd.ddsCaps.dwCaps|=DDSCAPS_VIDEOMEMORY;
			} else {
				sd.ddsCaps.dwCaps|=DDSCAPS_SYSTEMMEMORY;
			}
			sd.dwWidth=w;
			sd.dwHeight=h<<HostGfx_Shift(state);
			wxLogDebug("HostGfx_StartWindowedInternal: Create %ux%u back buffer %u/%u.",
				sd.dwWidth,sd.dwHeight,1+i,num_back_buffers);
			state->back_buffers[i]=NewSurface2(state,&sd);
			if(!state->back_buffers[i]) {
				hr=E_FAIL;
				break;
			}
		}
	}
	
	//Clipper
	wxLogDebug("HostGfx_StartWindowedInternal: CreateClipper.");
	hr=IDirectDraw2_CreateClipper(state->ddraw,0,&state->primary_clipper,0);
	if(FAILED(hr)) {
		goto stop_error;
	}
	wxLogDebug("HostGfx_StartWindowedInternal: Set clipper window.");
	hr=IDirectDrawClipper_SetHWnd(state->primary_clipper,0,state->target);
	if(FAILED(hr)) {
		goto stop_error;
	}
	wxLogDebug("HostGfx_StartWindowedInternal: Set clipper on primary surface.");
	hr=IDirectDrawSurface2_SetClipper(state->primary,state->primary_clipper);
	if(FAILED(hr)) {
		goto stop_error;
	}
	wxLogDebug("HostGfx_StartWindowedInternal: ok.");
	
	return S_OK;

stop_error:
#ifdef _DEBUG
	wxLogDebug("HostGfx_StartWindowedInternal: failed: %s.",DXGetErrorString9(hr));
#endif
	SafeRelease(state->primary_clipper);
	for(i=0;i<num_back_buffers;++i) {
		SafeRelease(state->back_buffers[i]);
	}
	SafeRelease(state->primary);
	return E_FAIL;
}

static HRESULT WINAPI EnumAttachedSurfacesCallback(LPDIRECTDRAWSURFACE lpDDSurface,
	LPDDSURFACEDESC lpDDSurfaceDesc,LPVOID lpContext)
{
	IDirectDrawSurface2 ***ptr=static_cast<IDirectDrawSurface2 ***>(lpContext);
	HRESULT hr;

	hr=IDirectDrawSurface_QueryInterface(lpDDSurface,IID_IDirectDrawSurface2,
		reinterpret_cast<void **>(*ptr));
	if(SUCCEEDED(hr)) {
		++*ptr;
	}
	return DIENUM_CONTINUE;
}

static void ShowPalette(FILE *h,const char *palette_name,const PALETTEENTRY *palette)
{
	fprintf(h,"Palette \"%s\":\n",palette_name);
	for(int i=0;i<256;++i) {
		const PALETTEENTRY *pe=&palette[i];
		//01234567890123456
		//XXX: (XX,XX,XX) 
		fprintf(h,"%03d: (%02X,%02X,%02X)      ",i,pe->peRed,pe->peGreen,pe->peBlue);
	}
}

static HRESULT HostGfx_StartFullScreenInternal(HostGfxState *state,int w,int h) {
	DDSURFACEDESC sd;
	HRESULT hr;
	PALETTEENTRY palette[256];
	unsigned i;
	//DDSCAPS dscaps;
	DDBLTFX bltfx;
	HDC screen_dc;
	DWORD coop;
	
	//Set video mode
	wxLogDebug("HostGfx_StartFullScreenInternal: SetCooperativeLevel.");
	coop=DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN;
#ifdef _DEBUG
//	coop|=DDSCL_NOWINDOWCHANGES;
#endif
	hr=IDirectDraw_SetCooperativeLevel(state->ddraw,state->owner,coop);
	if(FAILED(hr)) {
		goto stop_error;
	}
	wxLogDebug("HostGfx_StartFullScreenInternal: SetDisplayMode, %dHz.",
		state->full_screen_refresh_rate);
	hr=IDirectDraw2_SetDisplayMode(state->ddraw,800,600,8,
		state->full_screen_refresh_rate,0);
	if(FAILED(hr)) {
		goto stop_error;
	}

	//Primary surface and its description.
	//I want to avoid catering for flipping etc., so I don't bother
	memset(&sd,0,sizeof sd);
	sd.dwSize=sizeof sd;
	sd.dwFlags=DDSD_CAPS|DDSD_BACKBUFFERCOUNT;
	sd.dwBackBufferCount=1;
	sd.ddsCaps.dwCaps=DDSCAPS_PRIMARYSURFACE|DDSCAPS_COMPLEX|DDSCAPS_FLIP;
	wxLogDebug("HostGfx_StartFullScreenInternal: Create primary surface.");
	state->primary=NewSurface2(state,&sd);
	if(!state->primary) {
		hr=E_FAIL;
		goto stop_error;
	}

	//Set that as the front buffer.
	state->back_buffers[0]=state->primary;
	IDirectDrawSurface2_AddRef(state->back_buffers[0]);

	//Clear the buffer that's going to be used.
	memset(&bltfx,0,sizeof bltfx);
	bltfx.dwSize=sizeof bltfx;
	bltfx.dwFillColor=0;
	DXCHK(IDirectDrawSurface2_Blt(state->back_buffers[0],0,0,0,
		DDBLT_COLORFILL|DDBLT_WAIT,&bltfx));

	//Here the palette is retrieved, and 8 entries reset for use by the emulator,
	//then set as the current palette. The hope is that the GDI can draw onto the
	//primary surface, allowing use of normal Windows menu bar and status bar if
	//the user wants this.

	//This seems to be the best way to get the system palette.
	//http://groups.google.com/groups?q=default+palette+directdraw&hl=en&lr=&ie=UTF-8&oe=UTF-8&selm=38F675F48B36D4118E4D00508B5EBA37BE6BBB%40cpmsftmsgv21.microsoft.com&rnum=10
	//(was the first useful one from google)
	wxLogDebug("HostGfx_StartFullScreenInternal: determining default palette\n");
	wxLogDebug("HostGfx_StartFullScreenInternal:     IDirectDrawSurface2_GetDC...\n");
	hr=IDirectDrawSurface2_GetDC(state->primary,&screen_dc);
	if(FAILED(hr)) {
		goto stop_error;
	}
	wxLogDebug("HostGfx_StartFullScreenInternal:     GetSystemPaletteEntries...\n");
	if(GetSystemPaletteEntries(screen_dc,0,256,palette)!=256) {
		hr=E_FAIL;
		//but the DC should still be released.
	}
	IDirectDrawSurface2_ReleaseDC(state->primary,screen_dc);
	//ShowPalette(stdout,"system palette",palette);
	if(FAILED(hr)) {
		goto stop_error;
	}

	//Set the 8 entries that the emulator uses.
	for(i=0;i<8;++i) {
		PALETTEENTRY *pe=&palette[HostGfx_FULL_SCREEN_PALETTE_BASE+i];
		pe->peRed=i&1?255:0;
		pe->peGreen=i&2?255:0;
		pe->peBlue=i&4?255:0;
		pe->peFlags=PC_NOCOLLAPSE;
	}
	hr=IDirectDraw_CreatePalette(state->ddraw,DDPCAPS_8BIT|DDPCAPS_ALLOW256,palette,
		&state->primary_palette,0);
	if(FAILED(hr)) {
		goto stop_error;
	}

	//It's not clear what to do if this should fail. So hope it doesn't.
	IDirectDrawSurface_SetPalette(state->primary,state->primary_palette);
	
	return S_OK;

stop_error:
#ifdef _DEBUG
	wxLogDebug("HostGfx_StartFullScreenInternal: fail: %s",DXGetErrorString9(hr));
#endif
	IDirectDraw2_RestoreDisplayMode(state->ddraw);
	IDirectDraw2_SetCooperativeLevel(state->ddraw,state->owner,DDSCL_NORMAL);
	SafeRelease(state->primary_palette);
	SafeRelease(state->back_buffers[0]);
	SafeRelease(state->primary);
	return hr;
}

static bool HostGfx_StartInternal(HostGfxState *state,int w,int h) {
	IDirectDraw *ddraw;
	HRESULT hr;
	
	//DirectDraw object
	wxLogDebug("HostGfx_StartInternal: starting DirectDraw.");
	hr=DirectDrawCreate(0,&ddraw,0);
	if(FAILED(hr)) {
		ddraw->Release();
		goto stop_error;
	}
	wxLogDebug("HostGfx_StartInternal: QueryInterface for IDirectDraw2.");
	hr=IDirectDraw_QueryInterface(ddraw,IID_IDirectDraw2,
		reinterpret_cast<void **>(&state->ddraw));
	ddraw->Release();
	if(FAILED(hr)) {
		goto stop_error;
	}
	
	wxLogDebug("HostGfx_StartInternal: GetCaps.");
	memset(&state->halcaps,0,sizeof state->halcaps);
	state->halcaps.dwSize=sizeof state->halcaps;
	hr=IDirectDraw2_GetCaps(state->ddraw,&state->halcaps,0);
	if(FAILED(hr)) {
		goto stop_error;
	}
	
	DoSomeEvents();
	if(state->full_screen) {
		hr=HostGfx_StartFullScreenInternal(state,w,h);
		DoSomeEvents();
		if(FAILED(hr)) {
			//HostGfx_StopInternal(state);
			state->full_screen=false;
		}
	}

	if(!state->full_screen) {
		hr=HostGfx_StartWindowedInternal(state,w,h);
		DoSomeEvents();
	}
	if(FAILED(hr)) {
		goto stop_error;
	}

	if(state->back_buffers[0]) {
		memset(&state->back_buffer_desc,0,sizeof state->back_buffer_desc);
		state->back_buffer_desc.dwSize=sizeof state->back_buffer_desc;
		IDirectDrawSurface2_GetSurfaceDesc(state->back_buffers[0],
			&state->back_buffer_desc);
		state->buffers_ok=true;
		state->cpu_buffer=0;
		state->gpu_buffer=num_back_buffers-1;
	}

	HostGfx_SetScreenType(state,HostGfx_COLOUR);
//	HostGfx_SetScanlinesMode(state,HostGfx_SINGLE);
	return true;
	
stop_error:
#ifdef _DEBUG
	wxLogDebug("HostGfx_StartInternal: fail: %s",DXGetErrorString9(hr));
#endif
	HostGfx_StopInternal(state);
	return false;
}

HostGfxState *HostGfx_Start(wxWindow *owner,wxWindow *target) {
	HostGfxState *new_state=new HostGfxState;
	unsigned i;

	wxLogDebug("HostGfx_Start: owner window \"%s\"",owner->GetTitle().c_str());
	new_state->owner=HWND(owner->GetHandle());
	new_state->target=HWND(target->GetHandle());
	new_state->ddraw=0;
	new_state->primary=0;
	for(i=0;i<num_back_buffers;++i) {
		new_state->back_buffers[i]=0;
	}
	new_state->primary_clipper=0;
	new_state->primary_palette=0;
	new_state->buffers_ok=0;
	new_state->scanlines_mode=HostGfx_SINGLE;
	new_state->full_screen=false;
	new_state->full_screen_refresh_rate=0;

	if(!HostGfx_StartInternal(new_state,-1,-1)) {
		delete new_state;
		return 0;
	}

	OutputDebugStringA("HostGfx_Start: ok");
	return new_state;
}

void HostGfx_Stop(HostGfxState *state) {
	if(state) {
		HostGfx_StopInternal(state);
		delete state;
	}
}

void HostGfx_Present(HostGfxState *state,const unsigned char *bitmap_3bit,int w,int h,
	unsigned pitch)
{
	DDSURFACEDESC sd;
	HRESULT hr;
	bool right_width=true,right_height=true;
	int desired_height;

	if(state->buffers_ok&&!state->full_screen) {
		desired_height=h<<(state->scanlines_mode==HostGfx_DOUBLE?1:0);
		right_width=w==int(state->back_buffer_desc.dwWidth);
		right_height=desired_height==int(state->back_buffer_desc.dwHeight);
	}
	if(!state->buffers_ok||!right_width||!right_height) {
		wxLogDebug("HostGfx_Present: settings change, restarting DirectDraw.");
		HostGfx_StopInternal(state);
		HostGfx_StartInternal(state,w,h);
		//Anything might have happened, so drop out without doing anything so
		//that next time Present is called the emulator can check if full screen
		//succeeded.
		return;
	}

	if(!state->buffers_ok) {
		return;
	}

	memset(&sd,0,sizeof sd);
	sd.dwSize=sizeof sd;
	hr=IDirectDrawSurface2_Lock(state->back_buffers[state->cpu_buffer],0,&sd,
		DDLOCK_WRITEONLY|DDLOCK_WAIT|DDLOCK_SURFACEMEMORYPTR/*|DDLOCK_NOSYSLOCK*/,0);
	if(FAILED(hr)) {
		state->buffers_ok=false;
		return;
	}

	if(state->full_screen) {
		unsigned dest_pitch=sd.lPitch;
		if(state->scanlines_mode!=HostGfx_DOUBLE) {
			dest_pitch*=2;
		}
		t65::byte *dest=static_cast<t65::byte *>(sd.lpSurface);
		dest+=(300-h)*sd.lPitch;
		dest+=(400-w/2);
		static bool alternate=false;
		alternate=!alternate;
		if(alternate&&state->scanlines_mode==HostGfx_INTERLACE) {
			dest+=sd.lPitch;
		}
//		bbcVideo::SetPixelFixedBits(HostGfx_FULL_SCREEN_PALETTE_BASE);
		HostGfx_Copy8(state,dest,dest_pitch,bitmap_3bit,w,pitch,h,
			state->scanlines_mode==HostGfx_DOUBLE?1:0);
	} else {
//		bbcVideo::SetPixelFixedBits(0);
		switch(state->primary_desc.ddpfPixelFormat.dwRGBBitCount) {
		case 8:
			HostGfx_Convert8(state,static_cast<t65::byte *>(sd.lpSurface),sd.lPitch,
				bitmap_3bit,w,pitch,h);
			break;
		case 15:
		case 16:
			HostGfx_Convert16(state,static_cast<t65::word *>(sd.lpSurface),sd.lPitch,
				bitmap_3bit,w,pitch,h);
			break;
		case 24:
			HostGfx_Convert24(state,static_cast<t65::byte *>(sd.lpSurface),sd.lPitch,
				bitmap_3bit,w,pitch,h);
			break;
		case 32:
			HostGfx_Convert32(state,static_cast<t65::dword *>(sd.lpSurface),sd.lPitch,
				bitmap_3bit,w,pitch,h);
			break;
		}
	}
	IDirectDrawSurface2_Unlock(state->back_buffers[state->cpu_buffer],0);

	if(state->full_screen) {
		//IDirectDrawSurface_Flip(state->primary,0,DDFLIP_WAIT);
	} else {
		state->gpu_buffer=(state->gpu_buffer+1)%num_back_buffers;
		state->cpu_buffer=(state->cpu_buffer+1)%num_back_buffers;
		HostGfx_ShowWindowed(state);
	}
}

void HostGfx_ShowWindowed(HostGfxState *state) {
	POINT top_left={0,0};
	RECT dest;
	
	if(!state->buffers_ok) {
		return;
	}
	//HostGfx_ShowWindowed called if cfg_.video.full_screen, which doesn't imply
	//state->full_screen or indeed state->buffers_ok, so drop out if any of the
	//required surfaces are currently unavailable.
	if(!state->primary) {
		return;
	}
	if(!state->back_buffers[state->gpu_buffer]) {
		return;
	}
	GetClientRect(state->target,&dest);
	if(dest.right>0&&dest.bottom>0) {
		ClientToScreen(state->target,&top_left);
		OffsetRect(&dest,top_left.x,top_left.y);
		//primary_->Blt(&dest,back_[gpu_buffer_],0,DDBLT_WAIT,0);
		IDirectDrawSurface7_Blt(state->primary,&dest,
			state->back_buffers[state->gpu_buffer],0,DDBLT_WAIT,0);
	}
}

static void GetBitfield(DWORD value,int *shift,int *count) {
	*shift=0;
	*count=0;
	while(!(value&1)) {
		++*shift;
		value>>=1;
	}
	while(value&1) {
		++*count;
		value>>=1;
	}
}

HostGfx_BeebScreenType HostGfx_ScreenType(HostGfxState *state) {
	return state->beeb_screen_type;
}

//generates a 24bit table which is then cut down to size.
void HostGfx_SetScreenType(HostGfxState *state,HostGfx_BeebScreenType type) {
	int i;
	t65::byte pal24[8][3];
	int rshift,rcount;
	int gshift,gcount;
	int bshift,bcount;
	const DDPIXELFORMAT *pf=&state->primary_desc.ddpfPixelFormat;
	
	GetBitfield(pf->dwRBitMask,&rshift,&rcount);
	GetBitfield(pf->dwGBitMask,&gshift,&gcount);
	GetBitfield(pf->dwBBitMask,&bshift,&bcount);
	state->beeb_screen_type=type;
	for(i=0;i<8;++i) {
		float rmul=1.f,gmul=1.f,bmul=1.f;
		pal24[i][2]=pal24[i][1]=pal24[i][0]=0;

		if(state->beeb_screen_type==HostGfx_COLOUR) {
			if(i&1) {
				pal24[i][0]=255;
			}
			if(i&2) {
				pal24[i][1]=255;
			}
			if(i&4) {
				pal24[i][2]=255;
			}
		} else {
			t65::byte element=0;

			if(i&1) {
				element+=76;
			}
			if(i&2) {
				element+=151;
			}
			if(i&4) {
				element+=28;
			}
			//values from beebem :)
			switch(state->beeb_screen_type) {
			case HostGfx_AMBER:
				pal24[i][0]=element;
				pal24[i][1]=t65::byte(element*.8f);
				pal24[i][2]=t65::byte(element*.1f);
				break;
			case HostGfx_BNW:
				pal24[i][0]=element;
				pal24[i][1]=element;
				pal24[i][2]=element;
				break;
			case HostGfx_GREEN:
				pal24[i][0]=t65::byte(element*.2f);
				pal24[i][1]=t65::byte(element*.9f);
				pal24[i][2]=t65::byte(element*.1f);
				break;
			}
		}

	}

	//http://www.compuphase.com/palette.htm
	//I just use 0 for black and 249->255 for the rest -- they're fixed.
	state->lookup_8bit[0]=0;
	state->lookup_8bit[1]=249;
	state->lookup_8bit[2]=250;
	state->lookup_8bit[3]=251;
	state->lookup_8bit[4]=252;
	state->lookup_8bit[5]=253;
	state->lookup_8bit[6]=254;
	state->lookup_8bit[7]=255;

	//All are generated at once. Only the one that matches the display depth
	//is any use.
	for(i=0;i<8;++i) {
		t65::word *p16=&state->lookup_16bit[i];
		t65::dword *p24=reinterpret_cast<t65::dword *>(&state->lookup_24bit[i]);
		t65::dword *p32=&state->lookup_32bit[i];

		*p32=0;
		*p32|=(pal24[i][0]>>(8-rcount))<<rshift;
		*p32|=(pal24[i][1]>>(8-gcount))<<gshift;
		*p32|=(pal24[i][2]>>(8-bcount))<<bshift;

		*p24=0;
		*p24|=(pal24[i][0]>>(8-rcount))<<rshift;
		*p24|=(pal24[i][1]>>(8-gcount))<<gshift;
		*p24|=(pal24[i][2]>>(8-bcount))<<bshift;

		*p16=0;
		*p16|=(pal24[i][0]>>(8-rcount))<<rshift;
		*p16|=(pal24[i][1]>>(8-gcount))<<gshift;
		*p16|=(pal24[i][2]>>(8-bcount))<<bshift;
	}
}

void HostGfx_SetScanlinesMode(HostGfxState *state,
	HostGfx_ScanlinesMode new_scanlines_mode)
{
	if(state->scanlines_mode!=new_scanlines_mode) {
		state->buffers_ok=false;
	}
	state->scanlines_mode=new_scanlines_mode;
}

void HostGfx_SetFullScreen(HostGfxState *state,bool full_screen,
	int full_screen_refresh_rate)
{
#ifdef mbENABLE_FULL_SCREEN
	bool restart=false;

	if(full_screen!=state->full_screen||
		(state->full_screen&&
		full_screen_refresh_rate!=state->full_screen_refresh_rate))
	{
		HostGfx_StopInternal(state);
		restart=true;
	}
	state->full_screen=full_screen;
	if(state->full_screen) {
		state->full_screen_refresh_rate=full_screen_refresh_rate;
	}
//	if(restart) {
//		HostGfx_StartInternal(state,-1,-1);
//	}
#else
	state->full_screen=false;
#endif
}

bool HostGfx_IsFullScreen(HostGfxState *state) {
	return state->buffers_ok&&state->full_screen;
}

void HostGfx_Cls(HostGfxState *state) {
	if(state->full_screen) {
		DDBLTFX fx;

		memset(&fx,0,sizeof fx);
		fx.dwSize=sizeof fx;
		fx.dwFillColor=0;
		IDirectDrawSurface2_Blt(state->primary,0,0,0,DDBLT_COLORFILL|DDBLT_WAIT,&fx);
	}
}

//////////////////////////////////////////////////////////////////////////
struct HostSndState {
	HWND owner;
#ifdef DX3_ONLY
	IDirectSound *dsound;
	IDirectSoundBuffer *buffer;
#else
	IDirectSound8 *dsound;
	IDirectSoundBuffer8 *buffer;
#endif
	DSBUFFERDESC bd;
	WAVEFORMATEX fmt;
	DWORD last_play_cursor;
	bool playing;
	bool locked;
	void *lp0,*lp1;
	DWORD lnum0,lnum1;
	int volume;//0-100
	LONG dsound_volume;//dsound logarithmic
};

//////////////////////////////////////////////////////////////////////////
// Updates volume for given HostSndState. state->volume is the desired volume
// (linear scale, 0-100). Updates state->dsound_volume and, if the buffer is
// created, sets the buffer's volume too.
static void HostSnd_UpdateVolume(HostSndState *state) {
	wxASSERT(DSBVOLUME_MIN==-10000&&DSBVOLUME_MAX==0);//Yes, I _am_ paranoid.
	state->dsound_volume=DSBVOLUME_MIN;
	if(state->volume>0) {
		state->dsound_volume=LONG(-10*log10f(100.f/state->volume)*100);
	}
	//IDirectSoundBuffer_SetVolume(state->buffer,ds_volume);
//	dprintf("HostSnd_UpdateVolume: volume=%d state->dsound_volume=%ld\n",state->volume,state->dsound_volume);
	if(state->buffer) {
		LONG set_volume;
		state->buffer->SetVolume(DSBVOLUME_MIN);
		HRESULT hr=state->buffer->SetVolume(state->dsound_volume);
		state->buffer->GetVolume(&set_volume);
//		dprintf("HostSnd_UpdateVolume:     volume set, hr=0x%08X, GetVolume=%ld\n",hr,set_volume);
	}
}

//////////////////////////////////////////////////////////////////////////
// Resets the given state. Stops playing, frees DirectSound and buffers.
static void HostSnd_StopInternal(HostSndState *state) {
	wxASSERT(state);
	HostSnd_PlayStop(state);
	SafeRelease(state->buffer);
	SafeRelease(state->dsound);
}

//////////////////////////////////////////////////////////////////////////
// Restarts the given state.
static bool HostSnd_StartInternal(HostSndState *state) {
	HRESULT hr;

	wxASSERT(!state->dsound);
#ifdef DX3_ONLY
	hr=DXCHK(DirectSoundCreate(0,&state->dsound,0));
#else
	hr=DXCHK(DirectSoundCreate8(0,&state->dsound,0));
#endif
	if(FAILED(hr)) {
		goto stop_error;
	}
	hr=DXCHK(IDirectSound_SetCooperativeLevel(state->dsound,state->owner,DSSCL_NORMAL));
	if(FAILED(hr)) {
		goto stop_error;
	}
	return true;
stop_error:
	HostSnd_StopInternal(state);
	return false;
}
/*
HWND owner;
IDirectSound *dsound;
IDirectSoundBuffer *buffer;
DSBUFFERDESC bd;
WAVEFORMATEX fmt;
DWORD last_play_cursor;
bool playing;
bool locked;
void *lp0,*lp1;
DWORD lnum0,lnum1;
*/
HostSndState *HostSnd_Start(wxWindow *window) {
	HostSndState *state=new HostSndState;

	dprintf("HostSnd_Start: setting up.\n");
	state->owner=HWND(window->GetHandle());
	state->dsound=0;
	state->buffer=0;
	state->playing=false;
	state->locked=false;
	if(!HostSnd_StartInternal(state)) {
		dprintf("HostSnd_Start: HostSnd_StartInternal failed.\n");
		delete state;
		return 0;
	}
	HostSnd_SetVolume(state,0);
	dprintf("HostSnd_Start: ok, state=0x%p.\n",state);
	return state;
}

void HostSnd_Stop(HostSndState *state) {
	dprintf("HostSnd_Stop: state=0x%p\n",state);
	if(state) {
		HostSnd_StopInternal(state);
		delete state;
	}
}

void HostSnd_SetSoundParameters(HostSndState *state,int hz,int bits,int channels,
	int len_samples)
{
	HRESULT hr;
	void *p[2];
	DWORD num[2];
	unsigned char silence;
	IDirectSoundBuffer *sound_buffer=0;

	dprintf("HostSnd_SetSoundParameters: hz=%d bits=%d channels=%d len_samples=%d\n",
		hz,bits,channels,len_samples);
	
	wxASSERT(state->dsound);//HostSnd_Start must have been called first.
	HostSnd_PlayStop(state);
	SafeRelease(state->buffer);

	state->fmt.wFormatTag=WAVE_FORMAT_PCM;
	state->fmt.nChannels=channels;
	state->fmt.nSamplesPerSec=hz;
	state->fmt.wBitsPerSample=bits;
	
	state->fmt.nBlockAlign=state->fmt.nChannels*state->fmt.wBitsPerSample/8;
	state->fmt.nAvgBytesPerSec=state->fmt.nSamplesPerSec*state->fmt.nBlockAlign;
	state->fmt.cbSize=0;
	
	state->bd.dwBufferBytes=len_samples*state->fmt.nBlockAlign;
	state->bd.dwFlags=DSBCAPS_GETCURRENTPOSITION2|DSBCAPS_CTRLVOLUME|
		DSBCAPS_LOCSOFTWARE;
	state->bd.dwReserved=0;
	state->bd.dwSize=sizeof state->bd;
	state->bd.lpwfxFormat=&state->fmt;
#ifndef DX3_ONLY
	memset(&state->bd.guid3DAlgorithm,0,sizeof state->bd.guid3DAlgorithm);
#endif

	dprintf("HostSnd_SetSoundParameters: IDirectSound_CreateSoundBuffer...\n");
	hr=IDirectSound_CreateSoundBuffer(state->dsound,&state->bd,&sound_buffer,0);
	if(FAILED(hr)) {
#ifdef _DEBUG
		dprintf("HostSnd_SetSoundParameters: failed %s\n",DXGetErrorString9(hr));
#endif
		return;
	}
#ifdef DX3_ONLY
	state->buffer=sound_buffer;
#else
	dprintf("HostSnd_SetSoundParameters: IDirectSound_QueryInterface...\n");
	hr=IDirectSound_QueryInterface(sound_buffer,IID_IDirectSoundBuffer8,
		reinterpret_cast<void **>(&state->buffer));
	if(FAILED(hr)) {
		SafeRelease(sound_buffer);
		return;
	}
#endif

	dprintf("HostSnd_SetSoundParameters: fill buffer with silence...\n");
	hr=IDirectSoundBuffer_Lock(state->buffer,0,0,&p[0],&num[0],&p[1],&num[1],
		DSBLOCK_ENTIREBUFFER);
	if(FAILED(hr)) {
#ifdef _DEBUG
		dprintf("HostSnd_SetSoundParameters:     IDirectSoundBuffer_Lock failed: %s\n",
			DXGetErrorString9(hr));
#endif
		SafeRelease(state->buffer);
		return;
	}

	silence=bits==8?0x80:0;
	memset(p[0],silence,num[0]);
	memset(p[1],silence,num[1]);

	DXCHK(IDirectSoundBuffer_Unlock(state->buffer,p[0],num[0],p[1],num[1]));

	HostSnd_UpdateVolume(state);

	state->last_play_cursor=0;
	dprintf("HostSnd_SetSoundParameters: ok.\n");
}

void HostSnd_PlayStart(HostSndState *state) {
	wxASSERT(state);
/*	
	LONG tmp=-1;

	if(state->buffer) {
		state->buffer->GetVolume(&tmp);
	}
	dprintf("HostSnd_PlayStart: state->buffer=0x%p state->playing=%d curvol=%ld\n",
		state->buffer,state->playing,tmp);
*/
	HostSnd_UpdateVolume(state);
	if(state->buffer&&!state->playing) {
		IDirectSoundBuffer_Play(state->buffer,0,0,DSBPLAY_LOOPING);
		state->playing=true;
	}
}

void HostSnd_PlayStop(HostSndState *state) {
	wxASSERT(state);
//	dprintf("HostSnd_PlayStop: state->buffer=0x%p state->playing=%d\n",
//		state->buffer,state->playing);
	if(state->buffer&&state->playing) {
		IDirectSoundBuffer_Stop(state->buffer);
		state->playing=false;
	}
}

int HostSnd_GetWriteArea(HostSndState *state,void **p1,unsigned *len1,
	void **p2,unsigned *len2)
{
	wxASSERT(state);
	wxASSERT(state->buffer);

	if(!state->playing) {
		return 0;
	}

	unsigned num_bytes;
	DWORD this_play;
	HRESULT hr;

	hr=DXCHK(IDirectSoundBuffer_GetCurrentPosition(state->buffer,&this_play,0));
	if(FAILED(hr)) {
		return -1;
	}
	if(this_play>=state->last_play_cursor) {
		num_bytes=this_play-state->last_play_cursor;
	} else {
		num_bytes=(this_play+state->bd.dwBufferBytes)-state->last_play_cursor;
	}
	if(num_bytes==0) {
		return 0;
	}
	hr=DXCHK(IDirectSoundBuffer_Lock(state->buffer,state->last_play_cursor,num_bytes,
		&state->lp0,&state->lnum0,&state->lp1,&state->lnum1,0));
	if(FAILED(hr)) {
		return -1;
	}
	state->locked=true;
	if(p1) {
		*p1=state->lp0;
	}
	if(len1) {
		*len1=state->lnum0;
	}
	if(p2) {
		*p2=state->lp1;
	}
	if(len2) {
		*len2=state->lnum1;
	}
	state->last_play_cursor=this_play;
	return 1;
}

void HostSnd_CommitWriteArea(HostSndState *state) {
	if(state->locked) {
		DXCHK(IDirectSoundBuffer_Unlock(state->buffer,state->lp0,state->lnum0,
			state->lp1,state->lnum1));
		state->locked=false;
	}
}

void HostSnd_SetVolume(HostSndState *state,int volume) {

	wxASSERT(volume>=0&&volume<=100);
	wxASSERT(!state->locked);
	//http://download.insomniavisions.com/sources/uncompressed/c/out_ivdsound/plugin.c

	state->volume=volume;
	dprintf("HostSnd_SetVolume: state->volume=%d now\n",state->volume);
	HostSnd_UpdateVolume(state);

	return;
}

//////////////////////////////////////////////////////////////////////////
//
void HostTmr_Start() {
}

void HostTmr_Stop() {
}
/*
void HostTmr_ReadMilliseconds(int *value) {
	*value=GetTickCount();
}
*/

#ifndef READHF_INLINE
void HostTmr_ReadHf(HostTmr_HfValue *value) {
	static __int64 tot=0;
	static unsigned ncalls=0;
	LARGE_INTEGER tscstart,tscend;
	LARGE_INTEGER qpc;
	__asm cpuid;
	__asm rdtsc;
	__asm mov tscstart.LowPart,eax;
	__asm mov tscstart.HighPart,edx;
	QueryPerformanceCounter(&qpc);
	__asm cpuid;
	__asm rdtsc;
	__asm mov tscend.LowPart,eax;
	__asm mov tscend.HighPart,edx;
	*value=qpc.QuadPart;

	tot+=(tscend.QuadPart-tscstart.QuadPart);
	++ncalls;

	if(ncalls>5000) {
		char tmpbuf[1000];
		_snprintf(tmpbuf,sizeof tmpbuf,"HostTmr_ReadHf: avg %f cycles\n",
			tot/float(ncalls));
		OutputDebugStringA(tmpbuf);
		tot=0;
		ncalls=0;
	}
}
#endif

int HostTmr_GetHfFrequency(HostTmr_HfValue *frequency) {
	LARGE_INTEGER qpf;
	if(!QueryPerformanceFrequency(&qpf)) {
		return 0;
	}
	*frequency=qpf.QuadPart;
	return 1;
}


//////////////////////////////////////////////////////////////////////////
//
#ifdef DX3_ONLY
typedef IDirectInput2A IDirectInputType;
typedef IDirectInputDeviceA IDirectInputDeviceType;
#else
typedef IDirectInput8A IDirectInputType;
typedef IDirectInputDevice8A IDirectInputDeviceType;
#endif

struct HostInpState {
	HWND owner;
	IDirectInputType *dinput;
	IDirectInputDeviceType *kb;

	BYTE kb_state[256];
	BYTE last_kb_state[256];

#ifndef DX3_ONLY
	struct Joy {
		IDirectInputDeviceType *dev;
		DIJOYSTATE state;
		DIJOYSTATE mini,maxi;
	};
	
	std::vector<Joy> joysticks;
#endif//DX3_ONLY
};

#ifndef DX3_ONLY
static const unsigned joystate_axis_offsets[]={
	offsetof(DIJOYSTATE,lX),
	offsetof(DIJOYSTATE,lY),
	offsetof(DIJOYSTATE,lZ),
	offsetof(DIJOYSTATE,lRx),
	offsetof(DIJOYSTATE,lRy),
	offsetof(DIJOYSTATE,lRz),
	offsetof(DIJOYSTATE,rglSlider[0]),
	offsetof(DIJOYSTATE,rglSlider[1]),
};
static const unsigned num_joystate_axes=
	sizeof joystate_axis_offsets/sizeof joystate_axis_offsets[0];

static BOOL CALLBACK EnumDevicesCallback(LPCDIDEVICEINSTANCE lpddi,LPVOID pvRef) {
	HostInpState *state=static_cast<HostInpState *>(pvRef);
	HostInpState::Joy *joy;
	HRESULT hr;
	DIPROPDWORD pdw;
	unsigned i;

	state->joysticks.push_back(HostInpState::Joy());
	joy=&state->joysticks.back();
	OutputDebugStringA("HostInp_Start:    Instance: ");
	OutputDebugStringA(lpddi->tszInstanceName);
	OutputDebugStringA("\nHostInp_Start:    Product : ");
	OutputDebugStringA(lpddi->tszProductName);
	OutputDebugStringA("\n");
	joy->dev=0;
	hr=IDirectInput_CreateDevice(state->dinput,lpddi->guidInstance,&joy->dev,0);
	if(FAILED(hr)) {
		goto stop_error;
	}
	hr=IDirectInputDevice_SetCooperativeLevel(joy->dev,state->owner,
		DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
	if(FAILED(hr)) {
		goto stop_error;
	}
	hr=IDirectInputDevice_SetDataFormat(joy->dev,&c_dfDIJoystick);
	if(FAILED(hr)) {
		goto stop_error;
	}

	//Set absolute axis mode
	pdw.diph.dwSize=sizeof pdw;
	pdw.diph.dwHeaderSize=sizeof pdw.diph;
	pdw.diph.dwObj=0;
	pdw.diph.dwHow=DIPH_DEVICE;
	pdw.dwData=DIPROPAXISMODE_ABS;
	hr=IDirectInputDevice_SetProperty(joy->dev,DIPROP_AXISMODE,&pdw.diph);
	if(FAILED(hr)) {
		goto stop_error;
	}

	//Axes deadzone and saturation
	pdw.diph.dwSize=sizeof pdw;
	pdw.diph.dwHeaderSize=sizeof pdw.diph;
	pdw.diph.dwObj=0;
	pdw.diph.dwHow=DIPH_DEVICE;
	pdw.dwData=2500;
	hr=IDirectInputDevice_SetProperty(joy->dev,DIPROP_DEADZONE,&pdw.diph);
	if(FAILED(hr)) {
		goto stop_error;
	}
	pdw.dwData=9500;
	hr=IDirectInputDevice_SetProperty(joy->dev,DIPROP_SATURATION,&pdw.diph);
	if(FAILED(hr)) {
		goto stop_error;
	}
	
	//Determine ranges
	for(i=0;i<num_joystate_axes;++i) {
		DIPROPRANGE prop;
		prop.diph.dwSize=sizeof prop;
		prop.diph.dwHeaderSize=sizeof prop.diph;
		prop.diph.dwObj=joystate_axis_offsets[i];
		prop.diph.dwHow=DIPH_BYOFFSET;
		prop.lMin=0;
		prop.lMax=0;
		HRESULT hr2=IDirectInputDevice_GetProperty(joy->dev,DIPROP_RANGE,&prop.diph);
		unsigned offset=joystate_axis_offsets[i];
		LONG *mini=offsetwith<LONG>(&joy->mini,offset);
		LONG *maxi=offsetwith<LONG>(&joy->maxi,offset);
		//twprintf("JnbInput:\taxis %u: ",i);
		if(FAILED(hr2)) {
			//twprintf("%s",DXGetErrorString8A(hr2));
			*mini=*maxi=0;
		} else {
			*mini=prop.lMin;
			*maxi=prop.lMax;
			//twprintf("min=%u max=%u",*mini,*maxi);
		}
		//twprintf("\n");
	}			
	return DIENUM_CONTINUE;

stop_error:
	OutputDebugStringA("HostInp_Start:\tFailed: ");
//	OutputDebugStringA(DXGetErrorString9A(hr));
	OutputDebugStringA("\n");
	state->joysticks.pop_back();
	return DIENUM_CONTINUE;
}
#endif//DX3_ONLY

HostInpState *HostInp_Start(wxWindow *window) {
	HostInpState *state=new HostInpState;
	HRESULT hr;
#ifdef DX3_ONLY
	IDirectInputA *dinput;
#endif
	unsigned coop;
	
	state->owner=HWND(window->GetHandle());
	state->dinput=0;
	state->kb=0;
	memset(state->kb_state,0,sizeof state->kb_state);
	memset(state->last_kb_state,0,sizeof state->last_kb_state);
#ifdef DX3_ONLY
	hr=DirectInputCreate(GetModuleHandle(0),DIRECTINPUT_VERSION,&dinput,0);
#else
	hr=DirectInput8Create(GetModuleHandle(0),DIRECTINPUT_VERSION,IID_IDirectInput8A,
		reinterpret_cast<void **>(&state->dinput),0);
#endif
	if(FAILED(hr)) {
		goto stop_error;
	}
#ifdef DX3_ONLY
	hr=IDirectInput_QueryInterface(dinput,IID_IDirectInput2A,
		reinterpret_cast<void **>(&state->dinput));
	dinput->Release();
	if(FAILED(hr)) {
		goto stop_error;
	}
#endif
#if !(defined DX3_ONLY)&&(defined EMULATED_KEYBOARD)&&(defined _DEBUG)
	dprintf("HostInp_Start: trying GUID_SysKeyboardEm2.\n");
	hr=IDirectInput_CreateDevice(state->dinput,GUID_SysKeyboardEm2,&state->kb,0);
	if(FAILED(hr)) {
		dprintf("HostInp_Start: trying GUID_SysKeyboardEm.\n");
		hr=IDirectInput_CreateDevice(state->dinput,GUID_SysKeyboardEm,&state->kb,0);
		if(FAILED(hr)) {
			dprintf("HostInp_Start: trying GUID_SysKeyboard.\n");
			hr=IDirectInput_CreateDevice(state->dinput,GUID_SysKeyboard,&state->kb,0);
		}
	}
#else
	dprintf("HostInp_Start: trying GUID_SysKeyboard.\n");
	hr=IDirectInput_CreateDevice(state->dinput,GUID_SysKeyboard,&state->kb,0);
#endif
	if(FAILED(hr)) {
		goto stop_error;
	}
	coop=DISCL_FOREGROUND|DISCL_EXCLUSIVE;
#ifndef DX3_ONLY
	coop|=DISCL_NOWINKEY;
#endif
	hr=IDirectInputDevice_SetCooperativeLevel(state->kb,state->owner,coop);
	if(FAILED(hr)) {
		coop&=~DISCL_EXCLUSIVE;
		coop|=DISCL_NONEXCLUSIVE;
		hr=IDirectInputDevice_SetCooperativeLevel(state->kb,state->owner,coop);
	}
	if(FAILED(hr)) {
		goto stop_error;
	}
	hr=IDirectInputDevice_SetDataFormat(state->kb,&c_dfDIKeyboard);
	if(FAILED(hr)) {
		goto stop_error;
	}
#ifndef DX3_ONLY
	OutputDebugStringA("HostInp_Start: Enumerating joysticks...\n");
	hr=IDirectInput_EnumDevices(state->dinput,DI8DEVCLASS_GAMECTRL,
		&EnumDevicesCallback,state,DIEDFL_ATTACHEDONLY);
	if(FAILED(hr)) {
		goto stop_error;
	}
	OutputDebugStringA("HostInp_Start: joysticks done.\n");
#endif
	OutputDebugStringA("HostInp_Start: ok.\n");
	return state;

stop_error:
	HostInp_Stop(state);
	return false;
}

void HostInp_Stop(HostInpState *state) {
	if(state) {
		if(state->kb) {
			IDirectInputDevice_Unacquire(state->kb);
			IDirectInputDevice_Release(state->kb);
			state->kb=0;
		}
#ifndef DX3_ONLY
		for(unsigned i=0;i<state->joysticks.size();++i) {
			HostInpState::Joy *joy=&state->joysticks[i];
			if(joy->dev) {
				IDirectInputDevice_Unacquire(joy->dev);
				IDirectInputDevice_Release(joy->dev);
			}
		}
		state->joysticks.clear();
#endif//DX3_ONLY
		if(state->dinput) {
			IDirectInput_Release(state->dinput);
			state->dinput=0;
		}
	}
	delete state;
}

static void HostInp_PollInternal(IDirectInputDeviceType *dev,void *state,
	unsigned state_size)
{
	HRESULT hr;

	hr=IDirectInputDevice_Acquire(dev);
	if(SUCCEEDED(hr)) {
#ifndef DX3_ONLY
		//TODO -- poll possibly _never_ needed for system keyboard????
		hr=IDirectInputDevice2_Poll(dev);
		if(SUCCEEDED(hr))
#endif
		{
			hr=IDirectInputDevice_GetDeviceState(dev,state_size,state);
			if(SUCCEEDED(hr)) {
				return;
			}
		}
	}
	memset(state,state_size,0);
}

void HostInp_Poll(HostInpState *state) {
	memcpy(state->last_kb_state,state->kb_state,sizeof state->kb_state);
	HostInp_PollInternal(state->kb,state->kb_state,sizeof state->kb_state);
#ifndef DX3_ONLY
	for(unsigned i=0;i<state->joysticks.size();++i) {
		HostInpState::Joy *joy=&state->joysticks[i];
		if(joy->dev) {
			HostInp_PollInternal(joy->dev,&joy->state,sizeof joy->state);
		}
	}
#endif//DX3_ONLY
}

void HostInp_GetKeyboardState(HostInpState *state,unsigned char *keyflags) {
	memcpy(keyflags,state->kb_state,sizeof state->kb_state);
}

void HostInp_GetKeyboardDownFlags(HostInpState *state,unsigned char *downflags) {
	for(unsigned i=0;i<256;++i) {
		downflags[i]=state->kb_state[i]&&!state->last_kb_state[i];
	}
}


#ifndef DX3_ONLY
unsigned HostInp_NumJoysticks(HostInpState *state) {
	return state->joysticks.size();
}

static void DoRangeElement(HostInpState::Joy *joy,unsigned offset,float *f) {
	LONG v=*offsetwith<LONG>(&joy->state,offset);
	LONG mini=*offsetwith<LONG>(&joy->mini,offset);
	LONG maxi=*offsetwith<LONG>(&joy->maxi,offset);
	
	if(v<mini) {
		v=mini;
	} else if(v>maxi) {
		v=maxi;
	}
	if(maxi==mini) {
		*f=0.f;
	} else {
		*f=v-mini;
		*f/=maxi-mini;
		*f*=2.f;
		*f-=1.f;
	}
	assert(*f>=-1.f&&*f<=1.f);
}

void HostInp_GetJoystickState(HostInpState *state,unsigned index,
	HostInp_JoyState *joy_state)
{
	HostInpState::Joy *joy;
	
	if(index>=state->joysticks.size()||!(joy=&state->joysticks[index])) {
		joy_state->x=0.f;
		joy_state->y=0.f;
		joy_state->z=0.f;
		joy_state->rx=0.f;
		joy_state->ry=0.f;
		joy_state->rz=0.f;
		joy_state->sliders[0]=0.f;
		joy_state->sliders[1]=0.f;
		memset(joy_state->buttons,0,sizeof joy_state->buttons);
	} else {
		DoRangeElement(joy,offsetof(DIJOYSTATE,lX),&joy_state->x);
		DoRangeElement(joy,offsetof(DIJOYSTATE,lY),&joy_state->y);
		DoRangeElement(joy,offsetof(DIJOYSTATE,lZ),&joy_state->z);
		DoRangeElement(joy,offsetof(DIJOYSTATE,lRx),&joy_state->rx);
		DoRangeElement(joy,offsetof(DIJOYSTATE,lRy),&joy_state->ry);
		DoRangeElement(joy,offsetof(DIJOYSTATE,lRz),&joy_state->rz);
		DoRangeElement(joy,offsetof(DIJOYSTATE,rglSlider[0]),&joy_state->sliders[0]);
		DoRangeElement(joy,offsetof(DIJOYSTATE,rglSlider[1]),&joy_state->sliders[1]);
		memcpy(joy_state->buttons,&joy->state.rgbButtons,32);
	}
}
#endif//DX3_ONLY

struct KeyAndName {
	unsigned key_code;
	const char *key_name;
};

//The rationale behind all this is to keep the key names positional rather
//than based on keycap.
static const KeyAndName pc_key_details[]={
	{DIK_ESCAPE,"PC_ESC"},
	{DIK_F1,"PC_F1"},
	{DIK_F2,"PC_F2"},
	{DIK_F3,"PC_F3"},
	{DIK_F4,"PC_F4"},
	{DIK_F5,"PC_F5"},
	{DIK_F6,"PC_F6"},
	{DIK_F7,"PC_F7"},
	{DIK_F8,"PC_F8"},
	{DIK_F9,"PC_F9"},
	{DIK_F10,"PC_F10"},
	{DIK_F11,"PC_F11"},
	{DIK_F12,"PC_F12"},
	{DIK_SYSRQ,"PC_PRTSCR"},
	{DIK_SCROLL,"PC_SCROLLLOCK"},
	{DIK_PAUSE,"PC_PAUSE"},
	{DIK_GRAVE,"PC_BACKQUOTE"},
	{DIK_1,"PC_1"},
	{DIK_2,"PC_2"},
	{DIK_3,"PC_3"},
	{DIK_4,"PC_4"},
	{DIK_5,"PC_5"},
	{DIK_6,"PC_6"},
	{DIK_7,"PC_7"},
	{DIK_8,"PC_8"},
	{DIK_9,"PC_9"},
	{DIK_0,"PC_0"},
	{DIK_MINUS,"PC_MINUS"},
	{DIK_EQUALS,"PC_EQUALS"},
	{DIK_BACK,"PC_BACKSPACE"},
	{DIK_INSERT,"PC_INSERT"},
	{DIK_HOME,"PC_HOME"},
	{DIK_PRIOR,"PC_PGUP"},
	{DIK_TAB,"PC_TAB"},
	{DIK_Q,"PC_Q"},
	{DIK_W,"PC_W"},
	{DIK_E,"PC_E"},
	{DIK_R,"PC_R"},
	{DIK_T,"PC_T"},
	{DIK_Y,"PC_Y"},
	{DIK_U,"PC_U"},
	{DIK_I,"PC_I"},
	{DIK_O,"PC_O"},
	{DIK_P,"PC_P"},
	{DIK_LBRACKET,"PC_OPENBRACE"},
	{DIK_RBRACKET,"PC_CLOSEBRACE"},
	{DIK_RETURN,"PC_ENTER"},
	{DIK_DELETE,"PC_DELETE"},
	{DIK_END,"PC_END"},
	{DIK_NEXT,"PC_PGDN"},
	{DIK_CAPITAL,"PC_CAPSLOCK"},
	{DIK_A,"PC_A"},
	{DIK_S,"PC_S"},
	{DIK_D,"PC_D"},
	{DIK_F,"PC_F"},
	{DIK_G,"PC_G"},
	{DIK_H,"PC_H"},
	{DIK_J,"PC_J"},
	{DIK_K,"PC_K"},
	{DIK_L,"PC_L"},
	{DIK_SEMICOLON,"PC_SEMICOLON"},
	{DIK_APOSTROPHE,"PC_QUOTE"},
	{DIK_BACKSLASH,"PC_HASH"},
	{DIK_LSHIFT,"PC_LEFTSHIFT"},
	{DIK_OEM_102,"PC_BACKSLASH"},
	{DIK_Z,"PC_Z"},
	{DIK_X,"PC_X"},
	{DIK_C,"PC_C"},
	{DIK_V,"PC_V"},
	{DIK_B,"PC_B"},
	{DIK_N,"PC_N"},
	{DIK_M,"PC_M"},
	{DIK_COMMA,"PC_COMMA"},
	{DIK_PERIOD,"PC_STOP"},
	{DIK_SLASH,"PC_SLASH"},
	{DIK_RSHIFT,"PC_RIGHTSHIFT"},
	{DIK_LCONTROL,"PC_LEFTCTRL"},
	{DIK_LWIN,"PC_LEFT95"},
	{DIK_LMENU,"PC_ALT"},
	{DIK_SPACE,"PC_SPACE"},
	{DIK_RMENU,"PC_ALTGR"},
	{DIK_RWIN,"PC_RIGHT95"},
	{DIK_APPS,"PC_MENU95"},
	{DIK_RCONTROL,"PC_RIGHTCTRL"},
	{DIK_UP,"PC_UP"},
	{DIK_DOWN,"PC_DOWN"},
	{DIK_LEFT,"PC_LEFT"},
	{DIK_RIGHT,"PC_RIGHT"},
	{DIK_NUMLOCK,"PC_NUMLOCK"},
	{DIK_DIVIDE,"PC_KP_SLASH"},
	{DIK_MULTIPLY,"PC_KP_STAR"},
	{DIK_SUBTRACT,"PC_KP_MINUS"},
	{DIK_ADD,"PC_KP_PLUS"},
	{DIK_NUMPADENTER,"PC_KP_ENTER"},
	{DIK_NUMPAD0,"PC_KP_0"},
	{DIK_NUMPAD1,"PC_KP_1"},
	{DIK_NUMPAD2,"PC_KP_2"},
	{DIK_NUMPAD3,"PC_KP_3"},
	{DIK_NUMPAD4,"PC_KP_4"},
	{DIK_NUMPAD5,"PC_KP_5"},
	{DIK_NUMPAD6,"PC_KP_6"},
	{DIK_NUMPAD7,"PC_KP_7"},
	{DIK_NUMPAD8,"PC_KP_8"},
	{DIK_NUMPAD9,"PC_KP_9"},
};
static const unsigned num_pc_key_details=sizeof pc_key_details/sizeof pc_key_details[0];

const char *HostInp_KeyNameFromCode(unsigned key_code) {
	for(unsigned i=0;i<num_pc_key_details;++i) {
		if(pc_key_details[i].key_code==key_code) {
			return pc_key_details[i].key_name;
		}
	}
	return "";
}

unsigned HostInp_CodeFromKeyName(const char *key_name) {
	for(unsigned i=0;i<num_pc_key_details;++i) {
		if(stricmp(pc_key_details[i].key_name,key_name)==0) {
			return pc_key_details[i].key_code;
		}
	}
	return HostInp_bad_key;
}

