#include "pch.h"
#include "bbcComputer.h"
#include "bbcUserVIA.h"
#include "bbcSystemVIA.h"
#include "bbc1770.h"
#include "bbcSound.h"
#include "bbcKeyboardMatrix.h"
#include "bbcTeletextFont.h"

//////////////////////////////////////////////////////////////////////////
//
int bbcComputer::next_stop=0;
bbcRomSlot bbcComputer::rom_slots_[16];
bbcRomSlot bbcComputer::os_rom_;
t65::byte bbcComputer::romsel_=0;
//No ROM board by default.
//t65::byte bbcComputer::romsel_mask_;
//t65::byte bbcComputer::romsel_fixed_bits_;
t65::byte *bbcComputer::read_pages_[256];
t65::byte *bbcComputer::write_pages_[256];
t65::byte bbcComputer::ram_[128*1024];//TODO MAke private
bbcComputer::StateType bbcComputer::cpustate;
int bbcComputer::cycles;
unsigned bbcComputer::irq_flags_;
bbcWriteMmioFn bbcComputer::mmio_write_fns_[0x300];
bbcReadMmioFn bbcComputer::mmio_read_fns_[0x300];
bbcModel *bbcComputer::model_;
#ifdef bbcDEBUG_TRACE
bbcDebugTrace *bbcComputer::trace;
#endif
bool bbcComputer::vdusel_;
t65::byte *bbcComputer::shadow_read_pages_[256];
t65::byte *bbcComputer::shadow_write_pages_[256];
t65::byte **bbcComputer::pc_read_pages_[256];
t65::byte **bbcComputer::pc_write_pages_[256];
bool bbcComputer::paged_ram_sel_;//bit 7 of FE30, B+ only.

//////////////////////////////////////////////////////////////////////////
//

static void NullWriteMmio(t65::byte,t65::byte) {
}

static t65::byte NullReadMmio(t65::byte) {
	return 0;
}

const bbcComputer::MmioFnInit bbcComputer::mmio_fn_inits_[]={
	{0,0,0,0,}
};

static t65::byte reads_black_hole[256]={0,};
static t65::byte writes_black_hole[256];

void bbcComputer::SetMmioFn(t65::word addr,bbcReadMmioFn read_fn,
	bbcWriteMmioFn write_fn)
{
	BASSERT(addr>=0xfc00&&addr<=0xfeff);
	addr-=0xfc00;
	if(read_fn) {
		mmio_read_fns_[addr]=read_fn;
	}
	if(write_fn) {
		mmio_write_fns_[addr]=write_fn;
	}
}

void bbcComputer::Init(const bbc1770Interface *disc_interface,int video_width,
	int video_height,bbcModel *bbc_model)
{
	BASSERT(disc_interface);
	unsigned i;

	model_=bbc_model;

	for(i=0;i<256;++i) {
		read_pages_[i]=0;
		write_pages_[i]=0;
		shadow_read_pages_[i]=0;
		shadow_write_pages_[i]=0;
	}
	
	for(i=0;i<0x80;++i) {
		read_pages_[i]=&ram_[i<<8];
		write_pages_[i]=&ram_[i<<8];
	}

	for(i=0;i<0xC0;++i) {
		if(i<0x30) {
			//Bottom of RAM
			shadow_read_pages_[i]=read_pages_[i];
			shadow_write_pages_[i]=write_pages_[i];
		} else if(i<0x80) {
			//shadow area
			shadow_read_pages_[i]=&ram_[0x8000+(i<<8)];
			shadow_write_pages_[i]=&ram_[0x8000+(i<<8)];
		} else if(i<0xB0) {
			//sideways area, filled in with WriteRomSel below.
		} else {
			shadow_write_pages_[i]=writes_black_hole;
			shadow_read_pages_[i]=reads_black_hole;
		}
	}
	
	os_rom_.GetPages(&read_pages_[0xC0],&write_pages_[0xC0]);
	os_rom_.GetPages(&shadow_read_pages_[0xC0],&shadow_write_pages_[0xC0]);

	write_pages_[0xfc]=writes_black_hole;
	write_pages_[0xfd]=writes_black_hole;

	vdusel_=false;
	paged_ram_sel_=false;

	//No ROM board by default.
//	romsel_mask_=15;
//	romsel_fixed_bits_=0;
	//WriteRomSel(0,15);//initial value.
	std::fill(ram_,ram_+32768,0);
	cycles=0;
	irq_flags_=0;

	ResetMmioFns();
	AddMmioFns(mmio_fn_inits_);

	//////////////////////////////////////////////////////////////////////////
	// Initialise hardware
	bbcSound::Init();
	bbcKeyboardMatrix::Init();
	bbcTeletextFont::Init();
	bbcVideo::buffer_width=video_width;
	bbcVideo::buffer_height=video_height;
	bbcVideo::Init();
	bbcVideo::SetVideoRam(&ram_[0]);
	
	InstallOSHacks();//keep, needed for keylink hack

	//////////////////////////////////////////////////////////////////////////
	// Set up disc interface
	// TODO all this is a bit mucky :(
	for(i=0;i<4;++i) {
		SetMmioFn(disc_interface->fdc_start+i,&bbc1770::ReadFdc,&bbc1770::WriteFdc);
	}
	SetMmioFn(disc_interface->drive_ctrl,&bbc1770::ReadControl,&bbc1770::WriteControl);

	AddMmioFns(disc_interface->extra_mmio_fns);

	if(disc_interface->init_fn) {
		(*disc_interface->init_fn)();
	}
	bbc1770::Init(*disc_interface);

//ex mbMainFrame_intel
//	bbcModelBSim::Init();
//	bbcModelBSim::InitialiseCPUState(the_beeb);
	model_->Init();
	model_->InitialiseCpuState();
	
	for(i=0;i<16;++i) {
		SetMmioFn(0xfe30+i,model_->RomSelReadFn(),model_->RomSelWriteFn());
	}
	(*model_->RomSelWriteFn())(0x30,15);//TODO this is nasty!!

	for(i=0;i<256;++i) {
		BASSERT(read_pages_[i]);
		BASSERT(write_pages_[i]);
//		BASSERT(shadow_read_pages_[i]);
//		BASSERT(shadow_write_pages_[i]);
	}
}

void bbcComputer::ResetMmioFns() {
	for(unsigned i=0;i<0x300;++i) {
		mmio_read_fns_[i]=&NullReadMmio;
		mmio_write_fns_[i]=&NullWriteMmio;
	}
}

void bbcComputer::AddMmioFns(const MmioFnInit *inits) {
	if(inits) {
		for(unsigned i=0;inits[i].count>0;++i) {
			const MmioFnInit *p=&inits[i];
			for(unsigned j=0;j<p->count;++j) {
				SetMmioFn(p->start+j,p->read_fn,p->write_fn);
			}
		}
	}
}
/*
void bbcComputer::SetHasRomBoard(bool has_rom_board) {
	if(has_rom_board) {
		romsel_mask_=0xf;
		romsel_fixed_bits_=0;
	} else {
		romsel_mask_=3;//%0011
		romsel_fixed_bits_=12;//%1100
	}
//	if(has_rom_board) {
//		Log::i.Logf("bbcComputer: rom board installed.\n");
//	} else {
//		Log::i.Logf("bbcComputer: rom board not installed.\n");
//	}
}
*/
void bbcComputer::SetRomContents(int slot,const t65::byte *contents,unsigned size_contents) {
	BASSERT(slot>=0&&slot<16);
//	if(PhysicalRomSlot(slot)!=slot) {
//		Log::w.Logf("ROM in slot %d not accessible (slot %d will be seen instead)\n",slot,PhysicalRomSlot(slot));
//	}
	rom_slots_[slot].ResetContents();
	rom_slots_[slot].SetContents(contents,size_contents);
}



void bbcComputer::InstallOSHacks() {
	return;
	//TODO Reinstate this, better still let's have one for OS 2.0!!
	unsigned i;

	//nopify the memory clear routine at 0xd9ee or so (it takes ages)
//	for(i=0x19EE;i!=0x19FD;++i) {
//		os_rom_.WriteByte(i,0xEA);
//	}

	//there's this code in the os:
	//DC59    LDA     &0267   ;if BIT 0 set and DISC EXEC error
	//DC5C    ROR             ;occurs
	//DC5D    BCS     &DC5D   ;hang up machine!!!!
	//sometimes c ends up set (very embarrassing) and the machine
	//hangs. this pokes nops over the bcs to stop this.
	//
	//TODO I think this hack can be removed.
	os_rom_.WriteByte(0x1C5D,0xEA);//nop
	os_rom_.WriteByte(0x1C5E,0xEA);//nop

	//there's this code too:
	//DA79    BIT     &FC     ;if bit 6=1 then JSR &F055 (normally 0) 
	//DA7B    BVC     &DA80   ;else DA80
	//DA7D    JSR     &F055   ;F055 JMP (&FDFE) probably causes a BRK unless 
	//                        ;hardware there redirects it.
	//this removes the JSR. I'd be lying if I said this wasn't a fudge, or if
	//I said I knew what was going on...
	//
	//TODO I think this hack can be removed. At the very least poke 0x80,0xDA
	//into 0xFDFE :)
	os_rom_.WriteByte(0x1A7D,0xEA);
	os_rom_.WriteByte(0x1A7E,0xEA);
	os_rom_.WriteByte(0x1A7F,0xEA);

	//See end of file for what this is!
	static const t65::byte boot_hack[]={
		0xa2,0x9c,
		0xa0,0x8d,
		0xa5,0xfc,
		0x49,0xff,
		0x8d,0x8f,0x02,
		0x68,
		0xf0,0x09,
		0xa0,0x7e,
		0x90,0x0a,
		0xa0,0x87,
		0xee,0x8d,0x02,
		0xee,0x8d,0x02,
		0xa2,0x90,
	};
	for(i=0;i<sizeof boot_hack;++i) {
		BASSERT(0xda26+i<0xda42);
		os_rom_.WriteByte(0x1a26+i,boot_hack[i]);
	}
}

void bbcComputer::DebugDumpMemory() {
	printf("bbcComputer::DebugDumpMemory: ");
	FILE *h=fopen(".\\memory_dump.txt","wt");
	if(!h) {
		printf("failed to open output file.\n");
		return;
	}
	int buffer[16];
	t65::Word addr;
	for(unsigned i=0;i<65536;i+=16) {
		fprintf(h,"%04X: ",i);
		unsigned j;
		for(j=0;j<16;++j) {
			addr.w=i+j;
			if((addr.h)!=0xfe) {
				buffer[j]=bbcComputer::read_pages_[addr.h][addr.l];
			} else {
				buffer[j]=-1;
			}
		}
		for(j=0;j<16;++j) {
			if(buffer[j]>=0) {
				fprintf(h,"%02X",buffer[j]);
			} else {
				fprintf(h,"**");
			}
			fprintf(h," ");
		}
		for(j=0;j<16;++j) {
			char c;
			if(buffer[j]<0) {
				c=' ';
			} else if(isprint(buffer[j])) {
				c=buffer[j];
			} else {
				c='.';
			}
			fputc(c,h);
		}
		fprintf(h,"\n");
	}
	fclose(h);
	printf("ok.\n");
}

void bbcComputer::Update() {
	bbcVideo::Update(cycles);
	bbc_system_via.Update();
	bbc_user_via.Update();
	bbc1770::Update();
}

void bbcComputer::WriteMmio(t65::Word addr,t65::byte val) {
	if(addr.h==0xfe) {
#ifndef BBC_SYNCCLOCKS_VIA_ONLY
		bbcComputer::SyncClocks();
#endif
		//(*write_sheila_fns_[addr.l])(addr.l,val);
		if(addr.l>=0x40&&addr.l<=0x5F) {
			bbc_system_via.Write(addr.l,val);
		} else if(addr.l>=0x60&&addr.l<=0x7F) {
			bbc_user_via.Write(addr.l,val);
		} else if(addr.l<=0x07) {
			bbcVideo::Write6845(addr.l,val);
		} else if(addr.l==0x20) {
			bbcVideo::WriteUla(addr.l,val);
		} else if(addr.l==0x21) {
			bbcVideo::WritePalette(addr.l,val);
//		} else if(addr.l>=0x30&&addr.l<=0x3F) {
//			WriteRomSel(addr.l,val);
		} else {
			(*mmio_write_fns_[0x200+addr.l])(addr.l,val);//0xfe00-0xfc00=0x200
		}
	} else if(addr.h==0xfd) {
#ifdef BBC_SYNCCLOCKS_JIM
		bbcComputer::SyncClocks();
#endif
		(*mmio_write_fns_[0x100+addr.l])(addr.l,val);
	} else if(addr.h==0xfc) {
#ifdef BBC_SYNCCLOCKS_FRED
		bbcComputer::SyncClocks();
#endif
		(*mmio_write_fns_[addr.l])(addr.l,val);
	}
}

t65::byte bbcComputer::ReadMmio(t65::Word addr) {
	if(addr.h==0xfe) {
#ifndef BBC_SYNCCLOCKS_VIA_ONLY
		bbcComputer::SyncClocks();
#endif
		if(addr.l>=0x40&&addr.l<=0x5F) {
			return bbc_system_via.Read(addr.l);
		} else if(addr.l>=0x60&&addr.l<=0x7F) {
			return bbc_user_via.Read(addr.l);
		} else if(addr.l<=0x07) {
			return bbcVideo::Read6845(addr.l);
		} else if(addr.l==0x8) {
			return 3;//ACIA
		} else if(addr.l==0xC0) {
			return 64;//ADC
//		} else if(addr.l>=0x30&&addr.l<=0x3f) {
//			return romsel_|(vdusel_?0x80:0);
		} else {
			return (*mmio_read_fns_[0x200+addr.l])(addr.l);
		}
	} else if(addr.h==0xfd) {
#ifdef BBC_SYNCCLOCKS_JIM
		bbcComputer::SyncClocks();
#endif
		return (*mmio_read_fns_[0x100+addr.l])(addr.l);
	} else if(addr.h==0xfc) {
#ifdef BBC_SYNCCLOCKS_FRED
		bbcComputer::SyncClocks();
#endif
		return (*mmio_read_fns_[addr.l])(addr.l);
	}
	return 0;
}

/*
NEW KEYLINKS CODE
=================

//This replaces the keylink code in OS1.20. With this hack, the keylinks
//are reread every boot; otherwise (without this hack) the keylinks are
//read only on a cold boot, i.e. power-on reset or Ctrl+Break.

//NEw code:
15 lines read, no errors in pass 1.
da26 =                  	org $da26
da26 : a29c             	ldx #$9c ;a2 9c
da28 : a08d             	ldy #$8d ;a0 8d
da2a : a5fc             	lda $fc ;a5 fc
da2c : 49ff             	eor #$ff ;49 ff
da2e : 8d8f02           	sta $28f ;8d 8f 02
da31 : 68               	pla ; 68 ;get back a from $d9db
da32 : f009             	beq $da3d ;if a=0 power up reset x=$9c y=$8d
da34 : a07e             	ldy #$7e ;a0 7e ;else y=$7e
da36 : 900a             	bcc $da42 ;and if not ctrl-break da42 warm reset
da38 : a087             	ldy #$87 ;a0 87 ;else y=$87 cold reset
da3a : ee8d02           	inc $028d ;ee 8d 02 ;$28d=1
da3d : ee8d02           	inc $028d ;ee 8d 02
da40 : a290             	ldx #$90 ;a2 90

//Old code:
DA26    LDX     #&9C    ;
DA28    LDY     #&8D    ;
DA2A    PLA             ;get back A from &D9DB
DA2B    BEQ     &DA36   ;if A=0 power up reset so DA36 with X=&9C Y=&8D
DA2D    LDY     #&7E    ;else Y=&7E
DA2F    BCC     &DA42   ;and if not CTRL-BREAK DA42 WARM RESET
DA31    LDY     #&87    ;else Y=&87 COLD RESET
DA33    INC     &028D   ;&28D=1
DA36    INC     &028D   ;&28D=&28D+1
DA39    LDA     &FC     ;get keyboard links set
DA3B    EOR     #&FF    ;invert
DA3D    STA     &028F   ;and store at &28F
DA40    LDX     #&90    ;X=&90
*/
/*
enum {
	bplus_mode=1,//for testing
};
*/
//////////////////////////////////////////////////////////////////////////
//
t65::byte bbcComputer::ReadRomSelB(t65::byte offset) {
	return 0;
}

void bbcComputer::WriteRomSelB(t65::byte offset,t65::byte value) {
	romsel_=value&0xf;
	rom_slots_[romsel_].GetPages(&read_pages_[0x80],&write_pages_[0x80]);
}

//////////////////////////////////////////////////////////////////////////
//
t65::byte bbcComputer::ReadRomSelBPlus(t65::byte offset) {
	if((offset&0xf)==0x04) {
		return vdusel_?0x80:0;
	} else {
		return romsel_|(paged_ram_sel_?0x80:0);
	}
}

void bbcComputer::WriteRomSelBPlus(t65::byte offset,t65::byte value) {
	unsigned i;

	if((offset&0xf)==0x04) {
		vdusel_=!!(value&0x80);
	} else {
		romsel_=value&0xf;
		paged_ram_sel_=!!(value&0x80);
	}
	//TODO this only needs to be done once really.
	for(i=0;i<256;++i) {
		pc_write_pages_[i]=write_pages_;
		pc_read_pages_[i]=read_pages_;
	}
	if(vdusel_) {
		//First 0x20 pages of OS ROM area are considered VDU driver
		for(i=0xC0;i<0xE0;++i) {
			pc_write_pages_[i]=shadow_write_pages_;
			pc_read_pages_[i]=shadow_read_pages_;
		}
		//If paged RAM is selected too, pages A0-AF incl. are too
		if(paged_ram_sel_) {
			for(i=0;i<16;++i) {
				pc_write_pages_[0xA0+i]=shadow_write_pages_;
				pc_read_pages_[0xA0+i]=shadow_read_pages_;
			}
		}
	}
	if(!paged_ram_sel_) {
		rom_slots_[romsel_].GetPages(&read_pages_[0x80],&write_pages_[0x80]);
	} else {
		for(i=0x80;i<0xB0;++i) {
			read_pages_[i]=write_pages_[i]=&ram_[0x8000+(i<<8)];
		}
		rom_slots_[romsel_].GetTop4kPages(&read_pages_[0xB0],&write_pages_[0xB0]);
	}

	//If vdusel_, some pc_[write|read]_pages will point to the shadow page set.
	//So, copy what's in the sideways area to the shadow pages so that the CPU's
	//view of this area of memory is consistent.
	if(vdusel_) {
		for(i=0x80;i<0xC0;++i) {
			shadow_write_pages_[i]=write_pages_[i];
			shadow_read_pages_[i]=read_pages_[i];
		}
	}

	//Finally, get the video display from shadow or normal memory depending.
	bbcVideo::SetVideoRam(vdusel_?&ram_[0x8000]:&ram_[0]);
}

void bbcComputer::ResetHard() {
	bbcVideo::Reset();
	bbc_system_via.Reset();
	bbc_user_via.Reset();
	bbc1770::Reset();
	ResetSoft();
}

void bbcComputer::ResetSoft() {
	model_->Reset6502();
}

#ifdef bbcDEBUG_TRACE
void bbcComputer::AddIrqTrace() {
	bbcDebugTrace::Irq *irq=trace->NewIrq();
	irq->fdc_status=bbc1770::Status();
	irq->irqflags=irq_flags_;
	irq->sys.acr=bbc_system_via.Acr();
	irq->sys.ier=bbc_system_via.Ier();
	irq->sys.ifr=bbc_system_via.Ifr();
	irq->usr.acr=bbc_user_via.Acr();
	irq->usr.ier=bbc_user_via.Ier();
	irq->usr.ifr=bbc_user_via.Ifr();
}
#endif
