;---------------------------------------------------------------------------
;
; FILE     cpuspeed.asm
; TYPE     assembly, binary, non-relocatable
; CREATED  1996.03.06
; REVISED  1996.03.22
; VERSION  1.1
; AUTHOR   Michael C. Draeger 
; SUMMARY  compute CPU speed
;
; DESCRIPTION
;
;   This assembly file computes the CPU clock speed of a ns486SXF evaluation
;   board and stores the value in DRAM.  It then jumps to a definable user 
;   address. 
;
;   This file uses the NS486SXF timer to time how long it takes to transmit
;   1 character on the external "debug" UART.  The UART is programmed for
;   160bps.  The UART is programmed for 8 bit words with no parity and
;   one stop bit.  In total 10 bits are sent for each character.  Therefore,
;   at 160bps, it will take 1/16th of a second (62.5ms) to trasmit.
;
;   Therefore the timer is started, a character is sent, and when it is
;   shifted out of the initial register, the timer is started.  The
;   UART is programmed in LOOPBACK mode, so no data is spit out on the
;   serial ports.  (Some of the control signals will go to their
;   inactive state if they were already active).
;
;   The PIT is set to use 1/32th of the CPU clock (extern OSC_CLK/2).
;   So the number of ticks elapsed are multiplied by 512 (16 * 32) to
;   get the CPU speed.  The result is a double word in Hz.  So 20MHz
;   would be indicated by 20000000 (0x1312D00).
;
;   This routine is quite accurate.  The external UART is very accurate.
;   There is some time between where the character is written to the UART
;   and the timer is started, but this should not be significant.
;   In general, the PIT is probably going to return a value within 1
;   tick of the actual time, so this routine is probably accurate within
;   1kHz.  The resultant CPU speed in Hz could be rounded to the nearest
;   even MHz value (i.e., 20MHz, 25MHz) or just used directly.
;
;   The PIT can only count 64K ticks before rolling, so this routine
;   will fail if the CPU is running too fast.  If 64K counts of the
;   PIT are elapsed, the CPU speed is 33,553,920.  So this routine may
;   fail above 33MHz.  If you need to measure faster clock rates,
;   you could temporary divide the CPU clock by 4 ( PM_PMR1 ) and
;   tell the timer to use the cpu_clk instead of osc_clk/2 ( PM_PMR3 ).
;   This would make the timer count 1/4th the speed, so you would
;   multiply the count by 2K instead of 512.  In this scenario you
;   could mesaure up to 134,217,728 which is obviously sufficient.
;   Note: the evaluation board will not run at 25MHz right now anyhow
;   because the DRAM access are too quick.  (The Target Loader sets up the
;   DRAM in it's fastest mode of operation.)
;
;   I have tested this code at 20 and 25MHz with rev A, B0, and B2 on the
;   NS486SXF evaluation board, and gotten quite accurate results.
;
;   Note I use the UART THRE register to take the measurements.  The
;   UART accepts a character and then transmits it to the trasmitter
;   holding register.  If the code were to wait for the TEMT bit to
;   be set, the result would not be accurate.  However, I found that
;   the time between writing a character and the UART THRE bit being
;   set was accurate.  After the code is done timing, I wait > 1/16th
;   of second to make sure everything is flushed out.  When I did not
;   have this delay I was seeing a character transmitted when I took
;   the UART out of loopback mode.
;
;   Currently the output of this routine is written in DRAM:
;
;        0xFFF00 - 0xFFF03   CPU speed (32 bits, in Hz)
;        0xFFF04 - 0xFFF05   used to store the PIT count (temporary data)
;
; HISTORY/CONTRIBUTORS
;
;   1.0   1996.03.06  mcd  genesis - from boiler.asm 1.3
;   1.1   1996.03.22  mcd  uses debug UART in loopback mode
;
; COPYRIGHT
;
;      (c) 1996 National Semiconductor Corporation
;
; NOTES
;
;   This code saves and restores the BIU control register and
;   the UART LCR, MCR, DLL, and DLM registers.  I am not sure if having
;   FIFOs enabled would effect the results, so I would suggest that if
;   you have FIFOs enabled before running this code you should turn off
;   the FIFOs.
;
;   I suggest you just run this code before you setup the external UART
;   for use.  If you choose to run this after setting up the UART, you
;   may want to make sure the UART is completely restored to it's
;   original state.
;
;   Also the timer is programmed and used, so take note that after this 
;   code the timer is setup (but the GATE is disabled, so it will not
;   count).  I do not use interrupts.  Note the CLI instruction at the
;   start of the code.
;
;   Also I change the ESP value.  This is not necessary if you already
;   have a stack with a few words of space.
;
;   In conclusion: make sure this code does not disturb your environment,
;   and make sure there is nothing set up before running this that could
;   cause it to fail (i.e. PIT clock disabled, no external UART chip select
;   at COM2 address, etc).
;
;---------------------------------------------------------------------------

; allow all instructions in assembler

.486p

;---------------------------------------------------------------------------

; include

UART_BASE equ UART_COM2    ; use external DEBUG UART

include ns486reg.inc       ; NS486SXF registers
include io.inc             ; macros for port IO

;---------------------------------------------------------------------------
        
        ; where to jump after setup

jump_location equ 0FFFCFFF0h

;---------------------------------------------------------------------------

CODESG    SEGMENT PARA USE32 'code'
CPUSPEED  PROC    FAR
          ASSUME  CS:CODESG

;---------------------------------------------------------------------------
; ************ Code ********************************************************
;---------------------------------------------------------------------------

        cli              
        cld

;---------------------------------------------------------------------------
; compute cpu speed
;---------------------------------------------------------------------------
        
        ; set up stack - just in case it is pointing somewhere weird
        ; we are assuming SS and DS are linear 32 bit selectors.

        mov esp, 0fffch    

        ; turn on TIMER

        inpw BIU_CONTROL1               ; read current setting
        push ax                         ; save BIU setting
        outpw BIU_CONTROL1, 0408h       ; enable PIT and GPE 
        
        ; set up PIT

        outpb PIT_CLOCK,   003h         ; use OSC_CLK/2 / 32
        outpb PIT_TICR,    00Ah         ; set gate low - no count
        outpb PIT_CONTROL, 030h         ; rw lsbmsb mode 0
        outpb PIT_COUNT0,  0FFh         ; maximum lsb
        outpb PIT_COUNT0,  0FFh         ; maximum msb

        ; save UART info

        inpb UART_LCR        ; read LCR
        push ax              ; save
        inpb UART_MCR        ; read MCR
        push ax              ; save
        
        ; set up UART

        inpb UART_MCR        ; read MCR
        or    al, 010h
        outpb UART_MCR, al   ; turn on loopback mode

        outpb UART_LCR, 083h ; set DLAB, no parity, 1 stop, 8 bit word

        inpb UART_DLL        ; save DLL
        push ax
        
        inpb UART_DLM        ; save DLM
        push ax

        outpb UART_DLL, 0D0h ; set baud rate - set to 160bps
        outpb UART_DLM, 002h
        outpb UART_LCR, 003h ; DLAB off

        ; loop until UART ready for a character
        ; this is a dummy write to make sure we are up and running

loop1:
        inpb    UART_LSR
        and     al, 020h
        jz      loop1
        
        ; output first character

        outpb   UART_THR, '1'

        ; loop until UART ready again

loop2:
        inpb    UART_LSR
        and     al, 020h
        jz      loop2

        ; start transmit
        ; this is the real character we are timing!

        outpb   UART_THR, '2'
        
        ; start PIT

        outpb PIT_TICR,    01Ah         ; gate high
  
        ; wait till set
        
loop3:
        inpb    UART_LSR
        and     al, 020h
        jz      loop3

        ; stop count

        outpb PIT_TICR,    00Ah         ; gate low

        ; restore some UART registers

        outpb   UART_LCR, 083h          ; set DLAB
        pop     ax
        outpb   UART_DLM, al            ; restore DLM
        pop     ax
        outpb   UART_DLL, al            ; restore DLL
        pop     ax                      ; restore LCR
        outpb   UART_LCR, al

        ; make sure UART is flushed

loop4:
        inpb    UART_LSR
        and     al, 060h
        jz      loop4

        ; this loop makes sure UART is not transmitting when
        ; it is taken out of loopback.  

        ; this was timed at about 2M cycles, or .1 seconds, which is
        ; plenty of time.  (cache on.  with off, probably .2 seconds)
        ; this could probably be reduced quite a lot, but it is not
        ; a huge delay, so experiment if you want to reduce dead time
   
        mov     ecx, 500000
loopd1:
        dec ecx
        jnz loopd1

        ; restore MCR

        pop     ax
        outpb   UART_MCR, al

        ; read count
  
        outpb PIT_CONTROL, 000h         ; latch counter 0
        inpb  PIT_COUNT0
        mov   ds:[0FFF04h], al          ; save count lsb at 0xfff04
        inpb  PIT_COUNT0
        mov   ds:[0FFF05h], al          ; save count msb at 0xfff05

        ; subtract count from 0xFFFF

        mov   ax, ds:[0FFF04h]           ; read 16 bit count
        mov   ebx, 0FFFFh
        sub   bx, ax                     ; subtract from FFFF to get elapsed

        ; multiply by 512 - now equal to CPU speed
        ; 512 = 32*16.  32 is since the PIT clock is 1/32nd of the CPU
        ; clock.  16 is since we are at 160bps and trasmitted 10 bits.
        ; (1/16th of a second)

        shl   ebx, 9

        ; save in memory

        mov   ds:[0FFF00h], ebx        ; save in memory
        
        ; restore BIU settings

        pop ax                         ; get old BIU setting
        outpw BIU_CONTROL1, ax         ; restore

;---------------------------------------------------------------------------
; Jump to user code
;---------------------------------------------------------------------------

        mov eax, jump_location
        jmp eax
        
;---------------------------------------------------------------------------
; Thats it.
;---------------------------------------------------------------------------
        
        HLT

;---------------------------------------------------------------------------

CPUSPEED  ENDP
CODESG  ENDS
        
; end of program - list entry proc (FAR)

        END     CPUSPEED

;---------------------------------------------------------------------------
; END       cpuspeed.asm 
;---------------------------------------------------------------------------

