/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

	Test program to show:

	- How different colors for background, foreground, pen, and brush
	  affect different XVT graphic shapes (including text and icons).

	- How to construct a user interface that uses "icons" that react to
	  user mouse clicks. (These aren't draw_icon-type icons, but just small
	  graphical objects.)

	It will be much easier to understand this program if you try it out
	first to get familiar with its user interface.
****************************************************************************

Implementation Notes:
---------------------
Under XVT/CH the output of some drawing functions, which are used in this program, is different than what you might expect.  Since XVT/CH is a Character Based Window System, only horizontal and vertical lines are displayed accurately.  All other types of lines are forced to horizontal or vertical.

For more details please read the implemenation notes's section of the function call, in the XVT Programmer's Reference Manual.

***************************************************************************/

#define INTPTR_CHK 0				/* elim. requirement for INTPTR casts */
#include "xvt.h"					/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"				/* standard XVT menu tags */
#include "color.h"					/* contains ICON_ID */

APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"color",						/* application's name */
	W_DOC,							/* type of initial window */
	TRUE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	FALSE,							/* close box on initial window? */
	TRUE,							/* want std. font menu? (includes sizes) */
	TRUE							/* want std. style menu? */
};

/*
	LINE_CENTERED means lines are drawn centered side-to-side along the
	mathematical line, and no pixels extend beyond the mathematical endpoint.
	Alternative is the Mac method, in which pixels hang below and to the right
	of the mathematical line.
*/
#ifndef MAC
#define LINE_CENTERED
#endif

/*
	There are 7 types of icons (objects) that the user can click on. There are
	10 icons of the shape type.
*/
typedef enum {T_BACK_CLR, T_FORE_CLR, T_PEN_CLR, T_BRUSH_CLR, T_BRUSH_PAT,
  T_PEN_WIDTH, T_SHAPE} OBJ_TYPE;
typedef enum {S_LINE, S_POLYLINE, S_POLYGON, S_RECT, S_ROUNDRECT, S_OVAL, S_ARC,
  S_PIE, S_TEXT, S_ICON} SHAPE_TYPE;

/* Counts for the number of objects of each type, and total number of objects. */
#define N_COLORS 		11
#define N_COLOR_CHOICES	4
#define N_PATS			8
#define N_WIDTHS 		6
#define N_SHAPES		10
#define N_OBJS			(N_COLORS * N_COLOR_CHOICES + N_PATS + N_WIDTHS +\
						  N_SHAPES)
/* Initial object selections. */
#define SEL_BACK		(1)
#define SEL_FORE		(N_COLORS + 10)
#define SEL_PEN_CLR		(2 * N_COLORS )
#define SEL_BRUSH_CLR	(3 * N_COLORS + 1)
#define SEL_BRUSH_PAT	(N_COLORS * N_COLOR_CHOICES + 1)
#define SEL_WIDTH		(N_COLORS * N_COLOR_CHOICES + N_PATS + 2)
#define SEL_SHAPE		(N_COLORS * N_COLOR_CHOICES + N_PATS + N_WIDTHS + 4)
#define WIN_WIDTH		624
#define WIN_HEIGHT		208

/* Structure for each icon object. */
typedef struct {
	OBJ_TYPE type;					/* type of object (see list above) */
	RCT bound;						/* bounding rectangle rel. to std_win */
	BOOLEAN selected;				/* is it selected? */
	union {
		COLOR color;				/* color, if color-choosing icon */
		PAT_STYLE pat;				/* pattern, if brush-choosing icon */
		int width;					/* width, if pen-width-choosing icon */
		SHAPE_TYPE shape;			/* shape, if shape-choosing icon */
	} v;
} OBJ;

static OBJ obj[N_OBJS];				/* array of objects */
static FONT user_font;				/* font last chosen by user */

/* Following are layout constants. */
#define X_COLOR			448
#define X_COLOR_LABEL	(X_COLOR + 120)
#define Y_COLOR			24

#define X_PAT			248
#define X_PAT_LABEL		288
#define Y_PAT			0

#define X_WIDTH			(X_PAT + 96)
#define X_WIDTH_LABEL	(X_PAT_LABEL + 96)
#define Y_WIDTH			112

#define X_SHAPE			8
#define Y_SHAPE			104

#define OBJ_WIDTH		16
#define OBJ_GAP1		0
#define OBJ_GAP2		16
#define SHAPE_GAP		8
#define SHAPE_GAP2		16
#if XVTWS == MTFWS /* gets around a bug at the X level */
#define FRM_WIDTH		1
#else
#define FRM_WIDTH	    0
#endif

/* View rectangle in which to see sample shape, and rectangle for its frame. */
static RCT view = {8, 16, 88, 240};
  
static RCT view_frame = {0, 0, 96, 248};

#ifdef PROTO
STATICFCN void build_objs(void);
STATICFCN void draw_shape_label(RCT *, char *);
STATICFCN void draw_shape_icon(OBJ *);
STATICFCN void invert_selection_frame(OBJ *);
STATICFCN void invert_obj(int);
STATICFCN void show_objs(WINDOW);
STATICFCN void show_view_shape(SHAPE_TYPE);
STATICFCN void show_view(void);
STATICFCN void do_update(WINDOW);
STATICFCN void new_selection(OBJ *);
STATICFCN int obj_hit(PNT);
STATICFCN void do_menu(MENU_TAG);
#else
STATICFCN void build_objs();
STATICFCN void draw_shape_label();
STATICFCN void draw_shape_icon();
STATICFCN void invert_selection_frame();
STATICFCN void invert_obj();
STATICFCN void show_objs();
STATICFCN void show_view_shape();
STATICFCN void show_view();
STATICFCN void do_update();
STATICFCN void new_selection();
STATICFCN int obj_hit();
STATICFCN void do_menu();
#endif

/* Function to initialize array of icon objects. */
#ifdef FPROTO
static void build_objs(void)
#else
static void build_objs()
#endif
{
	static COLOR colors[] = {COLOR_RED, COLOR_GREEN, COLOR_BLUE, COLOR_CYAN,
	  COLOR_MAGENTA, COLOR_YELLOW, COLOR_BLACK, COLOR_DKGRAY, COLOR_GRAY,
	  COLOR_LTGRAY, COLOR_WHITE};
	static PAT_STYLE pats[] = {PAT_HOLLOW, PAT_SOLID, PAT_HORZ, PAT_VERT,
      PAT_FDIAG, PAT_BDIAG, PAT_CROSS, PAT_DIAGCROSS};
    static int widths[] = {0, 1, 3, 5, 7, -1};
    static OBJ_TYPE types[] = {T_BACK_CLR, T_FORE_CLR, T_PEN_CLR, T_BRUSH_CLR,
      T_BRUSH_PAT, T_PEN_WIDTH, T_SHAPE};
    static SHAPE_TYPE shapes[] = {S_LINE, S_POLYLINE, S_POLYGON, S_RECT,
      S_ROUNDRECT, S_OVAL, S_ARC, S_PIE, S_TEXT, S_ICON};
	int x, y;
	OBJ *p;

	for (x = 0; x < N_COLOR_CHOICES; x++)
		for (y = 0; y < N_COLORS; y++) {
			p = &obj[x * N_COLORS + y];
			p->type = types[x];
			p->bound.left = X_COLOR + (OBJ_WIDTH + OBJ_GAP2) * x;
			p->bound.top = Y_COLOR + (OBJ_WIDTH + OBJ_GAP1) * y;
			p->bound.right = p->bound.left + OBJ_WIDTH;
			p->bound.bottom = p->bound.top + OBJ_WIDTH;
			p->v.color = colors[y];
		}
	for (y = 0; y < N_PATS; y++) {
		p = &obj[N_COLOR_CHOICES * N_COLORS + y];
		p->type = T_BRUSH_PAT;
		p->bound.left = X_PAT;
		p->bound.top = Y_PAT + (OBJ_WIDTH + OBJ_GAP1) * y;
		p->bound.right = p->bound.left + OBJ_WIDTH + OBJ_GAP2;
		p->bound.bottom = p->bound.top + OBJ_WIDTH;
		p->v.pat = pats[y];
	}
	for (y = 0; y < N_WIDTHS; y++) {
		p = &obj[N_COLOR_CHOICES * N_COLORS + N_PATS + y];
		p->type = T_PEN_WIDTH;
		p->bound.left = X_WIDTH;
		p->bound.top = Y_WIDTH + (OBJ_WIDTH + OBJ_GAP1) * y;
		p->bound.right = p->bound.left + OBJ_WIDTH + OBJ_GAP2;
		p->bound.bottom = p->bound.top + OBJ_WIDTH;
		p->v.width = widths[y];
	}
	assert2("build_objs", N_SHAPES == 10); /* layout assumption */
	for (x = 0; x < 5; x++)
		for (y = 0; y < 2; y++) {
			p = &obj[N_COLOR_CHOICES * N_COLORS + N_PATS + N_WIDTHS + x + 5 * y];
			p->type = T_SHAPE;
			p->bound.left = X_SHAPE + x * (2 * OBJ_WIDTH + OBJ_GAP2);
			p->bound.top = Y_SHAPE + y * 2 * (OBJ_WIDTH + OBJ_GAP2);
			p->bound.right = p->bound.left + 2 * OBJ_WIDTH;
			p->bound.bottom = p->bound.top + 2 * OBJ_WIDTH;
			p->v.shape = shapes[x + 5 * y];
		}
		obj[SEL_BACK].selected = TRUE;
		obj[SEL_FORE].selected = TRUE;
		obj[SEL_PEN_CLR].selected = TRUE;
		obj[SEL_BRUSH_CLR].selected = TRUE;
		obj[SEL_BRUSH_PAT].selected = TRUE;
		obj[SEL_WIDTH].selected = TRUE;
		obj[SEL_SHAPE].selected = TRUE;
}

/* Function to draw a string below a rectangle. Used for shape-icon labels. */
#ifdef FPROTO
static void draw_shape_label(RCT *r,char *s)
#else
static void draw_shape_label(r, s)
RCT *r;
char *s;
#endif
{
	int ascent, w;

	set_font(&small_font, FALSE);
	get_font_metrics(NULL, &ascent, NULL);
	w = get_text_width(s, -1);
	draw_text((r->left + r->right - w) / 2, r->bottom + ascent, s, -1);
}

/* Function to draw a shape-choosing icon. */
#ifdef FPROTO
static void draw_shape_icon(OBJ *objp)
#else
static void draw_shape_icon(objp)
OBJ *objp;
#endif
{
	RCT r, rct;
	PNT pnt, p[4];
	int w, descent;
	CBRUSH brush;

	r = objp->bound;
	brush.pat = PAT_SOLID;
	brush.color = COLOR_LTGRAY;
	set_cbrush(&brush);
	switch (objp->v.shape) {
	case S_LINE:
		pnt.h = r.left + SHAPE_GAP;
		pnt.v = r.bottom - SHAPE_GAP;
		move_to(pnt);
		pnt.h = r.right - SHAPE_GAP;
		pnt.v = r.top + SHAPE_GAP;
		draw_line(pnt);
		draw_shape_label(&r, "Line");
		break;
	case S_POLYLINE:
		p[0].h = r.left + SHAPE_GAP;
		p[0].v = r.bottom - SHAPE_GAP;
		p[1].h = (r.left + r.right) / 2;
		p[1].v = r.top + SHAPE_GAP;
		p[2].h = r.right - SHAPE_GAP;
		p[2].v = (r.top + r.bottom) / 2;
		draw_polyline(p, 3);
		draw_shape_label(&r, "Polyln");
		break;
	case S_POLYGON:
		p[0].h = r.left + SHAPE_GAP;
		p[0].v = r.bottom - SHAPE_GAP;
		p[1].h = (r.left + r.right) / 2;
		p[1].v = r.top + SHAPE_GAP;
		p[2].h = r.right - SHAPE_GAP;
		p[2].v = (r.top + r.bottom) / 2;
		draw_polygon(p, 3);
		draw_shape_label(&r, "Polygn");
		break;
	case S_RECT:
		set_rect(&rct, r.left + SHAPE_GAP, r.top + SHAPE_GAP,
		  r.right - SHAPE_GAP, r.bottom - SHAPE_GAP);
		draw_rect(&rct);
		draw_shape_label(&r, "Rect");
		break;
	case S_ROUNDRECT:
		set_rect(&rct, r.left + SHAPE_GAP, r.top + SHAPE_GAP,
		  r.right - SHAPE_GAP, r.bottom - SHAPE_GAP);
		draw_roundrect(&rct, 2 * SHAPE_GAP, 2 * SHAPE_GAP);
		draw_shape_label(&r, "RndRct");
		break;
	case S_OVAL:
		set_rect(&rct, r.left + SHAPE_GAP, r.top + SHAPE_GAP,
		  r.right - SHAPE_GAP, r.bottom - SHAPE_GAP);
		draw_oval(&rct);
		draw_shape_label(&r, "Oval");
		break;
	case S_ARC:
		set_rect(&rct, r.left + SHAPE_GAP, r.top + SHAPE_GAP,
		  r.right - SHAPE_GAP, r.bottom - SHAPE_GAP);
		draw_arc(&rct, r.right, (r.top + r.bottom) / 2, r.left, r.top);
		draw_shape_label(&r, "Arc");
		break;
	case S_PIE:
		set_rect(&rct, r.left - SHAPE_GAP, r.top + SHAPE_GAP,
		  r.right - SHAPE_GAP, r.bottom + SHAPE_GAP);
		draw_pie(&rct, r.right, (r.top + r.bottom) / 2, r.left + SHAPE_GAP,
		  r.top);
		draw_shape_label(&r, "Pie");
		break;
	case S_TEXT:
		set_font(&normal_font, FALSE);
		get_font_metrics(NULL, NULL, &descent);
		w = get_text_width("XVT", -1);
		draw_text((r.left + r.right - w) / 2, (r.top + r.bottom) / 2 + descent,
		  "XVT", -1);
		draw_shape_label(&r, "Text");
		break;
	case S_ICON:
		draw_icon(r.left, r.top, ICON_ID);
		draw_shape_label(&r, "Icon");
	}
}

/* Function to show selected icon by inverting thick frame around it. */
#ifdef FPROTO
static void invert_selection_frame(OBJ *objp)
#else
static void invert_selection_frame(objp)
OBJ *objp;
#endif
{
	RCT rct;
	DRAW_CTOOLS save_tools, t;

	get_draw_ctools(&save_tools);
	get_normal_ctools(&t);
	t.pen.width = FRM_WIDTH;
	t.brush.pat = PAT_HOLLOW;
	t.mode = M_XOR;
	set_draw_ctools(&t);
	rct = objp->bound;
	rct.left -= FRM_WIDTH;
	rct.top -= FRM_WIDTH;
	rct.right += FRM_WIDTH;
	rct.bottom += FRM_WIDTH;
	draw_rect(&rct);
	set_draw_ctools(&save_tools);
}

/*
	Function to invert icon (vs. its frame; see previous function).
	Used to provide feedback while user is clicking mouse.
*/
#ifdef FPROTO
static void invert_obj(int n)
#else
static void invert_obj(n)
int n;
#endif
{
	DRAW_CTOOLS save_tools, t;

	if (n < 0)
		return;
	get_draw_ctools(&save_tools);
	get_normal_ctools(&t);
	t.pen.pat = PAT_HOLLOW;
	t.brush.pat = PAT_SOLID;
	t.brush.color = COLOR_BLACK;
	t.mode = M_XOR;
	set_draw_ctools(&t);
	draw_rect(&obj[n].bound);
	set_draw_ctools(&save_tools);
}

/* Function to draw all object icons. Called in response to update events. */
#ifdef FPROTO
static void show_objs(WINDOW win)
#else
static void show_objs(win)
WINDOW win;
#endif
{
	static char *color_labels[] = {"Red", "Green", "Blue", "Cyan", "Magenta",
	  "Yellow", "Black", "DkGray", "Gray", "LtGray", "White"};
	static char *choice_labels[] = {"Bak", "For", "Pen", "Brs"};
	static char *pat_labels[] = {"Hollow", "Solid", "Horz", "Vert", "FDiag",
	  "Bdiag", "Cross", "DgCross"};
	static char *width_labels[] = {"Hollow", "1 Pixl", "3 Pixls", "5 Pixls",
	  "7 Pixls", "Rubber"};
	int i, leading, ascent, descent, width_adj;
	DRAW_CTOOLS t;
	PNT pnt;
	RCT update_rect;

	set_draw_ctools(get_normal_ctools(&t));
	set_font(&small_font, FALSE);
	get_font_metrics(&leading, &ascent, &descent);
	for (i = 0; i < N_COLORS; i++)
		draw_text(X_COLOR_LABEL, Y_COLOR + leading + ascent +
		  i * (OBJ_WIDTH + OBJ_GAP1), color_labels[i], -1);
	for (i = 0; i < N_COLOR_CHOICES; i++)
		draw_text(X_COLOR + i * (OBJ_WIDTH + OBJ_GAP2), Y_COLOR - OBJ_GAP1 - descent,
		  choice_labels[i], -1);
	for (i = 0; i < N_PATS; i++)
		draw_text(X_PAT_LABEL, Y_PAT + leading + ascent +
		  i * (OBJ_WIDTH + OBJ_GAP1), pat_labels[i], -1);
	for (i = 0; i < N_WIDTHS; i++)
		draw_text(X_WIDTH_LABEL, Y_WIDTH + leading + ascent +
		  i * (OBJ_WIDTH + OBJ_GAP1), width_labels[i], -1);
	for (i = 0; i < N_OBJS; i++) {
		if (obj[i].type == T_SHAPE)
			set_rect(&update_rect, 
				obj[i].bound.left,
				obj[i].bound.top,
				obj[i].bound.right,
				obj[i].bound.bottom + (obj[i].bound.bottom - obj[i].bound.top));
		else
			set_rect(&update_rect, 
				obj[i].bound.left,
				obj[i].bound.top,
				obj[i].bound.right,
				obj[i].bound.bottom);
		if (!needs_update(win, &update_rect))
			continue;
		set_draw_ctools(get_normal_ctools(&t));
		switch (obj[i].type) {
		case T_BACK_CLR:
		case T_FORE_CLR:
		case T_PEN_CLR:
		case T_BRUSH_CLR:
			t.brush.color = obj[i].v.color;
			set_cbrush(&t.brush);
			break;
		case T_BRUSH_PAT:
			t.brush.color = COLOR_BLACK;
			t.brush.pat = obj[i].v.pat;
			set_cbrush(&t.brush);
			break;
		case T_PEN_WIDTH:
			t.brush.pat = PAT_HOLLOW;
			set_cbrush(&t.brush);
			switch (obj[i].v.width) {
			case -1:
				t.pen.pat = PAT_RUBBER;
				width_adj = 1;
				break;
			case 0:
				t.pen.pat = PAT_HOLLOW;
				break;
			default:
				t.pen.width = obj[i].v.width;
				width_adj = obj[i].v.width;
			}
			set_cpen(&t.pen);
#ifdef LINE_CENTERED
			width_adj = 0;
#endif
			pnt.h = obj[i].bound.left + OBJ_GAP1;
			pnt.v = (obj[i].bound.top + obj[i].bound.bottom - width_adj) / 2;
			move_to(pnt);
			pnt.h = obj[i].bound.right - width_adj - OBJ_GAP1;
			draw_line(pnt);
			break;
		case T_SHAPE:
			draw_shape_icon(&obj[i]);
			t.brush.pat = PAT_HOLLOW;
			set_cbrush(&t.brush);
		}
		t.pen.pat = PAT_SOLID;
		t.pen.width = 1;
		set_cpen(&t.pen);
		draw_rect(&obj[i].bound);
	}
	for (i = 0; i < N_OBJS; i++)
		if (obj[i].selected)
			invert_selection_frame(&obj[i]);
}

/* Function to draw sample shape in view rectangle. */
#ifdef FPROTO
static void show_view_shape(SHAPE_TYPE shape)
#else
static void show_view_shape(shape)
SHAPE_TYPE shape;
#endif
{
	RCT rct, r;
	PNT pnt, p[4];
	int descent, w;

	r = view;
	switch (shape) {
	case S_LINE:
		pnt.h = r.left + SHAPE_GAP2;
		pnt.v = r.bottom - SHAPE_GAP2;
		move_to(pnt);
		pnt.h = r.right - SHAPE_GAP2;
		pnt.v = r.top + SHAPE_GAP2;
		draw_line(pnt);
		break;
	case S_POLYLINE:
		p[0].h = r.left + SHAPE_GAP2;
		p[0].v = r.bottom - SHAPE_GAP2;
		p[1].h = (r.left + r.right) / 2;
		p[1].v = r.top + SHAPE_GAP2;
		p[2].h = r.right - SHAPE_GAP2;
		p[2].v = (r.top + r.bottom) / 2;
		draw_polyline(p, 3);
		break;
	case S_POLYGON:
		p[0].h = r.left + SHAPE_GAP2;
		p[0].v = r.bottom - SHAPE_GAP2;
		p[1].h = (r.left + r.right) / 2;
		p[1].v = r.top + SHAPE_GAP2;
		p[2].h = r.right - SHAPE_GAP2;
		p[2].v = (r.top + r.bottom) / 2;
		draw_polygon(p, 3);
		break;
	case S_RECT:
		set_rect(&rct, r.left + SHAPE_GAP2, r.top + SHAPE_GAP2,
		  r.right - SHAPE_GAP2, r.bottom - SHAPE_GAP2);
		draw_rect(&rct);
		break;
	case S_ROUNDRECT:
		set_rect(&rct, r.left + SHAPE_GAP2, r.top + SHAPE_GAP2,
		  r.right - SHAPE_GAP2, r.bottom - SHAPE_GAP2);
		draw_roundrect(&rct, 2 * SHAPE_GAP2, 2 * SHAPE_GAP2);
		break;
	case S_OVAL:
		set_rect(&rct, r.left + SHAPE_GAP2, r.top + SHAPE_GAP2,
		  r.right - SHAPE_GAP2, r.bottom - SHAPE_GAP2);
		draw_oval(&rct);
		break;
	case S_ARC:
		set_rect(&rct, r.left + SHAPE_GAP2, r.top + SHAPE_GAP2,
		  r.right - SHAPE_GAP2, r.bottom - SHAPE_GAP2);
		draw_arc(&rct, r.right, (r.top + r.bottom) / 2, r.left, r.top);
		break;
	case S_PIE:
		set_rect(&rct, r.left + SHAPE_GAP2, r.top + SHAPE_GAP2,
		  r.right - SHAPE_GAP2, r.bottom - SHAPE_GAP2);
		draw_pie(&rct, r.right, (r.top + r.bottom) / 2, r.left + SHAPE_GAP2,
		  r.top);
		break;
	case S_TEXT:
		set_font(&user_font, FALSE);
		get_font_metrics(NULL, NULL, &descent);
		w = get_text_width("XVT", -1);
		draw_text((r.left + r.right - w) / 2, (r.top + r.bottom) / 2 + descent,
		  "XVT", -1);
		break;
	case S_ICON:
		draw_icon(r.left + 50, r.top + 50, ICON_ID);
	}
}

/* Function to update view area. */
#ifdef FPROTO
static void show_view(void)
#else
static void show_view()
#endif
{
	int i;
	DRAW_CTOOLS t;

	get_normal_ctools(&t);
	t.pen.pat = PAT_HOLLOW;
	set_draw_ctools(&t);
	draw_rect(&view);
	t.pen.pat = PAT_SOLID;
	t.opaque_text = TRUE;
	for (i = 0; i < N_OBJS; i++)
		if (obj[i].selected)
			switch (obj[i].type) {
			case T_BACK_CLR:
				t.back_color = obj[i].v.color;
				break;
			case T_FORE_CLR:
				t.fore_color = obj[i].v.color;
				break;
			case T_PEN_CLR:
				t.pen.color = obj[i].v.color;
				break;
			case T_BRUSH_CLR:
				t.brush.color = obj[i].v.color;
				break;
			case T_BRUSH_PAT:
				t.brush.pat = obj[i].v.pat;
				break;
			case T_PEN_WIDTH:
				switch (obj[i].v.width) {
				case -1:
					t.pen.pat = PAT_RUBBER;
					break;
				case 0:
					t.pen.pat = PAT_HOLLOW;
					break;
				default:
					t.pen.width = obj[i].v.width;
				}
				break;
			case T_SHAPE:
				set_draw_ctools(&t);
				show_view_shape(obj[i].v.shape);
			}
}

/* Function to handle update events. */
#ifdef FPROTO
static void do_update(WINDOW win)
#else
static void do_update(win)
WINDOW win;
#endif
{
	RCT rct;
	DRAW_CTOOLS t;

	get_normal_ctools(&t);
	get_client_rect(win, &rct);
	t.pen.pat = PAT_HOLLOW;
	set_draw_ctools(&t);
	draw_rect(&rct);
	t.pen.pat = PAT_SOLID;
	t.pen.width = 2;
	set_draw_ctools(&t);
	if (needs_update(win, &view_frame)) {
		draw_rect(&view_frame);
		show_view();
	}
	show_objs(win);
}

/*
	Function to show new selection. Must invert current selection of object
	of same type (to restore appearance), and then invert new selection.
*/
#ifdef FPROTO
static void new_selection(OBJ *objp)
#else
static void new_selection(objp)
OBJ *objp;
#endif
{
	int i;

	for (i = 0; i < N_OBJS; i++)
		if (obj[i].selected && obj[i].type == objp->type) {
			invert_selection_frame(&obj[i]);
			obj[i].selected = FALSE;
			break;
		}
	invert_selection_frame(objp);
	objp->selected = TRUE;
	show_view();
}

/* Function to find object where mouse is, if any. */
#ifdef FPROTO
static int obj_hit(PNT pnt)
#else
static int obj_hit(pnt)
PNT pnt;
#endif
{
	int i;

	for (i = 0; i < N_OBJS; i++)
		if (pt_in_rect(&obj[i].bound, pnt))
			return(i);
	return(-1);
}
	
/* Function to handle menu command. */
#ifdef FPROTO
static void do_menu(MENU_TAG cmd)
#else
static void do_menu(cmd)
MENU_TAG cmd;
#endif
/* tag identifying menu item */
{
	switch (cmd) {
	case M_FILE_QUIT:
		terminate();
		break;
	}
}

/*
	Function called by XVT to initialize application. Must make sure
	window is big enough and build objects.
*/
BOOLEAN XVTENTRY appl_init BTCENTRY(void)
{
	RCT rct;
	PNT pnt;
	
#if XVTWS == WMWS
/*	fatal("The Color example is not supported for XVT/CH\n");*/
#endif	

	pnt.h = pnt.v = 0;
	local_global(std_win, &pnt, 1);
	set_rect(&rct, pnt.h, pnt.v, pnt.h + WIN_WIDTH, pnt.v + WIN_HEIGHT);
	move_window(std_win, &rct);
	build_objs();
	user_font = normal_font;
	set_font_menu(&user_font);
	return(TRUE);
}

/* Main event function. */
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;
EVENT_PTR ep;
#endif
{
	static int down_obj = -1;
	static BOOLEAN inverted;
	int new_obj;

	switch (ep->type) {
	case E_MOUSE_DOWN:
	case E_MOUSE_DBL:
		trap_mouse(win);
		down_obj = obj_hit(ep->v.mouse.where);
		if (down_obj >= 0) {
			invert_obj(down_obj);
			inverted = TRUE;
		}
		break;
	case E_MOUSE_MOVE:
		if (down_obj >= 0) {
			new_obj = obj_hit(ep->v.mouse.where);
			if (new_obj == down_obj) {
				if (!inverted)
					invert_obj(down_obj);
				inverted = TRUE;
			}
			else
				if (inverted) {
					invert_obj(down_obj);
					inverted = FALSE;
				}
		}
		break;
	case E_MOUSE_UP:
		release_mouse();
		if (down_obj >= 0) {
			new_obj = obj_hit(ep->v.mouse.where);
			if (inverted) {
				invert_obj(down_obj);
				if (!obj[new_obj].selected)
					new_selection(&obj[new_obj]);
			}
			down_obj = -1;
		}
		break;
	case E_UPDATE:
		do_update(win);
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag);
		break;
	case E_FONT:
		user_font = ep->v.font.font;
		set_font_menu(&user_font);
		show_view();
		break;
	case E_CLOSE:
		terminate();
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
	}
}

/* Function called by XVT just before terminating application. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
}

/* End. */
