/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  Selective_backup.c                                                       */
/*                                                                           */
/*  Selective volume backup for CTOS Archive utilities.                      */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  04/24/92  122H.14  D. Gilson     /  archive names under 12 chars need    */
/*                                   /  a dir spec but those over 12 don't.  */
/*                                      Make all need dir spec. P15513331    */
/*  03/27/92  122G.13  D. Gilson     /  Correct file length problem          */
/*                                      LfaEOF may not be close to FileLength*/
/*  03/20/92  122G.12  D. Gilson     /  Added param to create_archive_sent...*/
/*                                      This is so it will match the         */
/*                                      declaration in backup.c.             */
/*  12/21/91  130E.11  D. Gilson     /  Added multi archive volume support   */
/*  10/03/91  130D.10  D. Gilson     /  Correct Archive.Spec to only accept  */
/*                                      [TapeSpec]0 and [TapeSpec]+ and not  */
/*                                      [tapeSpec]n when backing up.  Various*/
/*                                      ercs happen when using [tapeSpec]n   */
/*                                      like 9074 after a few files are      */
/*                                      written to the tape. PLE 15278790    */
/*  07/15/91  130A.09  P. Johansson  /  '[Delete existing archive dataset?]' */
/*                                      must be 'Yes' if user interaction is */
/*                                      suppressed.                          */
/*  06/20/91  130A.08  P. Johansson  /  Store all passwords, even encrypted, */
/*                                      in the archive dataset.              */
/*  05/28/91  121J.07  P. Johansson  /  Terminate position string with zero  */
/*                                      so that atoi() works correctly.      */
/*  05/08/91  121J.06  C.G. Naik     /  Dynamically determine correct FHB    */
/*                                      code for GetFileStatus.              */
/*  03/25/91  121H.05  P. Johansson  /  Ignore 'fNoSave' in FHB if filename  */
/*                                      is not wildcarded.                   */
/*  02/05/91  121G.04  P. Johansson  /  'File list' is a required parameter. */
/*  01/30/91  121F.03  P. Johansson  /  Explicit OpenByteStream for [Vid] so */
/*                                      that video filtering works.          */
/*  01/14/91  121F.02  P. Johansson  /  Support invocation from "Selective   */
/*                                      Backup" and "Tape Selective Backup"  */
/*                                      command forms (old, 12.0 style).     */
/*  01/04/91  121F.01  P. Johansson  /  Check both creation and modification */
/*                                      dates if incremental backup.         */
/*  12/13/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* Suppress C run-time (only CTOS functionality needed) */

pragma Off(List);
#include <stub.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocExch
#define BuildFileSpec
#define CloseFile
#define CSubparams
#define CurrentOsVersion
#define GetDateTime
#define GetFileStatus
#define NlsULCmpB
#define OpenFile
#define ParseFileSpec
#define Read
#define RgParam
#define Send
#define ULCmpB
#define Wait
#define WildCardInit
#define WildCardNext

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType
#define sdType
#define UCBType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern iob_type *allocate_iob(unsigned buffer_size);
extern void create_archive_io_process(void);
extern void deallocate_iob(iob_type *iob);
extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern unsigned fetch_mfd(sdType *sd_filename);
extern void get_archive_datasets(unsigned param_num); 		/* m11 */
extern mfd_entry_type *find_mfd_entry(char directory[]);
extern void initialize_msgs(void);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern unsigned open_log_file(void);
extern void open_video(void);
extern unsigned parse_parameters(param_descr_type *param_descr, unsigned n);
extern Boolean proceed_sel_arch(flag *finish_flag);
extern sdType *standard_msg(unsigned msg_num);
extern vid_only_msg(unsigned nls_msg_index, unsigned signature,
                    sdType nls_parms[], unsigned nls_parms_len);

/* Error return codes used by this module */

#define FsErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern archive_type archive;
extern FhbType fhb;
extern journal_type journal;
extern char sbVerRun[];
extern vhb_type vhb;

/* Global variables exported by this manuscript */

unsigned internal_os_version;
path_type path;
summary_type summary;
target_type target;
vlpb_type vlpb;

/* Static variables global within this manuscript */

private unsigned fhb_code;

/* Function prototypes defined before the functions themselves are declared */

void backup_file(sdType *sd_filespec, sdType *sd_password, Boolean wildcard);
void backup_files(void);
void count_files(void);
void create_archive_EOD(void);
void create_archive_sentinel(unsigned long file_length);
void summarize_backup(void);
void update_path(unsigned fh);
void validate_parameters(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Short and sweet, just munch through all the necessary functions for archiving
 a selected group of files...*/

void main(void) {

   unsigned erc, os_version, os_revision;

   open_video();		/* Before ANY [Vid] use! (so filter works) */
   initialize_msgs();		/* Use a message file if one exists */
   if ((erc = CurrentOsVersion(&os_version, &os_revision)) != ercOK)
      exit_with_msg(erc, 0);
   internal_os_version = (os_version << 8) | os_revision;
   if ((erc = AllocExch(&target.msg_exch)) != ercOK)
      exit_with_msg(erc, 0);
   validate_parameters();	/* Check the syntax of the Executive form */
   create_archive_io_process();	/* Runs in parallel for efficiency */
   count_files();		/* See how many (maybe wildcards) to backup */
   backup_files();		/* Then go ahead an process them */
   summarize_backup();		/* Finally, report our findings... */
   exit_with_msg(ercOK, 0);	/* ...and leave gracefully */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The parameters for the executive command form for 'Selective Archive' are
 described in the table below.  As much as possible, they are parsed by the
 generic routine 'parse_parameters' and are transferred into appropriate
 fields in the "variable-length parameter block" (vlpb).  If the generic
 parsing is successful, other tests that involve dependencies between the
 parameters are applied. */

private void validate_parameters(void) {

   unsigned long current_time = 0;
   unsigned i;									/* m11 */
   char device[MAX_DEVICE_LENGTH], directory[MAX_DIRECTORY_LENGTH],
      filename[MAX_FILENAME_LENGTH], node[MAX_NODE_LENGTH],
      position[MAX_FILENAME_LENGTH], volume[MAX_VOLUME_LENGTH];
   unsigned device_len, directory_len, erc, filename_len, n_exec_param,
      node_len, password_len, position_len, spec_len, volume_len;
   Boolean floppy_default = FALSE;
   sdType nls_parms[] = {         NULL, 0,
                         (void *) &sbVerRun[1], sbVerRun[0],
                         (void *) &vlpb.backup_after,
                                  sizeof(vlpb.backup_after)};
   static param_descr_type exec_param[] = {
      NULL, 0, 0, 0,		/* Dummy corresponds to name of command */
      &vlpb.from_specs, files_param, sizeof(vlpb.from_specs), NLS_FILE_LIST,
      &vlpb.backup_after, date_param, sizeof(vlpb.backup_after),
       NLS_ARCHIVE_AFTER_DATE,
      &vlpb.confirm, Boolean_param, sizeof(vlpb.confirm), NLS_CONFIRM,
      NULL, 0, 0, 0,										/*  m11 */
/*    &archive.spec[0], ascii_param,
       sizeof(archive.spec[0]), NLS_ARCHIVE_DATASET,			m11 */
      &vlpb.overwrite_OK, Boolean_param, sizeof(vlpb.overwrite_OK),
       NLS_DELETE_EXISTING,
      &journal.filespec, ascii_param, sizeof(journal.filespec), NLS_LOG_FILE,
      &vlpb.verify_write, Boolean_param, sizeof(vlpb.verify_write), NLS_VERIFY,
      &vlpb.noninteractive, Boolean_param, sizeof(vlpb.noninteractive),
       NLS_NONINTERACTIVE};
   sdType *sd_archive, sd_case;

   RgParam(0, 0, &nls_parms[0]);
   if (nls_parms[0].pb != NULL && nls_parms[0].cb != 0)
      log_msg(NLS_COMMAND_VERSION, 1, nls_parms, sizeof(nls_parms));
   RgParam(0, 1, &sd_case);
   if (sd_case.cb == 2 && ULCmpB(sd_case.pb, "SD", 2) == 0xFFFF) {
      n_exec_param = last(exec_param);		/* "Selective Backup" form */
      floppy_default = TRUE;
   } else if (sd_case.cb == 2 && ULCmpB(sd_case.pb, "ST", 2) == 0xFFFF)
      n_exec_param = last(exec_param) - 1;	/* "Tape Selective Backup" */
   else
      n_exec_param = last(exec_param) + 1;	/* "Selective Archive" form */
   if ((erc = parse_parameters(exec_param, n_exec_param)) != ercOK)
      exit_with_msg(erc, 0);
   if ((erc = open_log_file()) != ercOK)
      exit_with_msg(erc, NLS_CANT_OPEN_LOG_FILE);
   if (CSubparams(FILE_LIST_PARAM) == 0)
      exit_with_msg(ercSyntax, NLS_FILE_LIST_REQUIRED);
/* if (CSubparams(SELECTIVE_ARCHIVE_DATASET_PARAM) > 1)
      exit_with_msg(ercSyntax, NLS_TOO_MANY_DATASETS);				m11 */
   get_archive_datasets(SELECTIVE_ARCHIVE_DATASET_PARAM);		/*  m11 */
   if (vlpb.backup_after != 0) {
      GetDateTime(&current_time);
      if (vlpb.backup_after > current_time)
         exit_with_msg(ercSyntax, NLS_FUTURE_DATE);
      log_msg(NLS_INCREMENTAL_FROM, 1, nls_parms, sizeof(nls_parms));
   }
   if (archive.spec[archive.device_num][0] == 0) {
      sd_archive = standard_msg((floppy_default) ? NLS_DEFAULT_FLOPPY
                                                 : NLS_DEFAULT_DATASET);
      memcpy(&archive.spec[archive.device_num][1], sd_archive->pb,
             archive.spec[archive.device_num][0] =
                _min(sd_archive->cb, last(archive.spec[archive.device_num])));
   }
   erc = ParseFileSpec(0, &archive.spec[archive.device_num][1],
                       archive.spec[archive.device_num][0], FALSE, node,
                       &node_len, device, &device_len, NULL, 0, position,
                       &position_len, &archive.password[archive.device_num][1],
                       &password_len,
                       FALSE, 3);
   archive.password[archive.device_num][0] = password_len;
   if (erc == ercOK) {
      archive.sequential = TRUE;
      BuildFileSpec(0, &archive.spec[archive.device_num][1], &spec_len,
                    NULL, last(archive.spec[archive.device_num]),
                    FALSE, node, node_len, device, device_len, NULL, 0, NULL,
                    0, FALSE, NULL, 0, FALSE, 0);
      archive.spec[archive.device_num][0] = spec_len;
      archive.current_position = 0xFFFF;	/* Unknown on the first tape */
      if (position_len == 1 && position[0] == '+')
         archive.target_position = 0xFFFF;	/* Sentinel value means '+' */
      else if (position_len != 0) { /* m10... */
         position[position_len] = 0;
         archive.target_position = atoi(position);
         if (archive.target_position != 0)
             exit_with_msg(ercSyntax, NLS_INVALID_TAPE_MARK_SPEC);
      } /* ...m10  m11... */
      for (i = 1; i < archive.device_num_last; i++) {
         erc = ParseFileSpec(0, &archive.spec[i][1],
                             archive.spec[i][0],
                             FALSE, node,
                             &node_len, device, &device_len, NULL, 0, position,
                             &position_len,
                             &archive.password[i][1],
                             &password_len,
                             FALSE, 3);
         archive.password[i][0] = password_len;
         if (erc == ercOK) {
            BuildFileSpec(0, &archive.spec[i][1], &spec_len, NULL,
                          last(archive.spec[i]), FALSE,
                          node, node_len, device,
                          device_len, NULL, 0, NULL, 0, FALSE, NULL, 0,
                          FALSE, 0);
            archive.spec[i][0] = spec_len;
         }
      }      											  /* ...m11 */
   } else {
      if (CSubparams(SELECTIVE_ARCHIVE_DATASET_PARAM) > 1)		/* m11 */
         exit_with_msg(ercSyntax, NLS_TOO_MANY_DATASETS);		/* m11 */
      exec_param[4].type = path_param;
      if ((erc = parse_parameters(exec_param, last(exec_param) + 1)) != ercOK)
         exit_with_msg(erc, 0);
      ParseFileSpec(0, &archive.spec[archive.device_num][1],
                    archive.spec[archive.device_num][0], FALSE, node,
                    &node_len, volume, &volume_len, directory, &directory_len,
                    filename, &filename_len,
                    &archive.password[archive.device_num][1],
                    &password_len, FALSE, 0);
      if (directory_len == 0 )									/* m14 */
         exit_with_msg(ercSyntax, NLS_DIRECTORY_ERROR);			/* m14 */
      archive.password[archive.device_num][0] = password_len;
      BuildFileSpec(0, &archive.spec[archive.device_num][1],
                    &spec_len, NULL, last(archive.spec[archive.device_num]),
                    (volume_len == 0), node, node_len, volume, volume_len,
                    directory, directory_len, filename, filename_len, FALSE,
                    NULL, 0, FALSE, 0);
      archive.spec[archive.device_num][0] = spec_len;
      sd_archive = standard_msg(NLS_ARCHIVE_SUFFIX);
      memcpy(
      &archive.spec[archive.device_num][archive.spec[archive.device_num][0]+1],
      sd_archive->pb,sd_archive->cb);
      archive.spec[archive.device_num][0] += sd_archive->cb;
   }
   if (vlpb.noninteractive && !vlpb.overwrite_OK)
      exit_with_msg(ercSyntax, NLS_MUST_OVERWRITE);
   vlpb.write_archive = TRUE;
   fhb_code = (internal_os_version >= 0x0B00) ? FILE_STATUS_FHB
                                              : FILE_STATUS_FHB_RESERVED;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 In order to give the user (who may be sitting in front of the video display)
 a warm, fuzzy feeling of progress, pre-process the wild card list(s) and
 count how many files match.  This information is used in the "Processing file
 M of N ..." message at the top of the screen. */

private void count_files(void) {

   Boolean wildcard_encountered = FALSE;
   unsigned erc, i;
   sdType sd_filespec;
   char wildcard_buffer[800];

   for (i = 0; i < CSubparams(FILE_LIST_PARAM); i++) {
      if ((erc = RgParam(1, i, &sd_filespec)) != ercOK)
         exit_with_msg(erc, 0);
      if ((erc = WildCardInit(sd_filespec.pb, sd_filespec.cb, wildcard_buffer,
                              sizeof(wildcard_buffer))) != ercOK)
         if (erc == ercWCBufferTooSmall)
            exit_with_msg(erc, NLS_TOO_MANY_WILDCARDS);
         else if (erc == ercWildCardTooComplicated)
            exit_with_msg(erc, NLS_WILDCARD_TOO_COMPLICATED);
         else
            exit_with_msg(erc, NLS_WILDCARD_EXPANSION_ERROR);
      if (wildcard_buffer[0]) {
         if (!wildcard_encountered) {
            wildcard_encountered = TRUE;
            log_msg(NLS_EXPANDING_WILDCARDS, 1, NULL, 0);
         }
         while ((erc = WildCardNext(wildcard_buffer, &sd_filespec)) != ercEOF)
            if (erc != ercOK)
               exit_with_msg(erc, NLS_WILDCARD_EXPANSION_ERROR);
            else
               summary.count[TOTAL_FILES]++;
      } else
         summary.count[TOTAL_FILES]++;
   }
   if (wildcard_encountered)
      log_msg(NLS_DONE, 1, NULL, 0);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Loop through all the parameters of the '[File list from]' entry on the
 command form.  If the parameter is not wildcarded, just pass it to the
 procedure that follows in order that it may be written to the archive
 dataset.  Otherwise, expand all the possible wildcard matches and pass each
 one to the archiving procedure.  When done, send a sentinel to the archive IO
 process that causes the dataset to be closed. */

private void backup_files(void) {

   unsigned erc, i;
   iob_type *iob;
   char password[MAX_PASSWORD_LENGTH];
   sdType sd_filespec, sd_null = {NULL, 0}, sd_password = {password};
   char wildcard_buffer[800];

   vid_only_msg(NLS_CLEAR_VID1, 1, NULL, 0);	/* Clear for 'Processing...' */
   iob = allocate_iob(0);
   iob->ctrl = MODE;
   iob->mode.write = TRUE;
   iob->mode.sequential = archive.sequential;
   Send(archive.msg_exch, iob);
   for (i = 0; i < CSubparams(FILE_LIST_PARAM); i++) {
      if ((erc = RgParam(1, i, &sd_filespec)) != ercOK)
         exit_with_msg(erc, 0);
      ParseFileSpec(0, sd_filespec.pb, sd_filespec.cb, FALSE, NULL, NULL,
                    NULL, NULL, NULL, NULL, NULL, NULL, sd_password.pb,
                    &sd_password.cb, FALSE, 0);
      path.volume_may_have_changed = TRUE;
      if ((erc = WildCardInit(sd_filespec.pb, sd_filespec.cb, wildcard_buffer,
                              sizeof(wildcard_buffer))) != ercOK)
         if (erc == ercWCBufferTooSmall)
            exit_with_msg(erc, NLS_TOO_MANY_WILDCARDS);
         else if (erc == ercWildCardTooComplicated)
            exit_with_msg(erc, NLS_WILDCARD_TOO_COMPLICATED);
         else
            exit_with_msg(erc, NLS_WILDCARD_EXPANSION_ERROR);
      if (wildcard_buffer[0])
         while ((erc = WildCardNext(wildcard_buffer, &sd_filespec)) != ercEOF)
            if (erc != ercOK)
               exit_with_msg(erc, NLS_WILDCARD_EXPANSION_ERROR);
            else
               backup_file(&sd_filespec, &sd_password, TRUE);
      else
         backup_file(&sd_filespec, &sd_null, FALSE);
   }
   create_archive_EOD();

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 In order to archive a particular file, we have to be able to both open it in
 read only mode and obtain its file header block (so we may save password
 information, size, etc., etc. into the archive dataset).  
 
 If both these things can be accomplished, the file is potentially
 eligible for backup.
 
 The only problem is that to get password information from the fileHeader
 the file must be opened modemodify.  This is not wanted.  So it is 
 correct for a NUL password to be written to tape and it is the 
 responsibility of RestoreArchive to properly see that it is taken care of.
 
 Then check to see if either of the creation or modification dates is more
 recent than the (optional) incremental date provided by the user.  Note an
 interesting twist in 'fNoSave' semantics from the file header block: this
 field is ignored if the file has been explicitly requested by the user (i.e.
 no wildcarding).  Otherwise, 'fNoSave' TRUE can prevent the file from being
 archived.  Once final eligibility has been decided, just munch through
 reading all the data and passing it to the archiveIO process. */

private void backup_file(sdType *sd_filespec, sdType *sd_password,
                         Boolean wildcard) {

   unsigned erc, xfer_size, xfer_count;
   unsigned long file_length;				/* m13 */
   unsigned erc1;							/* m13 */
   flag finish_flag;						/* m14 */
   iob_type *iob;
   Boolean io_error = FALSE, other_error = FALSE;
   sdType nls_parms[] = {(void *) &summary.count[CURRENT_FILE],
                                  sizeof(summary.count[CURRENT_FILE]),
                         (void *) &summary.count[TOTAL_FILES],
                                  sizeof(summary.count[TOTAL_FILES]),
                         (void *) &target.filespec[1], 0,
                         (void *) &erc, sizeof(erc),
                         (void *) &target.lfa, sizeof(target.lfa)};

   summary.count[CURRENT_FILE]++;
   vid_only_msg(NLS_PROCESSING_FILE_N_OF, 1, nls_parms, sizeof(nls_parms));
   erc = OpenFile(&target.handle, sd_filespec->pb, sd_filespec->cb,
                  sd_password->pb, sd_password->cb, modeRead);
   if (     erc == ercOK
         && (erc = GetFileStatus(target.handle, fhb_code, &fhb,
                                 sizeof(fhb))) == ercBadRequestCode)
      erc = GetFileStatus(target.handle, (fhb_code == FILE_STATUS_FHB)
                          ? FILE_STATUS_FHB_RESERVED : FILE_STATUS_FHB,
                          &fhb, sizeof(fhb));
   erc1 = GetFileStatus(target.handle, FILE_STATUS_LENGTH,	/* m13 */
                       &file_length, sizeof(file_length));	/* m13 */
   if ( (erc != ercOK) || (erc1 != ercOK) ) {
      char node[MAX_NODE_LENGTH], volume[MAX_VOLUME_LENGTH],
           directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH];
      unsigned node_len, volume_len, directory_len, filename_len;

      summary.count[NOT_BACKED_UP]++;
      ParseFileSpec(0, sd_filespec->pb, sd_filespec->cb, TRUE, node, &node_len,
                    volume, &volume_len, directory, &directory_len, filename,
                    &filename_len, NULL, 0, TRUE, 0);
      memset(target.filespec, 0, sizeof(target.filespec));
      BuildFileSpec(0, nls_parms[2].pb, &nls_parms[2].cb, NULL,
                    last(target.filespec), (volume_len == 0), node, node_len,
                    volume, volume_len, directory, directory_len, filename,
                    filename_len, FALSE, NULL, 0, FALSE, 0);
      target.filespec[0] = nls_parms[2].cb;
      log_msg(NLS_CANT_ARCHIVE, 1, nls_parms, sizeof(nls_parms));
      return;
   }
   update_path(target.handle);
   memset(target.filespec, 0, sizeof(target.filespec));
   BuildFileSpec(0, nls_parms[2].pb, &nls_parms[2].cb, NULL,
                 last(target.filespec), FALSE, &path.node[1], path.node[0],
                 &path.volume[1], path.volume[0], &path.directory[1],
                 path.directory[0], &fhb.fileName[1], fhb.fileName[0],
                 FALSE, NULL, 0, FALSE, 0);
   if (     (   fhb.modificationDT < vlpb.backup_after
             && fhb.creationDT < vlpb.backup_after)
         || (wildcard && fhb.fNoSave)) {
      CloseFile(target.handle);
      return;
   }
   if (vlpb.confirm) {
      log_msg(NLS_CONFIRM_ARCHIVE, 1, nls_parms, sizeof(nls_parms));
      if (!proceed_sel_arch(&finish_flag)) {
         CloseFile(target.handle);
         if (finish_flag) {
            create_archive_EOD(); 
            exit_with_msg(ercOK, 0);
         }
         return;
      }
   }
   log_msg(NLS_BACKING_UP, 0xFF01, nls_parms, sizeof(nls_parms));
   create_archive_sentinel(file_length);
   target.lfa = 0;
/* target.size = (fhb.endOfFileLfa + (PAGE_SIZE - 1)) & ~(PAGE_SIZE - 1);m13 */
   target.size = file_length;
   while (target.size > 0) {
      iob = allocate_iob(SEGMENT_SIZE - PAGE_SIZE);
      iob->ctrl = DATA;
      iob->data = iob->base;
      iob->available = 0;
      if (target.size > iob->size)
         xfer_size = iob->size;
      else
         xfer_size = target.size;
      do {
         erc = Read(target.handle, iob->data, xfer_size, target.lfa,
                    &xfer_count);
         offset_of(iob->data) += xfer_count;
         iob->available += xfer_count;
         target.lfa += xfer_count;
         target.size -= xfer_count;
         xfer_size -= xfer_count;
         if (erc == ercIoError)
            do {
               erc = Read(target.handle, iob->data, PAGE_SIZE, target.lfa,
                          &xfer_count);
               if (erc == ercIoError || erc == ercOK) {
                  if (erc == ercIoError) {
                     memset(iob->data, 0, PAGE_SIZE);
                     log_msg(NLS_FILE_IO_ERROR, 1, nls_parms,
                             sizeof(nls_parms));
                     io_error = TRUE;
                  }
                  offset_of(iob->data) += PAGE_SIZE;
                  iob->available += PAGE_SIZE;
                  target.lfa += PAGE_SIZE;
                  target.size -= PAGE_SIZE;
                  xfer_size -= PAGE_SIZE;
               } else {
                  other_error = TRUE;
                  memset(iob->data, 0, xfer_size);
                  offset_of(iob->data) += xfer_size;
                  iob->available += xfer_size;
                  target.lfa += xfer_size;
                  target.size -= xfer_size;
                  xfer_size = 0;
               }
            } while (erc == ercIoError && xfer_size > 0);
         else if (erc != ercOK) {
            other_error = TRUE;
            memset(iob->data, 0, xfer_size);
            offset_of(iob->data) += xfer_size;
            iob->available += xfer_size;
            target.lfa += xfer_size;
            target.size -= xfer_size;
            xfer_size = 0;
         }
      } while (xfer_size > 0);
      Send(archive.msg_exch, iob);
   }
   CloseFile(target.handle);
   if (io_error) {
      summary.count[IO_ERRORS]++;
      log_msg(NLS_DONE, 1, NULL, 0);
   } else if (other_error) {
      summary.count[OTHER_ERRORS]++;
      log_msg(NLS_FILE_ERROR, 1, nls_parms, sizeof(nls_parms));
   } else {
      summary.count[BACKED_UP_OK]++;
      log_msg(NLS_DONE, 1, NULL, 0);
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 One or more of the path components (node, volume or directory) has potentially
 changed at this point.  Obtain new information that corresponds to the file
 handle supplied in the call and then compare it to prior information saved in
 the archive control block.  If necessary, fetch a new MFD.Sys from the changed
 {Node}[Volume]<Sys> before searching for the new directory information. */

void update_path(unsigned fh) {

   unsigned i;
   mfd_entry_type *mfd_entry;
   char node[MAX_NODE_LENGTH + 1] = 0;
   sdType *sd_local, *sd_mstr;

   if (path.volume_may_have_changed) {
      GetFileStatus(fh, FILE_STATUS_NODE, node, sizeof(node));
      sd_local = standard_msg(NLS_LOCAL);
      sd_mstr = standard_msg(NLS_MASTER);
      if (     (   node[0] == sd_local->cb
                && NlsULCmpB(NULL, &node[1], sd_local->pb,
                             node[0], &i) == ercOK
                && i == 0xFFFF)
            || (   node[0] == sd_mstr->cb
                && NlsULCmpB(NULL, &node[1], sd_mstr->pb, node[0], &i) == ercOK
                && i == 0xFFFF))
         node[0] = 0;		/* Don't clutter display if CT-Net unused */
      GetFileStatus(fh, FILE_STATUS_VHB, &vhb, sizeof(vhb));
      path.volume_encrypted =    (vhb.magicWd == MAGIC_PC)
                              || (   (vhb.magicWd & (MAGIC_VM | MAGIC_ENCRYPT))
                                  == (MAGIC_VM | MAGIC_ENCRYPT));
      if (     node[0] != path.node[0]
            || NlsULCmpB(NULL, &node[1], &path.node[1], node[0], &i) != ercOK
            || i != 0xFFFF
            || vhb.volName[0] != path.volume[0]
            || NlsULCmpB(NULL, &vhb.volName[1], &path.volume[1],
                         vhb.volName[0], &i) != ercOK
            || i != 0xFFFF) {
         memcpy(path.node, node, node[0] + 1);	/* Node/volume has changed */
         memcpy(path.volume, &vhb.volName[0], vhb.volName[0] + 1);
         fetch_mfd(standard_msg(NLS_MFD));	/* Get the correct MFD */
         path.directory[0] = 0;		/* Force search for directory below */
      }
      path.volume_may_have_changed = FALSE;	/* OK for the time being... */
   }
   if (     fhb.dirName[0] != path.directory[0]
         || NlsULCmpB(NULL, &fhb.dirName[1], &path.directory[1],
                      fhb.dirName[0], &i) != ercOK
         || i != 0xFFFF) {
      if ((mfd_entry = find_mfd_entry((char *) fhb.dirName)) == NULL) {
         memcpy(path.directory, fhb.dirName, fhb.dirName[0] + 1);
         path.directory_password[0] = 0;
         path.directory_protection = DEFAULT_DIRECTORY_PROTECTION;
         path.directory_pages = DEFAULT_DIRECTORY_PAGES;
      } else {
         memcpy(path.directory, mfd_entry->sbDirectory,
                mfd_entry->sbDirectory[0] + 1);
         memcpy(path.directory_password, mfd_entry->sbPassword,
                mfd_entry->sbPassword[0] + 1);
         path.directory_protection = mfd_entry->defaultAccessCode;
         path.directory_pages = mfd_entry->cPages;
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 To signal the beginning of a new file within the archive dataset, write a
 sentinel record.  This is also the place that all the pertinent information
 about the file that we may want to restore is kept. */

void create_archive_sentinel(unsigned long file_length) {

   unsigned checksum = 0, i;
   archive_sentinel_type *archive_sentinel;
   iob_type *iob;

   iob = allocate_iob(sizeof(archive_sentinel_type));
   iob->ctrl = BOF;
   iob->available = sizeof(archive_sentinel_type);
   archive_sentinel = iob->data;
   memset(archive_sentinel, 0, sizeof(*archive_sentinel));
   archive_sentinel->signature = ARCHIVE_SIGNATURE;
   archive_sentinel->record_type = BOF;
/* archive_sentinel->pages_data = (fhb.endOfFileLfa + PAGE_SIZE - 1)
                                   / PAGE_SIZE; 					m13*/
   archive_sentinel->pages_data = 
         ((file_length != 0)
         ? file_length
         : (fhb.endOfFileLfa + PAGE_SIZE - 1)
                                   ) / PAGE_SIZE;
   memcpy(archive_sentinel->node, path.node, path.node[0] + 1);
   memcpy(archive_sentinel->volume, path.volume, path.volume[0] + 1);
   memcpy(archive_sentinel->directory, path.directory, path.directory[0] + 1);
   memcpy(archive_sentinel->directory_password, path.directory_password,
          path.directory_password[0] + 1);
   archive_sentinel->directory_protection = path.directory_protection;
   archive_sentinel->directory_pages = path.directory_pages;
   memcpy(archive_sentinel->filename, fhb.fileName, fhb.fileName[0] + 1);
   memcpy(archive_sentinel->file_password, fhb.password, fhb.password[0] + 1);
   archive_sentinel->directory_type = fhb.ObjectType;
   archive_sentinel->protection = fhb.accessProtection;
   archive_sentinel->control.suppress_backup = fhb.fNoSave;
   archive_sentinel->control.hidden_file = fhb.fNoDirPrint;
   archive_sentinel->control.prevent_delete = fhb.fNoDelete;
   archive_sentinel->control.read_only = fhb.fReadOnly;
   archive_sentinel->ms_dos_magic = fhb.bDosMagic;
   archive_sentinel->ms_dos_attribute = fhb.bDosAttribute;
   archive_sentinel->object_type = fhb.userObjectType;
   archive_sentinel->creation_date = fhb.creationDT;
   archive_sentinel->modification_date = fhb.modificationDT;
   archive_sentinel->access_date = fhb.accessDT;
   archive_sentinel->expiration_date = fhb.expirationDT;
   archive_sentinel->eof_lfa = fhb.endOfFileLfa;
   memcpy(archive_sentinel->distrix_info, fhb.rgbDistrix,
          sizeof(fhb.rgbDistrix));
   memcpy(archive_sentinel->application_info, fhb.application,
          sizeof(fhb.application));
   archive_sentinel->dataset_creation_time = archive.creation_time;
   for (i = 0; i < sizeof(archive_sentinel_type) / 2; i++)
      checksum += ((unsigned *) archive_sentinel)[i];
   archive_sentinel->checksum = -(checksum + ARCHIVE_SIGNATURE);
   Send(archive.msg_exch, iob);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 A final wrap up to all the data written to the archive dataset.  Necessary so
 Restore Archive can tell the difference between the end of one tape in a
 multiple volume set and the end of the dataset overall. */

void create_archive_EOD(void) {

   unsigned checksum = 0, i;
   archive_sentinel_type *archive_sentinel;
   iob_type *iob;

   iob = allocate_iob(sizeof(archive_sentinel_type));
   iob->ctrl = EOD;
   iob->mode.final_close = TRUE;
   iob->available = sizeof(archive_sentinel_type);
   archive_sentinel = iob->data;
   memset(archive_sentinel, 0, sizeof(*archive_sentinel));
   archive_sentinel->signature = ARCHIVE_SIGNATURE;
   archive_sentinel->record_type = EOD;
   for (i = 0; i < sizeof(archive_sentinel_type) / 2; i++)
      checksum += ((unsigned *) archive_sentinel)[i];
   archive_sentinel->checksum = -(checksum + ARCHIVE_SIGNATURE);
   Send(archive.msg_exch, iob);
   do
      Wait(target.msg_exch, &iob);
   while (iob->ctrl != SYNC);
   deallocate_iob(iob);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 As its name implies... */

private void summarize_backup(void) {

   sdType nls_parms[] = {(void *) &summary.count[BACKED_UP_OK],
                                  sizeof(summary.count[BACKED_UP_OK]),
                         (void *) &summary.count[IO_ERRORS],
                                  sizeof(summary.count[IO_ERRORS]),
                         (void *) &summary.count[OTHER_ERRORS],
                                  sizeof(summary.count[OTHER_ERRORS]),
                         (void *) &summary.count[NOT_BACKED_UP],
                                  sizeof(summary.count[NOT_BACKED_UP])};

   log_msg(NLS_FILES_ARCHIVED, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[IO_ERRORS] > 0)
      log_msg(NLS_FILES_IO_ERRORS, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[OTHER_ERRORS] > 0)
      log_msg(NLS_FILES_OTHER_ERRORS, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[NOT_BACKED_UP] > 0)
      log_msg(NLS_FILES_NOT_ARCHIVED, 1, nls_parms, sizeof(nls_parms));

}