/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  restore.c                                                                */
/*                                                                           */
/*  File restoration from CTOS archive dataset.                              */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  06/24/92  122J.29  D. Gilson     /  Added file_exists flag to properly   */
/*                                      handle creation and Mod Dates like   */
/*                                      Copy and LCopy.                      */
/*  06/11/92  122I.28  D. Gilson     /  Added lock_video and unlock_video    */
/*                                      to fix a problem where the both      */
/*                                      processes wanted to do a proceed at  */
/*                                      the same time.  This only happens w/ */
/*                                      confirmation set when Open_target_fil*/
/*                                      wanted to ask the user if they want  */
/*                                      that file and Archive_io is doing a  */
/*                                      read ahead and wants to mount the    */
/*                                      next archive volume.                 */
/*  05/14/92  122H.27  D. Gilson     /  P 15535504 from and to spec not      */
/*                                      getting restored correctly.          */
/*  05/13/92  122H.26  D. Gilson     /  passwords are getting changed to NUL */
/*  04/24/92  122H.25  D. Gilson     /  archive names under 12 chars need    */
/*                                   /  a dir spec but those over 12 don't.  */
/*                                      Make all need dir spec. P15513331    */
/*  04/16/92  122G.24  D. Gilson     /  Dont Allow FileHeaders.sys,  Log.sys */
/*                                   /  Mfd.sys,  ect to be overwritten.     */
/*  02/14/92  122F.23  W. Chiu       /  Don't check OS version to determine  */
/*                                   /  if Remove on close is supported      */
/*                                   /  because file may not be local.       */
/*  12/21/91  130E.22  D. Gilson     /  Added multi archive volume support   */
/*  08/22/91  130B.21  W. Chiu       /  Added Remove On Close when file      */
/*                                   /  opened for read by someone.          */
/*  07/15/91  130A.20  P. Johansson  /  [Overwrite ok?] cannot be left blank */
/*                                      if user interaction is suppressed.   */
/*  06/21/91  130A.19  P. Johansson  /  If the password from the archive     */
/*                                      dataset is encrypted, copy it into   */
/*                                      FHB only if the destination volume   */
/*                                      is encrypted also.                   */
/*  06/06/91  121J.18  P. Johansson  /  If the password from the archive     */
/*                                      dataset is not null, copy it into    */
/*                                      FHB unless a password was explicitly */
/*                                      supplied in '[File list to]'.        */
/*  05/31/91  121J.17  C. Naik       /  Use explicit or UCB password to gain */
/*                                      access to a file, not the password   */
/*                                      from the archive dataset.            */
/*  05/28/91  121J.16  P. Johansson  /  Terminate position string with zero  */
/*                                      so that atoi() works correctly.      */
/*  05/23/91  121J.15  P. Johansson  /  Clear 'canonical_filespec' before    */
/*                                      BuildFileSpec; fix multiple default  */
/*                                      file prefix bug.                     */
/*  05/14/91  121J.14  C. Naik       /  Added case "II" for installation     */
/*                                      manager. Exit on error for this case.*/
/*                                      Else this case functions as "RA".    */
/*  05/08/91  121J.13  C.G. Naik     /  Dynamically determine correct FHB    */
/*                                      code for Get/SetFileStatus; close    */
/*                                      message file (if necessary) so that  */
/*                                      it may be overwritten.               */
/*  04/15/91  121J.12  C.G. Naik     /  Permit the restoration of files      */
/*                                      from local and remote nodes present  */
/*                                      in same archive dataset. Optimization*/
/*                                      code in expand_from_spec modified to */
/*                                      handle this.                         */
/*  04/09/91  121J.11  P. Johansson  /  Show total files whenever possible.  */
/*  03/25/91  121H.10  P. Johansson  /  Permit the creation of (potentially) */
/*                                      empty directories if restoring from  */
/*                                      an archive dataset created by Volume */
/*                                      Archive.                             */
/*  03/11/91  121H.09  P. Johansson  /  Use hard-coded defaults to create a  */
/*                                      directory if no MFD data available.  */
/*  03/06/91  121G.08  P. Johansson  /  12.0 half-inch tape labels don't     */
/*                                      contain total files in dataset.      */
/*  01/30/91  121F.07  P. Johansson  /  Explicit OpenByteStream for [Vid] so */
/*                                      that video filtering works.          */
/*  01/28/91  121F.06  P. Johansson  /  FHB 'creationDT' set to current date.*/
/*  01/22/91  121F.05  P. Johansson  /  Dataset [XXX]+ not valid for restore.*/
/*  01/14/91  121F.04  P. Johansson  /  Support invocation from "Restore"    */
/*                                      and "Tape Restore" command forms.    */
/*  01/03/90  121F.03  P. Johansson  /  When ChangeFileLength gets an error, */
/*                                      substitute 'ercOK' if the file is    */
/*                                      already large enough.                */
/*  12/27/90  121F.02  P. Johansson  /  FHB 'modificationDT' enhancement     */
/*  12/20/90  121F.01  P. Johansson  /  When a directory doesn't exist,      */
/*                                      create it.                           */
/*  12/13/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* Suppress C run-time (only CTOS functionality needed) */

pragma Off(List);
#include <stub.h>
pragma Pop(List);

/* There are no procedures in the Archive utilities that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocExch
#define BuildFileSpec
#define ChangeFileLength
#define CloseFile
#define CloseMsgFile
#define CreateDir
#define CreateFile
#define CSubparams
#define CurrentOsVersion
#define GetDateTime
#define GetFileStatus
#define GetVHB
#define NlsULCmpB
#define OpenFile
#define ParseFileSpec
#define Read
#define RgParam
#define RgParamSetSimple
#define Send
#define SetFileStatus
#define ULCmpB
#define WildCardMatch
#define Wait
#define Write

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType
#define sdType
#define UCBType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern iob_type *allocate_iob(unsigned buffer_size);
extern void build_restore_spec(sdType *sd_archive_spec, sdType *sd_from,
                               sdType *sd_to, sdType *sd_restore);
extern void create_archive_io_process(void);
extern void deallocate_iob(iob_type *iob);
extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern void get_archive_datasets(unsigned param_num); 		/* m22 */
extern void initialize_msgs(void);
extern void lock_video(unsigned signature);
extern void log_msg(unsigned nls_msg_index, unsigned signature,
                    sdType nls_parms[], unsigned nls_parms_len);
extern unsigned open_log_file(void);
extern void open_video(void);
extern unsigned parse_parameters(param_descr_type *param_descr, unsigned n);
extern Boolean proceed(void);
extern sdType *standard_msg(unsigned msg_num);
extern void unlock_video(unsigned signature);
extern void vid_only_msg(unsigned nls_msg_index, unsigned signature,
                         sdType nls_parms[], unsigned nls_parms_len);
extern Boolean wildcards_mappable(sdType *sd_from, sdType *sd_to);


/* M21 */
extern ErcType HandleRmvOnClose(Pointer pFhOrBswa, Pointer pbFileSpec, 
                         Word cbFileSpec, Pointer pbPassword, 			
						 Word cbPassword, Word mode, FlagType fByBstrm,
                         Pointer pBufArea, Word sBufferArea,
						 FlagType fPromptUser,
						 ErcType (*PrmptUserFnc)(pointer, word, pointer), 
                         Pointer prgSd, Word srgSd,
                         Pointer pAnswer); 

/* Error return codes used by this module */

#define FsErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern archive_type archive;
extern char msg_file_spec[];
extern journal_type journal;
extern char sbVerRun[];

/* Global variables exported by this manuscript */

filesys_type filesys;
unsigned internal_os_version;
summary_type summary;
target_type target;
vlpb_type vlpb;

/* Static variables global within this manuscript */

private FhbType fhb;
private Boolean fInstall_Mngr_Case;
private unsigned fhb_code;
private sdType *sd_overwriting;


/* Function prototypes defined before the functions themselves are declared */

void close_target_file(Flag file_exists);
void expand_from_spec(unsigned subparam, sdType *sd_from, Boolean expand_node);
void expand_to_spec(unsigned subparam, sdType *sd_to, sdType *sd_password,
                    Boolean *volume_encrypted);
void open_target_file(archive_sentinel_type *archive_sentinel
                     ,Flag *file_exists);
void restore_file_data(void *data, unsigned length);
void restore_files(void);
void summarize_restore(void);
void validate_file_lists(void);
void validate_parameters(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Short and sweet, just munch through all the necessary functions for file
 restoration from an archive dataset (either direct or sequential access). */

void main(void) {

   unsigned erc, os_version, os_revision;

   open_video();		/* Before ANY [Vid] use! (so filter works) */
   initialize_msgs();		/* Use a message file if one exists */
   if ((erc = CurrentOsVersion(&os_version, &os_revision)) != ercOK)
      exit_with_msg(erc, 0);
   internal_os_version = (os_version << 8) | os_revision;

     
   if ((erc = AllocExch(&target.msg_exch)) != ercOK)
      exit_with_msg(erc, 0);
   sd_overwriting = standard_msg(NLS_OVERWRITING);
   validate_parameters();	/* Check the syntax of the Executive form */
   validate_file_lists();	/* Make sure 'From' and 'To' are compatable */
   create_archive_io_process();	/* Runs in parallel for efficiency */
   restore_files();		/* Then go ahead an process them */
   summarize_restore();		/* Finally, report our findings... */
   exit_with_msg(ercOK, 0);	/* ...and leave gracefully */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The parameters for the executive command form for Restore Archive' are
 described in the table below.  As much as possible, they are parsed by the
 generic routine 'parse_parameters' and are transferred into appropriate
 fields in the "variable-length parameter block" (vlpb).  If the generic
 parsing is successful, other tests that involve dependencies between the
 parameters are applied. */

private void validate_parameters(void) {

   char device[MAX_DEVICE_LENGTH], directory[MAX_DIRECTORY_LENGTH],
      filename[MAX_FILENAME_LENGTH], node[MAX_NODE_LENGTH],
      position[MAX_FILENAME_LENGTH], volume[MAX_VOLUME_LENGTH];
   unsigned device_len, directory_len, erc, filename_len, node_len,
      password_len, position_len, spec_len, volume_len;
   unsigned i;									/* m22 */
   Boolean floppy_default;
   sdType nls_parms[] = {         NULL, 0,
                         (void *) &sbVerRun[1], sbVerRun[0]};
   static param_descr_type exec_param[] = {
      NULL, 0, 0, 0,		/* Dummy corresponds to name of command */
      NULL, 0, 0, 0,											/*  m22 */
/*    &archive.spec[archive.device_num], ascii_param,
       sizeof(archive.spec[archive.device_num][0]), NLS_ARCHIVE_DATASET, m22 */
      &vlpb.from_specs, files_param, sizeof(vlpb.from_specs),
       NLS_FILE_LIST_FROM,
      &vlpb.to_specs, files_param, sizeof(vlpb.to_specs), NLS_FILE_LIST_TO,
      &vlpb.overwrite_OK, fuzzy_Boolean_param, sizeof(vlpb.overwrite_OK),
       NLS_OVERWRITE_OK,
      &vlpb.confirm, Boolean_param, sizeof(vlpb.confirm), NLS_CONFIRM,
      &archive.sequence, decimal_param, sizeof(archive.sequence),
       NLS_ARCHIVE_SEQUENCE,
      &vlpb.merge, Boolean_param, sizeof(vlpb.merge),
       NLS_MERGE_WITH_EXISTING_FILE,
      &vlpb.list_only, Boolean_param, sizeof(vlpb.list_only),
       NLS_LIST_FILES_ONLY,
      &journal.filespec, ascii_param, sizeof(journal.filespec), NLS_LOG_FILE,
      &vlpb.noninteractive, Boolean_param, sizeof(vlpb.noninteractive),
       NLS_NONINTERACTIVE};
   sdType *sd_archive, sd_case;

   RgParam(0, 0, &nls_parms[0]);
   if (nls_parms[0].pb != NULL && nls_parms[0].cb != 0)
      log_msg(NLS_COMMAND_VERSION, 1, nls_parms, sizeof(nls_parms));
   RgParam(0, 1, &sd_case);
   archive.sequence = 1;	/* Default if not specified by user */
   floppy_default = (sd_case.cb == 2 && ULCmpB(sd_case.pb, "RD", 2) == 0xFFFF);
   fInstall_Mngr_Case = (   sd_case.cb == 2			/* M14 */
                        && ULCmpB(sd_case.pb, "II", 2) == 0xFFFF);
   vlpb.overwrite_OK = 2;	/* Fuzzy Boolean (see NlsYesNoOrBlank) */
/* if (CSubparams(RESTORE_ARCHIVE_DATASET_PARAM) > 1)
      exit_with_msg(ercSyntax, NLS_TOO_MANY_DATASETS);			m22 */
   get_archive_datasets(RESTORE_ARCHIVE_DATASET_PARAM);		/*  m22 */
   if ((erc = parse_parameters(exec_param, last(exec_param) + 1)) != ercOK)
      exit_with_msg(erc, 0);
   if (archive.spec[archive.device_num][0] == 0) {
      sd_archive = standard_msg((floppy_default) ? NLS_DEFAULT_FLOPPY
                                                 : NLS_DEFAULT_DATASET);
      memcpy(&archive.spec[archive.device_num][1], sd_archive->pb,
             archive.spec[archive.device_num][0] =
             _min(sd_archive->cb, last(archive.spec[archive.device_num])));
   }
   erc = ParseFileSpec(0, &archive.spec[archive.device_num][1],
                       archive.spec[archive.device_num][0], FALSE, node,
                       &node_len, device, &device_len, NULL, NULL, position,
                       &position_len, &archive.password[archive.device_num][1],
                       &password_len,
                       FALSE, 3);
   archive.password[archive.device_num][0] = password_len;
   if (erc == ercOK) {
      archive.sequential = TRUE;
      BuildFileSpec(0, &archive.spec[archive.device_num][1],
                    &spec_len, NULL, last(archive.spec[archive.device_num]),
                    FALSE, node, node_len, device, device_len, NULL, 0, NULL,
                    0, FALSE, NULL, 0, FALSE, 0);
      archive.spec[archive.device_num][0] = spec_len;
      archive.current_position = 0xFFFF;	/* Unknown on the first tape */
      if (position_len == 1 && position[0] == '+')
         exit_with_msg(ercInvalidTapeSpecification, NLS_APPEND_INVALID);
      if (position_len != 0) {
         position[position_len] = 0;
         archive.target_position = atoi(position);
      }		/* m22... */
      for (i = 1; i < archive.device_num_last; i++) {
         erc = ParseFileSpec(0, &archive.spec[i][1],
                             archive.spec[i][0],
                             FALSE, node,
                             &node_len, device, &device_len, NULL, 0, position,
                             &position_len,
                             &archive.password[i][1],
                             &password_len,
                             FALSE, 3);
         archive.password[i][0] = password_len;
         if (erc == ercOK) {
            BuildFileSpec(0, &archive.spec[i][1], &spec_len, NULL,
                          last(archive.spec[i]), FALSE,
                          node, node_len, device,
                          device_len, NULL, 0, NULL, 0, FALSE, NULL, 0,
                          FALSE, 0);
            archive.spec[i][0] = spec_len;
         }
      }     	 											  /* ...m22 */
   } else {
      if (CSubparams(VOLUME_ARCHIVE_DATASET_PARAM) > 1)			/*  m22 */
         exit_with_msg(ercSyntax, NLS_TOO_MANY_DATASETS); 		/*	m22 */
      exec_param[1].type = path_param;	/* Prepare for second parsing... */
      vlpb.overwrite_OK = 2;	/* Fuzzy Boolean (see NlsYesNoOrBlank) */
      if ((erc = parse_parameters(exec_param, last(exec_param) + 1)) != ercOK)
         exit_with_msg(erc, 0);
      ParseFileSpec(0, &archive.spec[archive.device_num][1],
                    archive.spec[archive.device_num][0], FALSE, node,
                    &node_len, volume, &volume_len, directory, &directory_len,
                    filename, &filename_len,
                    &archive.password[archive.device_num][1],
                    &password_len, FALSE, 0);
      archive.password[archive.device_num][0] = password_len;
      BuildFileSpec(0, &archive.spec[archive.device_num][1],
                    &spec_len, NULL, last(archive.spec[archive.device_num]),
                    (volume_len == 0), node, node_len, volume, volume_len,
                    directory, directory_len, filename, filename_len, FALSE,
                    NULL, 0, FALSE, 0);
      if (directory_len == 0 )									/* m25 */
         exit_with_msg(ercSyntax, NLS_DIRECTORY_ERROR);			/* m25 */
      archive.spec[archive.device_num][0] = spec_len;
      sd_archive = standard_msg(NLS_ARCHIVE_SUFFIX);
      memcpy(
      &archive.spec[archive.device_num][archive.spec[archive.device_num][0]+1]
      ,sd_archive->pb,sd_archive->cb);
      archive.spec[archive.device_num][0] += sd_archive->cb;
   }
   switch (vlpb.overwrite_OK) {		/* Properly decode "fuzzy" parameter */
      case 0:
         vlpb.overwrite_OK = FALSE;
         vlpb.no_overwrite = TRUE;
         break;

      case 1:
         vlpb.overwrite_OK = TRUE;
         vlpb.no_overwrite = FALSE;
         break;

      case 2:
         if (vlpb.noninteractive)
            exit_with_msg(ercSyntax, NLS_OVERWRITE_EXPLICIT);
         vlpb.overwrite_OK = FALSE;
         vlpb.no_overwrite = FALSE;
         break;
   }
   if (vlpb.confirm && vlpb.noninteractive)
      exit_with_msg(ercSyntax, NLS_CONFIRM_INCONSISTENT);
   if ((erc = open_log_file()) != ercOK)
      exit_with_msg(erc, NLS_CANT_OPEN_LOG_FILE);
   archive.first_volume = TRUE;	/* One-time only flag for messages, etc. */
   if (archive.sequence == 0 || archive.sequence > 999)
      exit_with_msg(ercSyntax, NLS_SEQUENCE_INVALID);
   archive.sequence--;		/* Adjust for pre-increment on first open */
   fhb_code = (internal_os_version >= 0x0B00) ? FILE_STATUS_FHB
                                              : FILE_STATUS_FHB_RESERVED;
   filesys.memory_for_structures = 0;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 In order for files to be retrieved from the archive medium and successfully
 created as (possibly) new filenames on the destination medium, two basic
 conditions must be met.  First, all subparameters in 'File list from' must be
 of the form <Directory>Filename with the optional forms
 [Volume]<Directory>Filename and {Node}[Volume]<Directory>Filename (this
 requirement is present for historical reasons and is taken from Restore
 12.0).  Secondly, the set of files specified in the 'File list from'
 parameter must be capable of being mapped to the specifications in 'File list
 to'.  The rules used to determine this eligibility are summarized below:

    *	'File list to' is NULL.  Any valid wildcarding in 'File list from' is
	permissible and no additional verifications are made here.

    *	The number of subparameters in 'File list from' is equal to 'File list
	to'.  In this case, each subparameter from 'File list from' must be
	mappable to its corresponding subparameter in 'File list to'.

    *	There is exactly one subparameter in 'File list to', in which case all
	'File list from' subparameters must be mappable to this subparameter
	(when 'File list from' is NULL, it is treated as the single
	 subparameter '{*}[*]<*>*').

 All other cases are considered syntax errors (mismatched file
 specifications). */

private void validate_file_lists(void) {

   char directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH];
   unsigned directory_len, erc, filename_len, i;
   sdType sd_from, sd_to;

   if (vlpb.from_specs == 0) {
      vlpb.from_specs = 1;
      if (    (erc = RgParamSetSimple(FILE_LIST_FROM_PARAM,
                                      standard_msg(NLS_DEFAULT_WILDCARD)))
            != ercOK)
         exit_with_msg(erc, 0);
   } else
      for (i = 0; i < vlpb.from_specs; i++) {
         RgParam(FILE_LIST_FROM_PARAM, i, &sd_from);
         ParseFileSpec(0, sd_from.pb, sd_from.cb, FALSE, NULL, NULL, NULL,
                       NULL, &directory, &directory_len, &filename,
                       &filename_len, NULL, NULL, FALSE, 0);
         if (directory_len == 0 || filename_len == 0) {
            sdType nls_parms[] = {sd_from.pb, sd_from.cb};

            log_msg(NLS_FILE_LIST_FROM_FORMAT, 1, nls_parms,
                    sizeof(nls_parms));
            exit_with_msg(ercSyntax, 0);
         }
      }
   if (!vlpb.list_only) {
      if (vlpb.to_specs == 0) {
         vlpb.to_specs = 1;
         if (    (erc = RgParamSetSimple(FILE_LIST_TO_PARAM,
                                         standard_msg(NLS_DEFAULT_WILDCARD)))
               != ercOK)
            exit_with_msg(erc, 0);
      }
      if (vlpb.from_specs == vlpb.to_specs || vlpb.to_specs == 1)
         for (i = 0; i < vlpb.from_specs; i++) {
            RgParam(FILE_LIST_FROM_PARAM, i, &sd_from);
            if (i < vlpb.to_specs)
               RgParam(FILE_LIST_TO_PARAM, i, &sd_to);
            if (!wildcards_mappable(&sd_from, &sd_to)) {
               sdType nls_parms[] = {sd_from.pb, sd_from.cb,
                                     sd_to.pb, sd_to.cb};

               log_msg(NLS_WILDCARD_MISMATCH, 1, nls_parms, sizeof(nls_parms));
               exit_with_msg(ercSyntax, 0);
            }
         }
      else if (vlpb.to_specs != 1) {
         log_msg(NLS_FILE_LISTS_DISAGREE, 1, NULL, 0);
         exit_with_msg(ercSyntax, 0);
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 File restoration is driven by the arrival of data block(s) from the archive
 dataset I/O co-process.  Each block is accompanied by a sentinel, which
 indicates one of a) the beginning of a new file, b) data belonging to a file
 previously identified by a BOF token, c) the end of a file (optional---note
 that the beginning of file token may also delimit the end of the preceding
 file) or d) the end of the archive dataset.  After the accompanying
 information has been processed, return the empty I/O control token to the
 archive I/O co-process. */

private void restore_files(void) {

   char ctrl;
   unsigned erc;
   iob_type *iob;
   Flag file_exists = TRUE;

   vid_only_msg(NLS_CLEAR_VID1, 1, NULL, 0);	/* Clear for 'Processing...' */
   iob = allocate_iob(0);
   iob->ctrl = MODE;
   Send(archive.msg_exch, iob);
   do {
      if ((erc = Wait(target.msg_exch, &iob)) != ercOK)
         exit_with_msg(erc, 0);
      switch (ctrl = iob->ctrl) {
         case BOF:
            if (target.handle != 0)
               close_target_file(file_exists);
            open_target_file((archive_sentinel_type *) iob->base
                            ,&file_exists);
            break;

         case DATA:
            if (target.handle != 0)
               restore_file_data(iob->data, iob->available);
            break;

         case EOF:
         case EOD:
            if (target.handle != 0)
               close_target_file(file_exists);
            break;
      }
      deallocate_iob(iob);
   } while (ctrl != EOD);
   do
      Wait(target.msg_exch, &iob);
   while (iob->ctrl != SYNC);
   deallocate_iob(iob);

}

/* M21 This function will be passed to HandleRmvOnClose so that the
 * use can be prompted to replace an open file. HandleRmvOnClose will 
 * only prompt the user if the file can be replaced. HandleRmvOnClose
 * expects this function to return an ercOK if the user wants to replace. The 
 * panswer should be used to save the value which will indicate the answer
 * from the user.
 */ 
private ErcType prompt_replace (Pointer prgsd, Word srgsd, Pointer panswer) {

   ErcType erc;
   Boolean *ptmp;

   log_msg(NLS_REPLACE,1, prgsd, srgsd);
   ptmp = (Boolean *)panswer;
   *ptmp = proceed();

   if(*ptmp)
      erc = ercOK;
   else
      erc = ercOperatorIntervention;
   return(erc);
}
   

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure first performs the complicated matching of the descriptive
 file information obtained from the archive dataset (the original {Node},
 [Volume], <Directory> and filename that is the source of the data) to the
 'File list from' wildcard(s) to see if the file should be restored or
 listed.  If there is a legal match, this information is merged with the
 'File list to' wildcard(s) to produce a target file specification.
 Proceeding on the optimizing assumption that most file restorations are
 overwrites, an attempt is made to open the file in 'modeModify'.  Note that
 if this fails because the directory does not exist, an attempt is made to
 create the directory with the same parameters as the directory from which the
 file was archived.  If the OpenFile succeeds, the file exists---so check
 internal flag settings as to whether or not it is permissible to overwrite
 the file (with or without the user's further permission).  If the file
 doesn't exist, create it.  Finally, get a copy of the File Header block and
 transfer the sentinel information (this is used later when the file is
 closed). */

private void open_target_file(archive_sentinel_type *archive_sentinel
                             ,Flag *file_exists) {

   char archive_spec[MAX_FILE_SPECIFICATION_LENGTH],
      canonical_filespec[MAX_FILE_SPECIFICATION_LENGTH],
      directory[MAX_DIRECTORY_LENGTH],
      directory_spec[MAX_FILE_SPECIFICATION_LENGTH],
      filename[MAX_FILENAME_LENGTH], node[MAX_NODE_LENGTH],
      volume[MAX_VOLUME_LENGTH],
      test_filespec[MAX_FILE_SPECIFICATION_LENGTH],
      test_directory_spec[MAX_FILE_SPECIFICATION_LENGTH];
   Boolean destination_encrypted, directory_descriptor, explicit_password;
   unsigned canonical_filespec_len, directory_len, erc, filename_len, i, j,
      node_len, volume_len, test_directory_len, test_file_len;
   unsigned long file_length;

   Boolean answer;  /* m21 */
   Boolean fOverwrite = FALSE; /* m26 */

   sdType nls_parms[] = {(void *) &summary.count[PROCESSED],
                                  sizeof(summary.count[PROCESSED]),
                         (void *) &summary.count[TOTAL_FILES],
                                  sizeof(summary.count[TOTAL_FILES])};
   sdType sd_archive_spec = {archive_spec}, sd_from,
      sd_password, sd_restore = {&target.filespec[1]}, sd_to;

   *file_exists = TRUE;

   if (archive_sentinel->filename[0] == 0) {
      directory_descriptor = TRUE;
      archive_sentinel->filename[0] = 1;
      archive_sentinel->filename[1] = 0xFF;
   } else {
      directory_descriptor = FALSE;
      target.handle = target.lfa = 0;	/* 'target.size' set by coprocess! */
      summary.count[PROCESSED]++;
      vid_only_msg((summary.count[TOTAL_FILES] == 0)
                   ? NLS_PROCESSING_FILE_N : NLS_PROCESSING_FILE_N_OF, 1,
                   nls_parms, sizeof(nls_parms));
   }
   memset(archive_spec, 0, sizeof(archive_spec));
   BuildFileSpec(0, sd_archive_spec.pb, &sd_archive_spec.cb, NULL,
                 sizeof(archive_spec), FALSE, &archive_sentinel->node[1],
                 archive_sentinel->node[0], &archive_sentinel->volume[1],
                 archive_sentinel->volume[0], &archive_sentinel->directory[1],
                 archive_sentinel->directory[0],
                 &archive_sentinel->filename[1],
                 archive_sentinel->filename[0], FALSE, NULL, 0, FALSE, 0);
   for (i = 0; i < vlpb.from_specs; i++) {
      expand_from_spec(i, &sd_from, archive_sentinel->node[0] > 0);/* m27... */
      memset(test_filespec, 0, sizeof(test_filespec));
      memset(test_directory_spec, 0, sizeof(test_directory_spec));
      ParseFileSpec(0, sd_from.pb, sd_from.cb, FALSE, NULL,
                    NULL, NULL, NULL,
                    test_directory_spec, &test_directory_len,
                    test_filespec, &test_file_len,
                    NULL, NULL, FALSE, 0);   /* ...m27 */
      if    (WildCardMatch(sd_from.pb, sd_from.cb, sd_archive_spec.pb,
                           sd_archive_spec.cb)
         &&  WildCardMatch(test_filespec, test_file_len,
                           &archive_sentinel->filename[1],
                           archive_sentinel->filename[0])
         &&  WildCardMatch(test_directory_spec, test_directory_len,
                           &archive_sentinel->directory[1],
                           archive_sentinel->directory[0])) {
         sdType nls_parms[] = {(void *) sd_archive_spec.pb, sd_archive_spec.cb,
                               (void *) sd_restore.pb, 0,
                               (void *) &erc, sizeof(erc),
                                        NULL, 0};

         if (vlpb.list_only) {
            if (!directory_descriptor)
               log_msg(NLS_LIST_ONLY, 1, nls_parms, sizeof(nls_parms));
            return;
         }
         expand_to_spec(i, &sd_to, &sd_password, &destination_encrypted);
         explicit_password = (sd_password.cb != 0);
         memcpy(&target.password[1], sd_password.pb,
                target.password[0] = sd_password.cb);
         sd_restore.cb = last(target.filespec);
         build_restore_spec(&sd_archive_spec, &sd_from, &sd_to,
                            &sd_restore);
         target.filespec[0] = nls_parms[1].cb = sd_restore.cb;
         if (vlpb.confirm && !directory_descriptor) {
            lock_video(2);
            log_msg(NLS_CONFIRM_RESTORE, 2, nls_parms, sizeof(nls_parms));
            if (!proceed()){
               unlock_video(2);
               return;
            }
            unlock_video(2);
         }
         while ((erc = OpenFile(&target.handle, sd_restore.pb,
                                sd_restore.cb, &target.password[1],
                                target.password[0],
                                modeModify)) == ercNoSuchDir) {
            ParseFileSpec(0, sd_restore.pb, sd_restore.cb, FALSE, node,
                          &node_len, volume, &volume_len, directory,
                          &directory_len, NULL, NULL, NULL, NULL, FALSE, 0);
            memset(directory_spec, 0, sizeof(directory_spec));
            BuildFileSpec(0, nls_parms[3].pb = directory_spec,
                          &nls_parms[3].cb, NULL, sizeof(directory_spec),
                          FALSE, node, node_len, volume, volume_len,
                          directory, directory_len, NULL, 0, FALSE, NULL, 0,
                          FALSE, 0);
            log_msg(NLS_CREATING_DIRECTORY, 0xFF01, nls_parms,
                    sizeof(nls_parms));
            if (     (erc = (archive_sentinel->directory_pages == 0)
                         ? CreateDir(nls_parms[3].pb, nls_parms[3].cb, NULL, 0,
                                     NULL, 0, DEFAULT_DIRECTORY_PAGES,
                                     DEFAULT_DIRECTORY_PROTECTION)
                         : CreateDir(nls_parms[3].pb, nls_parms[3].cb, NULL, 0,
                                     &archive_sentinel->directory_password[1],
                                     archive_sentinel->directory_password[0],
                                     archive_sentinel->directory_pages,
                                     archive_sentinel->directory_protection))
                  != ercOK) {
               if (fInstall_Mngr_Case)
                  exit_with_msg(erc, 0);			/* M14 */
               log_msg(NLS_DIRECTORY_CREATION_ERROR, 1, nls_parms,
                       sizeof(nls_parms));
               break;
            } else
               log_msg(NLS_DONE, 1, NULL, 0);
            if (directory_descriptor)
               return;
         }
         if (directory_descriptor)	/* Only needed to create directory? */
            return;
         if (     erc == ercAccessDenied
               && target.password[0] == 0
               && archive_sentinel->directory_password[0] != 0) {
                                   /* No password provided so get 
                                      one from the directory sentinel */
            memcpy(target.password, archive_sentinel->directory_password,
                   archive_sentinel->directory_password[0] + 1);
            erc = OpenFile(&target.handle, sd_restore.pb, sd_restore.cb,
                           &target.password[1], target.password[0],
                           modeModify);
         } 
         if (erc == ercFileInUse) {
            ParseFileSpec(0, sd_restore.pb, sd_restore.cb, TRUE, node,
                          &node_len, volume, &volume_len, directory,
                          &directory_len, filename, &filename_len, NULL, NULL,
                          TRUE, 0);
            memset(canonical_filespec, 0, sizeof(canonical_filespec));
            BuildFileSpec(0, canonical_filespec, &canonical_filespec_len,
                          NULL, sizeof(canonical_filespec), (volume_len == 0),
                          node, node_len, volume, volume_len, directory,
                          directory_len, filename, filename_len, FALSE, NULL,
                          0, FALSE, 0);
            if (     canonical_filespec_len == msg_file_spec[0]
                  && NlsULCmpB(NULL, canonical_filespec, &msg_file_spec[1],
                               canonical_filespec_len, &j) == ercOK
                  && j == 0xFFFF) {
               CloseMsgFile();
               erc = OpenFile(&target.handle, sd_restore.pb, sd_restore.cb,
                              &target.password[1], target.password[0],
                              modeModify);
            }
         }
         if (erc == ercOK){   /* File exists, may we overwrite it? */
            if (vlpb.no_overwrite) {
               log_msg(NLS_DID_NOT_RESTORE, 1, nls_parms, sizeof(nls_parms));
               CloseFile(target.handle);
               target.handle = 0;
               return;
            }
            if (!vlpb.overwrite_OK) {
               log_msg(NLS_FILE_EXISTS, 1, nls_parms, sizeof(nls_parms));
               if (!proceed()) {
                  CloseFile(target.handle);
                  target.handle = 0;
                  return;
               }
               else
                  fOverwrite = TRUE;
            }
            else
               fOverwrite = TRUE;

         } /* M21  */
         else if (erc == ercFileInUse && 
                    !vlpb.no_overwrite) {
			 /*
              * user did not specifiy no overwrite. Try doing remove on close.
              */
             erc = HandleRmvOnClose(&target.handle, sd_restore.pb,
                                    sd_restore.cb, &target.password[1], 
                                    target.password[0], modeModify,
                                    FALSE, NULL, 0, !vlpb.overwrite_OK,
                                    prompt_replace, nls_parms, 
                                    sizeof(nls_parms),  &answer);

             /* Here we prompted the user and he did not want to replace 
              * the file.
              */
             if (!vlpb.overwrite_OK && erc == ercOK && !answer){
                target.handle = 0;
                return;
             }
         }/* M21 */
         if(erc == ercOK){
            nls_parms[3].pb = sd_overwriting->pb;
            nls_parms[3].cb = sd_overwriting->cb;
            erc = ChangeFileLength(target.handle,
                                   archive_sentinel->pages_data * PAGE_SIZE);
            if (     erc != ercOK
                  && GetFileStatus(target.handle, FILE_STATUS_LENGTH,
                                   &file_length, sizeof(file_length)) == ercOK
                  && file_length >= archive_sentinel->pages_data * PAGE_SIZE)
               erc = ercOK;		/* The file is already big enough */
         } else if (erc == ercNoSuchFile) {
            nls_parms[3].pb = NULL;
            nls_parms[3].cb = 0;
            *file_exists = FALSE;
            erc = CreateFile(sd_restore.pb, sd_restore.cb, &target.password[1],
                             target.password[0],
                             archive_sentinel->pages_data * PAGE_SIZE);
            if (     erc == ercAccessDenied
                  && target.password[0] == 0
                  && archive_sentinel->directory_password[0] != 0) {
               memcpy(target.password, archive_sentinel->directory_password,
                      archive_sentinel->directory_password[0] + 1);
               erc = CreateFile(sd_restore.pb, sd_restore.cb,
                                &target.password[1], target.password[0],
                                archive_sentinel->pages_data * PAGE_SIZE);
            }
            if (erc == ercOK)
               erc = OpenFile(&target.handle, sd_restore.pb, sd_restore.cb,
                              &target.password[1], target.password[0],
                              modeModify);
         }
         if (     erc == ercOK
               && (erc = GetFileStatus(target.handle, fhb_code, &fhb,
                                       sizeof(fhb))) == ercBadRequestCode)
            erc = GetFileStatus(target.handle, (fhb_code == FILE_STATUS_FHB)
                                ? FILE_STATUS_FHB_RESERVED : FILE_STATUS_FHB,
                                &fhb, sizeof(fhb));
         if (fhb.fNoSave && fhb.fNoDelete) erc = ercAccessDenied; /* m24 */
         if (erc == ercOK) {
            if (explicit_password)
               memcpy(fhb.password, target.password, target.password[0] + 1);
            else if (   destination_encrypted
                     || ((   archive_sentinel->file_password[0] != 12
                             && (!fOverwrite && 
                                  archive_sentinel->file_password[0] == 0))
                         || (  archive_sentinel->file_password[12]
                             & PASSWORD_ENCRYPTED) == 1))
               memcpy(fhb.password, archive_sentinel->file_password,
                      archive_sentinel->file_password[0] + 1);
            fhb.accessProtection = archive_sentinel->protection;
            if (!file_exists && (GetDateTime(&fhb.creationDT) != ercOK))
               fhb.creationDT = archive_sentinel->creation_date;
            fhb.modificationDT = archive_sentinel->modification_date;
            fhb.accessDT = archive_sentinel->access_date;
            fhb.expirationDT = archive_sentinel->expiration_date;
            fhb.fNoSave = archive_sentinel->control.suppress_backup;
            fhb.fNoDirPrint = archive_sentinel->control.hidden_file;
            fhb.fNoDelete = archive_sentinel->control.prevent_delete;
            fhb.endOfFileLfa = archive_sentinel->eof_lfa;
            fhb.userObjectType = archive_sentinel->object_type;
            memcpy(fhb.rgbDistrix, archive_sentinel->distrix_info,
                   sizeof(fhb.rgbDistrix));
            fhb.fReadOnly = archive_sentinel->control.read_only;
            fhb.bDosMagic = archive_sentinel->ms_dos_magic;
            fhb.bDosAttribute = archive_sentinel->ms_dos_attribute;
            fhb.ObjectType = archive_sentinel->directory_type;
            memcpy(fhb.application, archive_sentinel->application_info,
                   sizeof(fhb.application));
            if ((erc = SetFileStatus(target.handle, fhb_code, &fhb,
                                     sizeof(fhb))) == ercBadRequestCode)
               erc = SetFileStatus(target.handle, (fhb_code == FILE_STATUS_FHB)
                                   ? FILE_STATUS_FHB_RESERVED
                                   : FILE_STATUS_FHB, &fhb, sizeof(fhb));
         }
         if (erc != ercOK) {
            summary.count[UNABLE_TO_RESTORE]++;
            log_msg(NLS_CANT_RESTORE, 1, nls_parms, sizeof(nls_parms));
            CloseFile(target.handle);
            target.handle = 0;
            if (fInstall_Mngr_Case)
               exit_with_msg(erc, 0);				/* M14 */
            return;
         }
         summary.count[RESTORED_OK]++;
         log_msg(NLS_RESTORING, 0xFF01, nls_parms, sizeof(nls_parms));
         return;
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 A little fancy footwork is needed here to provide a long desired enhancement
 to the modification dates of restored CTOS files.  When the first CloseFile
 is performed, the CTOS File System automatically updates the modification
 date with the current date and time (this is because the File System has
 noted that writes have been performed to the DATA portion of the file, not
 just the File Header).  After the subsequent reopening of the file, the
 setting of the correct modification date is safely preserved when the file is
 once again closed.  Note that if the password first used to open or create
 the file fails to grant access for a subsequent OpenFile, a second attempt is
 made with the file password itself.  This is in case the particular
 combinations of protection level, directory password and file password
 require a change in password after the SetFileStatus performed in the
 open_target_file() procedure.  The target handle is zeroed as an excessive
 safety precaution. */

private void close_target_file(Flag file_exists) {

   CloseFile(target.handle);
   if (!file_exists
   && ((OpenFile(&target.handle, &target.filespec[1], target.filespec[0],
                     &target.password[1], target.password[0],
                     modeModify) == ercOK)
         || (   fhb.password[0] != 0
             && OpenFile(&target.handle, &target.filespec[1],
                         target.filespec[0], &fhb.password[1],
                         fhb.password[0], modeModify) == ercOK))) {
      SetFileStatus(target.handle, FILE_STATUS_MODIFICATION_DATE,
                    &fhb.modificationDT, sizeof(fhb.modificationDT));
      CloseFile(target.handle);
   }
   target.handle = 0;
   log_msg(NLS_DONE, 1, NULL, 0);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Each time a quantity of file data is sent by the archive dataset I/O
 co-process, transfer it as expeditiously as possible to the disk medium.
 Keep track of current LFA within the file so we may report errors, etc. */

private void restore_file_data(void *data, unsigned length) {

   unsigned erc, xfer_count;
   Boolean io_error = FALSE, other_error = FALSE;
   sdType nls_parms[] = {(void *) &target.filespec[1], target.filespec[0],
                         (void *) &erc, sizeof(erc),
                         (void *) &target.lfa, sizeof(target.lfa)};

   while (length > 0) {
      erc = Write(target.handle, data, length, target.lfa, &xfer_count);
      offset_of(data) += xfer_count;
      target.lfa += xfer_count;
      length -= xfer_count;
      if (erc == ercIoError)
         do {
            erc = Write(target.handle, data, PAGE_SIZE, target.lfa,
                        &xfer_count);
            if (erc == ercIoError || erc == ercOK) {
               if (erc == ercIoError) {
                  log_msg(NLS_FILE_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
                  io_error = TRUE;
               }
               offset_of(data) += xfer_count;
               target.lfa += xfer_count;
               length -= xfer_count;
            } else {
               other_error = TRUE;
               offset_of(data) += length;
               target.lfa += length;
               length = 0;
            }
         } while (erc == ercIoError && length > 0);
      else if (erc != ercOK) {
         other_error = TRUE;
         offset_of(data) += length;
         target.lfa += length;
         length = 0;
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure takes the 'File list from' specification supplied by the
 user and converts it to a form suitable for use with the WildCardMatch
 operation.  The only significant work is to add {*} and [*] wildcards on
 behalf of the user if the node and/or volume were not specified explicitly.
 There is also a minor local optimization to not repeat these calculations
 unless absolutely necessary. The calculation is however needed when the node
 name is present to allow restore from any combination of local and/or remote
 nodes. */

private void expand_from_spec(unsigned subparam, sdType *sd_from,
                              Boolean expand_node) {

   static char from_spec[MAX_FILE_SPECIFICATION_LENGTH];
   char node[MAX_NODE_LENGTH], volume[MAX_VOLUME_LENGTH],
      directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH];
   static unsigned from_spec_len, prev_subparam = 0xFFFF;
   unsigned node_len, volume_len, directory_len, filename_len;

   if (subparam != prev_subparam || expand_node) {
      RgParam(FILE_LIST_FROM_PARAM, prev_subparam = subparam, sd_from);
      ParseFileSpec(0, sd_from->pb, sd_from->cb, FALSE, node, &node_len,
                    volume, &volume_len, directory, &directory_len,
                    filename, &filename_len, NULL, NULL, FALSE, 0);
      if (!archive.old_format) {
         if (expand_node && node_len == 0) {
            node[0] = '*';
            node_len = 1;
            prev_subparam = 0xFFFF;
         }
         if (volume_len == 0) {
            volume[0] = '*';
            volume_len = 1;
         }
      }
      memset(from_spec, 0, sizeof(from_spec));
      BuildFileSpec(0, from_spec, &from_spec_len, NULL, sizeof(from_spec),
                    FALSE, node, node_len, volume, volume_len, directory,
                    directory_len, filename, filename_len, FALSE, NULL, 0,
                    FALSE, 0);
   }
   sd_from->pb = from_spec;
   sd_from->cb = from_spec_len;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function takes one of the 'File list to' specifications from the command
 from and expands it by adding the current user path information.  It also
 extracts the password, if one was explicitly specified.  Note that parsing
 and building does not need to be performed each and every time, just when a
 different 'File list to' parameter is requested.  This is also a convenient
 place to ascertain whether or not the destination volume is password
 encrypted. */

private void expand_to_spec(unsigned subparam, sdType *sd_to,
                            sdType *sd_password, Boolean *volume_encrypted) {

   static Boolean to_volume_encrypted;
   static char password[MAX_PASSWORD_LENGTH],
      to_spec[MAX_FILE_SPECIFICATION_LENGTH];
   char node[MAX_NODE_LENGTH], volume[MAX_VOLUME_LENGTH],
      directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH];
   static unsigned password_len, prev_subparam = 0xFFFF, to_spec_len;
   unsigned node_len, volume_len, directory_len, filename_len;
   vhb_type vhb;

   if (subparam > vlpb.to_specs - 1)
      subparam = vlpb.to_specs - 1;
   if (subparam != prev_subparam) {
      RgParam(FILE_LIST_TO_PARAM, prev_subparam = subparam, sd_to);
      ParseFileSpec(0, sd_to->pb, sd_to->cb, TRUE, node, &node_len, volume,
                    &volume_len, NULL, NULL, NULL, NULL, NULL, NULL, TRUE, 0);
      memset(to_spec, 0, sizeof(to_spec));
      BuildFileSpec(0, to_spec, &to_spec_len, NULL, sizeof(to_spec),
                    FALSE, node, node_len, volume, volume_len,
                    NULL, 0, NULL, 0, FALSE, NULL, 0, TRUE, 0);
      vhb.magicWd = 0;		/* In case GetVHB fails for some reason... */
      GetVHB(to_spec, to_spec_len, &vhb, sizeof(vhb));
      to_volume_encrypted =    (vhb.magicWd == MAGIC_PC)
                            || (   (vhb.magicWd & (MAGIC_VM | MAGIC_ENCRYPT))
                                == (MAGIC_VM | MAGIC_ENCRYPT));
      ParseFileSpec(0, sd_to->pb, sd_to->cb, FALSE, node, &node_len, volume,
                    &volume_len, directory, &directory_len, filename,
                    &filename_len, password, &password_len, FALSE, 0);
      memset(to_spec, 0, sizeof(to_spec));
      BuildFileSpec(0, to_spec, &to_spec_len, NULL, sizeof(to_spec),
                    (volume_len == 0), node, node_len, volume, volume_len,
                    directory, directory_len, filename, filename_len, FALSE,
                    NULL, 0, FALSE, 0);
   }
   sd_to->pb = to_spec;
   sd_to->cb = to_spec_len;
   *volume_encrypted = to_volume_encrypted;
   sd_password->pb = password;
   sd_password->cb = password_len;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 As its name implies... */

private void summarize_restore(void) {

   sdType nls_parms[] = {(void *) &summary.count[PROCESSED],
                                  sizeof(summary.count[PROCESSED]),
                         (void *) &summary.count[RESTORED_OK],
                                  sizeof(summary.count[RESTORED_OK]),
                         (void *) &summary.count[IO_ERRORS],
                                  sizeof(summary.count[IO_ERRORS]),
                         (void *) &summary.count[0], sizeof(summary.count[0]),
                         (void *) &summary.count[UNABLE_TO_RESTORE],
                                  sizeof(summary.count[UNABLE_TO_RESTORE]),
                         (void *) &summary.count[0], sizeof(summary.count[0])};

   log_msg(NLS_FILES_PROCESSED, 1, nls_parms, sizeof(nls_parms));
   if (!vlpb.list_only)
      log_msg(NLS_FILES_RESTORED, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[IO_ERRORS] > 0)
      log_msg(NLS_FILES_RESTORED_WITH_ERRORS, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[0] > 0)
      log_msg(NLS_FILES_WITHOUT_FHB, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[UNABLE_TO_RESTORE] > 0)
      log_msg(NLS_FILES_NOT_RESTORED, 1, nls_parms, sizeof(nls_parms));
   if (summary.count[0] > 0)
      log_msg(NLS_FILES_UNRECOVERABLE, 1, nls_parms, sizeof(nls_parms));

}